<?php
namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon, Auth, DB, Session, Validator, Hash;
use App\Imports\GuestDataImport;
use App\Models\GuestData, App\Models\UpcomingGuest;
use App\Models\User, App\Models\Booking;
use App\Models\Event, App\Models\Location;
use Illuminate\Support\Facades\Mail;
use App\Helpers;
use Ramsey\Uuid\Uuid;

class GuestDataController extends Controller
{
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls',
        ]);
        $import = new GuestDataImport();
        $import->guest_event_id = $request->guest_event; 
        
        Excel::import($import, $request->file('file'));
        
        return redirect()->back()->with('success', 'Guest data imported successfully.');
    }
	
	public function processGuestData()
    {
        $guestDataRecords = GuestData::where('status', 0)->get();

        foreach ($guestDataRecords as $guestData) {
            $this->processGuestDataRecord($guestData);
        }

        return redirect()->back()->with('success', 'Guest data processing completed.');
    }

    private function processGuestDataRecord($guestData)
    { 
	  if(isset($guestData->id)){
        $user = User::where('email', $guestData->email)->first();
        if ($user) {
            $new_user = false;
            $uuid = $user->uuid;
        }else{
            $new_user = true;
			$uuid = (string) Uuid::uuid4();
            $password = Helpers::generatePassword();
            $user = new User([
                'uuid' => $uuid, 
                'role' => 'guest', 
                'fname' => $guestData->fname,
                'lname' => $guestData->lname,
                'email' => $guestData->email,
                'status' => 1, 
                'password' => Hash::make($password), 
            ]);
            $user->save();
        }
        $event = Event::find($guestData->guest_event_id);
        $upcomingGuest = new UpcomingGuest([
            'location_id' => $event->location_id, 
            'invitation_type' => 'event', 
            'ref_id' => $guestData->guest_event_id,
            'guest_uuid' => $uuid,
            'host_uuid' => $event->user_uuid, 
            'invitation_date' => $event->event_date, 
            'invitation_time' => $event->event_time, 
            'status' => 0, 
        ]);
        $upcomingGuest->save();	
        $guestData->update(['status' => 1]);
		// invitation email
        $data = array();
        $data['email']      = $guestData->email;
        $data['name']       = $guestData->fname.' '.$guestData->lname;
        if($new_user){
            $data['password']   = $password;
        }
        $data['invcontent'] = 'You are invited as a guest for the event on '.$event->event_date.' '.$event->event_time;
        \Mail::send('emails.invitation', $data, function($message) use($data) {
            $message->to($data['email']);
            $message->subject('Invitation for the event');
        });
	  }
      return true;
    }
	
    public function invite_guest(Request $request, $type = null, $id = null)
    {
        $data = [];
        $data['id'] = $id;
        $data['type'] = $type;
        if($id && $type=='booking'){
            $booking = Booking::select('location_id','start_date','start_time')
            ->where('booking_id',$id)->where('created_by',auth()->user()->id)->first();
            if(is_null($booking)){
                abort(404);
            }
            $data['loc']   = $booking->location_id;
            $data['start_date'] = $booking->start_date;
            $data['start_time'] = $booking->start_time;
        }
        $locations = Location::pluck('location','id');
        $data['locations'] = $locations;
        
        return view("frontend.guest.invite")->with($data);
    }

	public function invite_guest_process(Request $request)
    {
	    $validator = Validator::make($request->all(), [
            'fname' => 'required',
            'lname' => 'required',
            'email' => 'required|email'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'error' => $validator->errors()
            ], 422);
        }

        if($request->num!="" && $request->type=="booking"){
            $booking = Booking::select('location_id','start_date','start_time')
            ->where('booking_id',$request->num)->where('created_by',auth()->user()->id)->first();
            $location_id      = $booking->location_id;
            $invitation_date  = $booking->start_date;
            $invitation_time  = $booking->start_time;
            
        }else{
            if($request->location_id=="" || $request->invitation_date=="" || $request->invitation_time==""){
                return response()->json([
                    'status' => false,
                    'error' => 'Required fields are missing.'
                ], 422);
            }
            $location_id     = $request->location_id;
            $invitation_date = $request->invitation_date;
            $invitation_time = $request->invitation_time;
        }
		
        $user = User::where('email', $request->email)->first();
        if ($user) {
            $new_user = false;
            $uuid = $user->uuid;
        }else{
            $new_user = true;
			$uuid = (string) Uuid::uuid4();
            $password = Helpers::generatePassword();
            $user = new User([
                'uuid' => $uuid, 
                'role' => 'guest', 
                'fname' => $request->fname,
                'lname' => $request->lname,
                'email' => $request->email,
                'status' => 1, 
                'password' => Hash::make($password), 
            ]);
            $user->save();
        }
		
        $upcomingGuest = new UpcomingGuest([
            'location_id' => $location_id, 
            'invitation_type' => 'Studio', 
            'ref_id' => 0, 
            'guest_uuid' => $uuid, 
            'host_uuid' => auth()->user()->uuid, 
            'invitation_date' => $invitation_date, 
            'invitation_time' => $invitation_time, 
            'status' => 0, 
        ]);
        $upcomingGuest->save();
        // invitation email
        $data = array();
        $data['email']      = $guestData->email;
        $data['name']       = $guestData->fname.' '.$guestData->lname;
        if($new_user){
            $data['password']   = $password;
        }
        $data['invcontent'] = 'You are invited as a guest for the event on '.$event->event_date.' '.$event->event_time;
        \Mail::send('emails.invitation', $data, function($message) use($data) {
            $message->to($data['email']);
            $message->subject('Invitation for the event');
        });
        
        return redirect('/invited-guest')->with('success', 'Guest invited successfully.');
    }


    public function invitations(Request $request)
    {
        $data = UpcomingGuest::select('upcoming_guests.*', 'users.fname', 'users.lname', 'users.email')
        ->join('users', 'host_uuid', '=', 'users.uuid')->with('location', 'host')
        ->where('guest_uuid', auth()->user()->uuid)->paginate(10);
        
        return view("frontend.guest.invitation")->with(['data' => $data]);
    }

    public function invited_guest(Request $request)
    {
        $data = UpcomingGuest::select('upcoming_guests.*', 'users.fname', 'users.lname', 'users.email')
        ->join('users', 'guest_uuid', '=', 'users.uuid')->with('location', 'host')
        ->where('host_uuid', auth()->user()->uuid)->paginate(10);
        
        return view("frontend.guest.invited")->with(['data' => $data]);
    }

    public function guest_data(Request $request)
    {
        try{
            $data = UpcomingGuest::select('upcoming_guests.*', 'users.fname', 'users.lname', 'users.email')
            ->join('users', 'guest_uuid', '=', 'users.uuid')->with('location', 'host')
            ->where('upcoming_guests.id', $request->upcoming_id)->first();
        
            $content = view("frontend.guest.guest-data")->with(['data' => $data])->render();
            return response()->json([
                'status' => true,
                'content' => $content
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
    
}