<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon, Auth, DB, Session, Validator, Hash;
use App\Models\User, App\Models\SubscriptionPlan;
use App\Models\Location, App\Models\Role, App\Models\RolePages;
use App\Models\Plan, App\Models\Point;
use App\Models\Currency, App\Models\UserPermission;
use Ramsey\Uuid\Uuid;
use App\Helpers;

class UserController extends Controller
{
    public function signup()
    {
        return view('auth.signup');
    }

    public function signup_validate(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'fname' => 'required',
            'lname' => 'required',
            'email' => 'required',
            'mobile_number' => 'required|unique:users,mobile_number,NULL,id',
            'password' => 'required|min:6|same:retype_password',
            'retype_password' => 'required|min:6',
        ],[
            'fname.required' => 'First name is required.',
            'lname.required' => 'Last name is required.',
            'mobile_number.required' => 'Mobile number is required.',
            'mobile_number.unique' => 'Mobile number already exist.',
            'password.required' => 'Password is required.',
            'retype_password.required' => 'Retype password is required.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        
        try{

            $uuid = (string) Uuid::uuid4();
            $password = Hash::make($request->password);
            $token = random_int(100000, 999999);
            $signup = [];
            $signup['uuid'] = $uuid;
            $signup['role'] = 'guest';
            $signup['fname'] = $request->fname;
            $signup['lname'] = $request->lname;
            $signup['email'] = $request->email;
            $signup['mobile_number'] = $request->mobile_number;
            $signup['password'] = $password;
            $signup['token'] = $token;
            $request->session()->put('signup_form',$signup);
            // send verification email
            $data = array();
            $data['email']   = $request->email;
            $data['name']    = $request->fname.' '.$request->lname;
            $data['token']   = $token;
            \Mail::send('emails.verification', $data, function($message) use($data) {
                $message->to($data['email']);
                $message->subject('Verify your email');
            });

            $res = Helpers::send_otp($request->mobile_number);
            return $res;
            
            return response()->json([
                'status' => true,
                'message' => 'success',
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function otp_verification()
    {
        if(Session::has('signup_form')){
            $signup_data = Session::get('signup_form');
            $mobile = substr($signup_data['mobile_number'], -4);
            $email  = substr($signup_data['email'], -8);
            return view('auth.otp-verification')->with(['mobile'=>$mobile, 'amob'=>$signup_data['mobile_number'], 'email'=>$email, 'aemail'=>$signup_data['email']]);
        }else{
            abort(403);
        }
    }
    
    public function signup_save(Request $request)
    {
        try{
            $otp_email = $request->eot1.$request->eot2.$request->eot3.$request->eot4.$request->eot5.$request->eot6;
            $otp_number = $request->first.$request->second.$request->third.$request->fourth.$request->fifth.$request->sixth;
            $signup_data = $request->session()->get('signup_form');
            if($otp_email!=$signup_data['token']){
                return response()->json([
                    'status' => false,
                    'error'=>'Email verification code is not valid.'
                ], 422);
            }
            $isVerified = Helpers::verify_otp($signup_data['mobile_number'], $otp_number);
            if ($isVerified) {
                $user = new User;
                $user->role  = 'guest';
                $user->uuid  = $signup_data['uuid'];
                $user->fname = $signup_data['fname'];
                $user->lname = $signup_data['lname'];
                $user->email = $signup_data['email'];
                $user->mobile_number = $signup_data['mobile_number'];
                $user->password = $signup_data['password'];
                $user->email_verified_at = date('Y-m-d H:i:s');
                $user->token = $signup_data['token'];
                if($user->save()){
                    Auth::loginUsingId($user->id);
                    $request->session()->forget('signup_form');
                    $request->session()->put('signup_flow',1);
                    return response()->json([
                        'status' => true,
                    ], 200);
                }else{
                    return response()->json([
                        'status' => false,
                        'error'=>'Technical error.'
                    ], 422);
                }
            }else{
                return response()->json([
                    'status' => false,
                    'error'=>'OTP is not valid.'
                ], 422);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function profile_picture()
    {
        return view('auth.profile-picture')->with(['sess_uuid'=>auth()->user()->uuid]);
    }

    public  function save_profile_picture(Request $request)
    {
        //$uuid = $request->uuid;
        $uuid = auth()->user()->uuid;
        $user = User::where('uuid',$uuid)->first();
        $storage_path = storage_path('app/public/profile-pic/');
        @unlink($storage_path.$user->profile_pic);
        if($request->has('type') && $request->type=='cam'){
            $img = $request->image;
            $storage_path = storage_path('app/public/profile-pic/');
            
            $image_parts = explode(";base64,", $img);
            $image_type_aux = explode("image/", $image_parts[0]);
            $image_type = $image_type_aux[1];
            
            $image_base64 = base64_decode($image_parts[1]);
            $fileName = time() . '.png';
            $file = $storage_path . $fileName;
            file_put_contents($file, $image_base64);
            
            User::where('uuid',$uuid)->update(['profile_pic'=>$fileName]);
            return response()->json(['status'=>true,'success'=>$fileName]);
        }else{
            $storage_path = storage_path('app/public/profile-pic/');
            $image = $request->file('file');
            $imageName = time().'.'.$image->extension();
            $image->move($storage_path, $imageName);
            User::where('uuid',$uuid)->update(['profile_pic'=>$imageName]);
            return response()->json(['status'=>true,'success'=>$imageName]);
        }
        
    }

    public function remove_profile_picture(Request $request)
    {
        if($request->has('uuid')){
            $user = User::where('uuid',$request->uuid)->first();
            $storage_path = storage_path('app/public/profile-pic/');
            @unlink($storage_path.$user->profile_pic);

            User::where('uuid',$request->uuid)->update(['profile_pic'=>null]);
            return response()->json(['status'=>true],200);
        }else{
            return response()->json(['status'=>false,'error'=>'Unable to delete profile picture.'],422);    
        }
    }

    public function additional_information()
    {
        $location = Location::select('id','location')->get();
        return view('auth.additional-information')->with(['location'=>$location]);
    }

    public function additional_info_save(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'home_city' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $creator_interest = "";
            if($request->has('creator_interest') && count($request->creator_interest)>0){
                $creator_interest = implode(',',$request->creator_interest);
            }
            $user = User::find(auth()->user()->id);
            $user->creator_interest = $creator_interest;
            $user->home_city        = $request->home_city;
            $user->profile_link     = $request->profile_link;
            $user->instagram_link   = $request->instagram_link;
            $user->facebook_link    = $request->facebook_link;
            $user->twitter_link     = $request->twitter_link;
            $user->sm_link          = $request->sm_link;
            $user->referred_by      = $request->referred_by;
            if($user->save()){
                return response()->json([
                    'status' => true,
                ], 200);
            }else{
                return response()->json([
                    'status' => false,
                    'error'=>'Technical error.'
                ], 422);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function consent()
    {
        return view('auth.consent');
    }

    public function congratulations()
    {
        return view('auth.congratulations');
    }
 
    public function login()
    {
        // $client = new client();
        // $rate = Currency::rates()
        //     ->latest()
        //     ->base('USD')// sets base currency default is 'EUR'
        //     ->get();
        // //$currencyFormat = Helpers::currecnyConversion('EUR', 50);
        // dd($rate);
        return view('auth.login');
    }

    public function login_save(Request $request)
    { 
        $validator = Validator::make($request->all(),[ 
            'mobile_number' => 'required',
            'password' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $credentials = $request->only('mobile_number', 'password');
            if (Auth::attempt($credentials)) {
                $user = auth()->user();
                $this->__login_data($user);
                return response()->json([
                    'status' => true,
                ], 200);
            }
            return response()->json([
                'status' => false,
                'error'=>'These credentials do not match our records.'
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function login_otp()
    {
        return view('auth.login-otp');
    }
    
    public function resetPassword()
    {
        return view('auth.reset-password');
    }

    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'email' => 'required|email',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        try{
            $token = Str::random(64);
  
            DB::table('password_resets')->insert([
                'email' => $request->email, 
                'token' => $token, 
                'created_at' => Carbon::now()
              ]);
    
            \Mail::send('emails.forget-Password', ['token' => $token], function($message) use($request){
                $message->to($request->email);
                $message->subject('Reset Password');
            });
            Session::flash('success','Password has been sent to your email address.');
    
            
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }   
    }

    public function showUser(Request $request)
    {
        try {
            $query = User::with('created_by','role')->orderBy("users.id", "DESC");
            if ($request->ajax()) {
                $data = $query->paginate(5);
                return view("user.addUser_ajax")->with(["data" => $data]);
            }

            $data = $query->paginate(5);
            $userType = Role::where('status',1)->pluck('role_name','role_id');
            return view("user.addUser")->with([
                "userType" => $userType,
                "data" => $data,
            ]);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function delete_user(Request $request)
    {
        try {
            User::find($request->userid)->delete();
            return response()->json([
                "status" => true,
            ],200);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function logout()
    {
        Session::flush();        
        Auth::logout();
        return redirect('/');
    }

    public function send_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'type' => 'required',
            'otpfield' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        
        try{
            if($request->type=='email' && strpos($request->otpfield, "@") !== false){
                if(isset(auth()->user()->id)){
                    $name = auth()->user()->fname.' '.auth()->user()->lname;
                    $token = auth()->user()->token;
                }else{
                    $user = User::where('email',$request->otpfield)->first();
                    if(is_null($user)){
                        return response()->json([
                            "status" => false,
                            "error" => "Email address is not registered with us."
                        ],422);    
                    }else{
                      $name = $user->fname.' '.$user->lname;
                      $token = $user->token;
                    }
                }
                $data = array();
                $data['email']   = $request->otpfield;
                $data['name']    = $name;
                $data['token']   = str_split($token);
                \Mail::send('emails.verification', $data, function($message) use($data) {
                    $message->to($data['email']);
                    $message->subject('Verify your email');
                });
                return response()->json([
                    "status" => true,
                    "message" => "Verification code has been sent to your email address."
                ],200);
            }else if($request->type=='mobile'){
                if(!isset(auth()->user()->id)){
                    $user = User::where('mobile_number',$request->otpfield)->first();
                    if(is_null($user)){
                        return response()->json([
                            "status" => false,
                            "error" => "Mobile number is not registered with us."
                        ],422);    
                    }
                }
                $res = Helpers::send_otp($request->otpfield);
                return $res;
            }
            
            return response()->json([
                "status" => false,
                "message" => "Invalid form data received."
            ],200);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }catch (\Swift_TransportException $m){
            return response()->json([
                "status" => false,
                "error" => $m->getMessage(),
            ],422);
        }
    }

    public function verify_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'type' => 'required',
            'mobile_number' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $otp_number = $request->first.$request->second.$request->third.$request->fourth.$request->fifth.$request->sixth;
            $isVerified = Helpers::verify_otp($request->mobile_number, $otp_number);
            if ($isVerified) {
                $user = User::where('mobile_number',$request->mobile_number)->first();
                if($user){
                    if($request->type=='login'){
                        Auth::loginUsingId($user->id);
                        $this->__login_data($user);
                    }
                    return response()->json([
                        "status" => true
                    ],200);
                }else{
                    return response()->json([
                        'status' => false,
                        'error' => 'Mobile number is not registered with us.',
                    ], 422);
                }
            }else{
                return response()->json([
                    "status" => false,
                    "error" => "OTP is not valid."
                ],422);
            }
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function userinfo(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        try{
            $data = User::with('user_role')->with('user_plan')->with('user_location')->where('uuid',$request->uuid)->first();
            if($request->type=='edit'){
                $plans = Plan::select('name','slug')->get();
                $location = Location::pluck('location','id');
                $user_permissions = [];
                $user_roles = UserPermission::Join('location','location_id','location.id')
                ->select('location_id','location',DB::raw('GROUP_CONCAT(role_id , ", ") AS roleids'))->Where('user_uuid',$request->uuid)->groupBy('location_id')->get();
                foreach($user_roles as $role){
                    $locroles = explode(',',$role->roleids);
                    $uroles = Role::whereIn('role_id',$locroles)->pluck('role_name')->toArray();
                    $permissions['location'] = $role->location;
                    $permissions['roles'] = $uroles;
                    $user_permissions[] = $permissions;
                }
                $current_balance = 0;
                $balance_data = Point::select('balance')->where('receivedby_uuid',$request->uuid)->orderBy('id','desc')->first();
                if($balance_data){
                    $current_balance = $balance_data->balance;
                }
                $content = view("global.useredit-card")->with(['data'=>$data, 'location'=>$location, 'plans'=>$plans, 'user_permissions'=>$user_permissions, 'current_balance'=>$current_balance])->render();
            }else{
                $content = view("global.userinfo-card")->with(['data'=>$data])->render();
            }
            return response()->json([
                'status' => true,
                'content' => $content,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function userroles(Request $request)
    {
        $uroles = [];
        $options = '';
        $uuid = $request->uuid;
        $location = $request->location;
        $roles = Role::select('role_id','role_name','role_slug')->whereIn('type',['access','resource'])->get();
        $user_roles = UserPermission::select('role_id')->Where('user_uuid',$uuid)->where('location_id',$location)->get();
        foreach($user_roles as $ur){
            $uroles[] = $ur->role_id;
        }
        foreach($roles as $role){
            if(in_array($role->role_id,$uroles)){
                $options .= '<option selected="selected" value="'.$role->role_id.'">'.$role->role_name.'</option>';
            }else{
                $options .= '<option value="'.$role->role_id.'">'.$role->role_name.'</option>';
            }
            
        }
        return response()->json([
            'status' => true,
            'content' => $options,
        ], 200);
    }

    public function add_permissions(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        try{
            UserPermission::where('user_uuid',$request->user_uuid)->where('location_id',$request->location)->delete();
            foreach($request->roles as $role){
                $up = new UserPermission;
                $up->user_uuid = $request->user_uuid;
                $up->location_id = $request->location;
                $up->role_id = $role;
                $up->pages = config('app.page_access')[$role];
                $up->save();
            }
            return response()->json([
                'status' => true,
                'message' => 'Permissions updated for this location.',
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function transfer_points(Request $request)
    {
        try{
            $users = User::select('uuid', 'fname', 'lname')->where('uuid','!=',auth()->user()->uuid)
            ->whereNotIn('role',['super-admin','admin'])->where('status',1)->get();
            $content = view("user.transfer-points")->with(['users'=>$users])->render();

            return response()->json([
                'status' => true,
                'content' => $content
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function add_points(Request $request)
    {
        try{
            //receivedby entry
            $rec_current_balance = 0;
            $rec_balance_data = Point::select('balance')->where('receivedby_uuid',$request->user_uuid)->orderBy('id','desc')->first();
            if($rec_balance_data){
                $rec_current_balance = $rec_balance_data->balance;
            }
            $point = new Point;
            $point->action = 'POINTS_RECEIVED';
            $point->receivedby_uuid = $request->user_uuid;
            $point->sentby_uuid = auth()->user()->uuid;
            $point->credit = $request->points;
            $point->balance = ($rec_current_balance + $request->points);
            $point->save();
            //sentby entry
            $sent_current_balance = 0;
            $sent_balance_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)->orderBy('id','desc')->first();
            if($sent_balance_data){
                $sent_current_balance = $sent_balance_data->balance;
            }
            $rpoint = new Point;
            $rpoint->action = 'POINTS_SENT';
            $rpoint->receivedby_uuid = auth()->user()->uuid;
            $rpoint->sentby_uuid = $request->user_uuid;
            $rpoint->debit = $request->points;
            $rpoint->balance = ($sent_current_balance - $request->points);
            $rpoint->save();

            return response()->json([
                'status' => true,
                'message' => 'Points has been transfered successfully.',
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
    
    public function user_app_action(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        try{
            $data = User::where('uuid',$request->uuid)->update(['status'=>$request->action]);
            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function update_user(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }

        $validator = Validator::make($request->all(),[ 
            'user_uuid' => 'required',
            'location' => 'required',
            'role' => 'required',
            'membership' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $user = User::where('uuid',$request->user_uuid)->first();
            if($user){
                $user->role = $request->role;
                $user->home_city = $request->location;
                $user->membership = $request->membership;
                //$user->points = $request->points;
                $user->save();
                if($request->points!=""){
                    //receivedby entry
                    $rec_current_balance = 0;
                    $rec_balance_data = Point::select('balance')->where('receivedby_uuid',$request->user_uuid)->orderBy('id','desc')->first();
                    if($rec_balance_data){
                        $rec_current_balance = $rec_balance_data->balance;
                    }
                    $point = new Point;
                    $point->action = 'POINTS_RECEIVED';
                    $point->receivedby_uuid = $request->user_uuid;
                    $point->sentby_uuid = auth()->user()->uuid;
                    $point->credit = $request->points;
                    $point->balance = ($rec_current_balance + $request->points);
                    $point->save();
                    //sentby entry
                    $sent_current_balance = 0;
                    $sent_balance_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)->orderBy('id','desc')->first();
                    if($sent_balance_data){
                        $sent_current_balance = $sent_balance_data->balance;
                    }
                    $rpoint = new Point;
                    $rpoint->action = 'POINTS_SENT';
                    $rpoint->receivedby_uuid = auth()->user()->uuid;
                    $rpoint->sentby_uuid = $request->user_uuid;
                    $rpoint->debit = $request->points;
                    $rpoint->balance = ($sent_current_balance - $request->points);
                    $rpoint->save();
                }

                return response()->json([
                    'status' => true,
                    'message' => 'User updated successfully.',
                ], 200);
            }

            return response()->json([
                'status' => false,
                'error' => 'Invalid form data received.',
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function set_currency(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'currency' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            session()->put('user_currency',$request->currency);
            if(isset(auth()->user()->uuid)){
                User::where('uuid',auth()->user()->uuid)->update(['home_currency'=>$request->currency]);
            }

            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function __login_data($user)
    {
        // setup page access for middleware
        $user_locations = [];
        $page_access = [];
        $user_access = UserPermission::select('location_id', DB::raw('GROUP_CONCAT(pages) AS pages'))->Where('user_uuid',$user->uuid)->groupBy('location_id')->get();
        if($user_access){
            foreach($user_access as $access){
                $pages = [];
                $dup_pages = explode(',',$access->pages);
                foreach($dup_pages as $page){
                    if(!in_array($page,$pages)){
                        $pages[] = $page;
                    }
                }
                $page_access[$access->location_id] = implode(',',$pages);
                $user_locations[] = $access->location_id;
            }
        }
        session()->put('page_access',$page_access);
        // setup currency
        session()->put('user_currency',$user->home_currency);
        $currency = Currency::pluck('rate','currency')->toArray();
        session()->put('currency',$currency);
        // setup location
        if($user->role=='super-admin' || $user->role=='admin'){
            $locations = Location::get();
            Session::put('user_locations',$locations);
        }else{
            $locations = Location::whereIn('id',$user_locations)->get();
            Session::put('user_locations',$locations);
        }
    }
}
