<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Location;
use App\Models\Language;
use App\Models\Point;
use App\Models\Plan;
use App\Helpers;
use Carbon\Carbon, Validator;
use Laravel\Cashier\Cashier;
class GlobalController extends Controller
{
    private $paginate = 20;

    public function applications(Request $request)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }
        
        $query = User::with('user_role')->with('user_location')->where('uuid','!=','06238d85-b98e-4ee2-8944-17cb88989117');
        if($request->has('name') && $request->name!=""){
            $query->whereRaw('concat(fname," ",lname) like "'.$request->name.'%"');
        }
        $data = $query->OrderBy('created_at', 'desc')->paginate($this->paginate);
        
        if ($request->is('backend/manage-users*')){
            return view("global.manage-users")->with(['data'=>$data]);
        }else{
            return view("global.applications")->with(['data'=>$data]);
        }
    }

    public function user_management(Request $request, $uuid)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }
        $current_balance = 0;
        $guests = config('app.guests');
        $points_action = config('app.points_action');
        $last_month_booking_cost = Point::where('receivedby_uuid',$uuid)->where('action','BOOK_EVENT')->whereBetween('created_at',[Carbon::now()->subMonth()->startOfMonth(), Carbon::now()->subMonth()->endOfMonth()])->sum('debit');
        $current_month_booking_cost = Point::where('receivedby_uuid',$uuid)->where('action','BOOK_EVENT')->whereBetween('created_at',[Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->sum('debit');
        $balance_data = Point::select('balance')->where('receivedby_uuid',$uuid)->orderBy('id','desc')->first();
        if($balance_data){
            $current_balance = $balance_data->balance;
        }
        $data = User::with('user_role')->with('user_plan')->where('uuid',$uuid)->first();
        $points = Point::with('sentby','plan')->where('receivedby_uuid',$uuid)->whereIn('action',['MEMBERSHIP_CREDIT', 'POINTS_PURCHASED', 'POINTS_RECEIVED', 'POINTS_SENT', 'BOOK_EVENT'])->get();
        //dd($points);
        $plan = Plan::where('slug',$data->membership)->first();
        
       return view("global.user-management")->with(['data'=>$data, 'points'=>$points,  'points_action' => $points_action, 'plan' => $plan, 'guests'=>$guests, 'last_month_booking_cost' => $last_month_booking_cost, 'current_month_booking_cost' => $current_month_booking_cost, 'current_balance' => $current_balance]);
    }

    public function billing(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        $points_action = config('app.points_action');
        $data = Point::with('receivedby','plan')->whereIn('action',['MEMBERSHIP_CREDIT', 'POINTS_PURCHASED'])->paginate($this->paginate);
        
        return view("global.billing")->with(['data'=>$data, 'points_action'=>$points_action]);
    }

    public function locations(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        
        $query = Location::orderBy('location','asc');
        if($request->has('name') && $request->name!=""){
            $query->where('location', 'like', $request->name.'%');
        }
        $data = $query->paginate($this->paginate);
        return view("global.locations")->with(['data'=>$data]);
    }

    public function location_detail(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }

        try{
            $data = null;
            if($request->has('lid') && $request->lid!=""){
                $data = Location::find($request->lid);
            }
            $content = view("global.location-card")->with(['data'=>$data])->render();    
            return response()->json([
                'status' => true,
                'content' => $content,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function save_location(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }
        $locationId = null;
        if($request->has("locationid") && $request->locationid != ""){
            $locationId = $request->locationid;
            $fields['location'] = array('required','unique:location,location,'.$locationId.',id');
        }else{
            $fields['location'] = array('required','unique:location,location,null,id');
        }
        $validator = Validator::make($request->all(),$fields);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            if($locationId){
                $data = Location::find($locationId);
                $msg = 'Location updated successfully.';
            }else{
                $data = new Location;
                $msg = 'Location created successfully.';
            }
            $data->location = $request->location;
            $data->save();
            return response()->json([
                'status' => true,
                'message' => $msg,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function remove_action(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }

        try{
            $isDeleted = false;
            if($request->type=='location' && $request->did!=''){
                $data = User::where('home_city',$request->did)->first();
                if($data){
                    return response()->json([
                        'status' => false,
                        'error' => "Location already used by some users, so you can't delete it.",
                    ], 422);
                }
                Location::where('location',$request->did)->delete();
                $isDeleted = true;
            }
            if($isDeleted){
                return response()->json([
                    'status' => true,
                    'message' => 'Record deleted successfully.',
                ], 200);
            }

            return response()->json([
                'status' => false,
                'error' => 'Invalid form data received.',
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
    
    public function save_profile(Request $request)
    {
        if(auth()->user()->role!='super-admin'){
            abort(403);
        }

        $validator = Validator::make($request->all(),[ 
            'user_uuid' => 'required',
            'fname' => 'required',
            'lname' => 'required',
            'phone_number' => 'required',
            'description' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $user = User::where('uuid',$request->user_uuid)->first();
            if($user){
                $user->fname = $request->fname;
                $user->lname = $request->lname;
                $user->phone_number = $request->phone_number;
                $user->description = $request->description;
                //$user->points = $request->points;
                $user->save();
                return response()->json([
                    'status' => true,
                    'message' => 'User updated successfully.',
                ], 200);
            }

            return response()->json([
                'status' => false,
                'error' => 'Invalid form data received.',
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function account_setting(Request $request)
    {
        $uuid = auth()->user()->uuid;
        $current_balance = 0;
        $guests = config('app.guests');
        $points_action = config('app.points_action');
        $last_month_booking_cost = Point::where('receivedby_uuid',$uuid)->where('action','BOOK_EVENT')->whereBetween('created_at',[Carbon::now()->subMonth()->startOfMonth(), Carbon::now()->subMonth()->endOfMonth()])->sum('debit');
        $current_month_booking_cost = Point::where('receivedby_uuid',$uuid)->where('action','BOOK_EVENT')->whereBetween('created_at',[Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->sum('debit');
        $balance_data = Point::select('balance')->where('receivedby_uuid',$uuid)->orderBy('id','desc')->first();
        //dd($balance_data);
        if($balance_data){
            $current_balance = $balance_data->balance;
        }
        $data = User::with('user_role')->with('user_plan')->where('uuid',$uuid)->first();
        $points = Point::with('sentby','plan')->where('receivedby_uuid',$uuid)->whereIn('action',['MEMBERSHIP_CREDIT', 'POINTS_PURCHASED', 'POINTS_RECEIVED', 'POINTS_SENT', 'BOOK_EVENT'])->get();
        //dd($points);
        $plan = Plan::where('slug',$data->membership)->first();
        
        $next_billing_date = '';
        $mysubscription = auth()->user()->subscriptions()->first();
        if($mysubscription){
            $current_period_end_timestamp = auth()->user()->subscription($mysubscription->name)->asStripeSubscription()->current_period_end;
            $next_billing_date = date('d M Y',$current_period_end_timestamp);
        }
        return view("user.account-setting")->with(['data'=>$data, 'points'=>$points,  'points_action' => $points_action, 'plan' => $plan, 'guests'=>$guests, 'last_month_booking_cost' => $last_month_booking_cost, 'current_month_booking_cost' => $current_month_booking_cost, 'current_balance' => $current_balance, 'next_billing_date'=>$next_billing_date]);
    }

    public function account_setting_save(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'fname' => 'required',
            'lname' => 'required',
            'description' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $user = User::where('uuid',auth()->user()->uuid)->first();
            if($user){
                $user->fname = $request->fname;
                $user->lname = $request->lname;
                $user->description = $request->description;
                $user->save();
                return response()->json([
                    'status' => true,
                    'message' => 'Profile updated successfully.',
                ], 200);
            }

            return response()->json([
                'status' => false,
                'error' => 'Invalid form data received.',
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function otp_verification_profile(Request $request)
    {
        $id = auth()->user()->id;
        $validator = Validator::make($request->all(),[ 
            'email' => 'required|unique:users,email,'.$id.',id',
            'phone_number' => 'required|min:12|unique:users,mobile_number,'.$id.',id',
        ],[
            'email.required' => 'Email is required.',
            'email.unique' => 'Email already exist.',
            'phone_number.required' => 'Mobile number is required.',
            'phone_number.unique' => 'Mobile number already exist.',
            'phone_number.min' => 'Mobile number is not valid.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $data = [];
            $user = User::select('fname', 'lname','mobile_number','email')->where('id',$id)
                    ->first();
            if($user){
                if($user->email!=$request->email){
                    $token = random_int(100000, 999999);
                    $user->token = $token;
                    $user->save();
                    $data['aemail'] = $request->email;
                    $data['email']  = substr($request->email, -8);
                    // send verification email
                    $data = array();
                    $data['email']   = $request->email;
                    $data['name']    = $user->fname.' '.$user->lname;
                    $data['token']   = str_split($token);
                    \Mail::send('emails.verification', $data, function($message) use($data) {
                        $message->to($data['email']);
                        $message->subject('Verify your email');
                    });
                }
                if($user->mobile_number!=$request->phone_number){
                    $data['amob'] = $request->phone_number;
                    $data['mobile'] = substr($request->phone_number, -4);
                    $res = Helpers::send_otp($request->phone_number);
                }
                $content = '';
                if(isset($data['mobile']) || isset($data['email'])){
                    $content = view('user.otp-verification-profile')->with($data)->render();
                }
                return response()->json([
                    'status' => true,
                    'content' => $content
                ], 200);
            }else{
                return response()->json([
                    'status' => false,
                    'error' => 'Invalid form data received.'
                ], 422);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
    
    public function verify_otp_profile(Request $request)
    {
        try{
            if(isset($request->eot1)){
                $otp_email = $request->eot1.$request->eot2.$request->eot3.$request->eot4.$request->eot5.$request->eot6;
                $user = User::where('id',auth()->user()->id)->first();
                if($user->token == $otp_email){
                    User::where('id',auth()->user()->id)->update(['email',$request->email]);
                }return response()->json([
                    'status' => false,
                    'error'=>'OTP is not valid for email.'
                ], 422);
            }
            if(isset($request->first)){
                $otp_number = $request->first.$request->second.$request->third.$request->fourth.$request->fifth.$request->sixth;
                $isVerified = Helpers::verify_otp($request->mobile, $otp_number);
                if ($isVerified) {
                    User::where('id',auth()->user()->id)->update(['mobile_number',$request->mobile]);
                }else{
                    return response()->json([
                        'status' => false,
                        'error'=>'OTP is not valid for mobile.'
                    ], 422);
                }
            }
            return response()->json([
                'status' => true,
                'error'=>'OTP verified successfully.'
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

}
