<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User, App\Models\Event;
use Illuminate\Support\Facades\Hash;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Helpers;
use Illuminate\Support\Facades\Validator;
use App\Models\Location, App\Models\LandingPage;
use App\Models\Studio, App\Models\Resources;
use Illuminate\Support\Facades\Storage;

/**
 * @OA\Info(
 *     title="Nomadics API",
 *     version="1.0.0",
 *     description="APIs for frontend development"
 * )
*/

class AuthController extends Controller
{
    /**
     * @OA\Post(
     *     path="/api/login",
     *     summary="User Login",
     *     tags={"Authentication"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"email", "password"},
     *             @OA\Property(property="email", type="string", format="email", example="user@example.com"),
     *             @OA\Property(property="password", type="string", format="password", example="password123")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Successful login",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(property="token", type="string", example="jwt_token_here"),
     *             @OA\Property(
     *                 property="user",
     *                 type="object",
     *                 @OA\Property(property="uuid", type="string", example="123e4567-e89b-12d3-a456-426614174000"),
     *                 @OA\Property(property="fname", type="string", example="John"),
     *                 @OA\Property(property="lname", type="string", example="Doe"),
     *                 @OA\Property(property="email", type="string", example="user@example.com"),
     *                 @OA\Property(property="role", type="string", example="admin")
     *             ),
     *             @OA\Property(property="expires_in", type="integer", example=3600)
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Invalid credentials",
     *         @OA\JsonContent(
     *             @OA\Property(property="error", type="string", example="Invalid credentials")
     *         )
     *     )
     * )
     */
    public function login(Request $request)
    {
        $credentials = $request->only(['email', 'password']);

        if (!$token = JWTAuth::attempt($credentials)) {
            return response()->json(['error' => 'Invalid credentials'], 401);
        }

        $user = JWTAuth::user();

        return response()->json([
            'status' => true,
            'token' => $token,
            'user' => [
                'uuid' => $user->uuid,
                'fname' => $user->fname,
                'lname' => $user->lname,
                'email' => $user->email,
                'role' => $user->role,
            ],
            'expires_in' => JWTAuth::factory()->getTTL() * 60,
        ]);
    }

    /**
     * @OA\Post(
     *     path="/api/send-otp",
     *     summary="Send OTP to mobile number",
     *     tags={"Authentication"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"mobile_number"},
     *             @OA\Property(property="mobile_number", type="string", example="9876543210")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="OTP sent successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="OTP sent successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error or server error",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="The mobile number field is required.")
     *         )
     *     )
     * )
     */
    public function send_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'mobile_number' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        
        try{
            $res = Helpers::send_otp($request->mobile_number);
            
            return $res;
            
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }catch (\Swift_TransportException $m){
            return response()->json([
                "status" => false,
                "error" => $m->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/verify-otp",
     *     summary="Verify OTP and login user",
     *     tags={"Authentication"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"request_type", "otp_number", "mobile_number"},
     *             @OA\Property(property="request_type", type="string", example="login"),
     *             @OA\Property(property="otp_number", type="string", example="123456"),
     *             @OA\Property(property="mobile_number", type="string", example="9876543210")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="OTP verified and token generated",
     *         @OA\JsonContent(
     *             @OA\Property(property="token", type="string", example="jwt_token_here"),
     *             @OA\Property(
     *                 property="user",
     *                 type="object",
     *                 @OA\Property(property="uuid", type="string", example="123e4567-e89b-12d3-a456-426614174000"),
     *                 @OA\Property(property="fname", type="string", example="John"),
     *                 @OA\Property(property="lname", type="string", example="Doe"),
     *                 @OA\Property(property="email", type="string", example="user@example.com"),
     *                 @OA\Property(property="role", type="string", example="user")
     *             ),
     *             @OA\Property(property="expires_in", type="integer", example=3600)
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="OTP verification failed or user not found",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Verification code is not valid.")
     *         )
     *     )
     * )
     */
    public function verify_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'request_type' => 'required',
            'otp_number' => 'required',
            'mobile_number' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $otp_number = $request->otp_number;
            $isVerified = Helpers::verify_otp($request->mobile_number, $otp_number);
            if ($isVerified) {
                if($request->request_type=='login'){
                    $user = User::where('mobile_number',$request->mobile_number)->first();
                    if($user){
                        $token = JWTAuth::fromUser($user);
                        return response()->json([
                            'token' => $token,
                            'user' => [
                                'uuid' => $user->uuid,
                                'fname' => $user->fname,
                                'lname' => $user->lname,
                                'email' => $user->email,
                                'role' => $user->role,
                            ],
                            'expires_in' => JWTAuth::factory()->getTTL() * 60,
                        ]);
                    }else{
                        return response()->json([
                            'status' => false,
                            'error' => 'Unable to find user details.',
                        ], 422);
                    }
                }else{
                    return response()->json([
                        'status' => false,
                        'error' => 'Invalid form data received.',
                    ], 422);
                }
            }else{
                return response()->json([
                    "status" => false,
                    "error" => "Verification code is not valid."
                ],422);
            }
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Post(
     *     path="/api/logout",
     *     summary="Logout user and invalidate JWT token",
     *     tags={"Authentication"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Response(
     *         response=200,
     *         description="Successfully logged out",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(property="message", type="string", example="Successfully logged out")
     *         )
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Failed to logout, token invalid or other error",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Failed to logout, token invalid.")
     *         )
     *     )
     * )
     */
    public function logout(Request $request)
    {
        try {
            JWTAuth::invalidate(JWTAuth::getToken());

            return response()->json([
                'status' => true,
                'message' => 'Successfully logged out'
            ], 200);
        } catch (\Tymon\JWTAuth\Exceptions\JWTException $e) {
            return response()->json([
                'status' => false,
                'error' => 'Failed to logout, token invalid.'
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/locations",
     *     summary="Get list of available locations with banner and description",
     *     tags={"Locations"},
     *     security={{"bearerAuth":{}}},
     * 
     *     @OA\Response(
     *         response=200,
     *         description="List of locations retrieved successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="location", type="string", example="Mumbai"),
     *                     @OA\Property(property="type", type="string", example="City"),
     *                     @OA\Property(property="coming_soon", type="boolean", example=false),
     *                     @OA\Property(property="banner_picture", type="string", example="https://example.com/storage/location/banner.jpg"),
     *                     @OA\Property(property="description", type="string", example="Welcome to Mumbai")
     *                 )
     *             )
     *         )
     *     ),
     * 
     *     @OA\Response(
     *         response=422,
     *         description="Error occurred",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Token invalid or unexpected error")
     *         )
     *     )
     * )
     */
    public function locations(Request $request)
    {
        try {
            $locations = Location::join('landing_pages', function($join){
                $join->on('location.id', '=', 'landing_pages.location_id')
                ->where('landing_pages.type','=','location');
            })->select('location.id','location.location', 'location.type','location.coming_soon','landing_pages.banner_picture','landing_pages.description')->get();

            foreach($locations as $location){
                if($location->banner_picture){
                    $location->banner_picture = url('/').Storage::url('location/'.$location->banner_picture);
                }
            }

            return response()->json([
                'status' => true,
                'data' => $locations
            ], 200);
        } catch (\Tymon\JWTAuth\Exceptions\JWTException $e) {
            return response()->json([
                'status' => false,
                'error' => 'Failed to logout, token invalid.'
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/studios/{location_id}",
     *     summary="Get list of studios by location",
     *     description="Returns all studios associated with a specific location ID along with their type, banner, and description.",
     *     tags={"Studios"},
     *     security={{"bearerAuth":{}}},
     *
     *     @OA\Parameter(
     *         name="location_id",
     *         in="path",
     *         required=true,
     *         description="ID of the location",
     *         @OA\Schema(type="integer", example=5)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Studios fetched successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="id", type="integer", example=101),
     *                     @OA\Property(property="name", type="string", example="Studio A"),
     *                     @OA\Property(property="studio_type", type="string", example="Podcast"),
     *                     @OA\Property(property="banner_picture", type="string", example="https://example.com/storage/studio/banner.jpg"),
     *                     @OA\Property(property="description", type="string", example="Studio A is equipped with modern audio tools.")
     *                 )
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=422,
     *         description="Error occurred while fetching studios",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Something went wrong")
     *         )
     *     )
     * )
    */
    public function studios(Request $request, $location_id)
    {
        try {

            $studios = Studio::Join('studio_types as st','studios.studio_type','st.id')->join('landing_pages', function($join){
                $join->on('studios.id', '=', 'landing_pages.studios')
                ->where('landing_pages.type','=','studio');
            })->where('studios.location_id',$location_id)
            ->select('studios.id','studios.name','st.name as studio_type','landing_pages.banner_picture','landing_pages.description')->get();


            foreach($studios as $studio){
                if($studio->banner_picture){
                    $studio->banner_picture = url('/').Storage::url('studio/'.$studio->banner_picture);
                }
            }

            return response()->json([
                'status' => true,
                'data' => $studios
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/peoples",
     *     summary="Get list of all resource people",
     *     description="Returns a list of resource people including name, profile image, type, location, and social media links.",
     *     tags={"People"},
     *     security={{"bearerAuth":{}}},
     *
     *     @OA\Response(
     *         response=200,
     *         description="People fetched successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="uuid", type="string", example="a4d53e3a-1b2d-4cbb-8430-9c7b193df8af"),
     *                     @OA\Property(property="fname", type="string", example="John"),
     *                     @OA\Property(property="lname", type="string", example="Doe"),
     *                     @OA\Property(property="profile_pic", type="string", example="https://example.com/storage/profile-pic/john.jpg"),
     *                     @OA\Property(property="type", type="string", example="Photographer"),
     *                     @OA\Property(property="location", type="string", example="Bangalore"),
     *                     @OA\Property(property="profile_link", type="string", example="https://example.com/johndoe"),
     *                     @OA\Property(property="instagram_link", type="string", example="https://instagram.com/johndoe"),
     *                     @OA\Property(property="facebook_link", type="string", example="https://facebook.com/johndoe"),
     *                     @OA\Property(property="twitter_link", type="string", example="https://twitter.com/johndoe"),
     *                     @OA\Property(property="sm_link", type="string", example="https://someothersocial.com/johndoe"),
     *                     @OA\Property(property="description", type="string", example="John is a professional photographer with 10 years of experience.")
     *                 )
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=422,
     *         description="Error occurred while fetching people",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Something went wrong")
     *         )
     *     )
     * )
    */
    public function peoples(Request $request)
    {
        try {
            $resources = Resources::Join('resource_types','resource_types.id','resources.resource_type_id')
            ->Join('users','users.id','resourceId')
            ->join('location','location.id','resources.location_id')
            ->select('uuid','fname','lname','profile_pic','resource_types.name as type','location.location','profile_link','instagram_link','facebook_link','twitter_link', 'sm_link', 'description')->get();

            foreach($resources as $resource){
                if($resource->profile_pic){
                    $resource->profile_pic = url('/').Storage::url('profile-pic/'.$resource->profile_pic);
                }
            }

            return response()->json([
                'status' => true,
                'data' => $resources
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    /**
     * @OA\Get(
     *     path="/api/events",
     *     summary="Get list of active events",
     *     description="Returns a list of events with optional filters for date and location. Also includes interested and going counts.",
     *     tags={"Events"},
     *     security={{"bearerAuth":{}}},
     *
     *     @OA\Parameter(
     *         name="filterDate",
     *         in="query",
     *         required=false,
     *         description="Filter events by date (YYYY-MM-DD)",
     *         @OA\Schema(type="string", format="date", example="2025-07-19")
     *     ),
     *     @OA\Parameter(
     *         name="filterLocation",
     *         in="query",
     *         required=false,
     *         description="Filter events by location ID",
     *         @OA\Schema(type="integer", example=3)
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Events retrieved successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=true),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     @OA\Property(property="id", type="integer", example=1),
     *                     @OA\Property(property="title", type="string", example="Music Night"),
     *                     @OA\Property(property="description", type="string", example="An exciting night of music and fun."),
     *                     @OA\Property(property="thumbnail", type="string", example="https://example.com/storage/location/event/event.jpg"),
     *                     @OA\Property(property="event_status", type="integer", example=1),
     *                     @OA\Property(property="interested_count", type="integer", example=50),
     *                     @OA\Property(property="going_count", type="integer", example=30),
     *                     @OA\Property(property="created_at", type="string", format="date-time", example="2025-07-18T10:00:00Z")
     *                 )
     *             )
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=422,
     *         description="Failed to fetch events",
     *         @OA\JsonContent(
     *             @OA\Property(property="status", type="boolean", example=false),
     *             @OA\Property(property="error", type="string", example="Unexpected error occurred")
     *         )
     *     )
     * )
    */
    public function events(Request $request)
    {
        try {

            $user = JWTAuth::user();
            $id = 0;
            if(!is_null($user)){
                $id = $user->id;
            }
            $query = Event::withCount(['interested','going'])->where('event_status',1);
            
            if($request->has('filterDate') && $request->filterDate!=""){
                $query->whereDate('created_at','=',$request->filterDate);
            }
            if($request->has('filterLocation') && $request->filterLocation!=""){
                $query->where('location_id',$request->filterLocation);
            }

            // $events = $query->with(['interested'=> function ($query) use($id){
            //     $query->where('user_id', $id);
            // }])->with(['going'=> function ($query) use($id){
            //     $query->where('user_id', $id);
            // }])
            $events = $query->get();

            foreach($events as $event){
                if($event->thumbnail){
                    $event->thumbnail = url('/').Storage::url('location/event/'.$event->thumbnail);
                }
            }

            return response()->json([
                'status' => true,
                'data' => $events
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

}
