<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Location, App\Models\Event, App\Models\News;
use App\Models\Studio, App\Models\Equipment, App\Models\User; 
use App\Models\Schedule, App\Models\Booking, App\Models\BookingDetail;
use App\Models\LandingPage, App\Models\Notification, App\Models\Point;
use App\Models\Plan, App\MOdels\EquipmentCheckin, App\Models\Gallery;
use App\Models\BookingNote, App\Models\UpcomingGuest;
use Carbon\Carbon, App\Helpers, App\Models\SiteConfig;
use Ramsey\Uuid\Uuid, Hash;

use Validator, Session, DB;

class BookingController extends Controller
{
    public function clear_booking_form(){
        session()->forget('booking-form');
        session()->forget('temp-booking-points');
        session()->forget('temp-booking-studio-selection');
        session()->forget('temp-booking-equipment-selection');
        session()->forget('temp-booking-resource-selection');
        return response()->json(['status' => true]);
    }
   
    public function index(Request $request)
    {
        $type = (isset($request->type)) ? $request->type : 'studio';
       
        $booking_form = [];
        //$booking_form['type'] = 'flow';
        $booking_form['location'] = $request->txt_location;
        $booking_form['start_date'] = $request->txt_start_date;
        //$booking_form['start_time'] = $request->txt_start_time;
        $booking_form['end_date'] = $request->txt_end_date;
        //$booking_form['end_time'] = $request->txt_end_time;
        $booking_form['start_time'] = Carbon::createFromFormat('h:i A', $request->txt_start_time)->format('H:i:s');
        $booking_form['end_time']   = Carbon::createFromFormat('h:i A', $request->txt_end_time)->format('H:i:s');

        session()->put('booking-form',$booking_form);
 
        $redirect = '/choose-your-preference/'.$type;
        
        if(isset($redirect)){
            return response()->json([
                'status' => true,
                'redirect' => $redirect,
            ], 200);
        }else{
            return response()->json([
                'status' => false,
                'redirect' => 'Invalid form data received.',
            ], 422);
        }
    }

    public function choose_your_preference($type){
        $editForm = false;

        if(auth()->user()->membership=='guest' ||  session()->get('is_team_member') >0){
            session()->flash('error','You are not authorized for booking.');
            return redirect('account-setting');
        }

        $advance_date_flag = "";
        if(!is_null(auth()->user()->membership)){
            $plan = Plan::where('slug',auth()->user()->membership)->first();
            if($plan->advance_duration > 0){
                if($plan->advance_frequency=='week'){
                    $advance_date_flag = Carbon::now()->addWeeks($plan->advance_duration)->format('Y-m-d');
                }else if($plan->advance_frequency=='month'){
                    $advance_date_flag = Carbon::now()->addMonths($plan->advance_duration)->format('Y-m-d');
                }
            }
        }

        if(session()->has('booking-form')){
            $dates = [];
            $booking_form = session()->get('booking-form');
            $from = Carbon::parse($booking_form['start_date']);
            $to = Carbon::parse($booking_form['end_date']);
            for($d = $from; $d->lte($to); $d->addDay()) {
                $dates[] = $d->format('Y-m-d');
            }
        }else{
            $nowTime = Carbon::now();
            $endTime = Carbon::now()->addHours(2);
            $booking_form = [];
            $booking_form['location']   = '';
            $booking_form['start_date'] = $nowTime->format('Y-m-d');
            $booking_form['start_time'] = $nowTime->format('H:i:00');
            $booking_form['end_date']   = $nowTime->format('Y-m-d');
            $booking_form['end_time']   = $endTime->format('H:i:00');
            session()->put('booking-form',$booking_form);
            $dates[] = $nowTime->format('Y-m-d');
        }
        
        $locations = Location::pluck('location','id');
        $points = [];
        $first_date = $dates[0];
        $last_date  = $dates[count($dates)-1];
        $total_days = count($dates);
        $start_time = $booking_form['start_time'];
        $end_time   = $booking_form['end_time'];
        
        $start = Carbon::createFromTimeString( $start_time);
        $end = Carbon::createFromTimeString($end_time);
        $minutes = $start->diffInMinutes($end) * count($dates);
        
        $point_per_usd = session()->get('siteconfig.points_per_usd');
        
        if(session()->has('temp-booking-points')){
            $points = session()->get('temp-booking-points');
        }
        
        if($type=='studio'){
            if(isset($points['studio'])){
                unset($points['studio']);
            }
            // get studios only with schedules and bookings if available
            $query = Studio::where('min_booking_minutes','<=',$minutes)->where('max_booking_minutes','>=',$minutes)
            ->with('landingPage:id,banner_picture,studios')
            ->whereHas('schedules', function($w) use($dates, $booking_form){
                $w->where('location_id',$booking_form['location']);
                $w->whereIn('start_date',$dates);
                $w->where('start_time','<=', $booking_form['start_time']);
                $w->where('end_time','>=', $booking_form['end_time']);
            }, '>=', count($dates))
            ->with(['schedules' => function($w) use($dates){
                $w->whereIn('start_date',$dates);
            }])->with(['bookings' => function($w) use($dates){
                $w->whereIn('start_date',$dates);
            }]);
            $studios = $query->paginate(10);
            //dd($studios);
            if($studios->count() > 0){
                $loop = 0;
                foreach($studios as $studio){
                    $studio_points = 0;
                    $available = true;
                    $booking_days = 0;
                    $booking_hours = 0;
                    foreach($studio->schedules as $sch){
                        //price calculation
                        if($total_days==1){
                            if($sch->start_time==$start_time && $sch->end_time==$end_time){
                                $studio_points = $sch->daily_rate*$point_per_usd;
                                $points[$type][$studio->id]['daily'][] = $studio_points;
                                $booking_days++;
                            }else{
                                $hours = $this->getHours($start_time, $end_time);
                                $booking_hours += $hours;
                                $studio_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $points[$type][$studio->id]['hourly'][] = "{$hours}|{$studio_points}";
                            }
                        }else if($sch->start_date==$first_date){
                            if($sch->start_time==$start_time){
                                $studio_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$studio->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($start_time, $sch->end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $studio_points += $calc_points;
                                $points[$type][$studio->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else if($sch->start_date==$last_date){
                            if($sch->end_time==$end_time){
                                $studio_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$studio->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($sch->start_time, $end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $studio_points += $calc_points;
                                $points[$type][$studio->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else{
                            $studio_points += $sch->daily_rate*$point_per_usd;
                            $points[$type][$studio->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                            $booking_days++;
                            //dump($studio_points);
                        }
                        //check availability
                        if($studio->bookings){
                            $sch_start_date = $sch->start_date;
                            $bookings = $studio->bookings;
                            $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                                return $item->start_date==$sch_start_date;
                            });
                            if($total_days==1){
                                $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                                    return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                                });
                                if($is_booked->count()>0){
                                    $available = false;
                                    break;
                                }
                            }else{
                                if($sch->start_date==$first_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($start_time){
                                        return $item->end_time > $start_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($sch->start_date==$last_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($end_time){
                                        return $item->end_time < $end_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($bookings_of_the_day->count()>0){
                                    $available = false;
                                    break;
                                }
                            }
                        }
                    }
                    $studios[$loop]->available = $available;
                    $studios[$loop]->booking_days = $booking_days;
                    $studios[$loop]->booking_hours = $booking_hours;
                    $studios[$loop]->studio_points = (int)$studio_points;
                    $loop++;
                }
            }
            //dump($points);
            //dump($studios);
            session()->put('temp-booking-points',$points);
            $redirect = url('choose-your-preference/equipment');
            return view("frontend.booking.{$type}")->with(['type'=>$type, 'locations'=>$locations, 'studios'=>$studios, 'redirect'=>$redirect, 'advance_date_flag'=>$advance_date_flag, 'editForm'=>$editForm]);
        }else if($type=='equipment'){
            if(isset($points['studio'])){
                $editForm = false;
            }
            if(isset($points['equipment'])){
                unset($points['equipment']);
            }
            // get equipment only with schedules and bookings if available
            $query = Equipment::with('resourceTypes:id,name')
            ->whereHas('schedules', function($w) use($dates, $booking_form){
                $w->where('location_id',$booking_form['location']);
                $w->whereIn('start_date',$dates);
                $w->where('start_time','<=', $booking_form['start_time']);
                $w->where('end_time','>=', $booking_form['end_time']);
            }, '>=', count($dates))
            ->with(['schedules' => function($w) use($dates){
                $w->whereIn('start_date',$dates);
            }])->with(['bookings' => function($w) use($dates){
                $w->where('start_date',$dates);
            }])->where('studios_id',0);
            $equipments = $query->paginate(10);
            //dump($equipments);
            if($equipments->count() > 0){
                $loop = 0;
                foreach($equipments as $equipment){
                    $equipment_points = 0;
                    $available = true;
                    $booking_days = 0;
                    $booking_hours = 0;
                    foreach($equipment->schedules as $sch){
                        //price calculation
                        if($total_days==1){
                            if($sch->start_time==$start_time && $sch->end_time==$end_time){
                                $equipment_points = $sch->daily_rate*$point_per_usd;
                                $points[$type][$equipment->id]['daily'][] = $equipment_points;
                                $booking_days++;
                            }else{
                                $hours = $this->getHours($start_time, $end_time);
                                $booking_hours += $hours;
                                $equipment_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $points[$type][$equipment->id]['hourly'][] = "{$hours}|{$equipment_points}";
                            }
                        }else if($sch->start_date==$first_date){
                            if($sch->start_time==$start_time){
                                $equipment_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$equipment->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($start_time, $sch->end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $equipment_points += $calc_points;
                                $points[$type][$equipment->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else if($sch->start_date==$last_date){
                            if($sch->end_time==$end_time){
                                $equipment_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$equipment->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($sch->start_time, $end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $equipment_points += $calc_points;
                                $points[$type][$equipment->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else{
                            $equipment_points += $sch->daily_rate*$point_per_usd;
                            $points[$type][$equipment->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                            $booking_days++;
                            //dump($studio_points);
                        }
                        //check availability
                        if($equipment->bookings){
                            $sch_start_date = $sch->start_date;
                            $bookings = $equipment->bookings;
                            $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                                return $item->start_date==$sch_start_date;
                            });
                            if($total_days==1){
                                $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                                    return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                                });
                                if($is_booked->count()>0){
                                    $available = false;
                                    break;
                                }
                            }else{
                                if($sch->start_date==$first_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($start_time){
                                        return $item->end_time > $start_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($sch->start_date==$last_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($end_time){
                                        return $item->end_time < $end_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($bookings_of_the_day->count()>0){
                                    $available = false;
                                    break;
                                }
                            }
                        }
                    }
                    $equipments[$loop]->available = $available;
                    $equipments[$loop]->booking_days = $booking_days;
                    $equipments[$loop]->booking_hours = $booking_hours;
                    $equipments[$loop]->equipment_points = (int)$equipment_points;
                    $loop++;
                }
            }
            // dump($points);
            // dump($equipments);
            session()->put('temp-booking-points',$points);
            $redirect = url('choose-your-preference/resource');
            return view("frontend.booking.{$type}")->with(['type'=>$type, 'locations'=>$locations, 'equipments'=>$equipments, 'redirect'=>$redirect, 'advance_date_flag'=>$advance_date_flag, 'editForm'=>$editForm]);
        }else if($type=='resource'){
            if(isset($points['equipment'])){
                $editForm = false;
            }
            if(isset($points['resource'])){
                unset($points['resource']);
            }
            // get resource only with schedules and bookings if available
            $query = User::with('resourceName')
            ->whereHas('schedules', function($w) use($dates, $booking_form){
                $w->where('location_id',$booking_form['location']);
                $w->whereIn('start_date',$dates);
                $w->where('start_time','<=', $booking_form['start_time']);
                $w->where('end_time','>=', $booking_form['end_time']);
            }, '>=', count($dates))
            ->with(['schedules' => function($w) use($dates){
                $w->whereIn('start_date',$dates);
            }])->with(['bookings' => function($w) use($dates){
                $w->where('start_date',$dates);
            }]);
            $resources = $query->paginate(10);
            //dump($resources);
            if($resources->count() > 0){
                $loop = 0;
                foreach($resources as $resource){
                    $resource_points = 0;
                    $available = true;
                    $booking_days = 0;
                    $booking_hours = 0;
                    foreach($resource->schedules as $sch){
                        //price calculation
                        if($total_days==1){
                            if($sch->start_time==$start_time && $sch->end_time==$end_time){
                                $resource_points = $sch->daily_rate*$point_per_usd;
                                $points[$type][$resource->id]['daily'][] = $resource_points;
                                $booking_days++;
                            }else{
                                $hours = $this->getHours($start_time, $end_time);
                                $booking_hours += $hours;
                                $resource_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                //dd($resource_points);
                                $points[$type][$resource->id]['hourly'][] = "{$hours}|{$resource_points}";
                            }
                        }else if($sch->start_date==$first_date){
                            if($sch->start_time==$start_time){
                                $resource_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$resource->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($start_time, $sch->end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $resource_points += $calc_points;
                                $points[$type][$resource->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else if($sch->start_date==$last_date){
                            if($sch->end_time==$end_time){
                                $resource_points += $sch->daily_rate*$point_per_usd;
                                $points[$type][$resource->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                                $booking_days++;
                                //dump($studio_points);
                            }else{
                                $hours = $this->getHours($sch->start_time, $end_time);
                                $calc_points = ($hours*$sch->hourly_rate)*$point_per_usd;
                                $resource_points += $calc_points;
                                $points[$type][$resource->id]['hourly'][] = "{$hours}|{$calc_points}";
                                $booking_hours += $hours;
                                //dump($studio_points);
                            }
                        }else{
                            $resource_points += $sch->daily_rate*$point_per_usd;
                            $points[$type][$resource->id]['daily'][] = $sch->daily_rate*$point_per_usd;
                            $booking_days++;
                            //dump($studio_points);
                        }
                        //check availability
                        if($resource->bookings){
                            $sch_start_date = $sch->start_date;
                            $bookings = $resource->bookings;
                            $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                                return $item->start_date==$sch_start_date;
                            });
                            if($total_days==1){
                                $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                                    return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                                });
                                if($is_booked->count()>0){
                                    $available = false;
                                    break;
                                }
                            }else{
                                if($sch->start_date==$first_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($start_time){
                                        return $item->end_time > $start_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($sch->start_date==$last_date){
                                    $is_booked = $bookings_of_the_day->filter(function($item) use($end_time){
                                        return $item->end_time < $end_time;
                                    });
                                    if($is_booked->count()>0){
                                        $available = false;
                                        break;
                                    }
                                }else if($bookings_of_the_day->count()>0){
                                    $available = false;
                                    break;
                                }
                            }
                        }
                    }
                    $resources[$loop]->available = $available;
                    $resources[$loop]->booking_days = $booking_days;
                    $resources[$loop]->booking_hours = $booking_hours;
                    $resources[$loop]->resource_points = (int)$resource_points;
                    $loop++;
                }
            }
            // dump($points);
            // dump($resources);
            session()->put('temp-booking-points',$points);
            $redirect = url('review-booking');
            return view("frontend.booking.{$type}")->with(['type'=>$type, 'locations'=>$locations, 'resources'=>$resources, 'redirect'=>$redirect, 'advance_date_flag'=>$advance_date_flag, 'editForm'=>$editForm]);
        }
    }

    public function make_selection(Request $request){
        if($request->type=='studio'){
            session()->put('temp-booking-studio-selection',$request->id);
        }else if($request->type=='equipment'){
            session()->put('temp-booking-equipment-selection',$request->id);
        }else if($request->type=='resource'){
            session()->put('temp-booking-resource-selection',$request->id);
        }

        $content = view("frontend.booking.price-box")->render();
        return response()->json([
            'status' => true,
            'content' => $content,
        ], 200);       
    }

    public function check_availability(Request $request){
        $validator = Validator::make($request->all(),[ 
            'selection' => 'required',
            'selection_type' => 'required',
        ],[
            'selection.required' => 'Selection is required.',
            'selection_type.required' => 'Selection type is required.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $id = $request->selection;
            $type = $request->selection_type;
            $booking_form = session()->get('booking-form');
            
            $start_time = $booking_form['start_time'].':00';
            $end_time   = $booking_form['end_time'].':00';
            
            if($request->year!="" && $request->month!=""){
                $start_date = $request->year.'-'.$request->month.'-01';
                $end_date = date($request->year.'-'.$request->month.'-t');
            }else{
                $nowTime = Carbon::now();
                $start_date = $nowTime->format('Y-m-d');
                $newTime = Carbon::now()->addMonths(6);
                $end_date = $newTime->format('Y-m-d');
            }
            //dump($start_date);dd($end_date);

            $available_dates = [];
            if($type=='studio'){
                // get studios only with schedules and bookings if available
                $query = Studio::with(['schedules' => function($w) use($start_date, $end_date, $start_time, $end_time, $booking_form){
                    $w->where('location_id', $booking_form['location']);
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                    $w->where('start_time','<=', $start_time)->where('end_time','>=', $end_time);
                }])->with(['bookings' => function($w) use($start_date, $end_date, $start_time, $end_time){
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                }]);
                $studio = $query->where('id',$id)->first();
                //dd($studio);
                foreach($studio->schedules as $sch){
                    $available = true;
                    $sch_start_date = $sch->start_date;
                    //check availability
                    if($studio->bookings){
                        $bookings = $studio->bookings;
                        $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                            return $item->start_date==$sch_start_date;
                        });
                        
                        $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                            return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                        });
                        if($is_booked->count()>0){
                            $available = false;
                            break;
                        }
                    }
                    
                    if($available){
                        $available_dates[] = $sch_start_date;
                    } 
                }
            }else if($type=='equipment'){
                // get studios only with schedules and bookings if available
                $query = Equipment::with(['schedules' => function($w) use($start_date, $end_date, $start_time, $end_time, $booking_form){
                    $w->where('location_id',$booking_form['location']);
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                    $w->where('start_time','<=', $start_time)->where('end_time','>=', $end_time);
                }])->with(['bookings' => function($w) use($start_date, $end_date, $start_time, $end_time){
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                }]);
                $equipment = $query->where('id',$id)->first();
                //dd($equipment);
                foreach($equipment->schedules as $sch){
                    $available = true;
                    $sch_start_date = $sch->start_date;
                    //check availability
                    if($equipment->bookings){
                        $bookings = $equipment->bookings;
                        $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                            return $item->start_date==$sch_start_date;
                        });
                        
                        $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                            return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                        });
                        if($is_booked->count()>0){
                            $available = false;
                            break;
                        }
                    }
                    
                    if($available){
                        $available_dates[] = $sch_start_date;
                    } 
                }
            }else if($type=='resource'){
                // get studios only with schedules and bookings if available
                $query = User::with(['schedules' => function($w) use($start_date, $end_date, $start_time, $end_time, $booking_form){
                    $w->where('location_id',$booking_form['location']);
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                    $w->where('start_time','<=', $start_time)->where('end_time','>=', $end_time);
                }])->with(['bookings' => function($w) use($start_date, $end_date, $start_time, $end_time){
                    $w->where('start_date','>=',$start_date)->where('end_date','<=',$end_date);
                }]);
                $resource = $query->where('id',$id)->first();
                //dd($resource);
                foreach($resource->schedules as $sch){
                    $available = true;
                    $sch_start_date = $sch->start_date;
                    //check availability
                    if($resource->bookings){
                        $bookings = $resource->bookings;
                        $bookings_of_the_day = $bookings->filter(function($item) use($sch_start_date){
                            return $item->start_date==$sch_start_date;
                        });
                        
                        $is_booked = $bookings_of_the_day->filter(function($item) use($start_time, $end_time){
                            return (($item->start_time <= $start_time && $item->end_time >= $start_time) || ($item->start_time <= $end_time && $item->end_time >= $end_time) || ($start_time <= $item->start_time && $end_time >= $item->end_time));
                        });
                        if($is_booked->count()>0){
                            $available = false;
                            break;
                        }
                    }
                    
                    if($available){
                        $available_dates[] = $sch_start_date;
                    } 
                }
            }

            return response()->json([
                'status' => true,
                'available_dates' => $available_dates,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function save_guests_session(Request $request){
        $validator = Validator::make($request->all(),[ 
            'guests' => 'required|array|min:1',
            'guests.*.fname' => 'required|string',
            'guests.*.lname' => 'required|string',
            'guests.*.email' => 'required|email',
        ],[
            'guests.required' => 'Please add at least one guest.',
            'guests.*.fname.required' => 'First name is required for each guest.',
            'guests.*.lname.required' => 'Last name is required for each guest.',
            'guests.*.email.required' => 'Email is required for each guest.',
            'guests.*.email.email' => 'Email is not valid for each guest.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error' => $validator->errors()->first()
            ], 422);
        }

        $newGuests = $request->guests;
        $existingGuests = session()->get('guests', []);

        // Merge and remove duplicates by email
        $allGuests = array_merge($existingGuests, $newGuests);
        $uniqueGuests = [];
        foreach ($allGuests as $guest) {
            $uniqueGuests[$guest['email']] = $guest;
        }
        $finalGuests = array_values($uniqueGuests);

        session()->put('guests', $finalGuests);
        return response()->json([
            'status' => true,
            'message' => "Guests saved successfully."
        ], 200);
    }

    public function invite_guest_save_session(Request $request){
        $validator = Validator::make($request->all(),[
            'location_id' => 'required',
            'start_date' => 'required',
            'start_time' => 'required',
            'end_date' => 'required',
            'end_time' => 'required', 
            'guests' => 'required|array|min:1',
            'guests.*.fname' => 'required|string',
            'guests.*.lname' => 'required|string',
            'guests.*.email' => 'required|email',
            'guests.*.mobile' => 'required',
        ],[
            'guests.required' => 'Please add at least one guest.',
            'guests.*.fname.required' => 'First name is required for each guest.',
            'guests.*.lname.required' => 'Last name is required for each guest.',
            'guests.*.email.required' => 'Email is required for each guest.',
            'guests.*.email.email' => 'Email is not valid for each guest.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error' => $validator->errors()->first()
            ], 422);
        }

        $newGuests = $request->guests;
        $existingGuests = session()->get('guests', []);
        
        // Merge and remove duplicates by email
        $allGuests = array_merge($existingGuests, $newGuests);
        $uniqueGuests = [];
        foreach ($allGuests as $guest) {
            $uniqueGuests[$guest['email']] = $guest;
        }
        $finalGuests = array_values($uniqueGuests);
        session()->put('guests', $finalGuests);
        
        // Set session regarding invite guest
        $location_id = $request->location_id;
        $start_date = $request->start_date;
        $start_time = $request->start_time;
        $end_date = $request->end_date;
        $end_time = $request->end_time;
        $host_uuid = auth()->user()->uuid;

        session()->put('igld',[
            'location_id' => $location_id,
            'start_date' => $start_date,
            'start_time' => $start_time,
            'end_date' => $end_date,
            'end_time' => $end_time,
            'type' => $request->type,
            'id' => $request->num
        ]);

        Helpers::invite_guest_session($host_uuid, $location_id, $start_date, $start_time, $end_date, $end_time);
        
        return response()->json([
            'status' => true,
            'message' => "Guests saved successfully."
        ], 200);
    }

    public function delete_invite_guest_session(Request $request){
        session()->forget('igld');
        session()->forget('guests');

        return response()->json([
            'status' => true,
            'message' => "Guest deleted successfully."
        ], 200);
    }

    public function booking_process(Request $request){
        try{
            $total_points = 0;
            $booking_id = time();
            $studio_data = [];
            $resource_data = [];
            $equipment_data = [];
            $booking_form = session()->get('booking-form');
            // get selected studio points
            if(isset(session()->get('temp-booking-points')['studio'][session()->get('temp-booking-studio-selection')])){
                $studio_points = 0;
                $studio = session()->get('temp-booking-points')['studio'][session()->get('temp-booking-studio-selection')];
                if(isset($studio['daily'])){
                    foreach($studio['daily'] as $daily){
                        $studio_points += $daily;
                    }
                }
                if(isset($studio['hourly'])){
                    foreach($studio['hourly'] as $hr){
                        $hourly_data = explode('|',$hr);
                        $hourly_points = $hourly_data[1]/$hourly_data[0];
                        $studio_points += $hourly_data[1];
                    }
                }
                $total_points += $studio_points;    
                $studio_data = [
                    'booking_id'    => $booking_id,
                    'location_id'   => $booking_form['location'],
                    'type'          => 'studio',
                    'studio_id'     => session()->get('temp-booking-studio-selection'),
                    'start_date'    => $booking_form['start_date'],
                    'end_date'      => $booking_form['end_date'],
                    'start_time'    => $booking_form['start_time'],
                    'end_time'      => $booking_form['end_time'],
                    'points'        => $studio_points,
                    'data'          => json_encode($studio),
                    'status'        => 0,
                    'created_by'    => auth()->user()->id,
                    'created_at'    => date('Y-m-d H:i:s')
                ];
            }
            // get selected equipment points
            if(isset(session()->get('temp-booking-points')['equipment'][session()->get('temp-booking-equipment-selection')])){
                $equipment_points = 0;
                $equipment = session()->get('temp-booking-points')['equipment'][session()->get('temp-booking-equipment-selection')];
                if(isset($equipment['daily'])){
                    foreach($equipment['daily'] as $daily){
                        $equipment_points += $daily;
                    }
                }
                if(isset($equipment['hourly'])){
                    foreach($equipment['hourly'] as $hr){
                        $hourly_data = explode('|',$hr);
                        $hourly_points = $hourly_data[1]/$hourly_data[0];
                        $equipment_points += $hourly_data[1];
                    }
                }
                $total_points += $equipment_points;
                $equipment_data = [
                    'booking_id'    => $booking_id,
                    'location_id'   => $booking_form['location'],
                    'type'          => 'rental',
                    'equipment_id'  => session()->get('temp-booking-equipment-selection'),
                    'start_date'    => $booking_form['start_date'],
                    'end_date'      => $booking_form['end_date'],
                    'start_time'    => $booking_form['start_time'],
                    'end_time'      => $booking_form['end_time'],
                    'points'        => $equipment_points,
                    'data'          => json_encode($equipment),
                    'status'        => 0,
                    'created_by'    => auth()->user()->id,
                    'created_at'    => date('Y-m-d H:i:s')
                ];
            }
            // get selected resources points
            if(isset(session()->get('temp-booking-points')['resource'][session()->get('temp-booking-resource-selection')])){
                $resource_points = 0;
                $resource = session()->get('temp-booking-points')['resource'][session()->get('temp-booking-resource-selection')];
                if(isset($resource['daily'])){
                    foreach($resource['daily'] as $daily){
                        $resource_points += $daily;
                    }
                }
                if(isset($resource['hourly'])){
                    foreach($resource['hourly'] as $hr){
                        $hourly_data = explode('|',$hr);
                        $hourly_points = $hourly_data[1]/$hourly_data[0];
                        $resource_points += $hourly_data[1];
                    }
                }
                $total_points += $resource_points;
                $resource_data = [
                    'booking_id'    => $booking_id,
                    'location_id'   => $booking_form['location'],
                    'type'          => 'resource',
                    'resource_id'   => session()->get('temp-booking-resource-selection'),
                    'start_date'    => $booking_form['start_date'],
                    'end_date'      => $booking_form['end_date'],
                    'start_time'    => $booking_form['start_time'],
                    'end_time'      => $booking_form['end_time'],
                    'points'        => $resource_points,
                    'data'          => json_encode($resource),
                    'status'        => 0,
                    'created_by'    => auth()->user()->id,
                    'created_at'    => date('Y-m-d H:i:s')
                ];
            }
            // get guest details
            $guest_data = [];
            $guests = session()->get('guests', []);
            if(count($guests) > 0){
                $guest_points = 0;
                $additional_guest_points = session()->get('additional_guest_points', 0);
                $free_guest = session()->get('free_guest', 0);
                $i = 0;
                foreach($guests as $guest){
                    $i++;
                    if($i > $free_guest){
                        $guest_points += $additional_guest_points;
                    }
                }
                $total_points += $guest_points;
                $new_guest_data = [];
                $new_guest_data['guests'] = $guests;
                $new_guest_data['free_guest'] = $free_guest;
                $new_guest_data['additiona_guest_points'] = $additional_guest_points;

                $guest_data = [
                    'booking_id'    => $booking_id,
                    'location_id'   => $booking_form['location'],
                    'type'          => 'guest',
                    'start_date'    => $booking_form['start_date'],
                    'end_date'      => $booking_form['end_date'],
                    'start_time'    => $booking_form['start_time'],
                    'end_time'      => $booking_form['end_time'],
                    'points'        => $guest_points,
                    'data'          => json_encode($new_guest_data),
                    'status'        => 0,
                    'created_by'    => auth()->user()->id,
                    'created_at'    => date('Y-m-d H:i:s')
                ];
            }
            //dd($total_points);
            if($total_points<=0){
                session()->flash('error','Please make selection to proceed further.');
            }else if(session()->get('point_balance') >= $total_points){
                DB::transaction(function () use($booking_id, $total_points, $booking_form, $studio_data, $resource_data, $equipment_data, $guest_data){
                    // Deduct points from user
                    // Ensure that the user has enough points to cover the booking
                    $remaining = $total_points;

                    $points = Point::where('receivedby_uuid', auth()->user()->uuid)
                        ->whereIn('action', ['MEMBERSHIP_CREDIT', 'POINTS_PURCHASED', 'BOOKING_CANCELLED'])
                        ->where('balance', '>', 0)
                        ->whereDate('expired_at', '>=', now())
                        ->orderBy('expired_at', 'asc')
                        ->lockForUpdate() // Prevent race conditions
                        ->get();

                    foreach ($points as $point) {
                        if ($remaining <= 0) break;

                        $usable = min($point->balance, $remaining);

                        $point->debit += $usable;
                        $point->balance -= $usable;
                        $point->save();

                        // You can optionally log the deduction in a separate table here

                        $remaining -= $usable;
                    }

                    if ($remaining > 0) {
                        // Not enough points
                        return false;
                    }
                    // Points added to admin
                    $nomadic_account_uuid = env('NOMADIC_ACCOUNT_UUID');
                    $point = new Point;
                    $point->action = 'BOOKING_CONFIRMED';
                    $point->receivedby_uuid = $nomadic_account_uuid;
                    $point->sentby_uuid = auth()->user()->uuid;
                    $point->credit = $total_points;
                    $point->amount = $total_points;
                    $point->balance = 0;
                    $point->booking_id = $booking_id;
                    $point->save();
                    // save data in master table
                    $bmaster = new Booking;
                    $bmaster->uuid          = auth()->user()->uuid;
                    $bmaster->booking_id    = $booking_id;
                    $bmaster->location_id   = $booking_form['location'];
                    $bmaster->start_date    = $booking_form['start_date'];
                    $bmaster->end_date      = $booking_form['end_date'];
                    $bmaster->start_time    = $booking_form['start_time'];
                    $bmaster->end_time      = $booking_form['end_time'];
                    $bmaster->status        = 0;
                    $bmaster->total_points  = (int)$total_points;
                    $bmaster->created_by    = auth()->user()->id;
                    $bmaster->save();
                    // save data in child table
                    if(count($studio_data) > 0){
                        BookingDetail::insert($studio_data);
                    }
                    if(count($equipment_data) > 0){
                        BookingDetail::insert($equipment_data);
                    }
                    if(count($resource_data) > 0){
                        BookingDetail::insert($resource_data);
                    }
                    if(count($guest_data) > 0){
                        BookingDetail::insert($guest_data);
                    }
                });
                
                $totalBalance  = Point::where('receivedby_uuid', auth()->user()->uuid)
                    ->whereIn('action', ['MEMBERSHIP_CREDIT', 'POINTS_PURCHASED','BOOKING_CANCELLED'])
                    ->where('balance', '>', 0)
                    ->whereDate('expired_at', '>=', now())
                    ->get()->sum('balance');

                session()->put('point_balance',$totalBalance);

                // invite guests
                $this->__guest_process($guests,$booking_form['location'], $booking_id, $booking_form['start_date'], $booking_form['start_time']);

                // notifications
                Notification::insert(['user'=>auth()->user()->uuid, 'type'=>'booking', 'action'=>'confirmed', 'content'=>'Your booking has been confirmed.','reference_id'=>$booking_id]);
                // emails
                if(auth()->user()->email){
                    // $edata = array();
                    // $edata['email'] = auth()->user()->email;
                    // $edata['name']  = auth()->user()->fname.' '.auth()->user()->lname;
                    // $edata['booking_id']    = $booking_id;
                    // $edata['heading'] = 'Booking Confirmed';
                    // $edata['content'] = 'Your booking <b>'.$booking_form['start_date'].' '.$booking_form['start_time'].'</b> to <b>'.$booking_form['end_date'].' '.$booking_form['end_time'].'</b> has been confirmed.';
                    // \Mail::send('emails.booking', $edata, function($message) use($edata) {
                    //     $message->to($edata['email']);
                    //     $message->subject('Your booking is confirmed!');
                    // });

                    $templateId = env('SENDGRID_TEMPLATE_ID3');
                    $template_data['subject'] = 'Your booking is confirmed!';
                    $template_data['content'] = '<p>Booking ID: '.$booking_id.'</p><p>Your booking <b>'.$booking_form['start_date'].' '.$booking_form['start_time'].'</b> to <b>'.$booking_form['end_date'].' '.$booking_form['end_time'].'</b> has been confirmed.</p>';
                    $res = Helpers::__send_email(auth()->user()->email, $template_data, $templateId);
                }
                // clear booking session
                session()->forget('booking-form');
                session()->forget('temp-booking-points');
                session()->forget('temp-booking-studio-selection');
                session()->forget('temp-booking-equipment-selection');
                session()->forget('temp-booking-resource-selection');
                
                return redirect('booking-confirmed/'.$booking_id);
            }else{
                session()->flash('error','Your booking is not confirmed! Please buy more points to proceed further.');
            }
        } catch (\PDOException $pde) {
            session()->flash('error',$pde->getMessage());
        } catch (\Exception $e) {
            session()->flash('error',$e->getMessage());
        }
        return redirect()->back();
    }

    public function review_booking(){
        $data['studio'] = $data['resource'] = $data['equipment'] = null;

        $data['booking_points'] = session()->get('temp-booking-points');
        $data['booking_form'] = session()->get('booking-form');
        if(!isset($data['booking_form']) || count($data['booking_form'])<=0){
            session()->flash('error','Please fill the booking form to proceed further.');
            return redirect('home');
        }

        $location = LandingPage::with('location')->where('type','location')->where('location_id',$data['booking_form']['location'])->firstOrFail();
        $data['location'] = $location;

        if(isset($data['booking_points']['studio'])){
            $data['studio'] = Studio::with('landingPage:id,banner_picture,studios,description')->where('studios.id',array_key_first($data['booking_points']['studio']))->select('studios.id','studios.name')->first();
        }
        if(isset($data['booking_points']['equipment'])){
            $data['equipment'] = Equipment::where('id',array_key_first($data['booking_points']['equipment']))->select('id','name','photo','description')->first();
        }
        if(isset($data['booking_points']['resource'])){
            $data['resource'] = User::with('resourceName')->where('users.id',array_key_first($data['booking_points']['resource']))->select('users.id','fname','lname','profile_pic','description')->first();
        }

        $additional_guest_points = (!empty($location->additional_guest_points)) ? $location->additional_guest_points : 0;
        session()->put('additional_guest_points', $additional_guest_points);
        $free_guest = 0;
        $plan = Plan::where('slug',auth()->user()->membership)->first();
        if(isset($plan->num_of_guest) && $plan->num_of_guest > 0){
            $free_guest = $plan->num_of_guest;
        }
        session()->put('free_guest', $free_guest);

        return view('frontend.booking.review-booking')->with($data);
    }

    public function booking_confirmed($number) {
        $bookingId = $number;
        $data = [];
        $data['bookingId'] = $bookingId;
        
        $bookings = BookingDetail::select('studio_id','equipment_id','resource_id','type','points','data','status')
        ->where('booking_id',$bookingId)->where('created_by',auth()->user()->id)->get();
        
        $booking = Booking::Join('landing_pages as lp', function($join){
            $join->on('lp.location_id','booking_master.location_id');
            $join->where('lp.type','location');
        })->join('location','location.id','lp.location_id')
        ->select('status','start_date','end_date','start_time','end_time','total_points','location','banner_picture')
        ->where('booking_id',$bookingId)->first();
        $data['booking'] = $booking;
        //dd($bookings);
        if($bookings->count() >0){
            $data['bookings'] = $bookings;
            foreach($bookings as $booking){
                if($booking->type=='studio'){
                    $studio = Studio::with('landingPage:id,banner_picture,studios,description')->where('studios.id',$booking->studio_id)->select('studios.id','studios.name')->first();
                    $data['studio'] = $studio;
                }
                if($booking->type=='rental'){
                    $equipment = Equipment::where('id',$booking->equipment_id)->select('id','name','photo','description')->first();
                    $data['equipment'] = $equipment;

                    $isCheckedIn = EquipmentCheckin::select('id')->where('booking_id',$bookingId)->where('equipment_id',$booking->equipment_id)->first();
                    $data['isCheckedIn'] = (isset($isCheckedIn->id)) ? false : true;
                }
                if($booking->type=='resource'){
                    $resource = User::with('resourceName')->where('users.id',$booking->resource_id)->select('users.id','fname','lname','profile_pic','description')->first();
                    $data['resource'] = $resource;
                }
                if($booking->type=='guest'){
                    $data['guests'] = json_decode($booking->data,true);
                }
            }
            //dump($data);
            return view('frontend.booking.booking-confirmed')->with($data);
        }else{
            abort(403);  
        }
    }

    public function __guest_process($guest_data, $location_id = null, $booking_id = null, $invitation_date = null, $invitation_time = null){
        
        foreach($guest_data as $request){
            $user = User::where('email', $request->email)->first();
            if ($user) {
                $new_user = false;
                $uuid = $user->uuid;
                $user_id = $user->id;
            }else{
                $new_user = true;
                $uuid = (string) Uuid::uuid4();
                $password = Helpers::generatePassword();
                $user = new User;
                $user->uuid = $uuid; 
                $user->role = 'guest';
                $user->fname = $request->fname;
                $user->lname = $request->lname;
                $user->email = $request->email;
                $user->mobile_number = (isset($request->mobile_number)) ? $request->mobile_number : null;
                $user->status = 1;
                $user->password = Hash::make($password);
                $user->save();
                $user_id = $user->id;
            }

            $upcomingGuest = new UpcomingGuest;
            $upcomingGuest->location_id = $location_id;
            $upcomingGuest->invitation_type = 'booking';
            $upcomingGuest->booking_id = $booking_id;
            $upcomingGuest->guest_uuid = $uuid;
            $upcomingGuest->host_uuid = auth()->user()->uuid;
            $upcomingGuest->invitation_date = $invitation_date;
            $upcomingGuest->invitation_time = $invitation_time;
            $upcomingGuest->comp = (isset($request->comp)) ? $request->comp : 0;
            $upcomingGuest->status = 0;
            $upcomingGuest->save();

            $content = "<p>You are invited as a guest for the event on {$invitation_date} {$invitation_time}</p>";
            if (is_null($user)) {
                $content .= "<p>Please use these credentials to login:</p>";
                $content .= "<p>Email:{$request->email}</br>Password:{$password}</p>";
            }
            $templateId = env('SENDGRID_TEMPLATE_ID3');
            $template_data['subject'] = 'Invitation for the event';
            $template_data['content'] = $content;
            $res = Helpers::__send_email($request->email, $template_data, $templateId);
        }
        return true;
    }

    public function booking_cancelled(Request $request){
        if($request->has('booking') && $request->booking!=""){
            $booking_id = $request->booking;
            $book = Booking::where('booking_id',$request->booking)->first();
            Booking::where('booking_id',$request->booking)->update(['status'=>2]);
            BookingDetail::where('booking_id',$request->booking)->update(['status'=>2]);
            // Points added to admin
            // expired at 
            $configVal = SiteConfig::where('key_name','points_expired_month')->first();
            $points_expired_month = $configVal->key_value;
            $expired_at = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s'). '+'.$points_expired_month.' month'));
            $nomadic_account_uuid = env('NOMADIC_ACCOUNT_UUID');
            $point = new Point;
            $point->action = 'BOOKING_CANCELLED';
            $point->receivedby_uuid = auth()->user()->uuid;
            $point->sentby_uuid = $nomadic_account_uuid;
            $point->credit = $book->total_points;
            $point->amount = $book->total_points;
            $point->balance = $book->total_points;
            $point->booking_id = $booking_id;
            $point->expired_at = $expired_at;
            $point->save();
            // notifications
            Notification::insert(['user'=>auth()->user()->uuid, 'type'=>'booking', 'action'=>'cancelled', 'content'=>'Your booking has been cancelled.','reference_id'=>$book->booking_id]);
            // emails
            if(auth()->user()->email){
                // $edata = array();
                // $edata['email'] = auth()->user()->email;
                // $edata['name']  = auth()->user()->fname.' '.auth()->user()->lname;
                // $edata['booking_id']    = $book->booking_id;
                // $edata['heading'] = 'Booking Cancelled';
                // $edata['content'] = 'Your booking <b>'.$book->start_date.' '.$book->start_time.'</b> to <b>'.$book->end_date.' '.$book->end_time.'</b> has been cancelled.';
                // \Mail::send('emails.booking', $edata, function($message) use($edata) {
                //     $message->to($edata['email']);
                //     $message->subject('Booking cancelled!');
                // });

                $templateId = env('SENDGRID_TEMPLATE_ID3');
                $template_data['subject'] = 'Booking cancelled!';
                $template_data['content'] = '<p>Booking ID: '.$book->booking_id.'</p><p>Your booking <b>'.$book->start_date.' '.$book->start_time.'</b> to <b>'.$book->end_date.' '.$book->end_time.'</b> has been cancelled.</p>';
                $res = Helpers::__send_email(auth()->user()->email, $template_data, $templateId);
            }
            session()->flash('success','Your booking has been cancelled successfully.');
            return response()->json([
                'status' => true,
                'message' => 'Booking cancelled.',
            ], 200);
        }
        return response()->json([
            'status' => false,
            'error' => 'Invalid form data received.',
        ], 422);
    }

    public function booking_completed(Request $request){
        if($request->has('booking') && $request->booking!=""){
            $book = Booking::where('booking_id',$request->booking)->first();
            Booking::where('booking_id',$request->booking)->update(['status'=>1]);
            BookingDetail::where('booking_id',$request->booking)->update(['status'=>1]);

            // notifications
            Notification::insert(['user'=>auth()->user()->uuid, 'type'=>'booking', 'action'=>'completed', 'content'=>'Your booking has been completed.','reference_id'=>$book->booking_id]);
            // emails
            if(auth()->user()->email){
                // $edata = array();
                // $edata['email'] = auth()->user()->email;
                // $edata['name']  = auth()->user()->fname.' '.auth()->user()->lname;
                // $edata['booking_id']    = $book->booking_id;
                // $edata['heading'] = 'Booking Cancelled';
                // $edata['content'] = 'Your booking <b>'.$book->start_date.' '.$book->start_time.'</b> to <b>'.$book->end_date.' '.$book->end_time.'</b> has been cancelled.';
                // \Mail::send('emails.booking', $edata, function($message) use($edata) {
                //     $message->to($edata['email']);
                //     $message->subject('Booking cancelled!');
                // });

                $templateId = env('SENDGRID_TEMPLATE_ID3');
                $template_data['subject'] = 'Booking completed!';
                $template_data['content'] = '<p>Booking ID: '.$book->booking_id.'</p><p>Your booking <b>'.$book->start_date.' '.$book->start_time.'</b> to <b>'.$book->end_date.' '.$book->end_time.'</b> has been completed.</p>';
                $res = Helpers::__send_email(auth()->user()->email, $template_data, $templateId);
            }
            session()->flash('success','Your booking has been completed successfully.');
            return response()->json([
                'status' => true,
                'message' => 'Booking completed.',
            ], 200);
        }
        return response()->json([
            'status' => false,
            'error' => 'Invalid form data received.',
        ], 422);
    }

    public function booking_list (Request $request){
        // bookings
        $bookings = Booking::with('loc','locLanding')->leftJoin('booking_details as studio', function($join){
            $join->on('booking_master.booking_id','studio.booking_id')->whereNotNull('studio.studio_id');
        })->leftJoin('booking_details as equip', function($join){
            $join->on('booking_master.booking_id','equip.booking_id')->whereNotNull('equip.equipment_id');
        })->leftJoin('booking_details as resource', function($join){
            $join->on('booking_master.booking_id','resource.booking_id')->whereNotNull('resource.resource_id');
        })->select('booking_master.*','studio.studio_id as studioNum','equip.equipment_id as equipNum','resource.resource_id as resourceNum')->where('uuid',auth()->user()->uuid)->paginate(10);

        return view("frontend.booking.booking-list")->with(['bookings'=>$bookings]);

    }

    public function getHours($time1, $time2){
        $timestamp1 = strtotime(date('Y-m-d').' '.$time1);
        $timestamp2 = strtotime(date('Y-m-d').' '.$time2);
        $hours = (int)(round(($timestamp2-$timestamp1)/3600));
        return $hours;
    }

    public function equipment_checkout(Request $request, $booking_id, $equipment_id){
        $booking = BookingDetail::where('booking_id',$booking_id)->where('equipment_id',$equipment_id)->first();
        if($booking){
            $isCheckedIn = EquipmentCheckin::where('booking_id',$booking_id)->where('equipment_id',$equipment_id)->first();
            if($isCheckedIn){
                // if rental manager, redirect to checkin page
                $rental_manager = DB::table('user_permissions')->where('user_uuid', auth()->user()->uuid)->where('role_id', 6)->first();
                if(!is_null($rental_manager)){
                    return redirect('equipment-checkin/'.$isCheckedIn->id);    
                }
                // if not rental manager, redirect to booking confirmed page
                return redirect('booking-confirmed/'.$booking_id)->with('error', 'This equipment is already checked in');
            }
            $data = Equipment::where('id',$equipment_id)->select('id','location_id','name','photo','description','serial_number','manual', 'manual_link')->first();
            
            $sessionId = session()->getId();

            $content = '';
            $photos = Gallery::where('session_id',$sessionId)->where('type','equip_checkout')->select('id','image_url')->get();
            foreach($photos as $photo){
                $image_url = url('storage/checkout/'.$photo->image_url);
                $content .= '<div class="col-md-4 mb-1">
                                <i class="bi bi-trash3 glyimg pointer galimgdel" data-id="'.$photo->id.'" data-type="equip_checkout" data-bs-toggle="tooltip" data-bs-placement="bottom" aria-label="Delete" data-bs-original-title="Delete"></i>
                                <img src="'.$image_url.'" class="img-fluid w-100 pointer enlarge-image" style="border-radius:5px;">
                            </div>';
            }

            return view('frontend.equipments.checkout')->with(['data'=>$data, 'content'=>$content, 'equipment_id'=>$equipment_id, 'booking_id'=>$booking_id, 'sessionId'=>$sessionId]);
        }

        return redirect('booking-confirmed/'.$booking_id)->with('error', 'No equipment detail found to check in');
    }

    public function equipment_checkout_save(Request $request){
        $booking_id = $request->booking_id;
        $equipment_id = $request->equipment_id;
        $description = $request->description;
        $sessionId = session()->getId();
        $validator = Validator::make($request->all(),[
            'booking_id' => 'required',
            'equipment_id' => 'required',
        ],[
            'booking_id.required' => 'Booking ID is required.',
            'equipment_id.required' => 'Equipment ID is required.',
        ]);
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        $booking = BookingDetail::where('booking_id',$booking_id)->where('equipment_id',$equipment_id)->first();
        if($booking){
            $isCheckedIn = EquipmentCheckin::where('booking_id',$booking_id)->where('equipment_id',$equipment_id)->first();
            if(!is_null($isCheckedIn)){
                return response()->json([
                    'status' => false,
                    'error'=>'This equipment is already checked in.'
                ], 422);
            }

            // check if any photos uploaded for this checkout
            $photos = Gallery::where('session_id',$sessionId)->where('type','equip_checkout')->pluck('id')->toArray();
            $checkout = new EquipmentCheckin;
            $checkout->booking_id = $booking_id;
            $checkout->equipment_id = $equipment_id;
            $checkout->description = $description;
            $checkout->checkout_date = $booking->end_date.' '.$booking->end_time;
            if(count($photos)>0){
                $checkout->photos = implode(',',$photos);
            }
            $checkout->created_by = auth()->user()->id;
            if($checkout->save()){
                Gallery::where('session_id',$sessionId)->where('type','equip_checkout')->update(['ref_id'=>$checkout->id, 'session_id'=>null]);

                session()->flash('success','Equipment checked out successfully.');
                return response()->json([
                    'status' => true,
                ], 200);
            }
        }
        
        return response()->json([
            'status' => false,
            'error'=>'No equipment detail found to check in.'
        ], 422);
    }

    public function equipment_checkin(Request $request, $checkin_id){
        $data = EquipmentCheckin::Join('equipments', 'equipments.id', 'equipment_checkin.equipment_id')
        ->Join('booking_details', function($join){
            $join->on('booking_details.booking_id', 'equipment_checkin.booking_id')->whereColumn('equipment_checkin.equipment_id', 'booking_details.equipment_id');
        })->select('equipment_checkin.*','equipments.location_id','equipments.serial_number','equipments.name as equipment_name','equipments.photo as equipment_photo','booking_details.booking_id','booking_details.start_date','booking_details.end_date','booking_details.start_time','booking_details.end_time')
        ->where('equipment_checkin.id',$checkin_id)->where('equipment_checkin.status',0)->first();
        if($data){
            $sessionId = session()->getId();
            
            //checkout photos if any
            $checkout_content = '';
            $checkout_photos = Gallery::where('ref_id',$checkin_id)->where('type','equip_checkout')->select('id','image_url')->get();
            foreach($checkout_photos as $photo){
                $image_url = url('storage/checkout/'.$photo->image_url);
                $checkout_content .= '<div class="col-md-4 mb-1">
                    <img src="'.$image_url.'" class="img-fluid w-100 pointer enlarge-image" style="border-radius:5px;">
                </div>';
            }

            // checkin photos from session
            $checkin_content = '';
            $checkin_photos = Gallery::where('session_id',$sessionId)->where('type','equip_checkin')->select('id','image_url')->get();
            foreach($checkin_photos as $photo){
                $image_url = url('storage/checkin/'.$photo->image_url);
                $checkin_content .= '<div class="col-md-4 mb-1">
                                <i class="bi bi-trash3 glyimg pointer galimgdel" data-id="'.$photo->id.'" data-type="equip_checkout" data-bs-toggle="tooltip" data-bs-placement="bottom" aria-label="Delete" data-bs-original-title="Delete"></i>
                                <img src="'.$image_url.'" class="img-fluid w-100 pointer enlarge-image" style="border-radius:5px;">
                            </div>';
            }

            return view('frontend.equipments.checkin')->with(['data'=>$data, 'checkout_content'=>$checkout_content, 'checkin_content'=>$checkin_content, 'sessionId'=>$sessionId]);
        }

       abort(404, 'Checkin not found or already checked in.');
    }

    public function equipment_checkin_save(Request $request){
        $sessionId = session()->getId();
        $validator = Validator::make($request->all(),[
            'checkin_id' => 'required',
            'checkin_comments' => 'required',
        ],[
            'checkin_id.required' => 'Checkedin ID is required.',
            'checkin_comments.required' => 'Checkin comments is required.',
        ]);
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        $checkin_id = $request->checkin_id;
        $checkin_comments = $request->checkin_comments;
        $checkin = EquipmentCheckin::find($checkin_id);
         if(!is_null($checkin)){
            $photos = Gallery::where('session_id',$sessionId)->where('type','equip_checkin')->pluck('id')->toArray();
           
            $checkin->checkin_comments = $checkin_comments;
            $checkin->status = 1; // checked in
            if(count($photos)>0){
                $checkin->checkin_photos = implode(',',$photos);
            }
            $checkin->updated_by = auth()->user()->id;
            if($checkin->save()){
                Gallery::where('session_id',$sessionId)->where('type','equip_checkin')->update(['ref_id'=>$checkin->id, 'session_id'=>null]);

                session()->flash('success','Equipment checked in successfully.');
                return response()->json([
                    'status' => true,
                ], 200);
            }
        }
        
        return response()->json([
            'status' => false,
            'error'=>'No equipment detail found to check in.'
        ], 422);
    }

    public function booking_detail(Request $request, $booking_id){
        // current booking
        $booking = Booking::with('booking_by', 'studios', 'resources', 'rentals', 'notes')->where('booking_master.booking_id',$booking_id)->first();
        // old booking notes
        $old_bookings = Booking::with('notes')->where('uuid',$booking->uuid)
        ->where('booking_master.booking_id','!=',$booking_id)->orderBy('id','desc')->get();

        //dump($booking);
        return view('frontend.booking.booking-detail')->with([ 'booking_id'=>$booking_id, 'booking'=>$booking, 'old_bookings' => $old_bookings]);
    }

    public function save_booking_notes(Request $request){
        $validator = Validator::make($request->all(),[
            'booking_id' => 'required',
            'notes' => 'required',
            'action' => 'required'
        ]);
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        $booking = Booking::where('booking_id',$request->booking_id)->first();
        if(is_null($booking)){
            session()->flash('error','Invalid booking ID');
            return response()->json([
                'status' => true
            ], 200);
        }

        if($request->action == 'delete'){
            // delete notes
            BookingNote::where('id',$request->note_id)->delete();
            return response()->json([
                'status' => true
            ], 200);
        }else if($request->action == 'add'){
            // add notes
            $note = new BookingNote;
            $note->booking_id = $request->booking_id;
            $note->notes = $request->notes;
            $note->created_at = date('Y-m-d H:i:s');
            $note->createdby = auth()->user()->id;
            if($note->save()){
                return response()->json([
                    'status' => true
                ], 200);
            }
        }else if($request->action == 'edit'){
            // update notes
            $note = BookingNote::find($request->note_id);
            if(!is_null($note)){
                $note->notes = $request->notes;
                $note->created_at = date('Y-m-d H:i:s');
                if($note->save()){
                    return response()->json([
                        'status' => true
                    ], 200);
                }
            }else{
                return response()->json([
                    'status' => false,
                    'error' => 'Invalid note ID'
                ], 422);
            }
        }

        return response()->json([
            'status' => false,
            'error' => 'Invalid form data received'
        ], 422);
    }
}
