<?php

namespace App\Http\Controllers;
  
use Illuminate\Http\Request;
use App\Models\Plan;
use App\Models\User;
Use App\Models\Point;
Use App\Models\SiteConfig;
use App\Models\Notification;
use Stripe\PaymentIntent;
use Stripe\PaymentMethod;
use App\Helpers;
use Session, DB, Log, Validator;
class PlanController extends Controller
{
    public function index(Request $request)
    {
        $plans = Plan::get();
        $guests = config('app.guests');
        return view("stripe.plans", compact("plans", "guests"));
    }  
  
    public function show(Plan $plan, Request $request)
    {
        //Session::flash('success','Payment captured successfully');
        $intent = auth()->user()->createSetupIntent();
  
        return view("stripe.subscription", compact("plan", "intent"));
    }
    
    public function subscription(Request $request)
    {
        try{
          
            // Retrieve the payment method ID sent by the frontend
            $paymentMethod = $request->token;
            
            if($request->session()->has('signup_flow')){  
              $user = auth()->user();
              
              // Attach the payment method to the customer
              $options = [
                'name' => $user->fname .' '.$user->lname,
                //'email' => $user->email,
                'payment_method' => $paymentMethod,
                'invoice_settings' => ['default_payment_method' => $paymentMethod],
                'metadata'=> ['uuid' => $user->uuid]
              ];
              $stripeCustomer = $user->createAsStripeCustomer($options);
              
              //$user->addPaymentMethod($paymentMethod);
              //$user->updateDefaultPaymentMethod($paymentMethod);
              $user->updateDefaultPaymentMethodFromStripe();
              
              return  redirect()->route('additional-information');
            
            }else{
              // create new subscription
              $plan = Plan::find($request->plan);
              
              //dd($stripe_data);
              //$subscription = $request->user()->newSubscription($plan->slug, $plan->stripe_plan)
              //              ->create($paymentMethod, ['email'=>auth()->user()->email],['metadata'=> $metadata]);
              
              User::where('id',auth()->user()->id)->update(['signup_membership'=>$plan->slug,'status'=>0]);
              Session::flash('success','Your current subscription is on hold, after verification new subscription will be active.');

              return  redirect('account-setting');
            }
        }catch(\Stripe\Error\Card $e){
            $body = $e->getJsonBody();
            $err  = $body['error'];
            return redirect()->back()->with('error', $err);
        }
    }

    public function subscription_cancel(Request $request)
    {
      try{
        if(auth()->user()->membership){
          auth()->user()->subscription(auth()->user()->membership)->cancel();
          
          // notifications
          Notification::insert(['user'=>auth()->user()->id, 'type'=>'subscription', 'action'=>'cancelled', 'content'=>'Your subscription has been cancelled successfully.']);
          // emails
          if(auth()->user()->email){
            // $edata = array();
            // $edata['email'] = auth()->user()->email;
            // $edata['name']  = auth()->user()->fname.' '.auth()->user()->lname;
            // $edata['heading'] = 'Subscription Cancelled';
            // $edata['content'] = 'Your subscription <b>'.auth()->user()->membership.'</b> had been cancelled.';
            // \Mail::send('emails.general', $edata, function($message) use($edata) {
            //     $message->to($edata['email']);
            //     $message->subject('Subscription cancelled!');
            // });

            $content = '<p>Your subscription <b>'.auth()->user()->membership.'</b> had been cancelled.</p>';
            $templateId = env('SENDGRID_TEMPLATE_ID3');
            $template_data['content'] = $content;
            $template_data['subject'] = 'Subscription cancelled!';
            $res = Helpers::__send_email(auth()->user()->email, $template_data, $templateId);
          }
          
          return response()->json([
            "status" => true,
            "message" => 'Subscription cancelled successfully.',
          ],200);
        }

        return response()->json([
          "status" => false,
          "error" => 'Invalid form data received.',
        ],422);

      } catch (\PDOException $pde) {
          return response()->json([
              "status" => false,
              "error" => $pde->getMessage(),
          ],422);
      } catch (\Exception $e) {
          return response()->json([
              "status" => false,
              "error" => $e->getMessage(),
          ],422);
      }
    }

    public function buy_more_points(Request $request)
    {
        try{
          $defaultMethod = auth()->user()->defaultPaymentMethod();

          $paymentMethods = auth()->user()->paymentMethods();
          
          $point = SiteConfig::where('key_name','points_per_usd')->first();
          $content = view("stripe.buy-more-points")->with(['points_per_usd'=>$point->key_value, 'paymentMethods' => $paymentMethods, 'defaultMethod' => $defaultMethod])->render(); 
          return response()->json([
              'status' => true,
              'content' => $content
          ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function purchase_points(Request $request, $points)
    {
      $pointVal = SiteConfig::where('key_name','points_per_usd')->first();
      $amount = $points/$pointVal->key_value;
      auth()->user()->createOrGetStripeCustomer();
      $intent = auth()->user()->createSetupIntent();
      return view("stripe.buy-points-subscription", compact("amount", "points", "intent"));
    }

    public function process_buy_more_points(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'points' => 'required',
            'token' => 'required'
        ]);
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $points = $request->points;
            $pointVal = SiteConfig::where('key_name','points_per_usd')->first();
            $amount = $points/$pointVal->key_value;
            auth()->user()->createOrGetStripeCustomer();
            auth()->user()->updateDefaultPaymentMethod($request->token);
           
            auth()->user()->charge($amount * 100, $request->token, ['metadata'=>['purchase_type'=>'POINTS_PURCHASED', 'purchase_points'=>$points]]);
            
            // $pt = new Point
            // $pt->action = 'POINTS_PURCHASED'; 
            // $pt->receivedby_uuid = auth()->user()->uuid;
            // $pt->receivedby_uuid

            //point balance
            $point_balance = $points;
            $points_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)
            ->orderBy('id','desc')->first();
            if($points_data){
                $point_balance = $point_balance + $points_data->balance;
            }
            session()->put('point_balance',$point_balance);

            Session::flash('success','Payment captured successfully');
            
            return response()->json([
                'status' => true,
                'message'=> 'Payment captured successfully'
            ], 200);
        }catch(\Stripe\Error\Card $e){
            $body = $e->getJsonBody();
            $err  = $body['error'];
            return response()->json([
                'status' => false,
                'message'=> $err
            ], 422);
        }
    }

    public function add_payment_method(Request $request)
    {
      auth()->user()->createOrGetStripeCustomer();
      $intent = auth()->user()->createSetupIntent();
      return view("stripe.add-payment-method", compact("intent"));
    }

    public function add_payment_method_process(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'name' => 'required',
            'token' => 'required'
        ]);
        if($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors());
        }

        try{
            $user = $request->user();

            // Attach the payment method
            $user->addPaymentMethod($request->token);

            // Optional: set as default
            $user->updateDefaultPaymentMethod($request->token);

            Session::flash('success','Payment method added successfully');
            
            return redirect()->route('account-setting')->with('sucess', 'Payment method added successfully');
        }catch(\Stripe\Error\Card $e){
            $body = $e->getJsonBody();
            $err  = $body['error'];
            return redirect()->back()->with('error', $err);
        }
    }

    public function payment_method_delete(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
          'pmid' => 'required'
        ]);
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $user = $request->user();

            $user->deletePaymentMethod($request->pmid);
            
            return response()->json([
                'status' => true,
                'message'=> 'Payment method deleted successfully'
            ], 200);
        }catch(\Stripe\Error\Card $e){
            $body = $e->getJsonBody();
            $err  = $body['error'];
            return response()->json([
                'status' => false,
                'message'=> $err
            ], 422);
        }
    }

    public function stripe_webhook(Request $request)
    {
        $payload = @file_get_contents('php://input');
        $event = null;
        try {
            $event = \Stripe\Event::constructFrom(
                json_decode($payload, true)
            );
            //dd($event);
        } catch(\UnexpectedValueException $e) {
            // Invalid payload
            http_response_code(400);
            exit();
        }
        //Log::info($event);
        // Handle the event
        switch ($event->type) {
            case 'charge.succeeded':
              $charge = $event->data->object;
              $customer_id = $charge->customer;
              $invoice_id = $charge->invoice;
              $amount =  $charge->amount/100;
              $user = User::where('stripe_id',$customer_id)->first();
              $balance_point = 0;
              $available_points = Point::select('balance')->where('receivedby_uuid',$user->uuid)->orderBy('id','desc')->first();
              if(isset($charge->metadata->purchase_type) && $charge->metadata->purchase_type=='POINTS_PURCHASED'){
                if($available_points){
                  $balance_point = $available_points->balance + $charge->metadata->purchase_points;
                }else{
                  $balance_point = $charge->metadata->purchase_points;
                }
                // expired at 
                $configVal = SiteConfig::where('key_name','points_expired_month')->first();
                $points_expired_month = $configVal->key_value;
                $expired_at = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s'). '+'.$points_expired_month.' month'));
                // save points
                $point = new Point;
                $point->action = $charge->metadata->purchase_type;
                $point->receivedby_uuid = $user->uuid;
                $point->invoice_id = $invoice_id;
                $point->credit  = $charge->metadata->purchase_points;
                $point->balance = $balance_point;
                $point->amount = $amount;
                $point->expired_at = $expired_at;
                $point->save();
                $pointId = $point->id;
                // notifications
                Notification::insert(['user'=>$user->uuid, 'type'=>'point', 'action'=>'POINTS_PURCHASED', 'content'=>'Your purchased points ('.$charge->metadata->purchase_points.') has been credited to your wallet.','reference_id'=>$pointId]);
                // emails
                // if($user->email){
                //   $edata = array();
                //   $edata['email'] = $user->email;
                //   $edata['name']  = $user->fname.' '.$user->lname;
                //   $edata['heading'] = 'Points purchased';
                //   $edata['content'] = 'Your purchased points ('.$charge->metadata->purchase_points.') had been successfully credited to your wallet.';
                //   \Mail::send('emails.general', $edata, function($message) use($edata) {
                //       $message->to($edata['email']);
                //       $message->subject('Points purchased!');
                //   });
                // }

                $content = '<p>Your purchased points ('.$charge->metadata->purchase_points.') had been successfully credited to your wallet.</p>';
                $templateId = env('SENDGRID_TEMPLATE_ID3');
                $template_data['content'] = $content;
                $template_data['subject'] = 'Points purchased!';
                $res = Helpers::__send_email($user->email, $template_data, $templateId);
              }else{
                $plan = Plan::where('price',$amount)->first();
                if($available_points){
                  $balance_point = $available_points->balance + $plan->points;
                }else{
                  $balance_point = $plan->points;
                }
                // add into points table
                $point = new Point;
                $point->action = 'MEMBERSHIP_CREDIT';
                $point->receivedby_uuid = $user->uuid;
                $point->invoice_id = $invoice_id;
                $point->plan_id = $plan->id;
                $point->credit  = $plan->points;
                $point->balance = $balance_point;
                $point->amount = $amount;
                $point->save();
                $pointId = $point->id;
                // notifications
                Notification::insert(['user'=>$user->uuid, 'type'=>'point', 'action'=>'MEMBERSHIP_CREDIT', 'content'=>'Your membership points ('.$plan->points.') has been credited successfully.','reference_id'=>$pointId]);
                // emails
                // if($user->email){
                //   $edata = array();
                //   $edata['email'] = $user->email;
                //   $edata['name']  = $user->fname.' '.$user->lname;
                //   $edata['heading'] = 'Membership Credit';
                //   $edata['content'] = 'Your membership points ('.$plan->points.') had been successfully credited to your wallet.';
                //   \Mail::send('emails.general', $edata, function($message) use($edata) {
                //       $message->to($edata['email']);
                //       $message->subject('Membership credited!');
                //   });
                // }

                $content = '<p>Your membership points ('.$plan->points.') had been successfully credited to your wallet.</p>';
                $templateId = env('SENDGRID_TEMPLATE_ID3');
                $template_data['content'] = $content;
                $template_data['subject'] = 'Membership credited!';
                $res = Helpers::__send_email($user->email, $template_data, $templateId);
              }
              break;
            case 'charge.failed':
              Log::info('failed');
              $charge = $event->data->object;
              $customer_id = $charge->customer;
              $user = User::where('stripe_id',$customer_id)->first();
              DB::table('subscriptions')->where('user_id',$user->id)->update(['stripe_status'=>'inactive']);
              break;
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object; // contains a \Stripe\PaymentIntent
                // Then define and call a method to handle the successful payment intent.
                // handlePaymentIntentSucceeded($paymentIntent);
                break;
            case 'payment_method.attached':
                $paymentMethod = $event->data->object; // contains a \Stripe\PaymentMethod
                // Then define and call a method to handle the successful attachment of a PaymentMethod.
                // handlePaymentMethodAttached($paymentMethod);
                break;
            // ... handle other event types
            default:
                echo 'Received unknown event type ' . $event->type;
        }

        http_response_code(200);
    }
}
