<?php
namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon, Auth, DB, Session, Validator, Hash;
use App\Models\User, App\Models\SubscriptionPlan;
use App\Models\Location, App\Models\Role, App\Models\RolePages, App\Models\Studio;
use App\Models\Plan, App\Models\Point, App\Models\StudioTypes;
use App\Models\Currency, App\Models\UserPermission;
use App\Models\SiteConfig, App\Models\Notification;
use Ramsey\Uuid\Uuid;
use App\Models\ResourceTypes;
use App\Models\EquipmentSpecialist;
use App\Helpers;
use App\SendBird, Storage;
class UserController extends Controller
{
    public function choose_your_plan()
    {
        $plans = Plan::get();
        $guests = config('app.guests'); 
        return view("auth.choose-your-plan", compact("plans", "guests"));
    }

    public function change_plan()
    {
        $plans = Plan::get();
        $guests = config('app.guests'); 
        return view("auth.change-plan", compact("plans", "guests"));
    }

    public function signup($plan)
    {
        return view('auth.signup')->with('plan',$plan);
    }

    public function signup_validate(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'fname' => 'required',
            'lname' => 'required',
            'email' => 'required|email',
            'mobile_number' => 'required|unique:users,mobile_number,NULL,id',
            //'password' => 'required|min:6|same:retype_password',
            //'retype_password' => 'required|min:6',
        ],[
            'fname.required' => 'First name is required.',
            'lname.required' => 'Last name is required.',
            'mobile_number.required' => 'Mobile number is required.',
            'mobile_number.unique' => 'Mobile number already exist.',
            //'password.required' => 'Password is required.',
            //'retype_password.required' => 'Retype password is required.',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        
        try{
            $uuid = (string) Uuid::uuid4();
            $token = random_int(100000, 999999);
            $signup = [];
            $signup['uuid'] = $uuid;
            $signup['role'] = 'guest';
            $signup['selected-role'] = $request->plan;
            $signup['fname'] = $request->fname;
            $signup['lname'] = $request->lname;
            $signup['email'] = $request->email;
            $signup['mobile_number'] = $request->mobile_number;
            $signup['token'] = $token;
            $request->session()->put('signup_form',$signup);
            // send verification email
            // $data = array();
            // $data['email']   = $request->email;
            // $data['name']    = $request->fname.' '.$request->lname;
            // $data['token']   = $token;
            // \Mail::send('emails.verification', $data, function($message) use($data) {
            //     $message->to($data['email']);
            //     $message->subject('Verify your email');
            // });
            $res = Helpers::send_otp($request->mobile_number);
            return $res;
            return response()->json([
                'status' => true,
                'message' => 'success',
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function otp_verification()
    {
        if(Session::has('signup_form')){
            $signup_data = Session::get('signup_form');
            $mobile = substr($signup_data['mobile_number'], -4);
            return view('auth.otp-verification')->with(['mobile'=>$mobile, 'amob'=>$signup_data['mobile_number']]);
        }else{
            abort(403);
        }
    }
    
    public function signup_save(Request $request)
    {
        try{
            $otp_number = $request->first.$request->second.$request->third.$request->fourth.$request->fifth.$request->sixth;
            $signup_data = $request->session()->get('signup_form');
            $isVerified = Helpers::verify_otp($signup_data['mobile_number'], $otp_number);
            
            if ($isVerified) {
                $user = new User;
                $user->role  = 'guest';
                $user->membership = 'guest';
                $user->signup_membership = $signup_data['selected-role'];
                $user->uuid  = $signup_data['uuid'];
                $user->fname = $signup_data['fname'];
                $user->lname = $signup_data['lname'];
                $user->email = $signup_data['email'];
                $user->mobile_number = $signup_data['mobile_number'];
                //$user->password = $signup_data['password'];
                //$user->email_verified_at = date('Y-m-d H:i:s');
                $user->token = $signup_data['token'];
                $user->status = 0;
                if($user->save()){
                    Auth::loginUsingId($user->id);
                    // send email to admin
                    $support_email = env('SUPPORT_EMAIL');
                    $templateId = 'd-30d5c59ac5ed450f83808ca0870143a9';
                    $template_data['signup'] = 1;
                    $template_data['role'] = $signup_data['selected-role'];
                    $template_data['user'] = $signup_data['fname'].' '.$signup_data['lname'];
                    $res = Helpers::__send_email($support_email, $template_data, $templateId);
                    $request->session()->forget('signup_form');
                    $request->session()->put('signup_flow',$signup_data['selected-role']);
                    return response()->json([
                        'status' => true,
                    ], 200);
                }else{
                    return response()->json([
                        'status' => false,
                        'error'=>'Technical error.'
                    ], 422);
                }
            }else{
                return response()->json([
                    'status' => false,
                    'error'=>'Verification code is not valid.'
                ], 422);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function profile_picture(Request $request)
    {
        return view('auth.profile-picture')->with(['sess_uuid'=>auth()->user()->uuid, 'selected_plan'=>$request->session()->get('signup_flow')]);
    }

    public  function save_profile_picture(Request $request)
    {
        //$uuid = $request->uuid;
        if (!$request->hasFile('file')) {
            return response()->json(['status' => false, 'error' => 'No file uploaded.'], 422);  
        }
        $uuid = auth()->user()->uuid;
        $user = User::where('uuid',$uuid)->first();
        $storage_path = storage_path('app/public/profile-pic/');
        @unlink($storage_path.$user->profile_pic);
        if($request->has('type') && $request->type=='cam'){
            $img = $request->image;
            $storage_path = storage_path('app/public/profile-pic/');
            
            $image_parts = explode(";base64,", $img);
            $image_type_aux = explode("image/", $image_parts[0]);
            $image_type = $image_type_aux[1];
            
            $image_base64 = base64_decode($image_parts[1]);
            $fileName = time() . '.png';
            $file = $storage_path . $fileName;
            file_put_contents($file, $image_base64);
            
            User::where('uuid',$uuid)->update(['profile_pic'=>$fileName]);
            return response()->json(['status'=>true,'success'=>$fileName]);
        }else{
            $storage_path = storage_path('app/public/profile-pic/');
            
            $image = $request->file('file');
            $imageName = time().'.'.$image->extension();

            $image->storeAs('profile-pic', $imageName, 'public');

            User::where('uuid',$uuid)->update(['profile_pic'=>$imageName]);
            return response()->json(['status'=>true,'success'=>$imageName]);
        }
        
    }

    public function remove_profile_picture(Request $request)
    {
        if($request->has('uuid')){
            $user = User::where('uuid',$request->uuid)->first();
            $storage_path = storage_path('app/public/profile-pic/');
            @unlink($storage_path.$user->profile_pic);

            User::where('uuid',$request->uuid)->update(['profile_pic'=>null]);
            return response()->json(['status'=>true],200);
        }else{
            return response()->json(['status'=>false,'error'=>'Unable to delete profile picture.'],422);    
        }
    }

    public function additional_information()
    {
        $location = Location::select('id','location')->get();
        $studios = StudioTypes::select('id','name')->where('status',1)->get();
        return view('auth.additional-information')->with(['location'=>$location, 'studios'=>$studios]);
    }

    public function additional_info_save(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'home_city' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $creator_interest = "";
            if($request->has('creator_interest') && count($request->creator_interest)>0){
                $creator_interest = implode(',',$request->creator_interest);
            }
            $user = User::find(auth()->user()->id);
            $user->creator_interest = $creator_interest;
            $user->home_city        = $request->home_city;
            $user->profile_link     = $request->profile_link;
            $user->instagram_link   = $request->instagram_link;
            $user->facebook_link    = $request->facebook_link;
            $user->twitter_link     = $request->twitter_link;
            $user->sm_link          = $request->sm_link;
            $user->referred_by      = $request->referred_by;
            if($user->save()){
                return response()->json([
                    'status' => true,
                ], 200);
            }else{
                return response()->json([
                    'status' => false,
                    'error'=>'Technical error.'
                ], 422);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function consent()
    {
        return view('auth.consent');
    }

    public function congratulations()
    {
        return view('auth.congratulations');
    }
 
    public function login()
    {
        return view('auth.login-otp');
    }

    public function login_manual()
    {
        return view('auth.login');
    }

    public function login_save(Request $request)
    { 
        $validator = Validator::make($request->all(),[ 
            'mobile_number' => 'required',
            'password' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
			if(filter_var($request->mobile_number, FILTER_VALIDATE_EMAIL)) {
                $credentials = [
                    'email' => $request->mobile_number, 'password' => $request->password, 'status' => 1
                ];
            }else{
                $credentials = [
                    'mobile_number' => $request->mobile_number, 'password' => $request->password, 'status' => 1
                ];
            }
            $remember_me = $request->has('remember_token') ? true : false;
            if (Auth::attempt($credentials, $remember_me)) {
                $user = auth()->user();
                $this->__login_data($user);

                return response()->json([
                    'status' => true,
                ], 200);
            }
            return response()->json([
                'status' => false,
                'error'=>'These credentials do not match our records.'
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }    
    }

    public function resetPassword()
    {
        return view('auth.reset-password');
    }

    public function submitResetPassword(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'email' => 'required|email',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        try{
            $token = Str::random(64);
            
            DB::table('password_resets')->insert([
                'email' => $request->email, 
                'token' => $token, 
                'created_at' => Carbon::now()
            ]);

            // send email by twillio
            $template_data['type'] = 0;
            $template_data['reset_url'] = url('update-password').'/'.$token;
            $res = Helpers::__send_email($request->email, $template_data);

            Session::flash('success','Password reset link has been sent to your email address.');

            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }   
    }

    public function updatePassword(Request $request, $token)
    {
        $data = DB::table('password_resets')->where(['token' => $token])->first();
        return view('auth.update-password')->with(['data'=>$data]);
    }

    public function submitUpdatePassword(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'reset_token' => 'required',
            'new_password' => 'required|string|min:6',
            'confirm_password' => 'required|same:new_password'
        ]);

        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
  
        $updatePassword = DB::table('password_resets')->where(['token' => $request->reset_token])->first();
  
        if(!$updatePassword){
            return response()->json([
                'status' => false,
                'error'=> 'Invalid token received.'
            ], 422);
        }
  
        $user = User::where('email', $updatePassword->email)
                      ->update(['password' => Hash::make($request->new_password)]);
 
        DB::table('password_resets')->where(['token'=> $request->reset_token])->delete();
        Session::flash('success','Your password has been changed.');
        
        return response()->json([
            'status' => true
        ], 200);
    }

    public function showUser(Request $request)
    {
        try {
            $query = User::with('created_by','role')->orderBy("users.id", "DESC");
            if ($request->ajax()) {
                $data = $query->paginate(5);
                return view("user.addUser_ajax")->with(["data" => $data]);
            }

            $data = $query->paginate(5);
            $userType = Role::where('status',1)->pluck('role_name','role_id');
            return view("user.addUser")->with([
                "userType" => $userType,
                "data" => $data,
            ]);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function delete_user(Request $request)
    {
        try {
            User::find($request->userid)->delete();
            return response()->json([
                "status" => true,
            ],200);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function logout()
    {
        Session::flush();        
        Auth::logout();
        return redirect('/');
    }

    public function send_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'type' => 'required',
            'otpfield' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }
        
        try{
            if($request->type=='email' && strpos($request->otpfield, "@") !== false){
                if(isset(auth()->user()->id)){
                    $name = auth()->user()->fname.' '.auth()->user()->lname;
                    $token = auth()->user()->token;
                }else{
                    $user = User::where('email',$request->otpfield)->first();
                    if(is_null($user)){
                        return response()->json([
                            "status" => false,
                            "error" => "Email address is not registered with us."
                        ],422);    
                    }else{
                      $name = $user->fname.' '.$user->lname;
                      $token = $user->token;
                    }
                }
                // $data = array();
                // $data['email']   = $request->otpfield;
                // $data['name']    = $name;
                // $data['token']   = str_split($token);
                // \Mail::send('emails.verification', $data, function($message) use($data) {
                //     $message->to($data['email']);
                //     $message->subject('Verify your email');
                // });

                $content = '<p>This is your verification code:</p><p><h2>'.$token.'</h2></p>';
                $templateId = env('SENDGRID_TEMPLATE_ID3');
                $template_data['content'] = $content;
                $template_data['subject'] = 'Verify your email!';
                $res = Helpers::__send_email($request->otpfield, $template_data, $templateId);

                return response()->json([
                    "status" => true,
                    "message" => "Verification code has been sent to your email address."
                ],200);
            }else if($request->type=='mobile'){
                $res = Helpers::send_otp($request->otpfield);
                return $res;
            }
            
            return response()->json([
                "status" => false,
                "message" => "Invalid form data received."
            ],200);
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }catch (\Swift_TransportException $m){
            return response()->json([
                "status" => false,
                "error" => $m->getMessage(),
            ],422);
        }
    }

    public function verify_otp(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'type' => 'required',
            'mobile_number' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $otp_number = $request->first.$request->second.$request->third.$request->fourth.$request->fifth.$request->sixth;
            $isVerified = Helpers::verify_otp($request->mobile_number, $otp_number);
            if ($isVerified) {
                $user = User::where('mobile_number',$request->mobile_number)->first();
                if($user){
                    if($request->type=='login'){
                        $remember_me = $request->has('remember_token') ? true : false;
                        Auth::loginUsingId($user->id, $remember_me);
                        $this->__login_data($user);
                    }
                    return response()->json([
                        "status" => true
                    ],200);
                }else{
                    return response()->json([
                        'status' => false,
                        'error' => 'Mobile number is not registered with us.',
                    ], 422);
                }
            }else{
                return response()->json([
                    "status" => false,
                    "error" => "Verification code is not valid."
                ],422);
            }
        } catch (\PDOException $pde) {
            return response()->json([
                "status" => false,
                "error" => $pde->getMessage(),
            ],422);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "error" => $e->getMessage(),
            ],422);
        }
    }

    public function userinfo(Request $request)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }
        try{
            $data = User::with('user_role')->with('user_plan')->with('user_location')->where('uuid',$request->uuid)->first();
            $creator_interest = '';
            if($data->creator_interest){
                $studio_arr = [];
                $studios = StudioTypes::where('status',1)->whereIn('id',explode(',',$data->creator_interest))
                ->get();
                foreach($studios as $st){
                    $studio_arr[] = $st->name;
                }
                $creator_interest = implode(',',$studio_arr);
            }
            if($request->type=='edit'){
                $plans = Plan::select('name','slug')->get();
                $location = Location::pluck('location','id');
                $user_permissions = [];
                $user_roles = UserPermission::Join('location','location_id','location.id')
                ->select('location_id','location',DB::raw('GROUP_CONCAT(role_id , ", ") AS roleids'))->Where('user_uuid',$request->uuid)->groupBy('location_id')->get();
                foreach($user_roles as $role){
                    $locroles = explode(',',$role->roleids);
                    $uroles = Role::whereIn('role_id',$locroles)->pluck('role_name')->toArray();
                    $permissions['location'] = $role->location;
                    $permissions['roles'] = $uroles;
                    $user_permissions[] = $permissions;
                }
                $current_balance = 0;
                $balance_data = Point::select('balance')->where('receivedby_uuid',$request->uuid)->orderBy('id','desc')->first();
                if($balance_data){
                    $current_balance = $balance_data->balance;
                }
                $content = view("global.useredit-card")->with(['data'=>$data, 'location'=>$location, 'plans'=>$plans, 'user_permissions'=>$user_permissions, 'current_balance'=>$current_balance])->render();
            }else{
                $content = view("global.userinfo-card")->with(['data'=>$data, 'creator_interest'=>$creator_interest])->render();
            }
            return response()->json([
                'status' => true,
                'content' => $content,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
	
    public function save_team(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'daily_rate' => 'required',
            'locationId' => 'required',
            'hourly_rate' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'error' => $validator->errors()
            ], 422);
        }

        try {
            UserPermission::where('user_uuid',$request->user_uuid)->where('location_id',$request->locationId) ->delete();
            foreach($request->roles as $rol){
                $up = new UserPermission;
                $up->user_uuid = $request->user_uuid;
                $up->location_id = $request->locationId;
                $up->role_id = $rol;
                $up->pages = config('app.page_access')[$rol];
                $up->save();
            }
				
            EquipmentSpecialist::where('user_uuid',$request->user_uuid)->where('location_id',$request->locationId) ->delete();
            foreach($request->specialists as $spcl){
                $Ep = new EquipmentSpecialist;
                $Ep->user_uuid = $request->user_uuid;
                $Ep->location_id = $request->locationId;
                $Ep->resource_type = $spcl;
                $Ep->save();
            }
            if ($request->userID) {
                $data = User::find($request->userID);
                $data->daily_rate = Helpers::convertCurrencyToUSD($request->daily_rate);
                $data->hourly_rate = Helpers::convertCurrencyToUSD($request->hourly_rate);
                $data->save();
            }
            
            return response()->json([
                'status' => true,
                'message' => "Successfully Save changes",
            ], 200);
        } catch (\PDOException $pde) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
	
    public function teaminfo(Request $request)
    {
        $uroles = []; 
        $eq_specialist=[];
        $options = ''; $specialist_option = '';
        $uuid = $request->uuid;
        $location=$request->locationId;
        if (!in_array(auth()->user()->role, ['super-admin', 'admin'])) {
            abort(403);
        }

        try {
            $data = User::with(['user_role', 'user_plan', 'user_location'])->where('uuid', $request->uuid)->first();
            $creator_interest = '';
            $plans = Plan::select('name', 'slug')->get();
            $locations = Location::pluck('location', 'id');
            $user_permissions = [];
   
            $roles = Role::select('role_id','role_name','role_slug')->whereIn('type',['access'])->get();
            $user_roles = UserPermission::select('role_id')->Where('user_uuid',$uuid)->where('location_id',$location)->get();
            foreach($user_roles as $ur){
                $uroles[] = $ur->role_id;
            }
            foreach($roles as $role){
                if(in_array($role->role_id,$uroles)){
                    $options .= '<option selected="selected" value="'.$role->role_id.'">'.$role->role_name.'</option>';
                }else{
                    $options .= '<option value="'.$role->role_id.'">'.$role->role_name.'</option>';
                }
                
            }
            
            $specialist = ResourceTypes::select('id','name')->where('status',1)->get();
            $equipment_specialist = EquipmentSpecialist::select('resource_type')->Where('user_uuid',$uuid)->where('location_id',$location)->get();
            foreach($equipment_specialist as $eqs){
                $eq_specialist[] = $eqs->resource_type;
            }
            foreach($specialist as $spcl){
                if(in_array($spcl->id,$eq_specialist)){
                    $specialist_option.= '<option selected="selected" value="'.$spcl->id.'">'.$spcl->name.'</option>';
                }else{
                    $specialist_option.= '<option value="'.$spcl->id.'">'.$spcl->name.'</option>';
                }
                
            }
            
            $content = view("team.teamedit-card")->with([
                'data' => $data,
                'locationId' => $request->locationId,
                'user_permissions'=>$user_permissions, 
            ])->render();

            return response()->json([
                'status' => true,
                'content' => $content,
                'options' => $options,
                'specialist_option' => $specialist_option,
            ], 200);
        } catch (\PDOException $pde) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function userroles(Request $request)
    {
        $uroles = [];
        $options = '';
        $uuid = $request->uuid;
        $location = $request->location;
        if($location=='all'){
            $user = User::select('role')->where('uuid',$uuid)->first();
            $roles = Role::select('role_id','role_name','role_slug')->whereIn('type',['usertype'])->get();
            foreach($roles as $role){
                if($role->role_slug==$user->role){
                    $options .= '<option selected="selected" value="'.$role->role_slug.'">'.$role->role_name.'</option>';
                }else{
                    $options .= '<option value="'.$role->role_slug.'">'.$role->role_name.'</option>';
                }
            }
        }else{
            $roles = Role::select('role_id','role_name','role_slug')->whereIn('type',['access'])->get();
            $user_roles = UserPermission::select('role_id')->Where('user_uuid',$uuid)->where('location_id',$location)->get();
            foreach($user_roles as $ur){
                $uroles[] = $ur->role_id;
            }
            foreach($roles as $role){
                if(in_array($role->role_id,$uroles)){
                    $options .= '<option selected="selected" value="'.$role->role_id.'">'.$role->role_name.'</option>';
                }else{
                    $options .= '<option value="'.$role->role_id.'">'.$role->role_name.'</option>';
                }
                
            }
        }
        return response()->json([
            'status' => true,
            'content' => $options,
        ], 200);
    }

    public function add_permissions(Request $request)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }
        try{
            if($request->location=='all'){
                User::where('uuid',$request->user_uuid)->update(['role'=>$request->roles]);
				UserPermission::where('user_uuid',$request->user_uuid)->delete();
                return response()->json([
                    'status' => true,
                    'message' => 'Role has been updated for this user.',
                ], 200);    
            }else{
                UserPermission::where('user_uuid',$request->user_uuid)->where('location_id',$request->location)
                ->delete();
                if(!is_null($request->roles)){
                    foreach($request->roles as $role){
                        $up = new UserPermission;
                        $up->user_uuid = $request->user_uuid;
                        $up->location_id = $request->location;
                        $up->role_id = $role;
                        $up->pages = config('app.page_access')[$role];
                        $up->save();
                    }
                }
                return response()->json([
                    'status' => true,
                    'message' => 'Permissions updated for this location.',
                ], 200);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function search_user(Request $request)
    {
        $search_text = $request->search_text;
        try{
            $user = User::Select('uuid', 'email')->where('uuid','!=',auth()->user()->uuid)->where('status',1)
            ->whereNotIn('role',['super-admin','admin'])->where('email',$search_text)->first();
            
            if($user){
                return response()->json([
                    'status' => true,
                    'uuid' => $user->uuid,
                ], 200);
            }else{
                return response()->json([
                    'status' => true,
                    'uuid' => "",
                ], 200);
            }
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function transfer_points(Request $request)
    {
        try{
            $content = view("user.transfer-points")->render(); 
            return response()->json([
                'status' => true,
                'content' => $content
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function add_points(Request $request)
    {
        try{
            // points expired at
            $configVal = SiteConfig::where('key_name','points_expired_month')->first();
            $points_expired_month = $configVal->key_value;
            $expired_at = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s'). '+'.$points_expired_month.' month'));
            //receivedby entry
            $rec_current_balance = 0;
            $rec_balance_data = Point::select('balance')->where('receivedby_uuid',$request->user_uuid)->orderBy('id','desc')->first();
            if($rec_balance_data){
                $rec_current_balance = $rec_balance_data->balance;
            }
            $point = new Point;
            $point->action = 'POINTS_RECEIVED';
            $point->receivedby_uuid = $request->user_uuid;
            $point->sentby_uuid = auth()->user()->uuid;
            $point->credit = $request->points;
            $point->balance = ($rec_current_balance + $request->points);
            $point->expired_at = $expired_at;
            $point->save();
            $recPointId = $point->id;
            //notification
            Notification::insert(['user'=>$request->user_uuid, 'type'=>'point', 'action'=>'POINTS_RECEIVED', 'content'=>'Your have received ('.$request->points.') points from <b>'.auth()->user()->fname.' '.auth()->user()->lname.'</b>.','reference_id'=>$recPointId]);
            //sentby entry
            $sent_current_balance = 0;
            $sent_balance_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)->orderBy('id','desc')->first();
            if($sent_balance_data){
                $sent_current_balance = $sent_balance_data->balance;
            }
            $rpoint = new Point;
            $rpoint->action = 'POINTS_SENT';
            $rpoint->receivedby_uuid = auth()->user()->uuid;
            $rpoint->sentby_uuid = $request->user_uuid;
            $rpoint->debit = $request->points;
            $rpoint->balance = ($sent_current_balance - $request->points);
            $rpoint->save();
            $sentPointId = $point->id;
            //notification
            $rec_user = User::select('fname','lname')->where('uuid',$request->user_uuid)->first();
            Notification::insert(['user'=>auth()->user()->uuid, 'type'=>'point', 'action'=>'POINTS_SENT', 'content'=>'Your have sent ('.$request->points.') to <b>'.$rec_user->fname.' '.$rec_user->lname.'</b>.','reference_id'=>$sentPointId]);
            
            //point balance
            $point_balance = 0;
            $points_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)
            ->orderBy('id','desc')->first();
            if($points_data){
                $point_balance = $point_balance + $points_data->balance;
            }
            session()->put('point_balance',$point_balance);
            
            return response()->json([
                'status' => true,
                'message' => 'Points has been transfered successfully.',
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }
    
    public function user_app_action(Request $request)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }
        try{
            if($request->action==2){
                // start subscription
                $user = User::where('uuid',$request->uuid)->first();
                // if subscription exist then cancel
                if($user->membership){
                    $user->subscription($user->membership)->cancel();
                }
                // create new subscription
                $plan = Plan::where('slug',$user->signup_membership)->first();
                $user->newSubscription($plan->slug, $plan->stripe_plan)->create();
                User::where('uuid',$request->uuid)->update(['status'=>$request->action, 'membership'=>$user->signup_membership]);
                // notified to customer
                $message = 'Your membership has been approved and subscription started.';
                //$res = Helpers::send_sms($user->mobile_number, $message);
                //return $res;
            }else if($request->action==3){
                $user = User::where('uuid',$request->uuid)->first();
                // notified to customer
                $message = 'Your membership has been declined, you will be notified when new membership opportunities are available.';
                //$res = Helpers::send_sms($user->mobile_number, $message);
                //return $res;
            }else{
                User::where('uuid',$request->uuid)->update(['status'=>$request->action]);
            }
            
            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function update_user(Request $request)
    {
        if(!in_array(auth()->user()->role,['super-admin', 'admin'])){
            abort(403);
        }

        $validator = Validator::make($request->all(),[ 
            'user_uuid' => 'required',
            'location' => 'required',
            'role' => 'required',
            'membership' => 'required'
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            $user = User::where('uuid',$request->user_uuid)->first();
            if($user){
                $user->role = $request->role;
                $user->home_city = $request->location;
                $user->membership = $request->membership;
                //$user->points = $request->points;
                $user->save();
                if($request->points!=""){
                    //receivedby entry
                    $rec_current_balance = 0;
                    $rec_balance_data = Point::select('balance')->where('receivedby_uuid',$request->user_uuid)->orderBy('id','desc')->first();
                    if($rec_balance_data){
                        $rec_current_balance = $rec_balance_data->balance;
                    }
                    $point = new Point;
                    $point->action = 'POINTS_RECEIVED';
                    $point->receivedby_uuid = $request->user_uuid;
                    $point->sentby_uuid = auth()->user()->uuid;
                    $point->credit = $request->points;
                    $point->balance = ($rec_current_balance + $request->points);
                    $point->save();
                    //sentby entry
                    $sent_current_balance = 0;
                    $sent_balance_data = Point::select('balance')->where('receivedby_uuid',auth()->user()->uuid)->orderBy('id','desc')->first();
                    if($sent_balance_data){
                        $sent_current_balance = $sent_balance_data->balance;
                    }
                    $rpoint = new Point;
                    $rpoint->action = 'POINTS_SENT';
                    $rpoint->receivedby_uuid = auth()->user()->uuid;
                    $rpoint->sentby_uuid = $request->user_uuid;
                    $rpoint->debit = $request->points;
                    $rpoint->balance = ($sent_current_balance - $request->points);
                    $rpoint->save();
                }

                return response()->json([
                    'status' => true,
                    'message' => 'User updated successfully.',
                ], 200);
            }

            return response()->json([
                'status' => false,
                'error' => 'Invalid form data received.',
            ], 422);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function set_currency(Request $request)
    {
        $validator = Validator::make($request->all(),[ 
            'currency' => 'required',
        ]);
        
        if($validator->fails()) {
            return response()->json([
                'status' => false,
                'error'=>$validator->errors()
            ], 422);
        }

        try{
            session()->put('user_currency',$request->currency);
            if(isset(auth()->user()->uuid)){
                User::where('uuid',auth()->user()->uuid)->update(['home_currency'=>$request->currency]);
            }

            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function read_notification(Request $request)
    {
        try{
            Notification::where('user',auth()->user()->uuid)->where('status',0)->update(['status'=>1]);
            session()->put('unread_noti',0);

            return response()->json([
                'status' => true,
            ], 200);
        }catch(\PDOException $pde ) {
            return response()->json([
                'status' => false,
                'error' => $pde->getMessage(),
            ], 422);
        }catch(\Exception $e){
            return response()->json([
                'status' => false,
                'error' => $e->getMessage(),
            ], 422);
        }
    }

    public function sendbird_token(){
        $token = $this->__sendbird_session_token();
        
        return response()->json([
            'status' => true,
            'sbtoken' => $token,
        ], 200);
    }

    public function __sendbird_session_token(){
        $token = '';
        $sessionTime = Carbon::now()->addMinutes(120)->timestamp*1000;
        $sendbird = new SendBird();
        $res = $sendbird->POST_Call('users/'.auth()->user()->uuid.'/token',['expires_at'=>$sessionTime]);
        //dd($res);
        if($res['status']){
            $token = $res['data']['token'];
        }

        session()->put('sendbird_token',$token);
        return $token;
    }

    public function __login_data($user)
    {
        // setup page access for middleware
        $user_locations = [];
        $page_access = [];
        $user_access = UserPermission::select('location_id', DB::raw('GROUP_CONCAT(pages) AS pages'))->Where('user_uuid',$user->uuid)->groupBy('location_id')->get();
        if($user_access){
            foreach($user_access as $access){
                $pages = [];
                $dup_pages = explode(',',$access->pages);
                foreach($dup_pages as $page){
                    if(!in_array($page,$pages)){
                        $pages[] = $page;
                    }
                }
                $page_access[$access->location_id] = implode(',',$pages);
                $user_locations[] = $access->location_id;
            }
        }
        session()->put('page_access',$page_access);
        // set sendbird session token
        $sendbird_token = $this->__sendbird_session_token();
        // setup currency
        session()->put('user_currency',$user->home_currency);
        $currency = Currency::pluck('rate','currency')->toArray();
        session()->put('currency',$currency);
        // setup location
        if($user->role=='super-admin' || $user->role=='admin'){
			$locations = Location::with('Studio')->get();
            Session::put('user_locations',$locations);
        }else{
            $locations = Location::whereIn('id',$user_locations)->with('Studio')->get();
            Session::put('user_locations',$locations);
        }
        //site config
        $point = SiteConfig::where('key_name','points_per_usd')->first();
        session()->put('siteconfig.points_per_usd',$point->key_value);
        //point balance
        $point_balance = 0;
        $points_data = Point::select('balance')->where('receivedby_uuid',$user->uuid)
        ->orderBy('id','desc')->first();
        if($points_data){
            $point_balance = $points_data->balance;
        }
        session()->put('point_balance',$point_balance);
        //notifications
        $unread_noti = Notification::where('user',auth()->user()->uuid)->where('status',0)->count();
        session()->put('unread_noti',$unread_noti);
        $notifications = Notification::where('user',auth()->user()->uuid)
        ->orderBy('id','desc')->limit(10)->get();
        session()->put('notifications',$notifications);
    }
}
