<?php

namespace App\Models;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Event extends Model
{
    protected $table = 'events';
    protected $primaryKey = 'event_id';


    // Mutator: convert to 24-hour format before storing
    public function setEventTimeAttribute($value)
    {
        try {
            $this->attributes['event_time'] = \Carbon\Carbon::createFromFormat('h:i A', $value)->format('H:i:s');
        } catch (\Exception $e) {
            $this->attributes['event_time'] = $value; // fallback for raw 24-hour input
        }
    }

    // Accessor: (optional) convert back to 12-hour format on read
    public function getEventTimeAttribute($value)
    {
        if (!$value) return null;

        try {
            return \Carbon\Carbon::createFromFormat('H:i:s', $value)->format('h:i A');
        } catch (\Exception $e) {
            return $value; // fallback if format unexpected
        }
    }


    public function createdBy()
    {
        return $this->hasOne(User::class, 'id', 'created_by')->select('id','fname','lname');
    }

    public function updatedBy()
    {
        return $this->hasOne(User::class, 'id', 'updated_by')->select('id','fname','lname');
    }

    public function formattedDate()
    {
        return Carbon::parse($this->event_date.' '.$this->event_time)->isoFormat('dddd, D MMM [AT] h:mm A');
    }

    public function participants()
    {
        return $this->belongsToMany(User::class, 'participants', 'event_id', 'user_id')->withPivot('status');
           
    }

    public function interested()
    {
        return $this->hasMany(Participant::class, 'event_id', 'event_id')->where('participants.status','interested');
    }

    public function going()
    {
        return $this->hasMany(Participant::class, 'event_id', 'event_id')->where('participants.status','going');
    }
}
