<?php
namespace App\Services;

use Aws\IVS\IVSClient;
use Aws\Exception\AwsException;

class AwsIvsService
{
    protected $ivsClient;

    public function __construct()
    {
        // Initialize the AWS IVS client
        $this->ivsClient = new IVSClient([
            'region' => env('AWS_DEFAULT_REGION'),
            'version' => '2020-07-14',
            'http'     => [
                'verify' => false,  // Disable SSL verification
            ]
        ]);
    }

    public function createChannel($channelName)
    {
        try {
            $result = $this->ivsClient->createChannel([
                'name' => $channelName,    // Channel name
                'latencyMode' => 'LOW',    // Set 'LOW' or 'NORMAL'
                'type' => 'STANDARD',      // Set 'STANDARD' or 'BASIC'
                'recordingConfiguration' => [
                    'destinationConfiguration' => [
                        's3' => [
                            'bucketName' => 'nomadic-live-stream', // The name of your S3 bucket
                            'prefix' => 'ivs-recordings/' // Optional: specify a prefix for folder structure in the bucket
                        ]
                    ]
                ]
            ]);
            
            // Return channel ARN and stream key
            return [
                'channel_arn' => $result['channel']['arn'],
                'playbackUrl' => $result['channel']['playbackUrl'],
                'ingestEndpoint' => $result['channel']['ingestEndpoint'],
                'stream_arn' => $result['streamKey']['arn'],
                'stream_key' => $result['streamKey']['value'],
            ];
        } catch (AwsException $e) {
            // Output error message if an exception occurs
            return ['error' => $e->getMessage()];
        }
    }

    public function createStreamKey($channelArn)
    {
        try {
            $result = $this->ivsClient->createStreamKey([
                'channelArn' => $channelArn,
            ]);
           
            // Return new stream key
            return [
                'stream_arn' => $result['streamKey']['arn'],
                'stream_key' => $result['streamKey']['value'],
            ];
        } catch (AwsException $e) {
            // Output error message if an exception occurs
            return ['error' => $e->getMessage()];
        }
    }

    public function deleteStreamKey($channelArn)
    {
        try {
            $result = $this->ivsClient->deleteStreamKey([
                'arn' => $channelArn,
            ]);

            return $result;
        } catch (AwsException $e) {
            // Output error message if an exception occurs
            return ['error' => $e->getMessage()];
        }
    }

    public function deleteChannel($channelArn)
    {
        try {
            $result = $this->ivsClient->deleteChannel([
                'arn' => $channelArn,
            ]);

            return $result;
        } catch (AwsException $e) {
            // Output error message if an exception occurs
            return ['error' => $e->getMessage()];
        }
    }


}
