import { Link, useLocation, useNavigate } from "@remix-run/react";
import { Button } from "./ui/button";
import { MenuIcon } from "lucide-react";
import { useEffect, useState } from "react";
import {
  NavigationMenu,
  NavigationMenuContent,
  NavigationMenuItem,
  NavigationMenuLink,
  NavigationMenuList,
  NavigationMenuTrigger,
  navigationMenuTriggerStyle,
} from "./ui/navigation-menu";
import { Sheet, SheetContent, SheetHeader, SheetTrigger } from "./ui/sheet";
import { LanguageSelector } from "./LanguageSelector";
import { ProfileImage } from "./shared";
// Dropdown components are implemented natively
import { getUserData, isAuthenticated, handleLogout } from "~/lib/api";
import { useLanguage } from "~/hooks/useLanguage";

interface AppHeaderProps {
  theme?: "dark" | "white";
}

export function AppHeader({ theme = "dark" }: AppHeaderProps) {
  const location = useLocation();
  const navigate = useNavigate();
  const [isScrolled, setIsScrolled] = useState(false);
  const [isHovered, setIsHovered] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [user, setUser] = useState<any>(null);
  const [showProfileDropdown, setShowProfileDropdown] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [showNotificationModal, setShowNotificationModal] = useState(false);
  const { t, currentLanguage } = useLanguage();

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 0);
    };

    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    const checkAuthStatus = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);

      if (authenticated) {
        const userData = getUserData();
        setUser(userData);
      } else {
        setUser(null);
      }
    };

    checkAuthStatus();

    // Check auth status on mount and when route changes only
    // No need for interval polling
  }, [location.pathname]); // Re-check when route changes

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (showProfileDropdown) {
        setShowProfileDropdown(false);
      }
    };

    document.addEventListener("click", handleClickOutside);
    return () => document.removeEventListener("click", handleClickOutside);
  }, [showProfileDropdown]);

  const handleLogoutClick = async () => {
    try {
      await handleLogout();
      setIsLoggedIn(false);
      setUser(null);
      // Redirect to home page after logout
      window.location.href = "/";
    } catch (error) {
      console.error("Logout error:", error);
      // Even if API call fails, clear local state and redirect
      setIsLoggedIn(false);
      setUser(null);
      window.location.href = "/";
    }
  };

  const handleWalletClick = () => {
    navigate("/account-settings?tab=points");
  };

  const handleNotificationClick = () => {
    setShowNotificationModal(!showNotificationModal);
  };
  const navItems = [
    {
      label: t("nav.studios"),
      href: "/studios",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.videoStudios"), href: "/studios/video" },
        { label: t("nav.musicStudios"), href: "/studios/music" },
        { label: t("nav.photoStudios"), href: "/studios/photo" },
      ],
    },
    {
      label: t("nav.locations"),
      href: "/locations",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.bali"), href: "/locations/bali" },
        { label: t("nav.bangkok"), href: "/locations/bangkok" },
        { label: t("nav.mexico"), href: "/locations/mexico" },
        { label: t("nav.costaRica"), href: "/locations/costa-rica" },
        { label: t("nav.newYork"), href: "/locations/new-york" },
        { label: t("nav.mumbai"), href: "/locations/mumbai" },
      ],
    },
    { label: t("nav.people"), href: "/people" },
    { label: t("nav.equipment"), href: "/equipment" },
    { label: t("nav.events"), href: "/events" },
    { label: t("nav.services"), href: "/services" },
    { label: t("nav.news"), href: "/news" },
  ];

  const isDark = theme === "dark";
  const bgColor = isDark ? "bg-black" : "bg-white";
  const textColor = isDark ? "text-white" : "text-black";
  const borderColor = isDark ? "border-white/20" : "border-black/20";
  const hoverBg = isDark ? "hover:bg-white/10" : "hover:bg-black/10";
  const logoSrc = isDark ? "/images/logo-white.png" : "/images/logo-black.png";

  return (
    <header
      className={`fixed top-0 left-0 right-0 ${bgColor} z-50 transition-all duration-300 ${
        isScrolled && !isHovered
          ? `${isDark ? "bg-black/90" : "bg-white/90"} backdrop-blur-sm`
          : bgColor
      }`}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      <div className="mx-auto w-full max-w-[1328px]">
        <div className="flex items-center justify-between px-4 lg:px-14 h-16  lg:h-20 min-w-0">
          {/* Logo */}
          <Link to={isLoggedIn ? "/home" : "/"} className="block flex-shrink-0">
            <img
              src={logoSrc}
              alt={t("common.logo")}
              className="h-10 lg:h-12 w-auto"
            />
          </Link>

          {/* Desktop Navigation */}
          <div className="hidden xl:flex items-center gap-4 flex-1 justify-center mx-4 min-w-0">
            <NavigationMenu viewport={true} className="max-w-3xl">
              <NavigationMenuList className="gap-1">
                {navItems.map((item) => (
                  <NavigationMenuItem key={item.href}>
                    {item.hasDropdown ? (
                      <>
                        <NavigationMenuTrigger>
                          {item.label}
                        </NavigationMenuTrigger>
                        <NavigationMenuContent
                          className={`${bgColor} border ${borderColor} rounded-md overflow-hidden`}
                        >
                          <ul className="grid w-[600px] gap-3 p-4 grid-cols-2">
                            {item.dropdownItems?.map((dropdownItem) => (
                              <li key={dropdownItem.href}>
                                <NavigationMenuLink asChild>
                                  <Link
                                    to={dropdownItem.href}
                                    className={`block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors ${hoverBg} ${
                                      isDark
                                        ? "hover:text-white focus:text-white text-white/90"
                                        : "hover:text-black focus:text-black text-black/90"
                                    } ${
                                      isDark
                                        ? "focus:bg-white/10"
                                        : "focus:bg-black/10"
                                    }`}
                                  >
                                    <div className="text-sm font-medium leading-none uppercase tracking-[0.02em]">
                                      {dropdownItem.label}
                                    </div>
                                  </Link>
                                </NavigationMenuLink>
                              </li>
                            ))}
                          </ul>
                        </NavigationMenuContent>
                      </>
                    ) : (
                      <Link to={item.href}>
                        <NavigationMenuLink
                          className={navigationMenuTriggerStyle()}
                        >
                          {item.label}
                        </NavigationMenuLink>
                      </Link>
                    )}
                  </NavigationMenuItem>
                ))}
              </NavigationMenuList>
            </NavigationMenu>
          </div>

          {/* Mobile Navigation */}
          <div className="xl:hidden">
            <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
              <SheetTrigger asChild>
                <button className={`${textColor} ${hoverBg} p-2 rounded`}>
                  <MenuIcon className="h-6 w-6" />
                  <span className="sr-only">{t("header.openMenu")}</span>
                </button>
              </SheetTrigger>
              <SheetContent
                side="right"
                className={`${bgColor} border-l ${borderColor} ${textColor} w-full sm:w-96 overflow-y-auto`}
                onInteractOutside={(e) => {
                  // Allow closing when clicking outside
                }}
              >
                <div className="px-4 sm:px-8 py-6 sm:py-12 min-h-full">
                  <SheetHeader className="p-0 mb-6 sm:mb-8">
                    <div className="flex items-center justify-between">
                      <img
                        src={logoSrc}
                        alt={t("common.companyName")}
                        className="h-8 sm:h-10"
                      />
                    </div>
                  </SheetHeader>

                  <div
                    className={`space-y-3 sm:space-y-4 mb-6 sm:mb-8 pb-6 sm:pb-8 border-b ${
                      isDark ? "border-white/10" : "border-black/10"
                    }`}
                  >
                    {!isLoggedIn ? (
                      <>
                        <Button
                          size="sm"
                          asChild
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          <Link
                            to="/login"
                            onClick={() => setMobileMenuOpen(false)}
                          >
                            {t("auth.login")}
                          </Link>
                        </Button>
                        <Button
                          size="sm"
                          asChild
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          <Link
                            to="/signup"
                            onClick={() => setMobileMenuOpen(false)}
                          >
                            {t("auth.signUp")}
                          </Link>
                        </Button>
                      </>
                    ) : (
                      <div className="space-y-3 sm:space-y-4">
                        <div className="flex items-center gap-3 p-2.5 sm:p-3 rounded-lg bg-white/5">
                          <ProfileImage
                            src={user?.profile_pic}
                            alt={t("header.profile")}
                            className="w-10 h-10 sm:w-12 sm:h-12 rounded-full object-cover"
                          />
                          <div>
                            <p
                              className={`font-medium text-sm sm:text-base ${textColor}`}
                            >
                              {user?.fname && user?.lname ? `${user.fname} ${user.lname}` : user?.email || 'User'}
                            </p>
                            <p
                              className={`text-xs sm:text-sm ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              0 pts
                            </p>
                          </div>
                        </div>

                        {/* Mobile Action Buttons */}
                        <div className="flex justify-between items-center gap-2 sm:gap-4 p-2.5 sm:p-3 rounded-lg bg-white/5">
                          <button className="flex flex-col items-center gap-1" onClick={handleNotificationClick}>
                            <div className="relative">
                              <svg
                                className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor}`}
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                              >
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
                                />
                              </svg>
                              <div className="absolute -top-1 -right-1 w-1.5 h-1.5 sm:w-2 sm:h-2 bg-red-500 rounded-full"></div>
                            </div>
                            <span
                              className={`text-xs ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.notifications")}
                            </span>
                          </button>

                          <button className="flex flex-col items-center gap-1" onClick={() => navigate("/discuss")}>
                            <svg
                              className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor}`}
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                              />
                            </svg>
                            <span
                              className={`text-xs ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.chat")}
                            </span>
                          </button>

                          <button className="flex flex-col items-center gap-1">
                            <svg
                              className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor}`}
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"
                              />
                            </svg>
                            <span
                              className={`text-xs ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.wallet")}
                            </span>
                          </button>
                        </div>

                        <Button
                          size="sm"
                          onClick={handleLogoutClick}
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          {t("auth.signOut")}
                        </Button>
                      </div>
                    )}
                  </div>

                  <nav className="space-y-4 sm:space-y-6 pb-8 sm:pb-10">
                    {navItems.map((item) => (
                      <div key={item.href} className="space-y-2 sm:space-y-3">
                        <Link
                          to={item.href}
                          onClick={() => setMobileMenuOpen(false)}
                          className={`block ${textColor} ${
                            isDark
                              ? "hover:text-white/70"
                              : "hover:text-black/70"
                          } transition-colors text-base sm:text-lg font-medium uppercase tracking-wide`}
                        >
                          {item.label}
                        </Link>
                        {item.hasDropdown && item.dropdownItems && (
                          <div
                            className={`pl-3 sm:pl-4 space-y-1.5 sm:space-y-2 border-l ${
                              isDark ? "border-white/10" : "border-black/10"
                            }`}
                          >
                            {item.dropdownItems.map((dropdownItem) => (
                              <Link
                                key={dropdownItem.href}
                                to={dropdownItem.href}
                                onClick={() => setMobileMenuOpen(false)}
                                className={`block ${
                                  isDark
                                    ? "text-white/60 hover:text-white"
                                    : "text-black/60 hover:text-black"
                                } transition-colors text-xs sm:text-sm uppercase tracking-wide`}
                              >
                                {dropdownItem.label}
                              </Link>
                            ))}
                          </div>
                        )}
                      </div>
                    ))}
                  </nav>

                  {/* Language Selector for Mobile */}
                  <div
                    className={`mt-8 pt-6 border-t ${
                      isDark ? "border-white/10" : "border-black/10"
                    }`}
                  >
                    <div className="flex items-center justify-between">
                      <span className={`text-sm font-medium ${textColor}`}>
                        {t(
                          "languageSelector.languageAndRegion",
                          "Language & Region"
                        )}
                      </span>
                      <LanguageSelector />
                    </div>
                  </div>
                </div>
              </SheetContent>
            </Sheet>
          </div>

          {/* Desktop Auth Buttons / Profile */}
          <div className="hidden xl:flex items-center gap-3 flex-shrink-0 min-w-fit">
            {!isLoggedIn ? (
              <>
                <Button
                  className={`${isDark ? "!text-white/90" : "!text-black/90"} whitespace-nowrap ${currentLanguage === 'sp' ? 'min-w-[120px]' : ''}`}
                  asChild
                  variant={isDark ? "fill-black" : "outline-black"}
                  responsive
                >
                  <Link to="/login">{t("auth.login")}</Link>
                </Button>
                <Button
                  className={`${isDark ? "!text-white/90" : "!text-black/90"} whitespace-nowrap ${currentLanguage === 'sp' ? 'min-w-[120px]' : ''}`}
                  asChild
                  variant={isDark ? "fill-black" : "outline-black"}
                  responsive
                >
                  <Link to="/signup">{t("auth.signUp")}</Link>
                </Button>
              </>
            ) : (
              <div className="flex items-center gap-6">
                {/* Notification Bell */}
                <button className="relative" onClick={handleNotificationClick}>
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
                    />
                  </svg>
                  {/* <div className="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full"></div> */}
                </button>

                {/* Chat Icon */}
                <button className="" onClick={() => navigate("/discuss")}>
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                    />
                  </svg>
                </button>

                {/* Points */}
                <button 
                  className={`flex items-center gap-2 ${textColor} hover:opacity-80 transition-opacity`}
                  onClick={handleWalletClick}
                >
                  <span className="text-sm font-medium">0 pts</span>
                </button>

                {/* Wallet Icon */}
                <button className="" onClick={handleWalletClick}>
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"
                    />
                  </svg>
                </button>

                {/* Profile Dropdown */}
                <div className="relative">
                  <button
                    onClick={() => setShowProfileDropdown(!showProfileDropdown)}
                    className="w-10 h-10 rounded-full overflow-hidden focus:outline-none focus:ring-2 focus:ring-white/20"
                  >
                    <ProfileImage
                      src={user?.profile_pic}
                      alt={t("header.profile")}
                      className="w-full h-full object-cover"
                    />
                  </button>

                  {showProfileDropdown && (
                    <div
                      className={`absolute right-0 mt-2 w-56 sm:w-64 ${bgColor} border ${borderColor} rounded-lg shadow-lg z-50`}
                    >
                      {/* User Name Header */}
                      <div className="px-3 sm:px-4 py-2.5 sm:py-3 border-b border-gray-200">
                        <p
                          className={`font-medium text-sm sm:text-base ${textColor}`}
                        >
                          {user?.fname && user?.lname ? `${user.fname} ${user.lname}` : user?.email || 'User'}
                        </p>
                      </div>

                      {/* Menu Items */}
                      <div className="py-1">
                        <Link
                          to="/home"
                          className={`px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor} block`}
                          onClick={() => setShowProfileDropdown(false)}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("nav.home")}
                          </span>
                        </Link>

                        <Link
                          to="/account-settings"
                          className={`px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor} block`}
                          onClick={() => setShowProfileDropdown(false)}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"
                            />
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("header.accountSettings")}
                          </span>
                        </Link>

                        <button
                          onClick={() => {
                            setShowProfileDropdown(false);
                            handleLogoutClick();
                          }}
                          className={`w-full text-left px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor}`}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("auth.signOut")}
                          </span>
                        </button>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Notification Modal */}
      {showNotificationModal && (
        <div className="absolute top-full right-0 z-50 mt-2 mr-4 lg:mr-8" style={{ width: "20%" }}>
          <div className={`${isDark ? 'bg-white/10 border-white/20 text-white' : 'bg-black/90 border-black/20 text-white'} backdrop-blur-lg border rounded-2xl p-8 shadow-2xl`}>
            <div className="text-center">
              <h3 className="text-lg font-medium mb-4">Notifications</h3>
              <p className="text-white/60 text-sm mb-8">
                Notifications not available.
              </p>
              <button
                onClick={() => setShowNotificationModal(false)}
                className={`px-6 py-2 ${isDark ? 'bg-white/20 hover:bg-white/30' : 'bg-white/20 hover:bg-white/30'} text-white rounded-lg transition-colors text-sm`}
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </header>
  );
}
