import React, { useState, useEffect } from 'react';
import { Globe } from 'lucide-react';
import { useLanguage, type Language } from '~/hooks/useLanguage';
import {
  Dialog,
  DialogContent,
  DialogTrigger,
  DialogClose,
} from './ui/dialog';

const CURRENCY_STORAGE_KEY = 'nomadic-currency';
const DEFAULT_CURRENCY = 'USD';

interface LanguageOption {
  code: Language;
  label: string;
  country: string;
  currency: string;
  currencySymbol: string;
}

interface CurrencyOption {
  code: string;
  name: string;
  symbol: string;
}

// We'll define languages without translations first, then use t() function for display
const getLanguages = (t: (key: string, fallback: string) => string): LanguageOption[] => [
  { 
    code: 'en', 
    label: t('languageSelector.languages.english', 'English'), 
    country: t('languageSelector.countries.unitedStates', 'United States'), 
    currency: 'USD', 
    currencySymbol: '$' 
  },
  { 
    code: 'sp', 
    label: t('languageSelector.languages.spanish', 'Spanish'), 
    country: t('languageSelector.countries.spain', 'Spain'), 
    currency: 'EUR', 
    currencySymbol: '€' 
  },
];

// Currency options as requested
const currencies: CurrencyOption[] = [
  { code: 'USD', name: 'United States Dollar', symbol: '$' },
  { code: 'AUD', name: 'Australian Dollar', symbol: '$' },
  { code: 'BRL', name: 'Brazilian Dollar', symbol: 'R$' },
];

interface LanguageSelectorProps {
  className?: string;
}

export function LanguageSelector({ className = "" }: LanguageSelectorProps) {
  const { currentLanguage, changeLanguage, t } = useLanguage();
  const [isOpen, setIsOpen] = useState(false);
  const [activeTab, setActiveTab] = useState<'language' | 'currency'>('language');
  const [selectedCurrency, setSelectedCurrency] = useState(DEFAULT_CURRENCY);

  const languages = getLanguages(t);
  const currentOption = languages.find((lang: LanguageOption) => lang.code === currentLanguage) || languages[0];

  // Load saved currency from localStorage on mount
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedCurrency = localStorage.getItem(CURRENCY_STORAGE_KEY);
      if (savedCurrency) {
        setSelectedCurrency(savedCurrency);
      }
    }
  }, []);

  const handleLanguageChange = (language: Language) => {
    changeLanguage(language);
    setIsOpen(false);
  };

  const handleCurrencyChange = (currencyCode: string) => {
    setSelectedCurrency(currencyCode);
    // Save currency to localStorage
    if (typeof window !== 'undefined') {
      localStorage.setItem(CURRENCY_STORAGE_KEY, currencyCode);
    }
    // You can add additional currency change logic here if needed
    setIsOpen(false);
  };

  return (
    <div className={className}>
      <Dialog open={isOpen} onOpenChange={setIsOpen}>
        <DialogTrigger asChild>
          <button className="flex items-center gap-2 text-footer-bottom text-white/60 hover:text-white transition-colors cursor-pointer">
            <Globe className="h-4 w-4" />
          </button>
        </DialogTrigger>
        <DialogContent className="max-w-4xl bg-white text-black p-0 border-0">

          {/* Tabs */}
          <div className="flex border-b border-gray-200">
            <button
              onClick={() => setActiveTab('language')}
              className={`flex-1 px-6 py-4 text-left font-medium border-b-2 transition-colors ${
                activeTab === 'language'
                  ? 'text-orange-500 border-orange-500'
                  : 'text-gray-600 border-transparent hover:text-gray-800'
              }`}
            >
              {t('languageSelector.languageAndRegion', 'Language and region')}
            </button>
            <button
              onClick={() => setActiveTab('currency')}
              className={`flex-1 px-6 py-4 text-left font-medium border-b-2 transition-colors ${
                activeTab === 'currency'
                  ? 'text-orange-500 border-orange-500'
                  : 'text-gray-600 border-transparent hover:text-gray-800'
              }`}
            >
              {t('languageSelector.currency', 'Currency')}
            </button>
          </div>

          <div className="p-6">
            {activeTab === 'language' && (
              <div>
                <h3 className="text-lg font-medium text-gray-900 mb-6">
                  {t('languageSelector.chooseLanguageAndRegion', 'Choose a language and region')}
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  {languages.map((language: LanguageOption) => (
                    <button
                      key={language.code}
                      onClick={() => handleLanguageChange(language.code)}
                      className={`p-4 rounded-lg border-2 text-left transition-all hover:shadow-md ${
                        currentLanguage === language.code
                          ? 'border-orange-500 bg-orange-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <div className="font-medium text-orange-500">{language.label}</div>
                      <div className="text-gray-600 text-sm">{language.country}</div>
                    </button>
                  ))}
                </div>
              </div>
            )}

            {activeTab === 'currency' && (
              <div>
                <h3 className="text-lg font-medium text-gray-900 mb-6">
                  {t('languageSelector.chooseCurrency', 'Choose a currency')}
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  {currencies.map((currency) => (
                    <button
                      key={currency.code}
                      onClick={() => handleCurrencyChange(currency.code)}
                      className={`p-4 rounded-lg border-2 text-left transition-all hover:shadow-md ${
                        selectedCurrency === currency.code
                          ? 'border-orange-500 bg-orange-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <div className="font-medium text-gray-900">{currency.name}</div>
                      <div className="text-gray-600 text-sm">{currency.code} - {currency.symbol}</div>
                    </button>
                  ))}
                </div>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}