import { useEffect } from "react";
import { useNavigate, useLocation } from "@remix-run/react";
import { isAuthenticated } from "~/lib/api";

interface RouteProtectionProps {
  children: React.ReactNode;
  requireAuth?: boolean; // true = requires login, false = only for logged out users
}

export function RouteProtection({
  children,
  requireAuth = true,
}: RouteProtectionProps) {
  const navigate = useNavigate();
  const location = useLocation();

  useEffect(() => {
    // Check authentication after component mounts to ensure localStorage is available
    const checkAndRedirect = () => {
      const authenticated = isAuthenticated();

      if (requireAuth && !authenticated) {
        // User needs to be logged in but isn't - redirect to login
        navigate("/login");
      } else if (!requireAuth && authenticated) {
        // User shouldn't be on this page when logged in - redirect to home
        navigate("/home");
      }
    };

    // Run the check immediately and also after a small delay to catch any async updates
    checkAndRedirect();
    const timeout = setTimeout(checkAndRedirect, 100);

    return () => clearTimeout(timeout);
  }, [requireAuth, navigate, location.pathname]);

  // Check authentication for rendering
  const authenticated = isAuthenticated();

  // Don't render anything during redirect
  if (requireAuth && !authenticated) {
    return null;
  }

  if (!requireAuth && authenticated) {
    return null;
  }

  return <>{children}</>;
}

// Wrapper for protected routes that require authentication
export function ProtectedRoute({ children }: { children: React.ReactNode }) {
  return <RouteProtection requireAuth={true}>{children}</RouteProtection>;
}

// Wrapper for public routes that are accessible to everyone (logged in or not)
export function PublicRoute({ children }: { children: React.ReactNode }) {
  // Public routes should be accessible to everyone - no redirection needed
  return <>{children}</>;
}

// Wrapper for guest-only routes (login, signup) that should redirect if user is logged in
export function GuestOnlyRoute({ children }: { children: React.ReactNode }) {
  return <RouteProtection requireAuth={false}>{children}</RouteProtection>;
}
