import React from 'react';
import { useNavigate } from '@remix-run/react';
import { handleLogout, getUserData, isAuthenticated } from '~/lib/api';

interface AuthStatusProps {
  className?: string;
}

export const AuthStatus: React.FC<AuthStatusProps> = ({ className = "" }) => {
  const navigate = useNavigate();
  const [user, setUser] = React.useState<any>(null);
  const [isLoggedIn, setIsLoggedIn] = React.useState(false);

  React.useEffect(() => {
    const checkAuthStatus = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);
      
      if (authenticated) {
        const userData = getUserData();
        setUser(userData);
      }
    };

    checkAuthStatus();
    
    // Check auth status periodically
    const interval = setInterval(checkAuthStatus, 5000);
    
    return () => clearInterval(interval);
  }, []);

  const handleLogoutClick = async () => {
    try {
      await handleLogout();
      setIsLoggedIn(false);
      setUser(null);
      navigate('/login');
    } catch (error) {
      console.error('Logout error:', error);
    }
  };

  if (!isLoggedIn) {
    return (
      <div className={`flex gap-2 ${className}`}>
        <button
          onClick={() => navigate('/login')}
          className="px-4 py-2 text-sm text-gray-600 hover:text-gray-800"
        >
          Login
        </button>
        <button
          onClick={() => navigate('/signup')}
          className="px-4 py-2 text-sm bg-orange-400 text-white rounded-md hover:bg-orange-500"
        >
          Sign Up
        </button>
      </div>
    );
  }

  return (
    <div className={`flex items-center gap-3 ${className}`}>
      {user && (
        <span className="text-sm text-gray-600">
          Welcome, {user.fname}!
        </span>
      )}
      <button
        onClick={handleLogoutClick}
        className="px-4 py-2 text-sm text-red-600 hover:text-red-800"
      >
        Logout
      </button>
    </div>
  );
};

interface ProtectedRouteProps {
  children: React.ReactNode;
  fallback?: React.ReactNode;
}

export const ProtectedRoute: React.FC<ProtectedRouteProps> = ({ 
  children, 
  fallback 
}) => {
  const navigate = useNavigate();
  const [isLoggedIn, setIsLoggedIn] = React.useState<boolean | null>(null);

  React.useEffect(() => {
    const authenticated = isAuthenticated();
    setIsLoggedIn(authenticated);
    
    if (!authenticated) {
      navigate('/login');
    }
  }, [navigate]);

  if (isLoggedIn === null) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-orange-500"></div>
      </div>
    );
  }

  if (!isLoggedIn) {
    return fallback || null;
  }

  return <>{children}</>;
};