import React, { useState, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { authAPI } from '~/lib/api';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

const ResetPassword: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [formData, setFormData] = useState({
    email: ''
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  useEffect(() => {
    setIsVisible(true);
  }, []);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);

    try {
      const response = await authAPI.forgotPassword(formData.email);
      
      if (response.status) {
        // Navigate to OTP verification page for password reset
        navigate('/reset-password-otp', {
          state: {
            email: formData.email,
            mobile_number: response.mobile_number || '',
            requestType: 'reset-password'
          }
        });
      } else {
        setError(response.error || t("auth.resetPassword.errors.sendFailed", "Failed to send OTP. Please try again."));
      }
    } catch (err) {
      console.error('Forgot password error:', err);
      setError(err instanceof Error ? err.message : t("auth.resetPassword.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.resetPassword.title", "RESET PASSWORD")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.resetPassword.subtitle", "Enter your email to receive OTP on your mobile")}
          </p>
        </div>

        {/* Reset Password Card */}
        <div className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            <h2 className="text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider">{t("auth.resetPassword.cardTitle", "Reset Password")}</h2>
            
            {/* Success Message */}
            {success && (
              <div className="mb-6 p-4 bg-green-500/20 border border-green-500/30 text-green-200 rounded-xl text-sm backdrop-blur-sm text-center">
                <div className="mb-2">✅ OTP sent to your mobile!</div>
                <div>Check your mobile for the verification code.</div>
              </div>
            )}
            
            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            {!success ? (
              /* Reset Password Form */
              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Email */}
                <div className="space-y-2">
                  <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.resetPassword.emailAddress", "Email Address")}</label>
                  <input
                    type="email"
                    name="email"
                    placeholder={t("auth.resetPassword.emailPlaceholder", "Please enter your email")}
                    value={formData.email}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                  />
                </div>

                {/* Submit Button */}
                <Button
                  type="submit"
                  variant="fill-white"
                  size="lg"
                  className="w-full"
                  disabled={isLoading}
                  arrowRight={!isLoading}
                >
                  {isLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                      {t("auth.resetPassword.sending", "Sending...")}
                    </>
                  ) : (
                    t("auth.resetPassword.sendOTP", "Send OTP to Mobile")
                  )}
                </Button>

                {/* Back to Login */}
                <div className="text-center pt-4">
                  <Button
                    type="button"
                    variant="outline-white"
                    size="sm"
                    onClick={() => navigate('/login')}
                  >
                    {t("auth.resetPassword.backToLogin", "Back to Login")}
                  </Button>
                </div>
              </form>
            ) : (
              /* Success State */
              <div className="text-center space-y-6">
                <div className="text-6xl mb-4">📧</div>
                <p className="text-white/80 mb-6">
                  We've sent password reset instructions to your email address.
                </p>
                <Button
                  variant="fill-white"
                  size="lg"
                  className="w-full"
                  onClick={() => navigate('/login')}
                  arrowRight={true}
                >
                  Back to Login
                </Button>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default ResetPassword;