import { useState } from "react";
import { Button } from "~/components/ui/button";
import { Input } from "~/components/ui/input";
import { Textarea } from "~/components/ui/textarea";

export function MyProfileTab() {
  const [profileImage, setProfileImage] = useState<string | null>(null);

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onloadend = () => {
        setProfileImage(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  return (
    <div className="space-y-8 lg:space-y-12">
      {/* Profile Picture Section */}
      <div>
        <h3 className="text-heading-5 text-black mb-6">Your Profile Picture</h3>
        <div className="flex flex-col items-center">
          <div className="w-48 h-48 lg:w-64 lg:h-64 rounded-full overflow-hidden bg-neutral-100 mb-6">
            {profileImage ? (
              <img
                src={profileImage}
                alt="Profile"
                className="w-full h-full object-cover"
              />
            ) : (
              <div className="w-full h-full flex items-center justify-center text-neutral-400">
                <svg
                  width="80"
                  height="80"
                  viewBox="0 0 24 24"
                  fill="none"
                  stroke="currentColor"
                  strokeWidth="1"
                >
                  <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2" />
                  <circle cx="12" cy="7" r="4" />
                </svg>
              </div>
            )}
          </div>
          <div className="text-center">
            <label
              htmlFor="profile-upload"
              className="cursor-pointer text-[#D4845C] hover:text-[#C47550] text-body-3 uppercase underline"
            >
              Click to upload
            </label>
            <span className="text-black/60 text-body-3"> or drag and drop</span>
            <p className="text-black/60 text-body-3 mt-2">
              PNG, JPG or GIF (max. 800x400px)
            </p>
            <input
              id="profile-upload"
              type="file"
              accept="image/*"
              onChange={handleImageUpload}
              className="hidden"
            />
          </div>
        </div>
      </div>

      {/* Personal Info Section */}
      <div>
        <h3 className="text-heading-5 text-black mb-6">Personal Info</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 max-w-3xl">
          <Input
            placeholder="First name"
            defaultValue="Kunal"
            className="bg-white border-black/20 text-black placeholder:text-black/40"
          />
          <Input
            placeholder="Last name"
            defaultValue="Kumar"
            className="bg-white border-black/20 text-black placeholder:text-black/40"
          />
        </div>
      </div>

      {/* Bio Section */}
      <div>
        <h3 className="text-heading-5 text-black mb-6">Bio</h3>
        <Textarea
          placeholder="Tell us about yourself..."
          rows={6}
          className="max-w-3xl bg-white border-black/20 text-black placeholder:text-black/40 resize-none"
        />
      </div>

      {/* Price Section */}
      <div>
        <h3 className="text-heading-5 text-black mb-6">Price (USD)</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 max-w-3xl">
          <Input
            type="number"
            placeholder="0"
            defaultValue="0"
            className="bg-white border-black/20 text-black placeholder:text-black/40"
          />
          <Input
            type="number"
            placeholder="0.00"
            defaultValue="0.00"
            step="0.01"
            className="bg-white border-black/20 text-black placeholder:text-black/40"
          />
        </div>
      </div>

      {/* Save Button */}
      <div className="flex justify-center pt-4">
        <Button className="bg-[#D4845C] hover:bg-[#C47550] text-white px-12 py-6 text-body-3 uppercase">
          Save changes
        </Button>
      </div>
    </div>
  );
}
