import { useState, useEffect } from "react";
import { Button } from "~/components/ui/button";
import { SectionHeader } from "~/components/shared/SectionHeader";
import { PersonCard } from "./components/PersonCard";
import { PeopleToolbar } from "./components/PeopleToolbar";
import { peopleAPI, type Person as APIPerson } from "~/lib/api";

export interface Person {
  uuid: string;
  fname: string;
  lname: string;
  profile_pic: string;
  type: string;
  location: string;
  profile_link: string | null;
  instagram_link: string | null;
  facebook_link: string | null;
  twitter_link: string | null;
  sm_link: string | null;
  description: string;
}

export interface FilterOptions {
  locations: string[];
  roles: string[];
}

interface PeopleSectionProps {
  eyebrow?: string;
  title?: string;
  buttonText?: string;
  showFilters?: boolean;
}

export function PeopleSection({
  eyebrow = "Our Community",
  title = "Meet Our People",
  buttonText,
  showFilters = true,
}: PeopleSectionProps = {}) {
  const [people, setPeople] = useState<Person[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [filters, setFilters] = useState({
    location: "",
    role: "",
  });

  // Get unique locations and types for filtering
  const filterOptions = {
    locations: Array.from(new Set(people.map(person => person.location))).filter(Boolean),
    roles: Array.from(new Set(people.map(person => person.type))).filter(Boolean),
  };

  useEffect(() => {
    const fetchPeople = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await peopleAPI.getAll();
        
        if (response.status && response.data) {
          setPeople(response.data);
        } else {
          setError('Failed to fetch people data');
        }
      } catch (err) {
        console.error('Error fetching people:', err);
        setError(err instanceof Error ? err.message : 'Failed to fetch people data');
      } finally {
        setLoading(false);
      }
    };

    fetchPeople();
  }, []);

  const filteredPeople = people.filter((person) => {
    const locationMatch = !filters.location || person.location === filters.location;
    const roleMatch = !filters.role || person.type === filters.role;
    return locationMatch && roleMatch;
  });

  const handleFilterChange = (newFilters: typeof filters) => {
    setFilters(newFilters);
  };

  return (
    <section className="relative w-full bg-white">
      <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-8 lg:py-24">
        {/* Header */}
        <div className="mb-6 lg:mb-16">
          {/* Mobile Header Layout */}
          <div className="lg:hidden">
            {title && (
              <SectionHeader
                layout="vertical"
                eyebrow={eyebrow}
                title={title}
                className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black [&_h2]:text-[24px] [&_h2]:tracking-[-0.02em] mb-4"
              />
            )}
            {buttonText && (
              <Button
                variant="outline-black"
                size="md"
                className="border-black/30 hover:border-black/50 uppercase tracking-wider text-button-small !text-black"
                arrowRight
              >
                {buttonText}
              </Button>
            )}
          </div>

          {/* Desktop Header Layout */}
          <div className="hidden lg:flex lg:items-end lg:justify-between gap-4">
            {title && (
              <SectionHeader
                layout="vertical"
                eyebrow={eyebrow}
                title={title}
                className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black"
              />
            )}

            {/* CTA Button */}
            {buttonText && (
              <div className="flex items-center">
                <Button
                  variant="outline-black"
                  size="lg"
                  className="border-black/30 hover:border-black/50 uppercase tracking-wider font-medium !text-black"
                  arrowRight
                >
                  {buttonText}
                </Button>
              </div>
            )}
          </div>
        </div>

        {/* Toolbar - Hidden on mobile */}
        {showFilters && !loading && !error && (
          <div className="hidden lg:block">
            <PeopleToolbar
              filters={filters}
              filterOptions={filterOptions}
              onChange={handleFilterChange}
            />
          </div>
        )}

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-16">
            <div className="text-gray-500">Loading people...</div>
          </div>
        )}

        {/* Error State */}
        {error && (
          <div className="flex justify-center items-center py-16">
            <div className="text-red-500">Error: {error}</div>
          </div>
        )}

        {/* People Grid */}
        {!loading && !error && (
          <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 lg:gap-6">
            {filteredPeople.length > 0 ? (
              filteredPeople.map((person) => (
                <PersonCard 
                  key={person.uuid} 
                  photo={person.profile_pic}
                  name={person.fname}
                  surname={person.lname}
                  role={person.type}
                  location={person.location}
                  href={`/profile/${person.uuid}`}
                />
              ))
            ) : (
              <div className="col-span-full flex justify-center items-center py-16">
                <div className="text-gray-500">No people found matching your filters.</div>
              </div>
            )}
          </div>
        )}
      </div>
    </section>
  );
}
