import { useState, useEffect, useCallback } from 'react';

type Language = 'en' | 'sp';

interface LanguageData {
  [key: string]: any;
}

const LANGUAGE_STORAGE_KEY = 'nomadic-language';
const DEFAULT_LANGUAGE: Language = 'en';

export function useLanguage() {
  const [currentLanguage, setCurrentLanguage] = useState<Language>(DEFAULT_LANGUAGE);
  const [translations, setTranslations] = useState<LanguageData>({});
  const [loading, setLoading] = useState(true);

  // Load saved language from localStorage on mount
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedLanguage = localStorage.getItem(LANGUAGE_STORAGE_KEY) as Language;
      if (savedLanguage && (savedLanguage === 'en' || savedLanguage === 'sp')) {
        setCurrentLanguage(savedLanguage);
      }
    }
  }, []);

  // Load translations when language changes
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        setLoading(true);
        console.log(`Loading translations for language: ${currentLanguage}`);
        const response = await import(`../config/lang/${currentLanguage}.json`);
        const newTranslations = response.default || response;
        console.log(`Loaded translations for ${currentLanguage}:`, Object.keys(newTranslations));
        setTranslations(newTranslations);
        setLoading(false);
      } catch (error) {
        console.error(`Failed to load translations for ${currentLanguage}:`, error);
        // Fallback to English if current language fails
        if (currentLanguage !== 'en') {
          try {
            const fallback = await import('../config/lang/en.json');
            const fallbackTranslations = fallback.default || fallback;
            console.log(`Loaded fallback translations:`, Object.keys(fallbackTranslations));
            setTranslations(fallbackTranslations);
          } catch (fallbackError) {
            console.error('Failed to load fallback English translations:', fallbackError);
          }
        }
        setLoading(false);
      }
    };

    loadTranslations();
  }, [currentLanguage]);

  // Change language function
  const changeLanguage = useCallback((newLanguage: Language) => {
    if (newLanguage === currentLanguage) {
      console.log(`Language already set to ${newLanguage}, skipping change`);
      return; // Avoid unnecessary changes
    }
    
    console.log(`Changing language from ${currentLanguage} to ${newLanguage}`);
    setCurrentLanguage(newLanguage);
    if (typeof window !== 'undefined') {
      localStorage.setItem(LANGUAGE_STORAGE_KEY, newLanguage);
      // Reload the page after changing language
      window.location.reload();
    }
  }, [currentLanguage]);

  // Get translation by key with dot notation support
  const t = useCallback((key: string, defaultValue?: string): string => {
    if (!translations || Object.keys(translations).length === 0) {
      return defaultValue || key;
    }
    
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return defaultValue || key;
      }
    }
    
    return typeof value === 'string' ? value : defaultValue || key;
  }, [translations]);

  // Get nested object by key
  const getSection = useCallback((key: string): any => {
    if (!translations || Object.keys(translations).length === 0) {
      return {};
    }
    
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return {};
      }
    }
    
    return value || {};
  }, [translations]);

  return {
    currentLanguage,
    changeLanguage,
    t,
    getSection,
    loading,
    translations
  };
}

export type { Language };