// API configuration and utility functions
// const API_BASE_URL = "https://nomadicstudios.net/api";
export const API_BASE_URL = "https://nomadicstudios.net:444/api";
export const IMAGE_BASE_URL = "https://nomadicstudios.net:444";

// Types for API responses
export interface AuthResponse {
  status: boolean;
  token?: string;
  user?: {
    uuid: string;
    fname: string;
    lname: string;
    email: string;
    role: string;
  };
  expires_in?: number;
  error?: string;
  message?: string;
  mobile_number?: string; // For forgot password response
  uuid?: string; // For signup response
}

export interface LogoutResponse {
  status: boolean;
  message?: string;
  error?: string;
}

export interface OTPResponse {
  status: boolean;
  message?: string;
  error?: string;
}

export interface Studio {
  id: number;
  name: string;
  studio_type: string;
  banner_picture: string;
  description: string;
}

export interface Person {
  uuid: string;
  fname: string;
  lname: string;
  profile_pic: string;
  type: string;
  location: string;
  profile_link: string | null;
  instagram_link: string | null;
  facebook_link: string | null;
  twitter_link: string | null;
  sm_link: string | null;
  description: string;
}

export interface Event {
  event_id: number;
  location_id: number;
  user_uuid: string;
  event_type: "online" | "location";
  public_private: "public" | "private";
  stream_url: string | null;
  event_title: string;
  event_date: string;
  event_time: string;
  duration: string | null;
  address: string;
  ticket_price: number;
  ticket_url: string | null;
  thumbnail: string;
  picture?: string;
  description: string;
  event_status: number;
  created_by: number;
  updated_by: number | null;
  created_at: string;
  updated_at: string;
  channel_arn: string | null;
  stream_arn: string | null;
  playbackurl: string | null;
  stream_key: string | null;
  interested_count: number;
  going_count: number;
}

export interface News {
  id: number;
  location_id: number;
  name: string;
  tags: string;
  banner: string;
  heading_intro: string;
  para_intro: string;
  heading_elaboration: string;
  para_elaboration: string;
  conclusion_picture: string;
  heading_conclusion: string | null;
  para_conclusion: string | null;
  status: number;
  created_by: {
    id: number;
    fname: string;
    lname: string;
  };
  updated_by: number;
  created_at: string;
  updated_at: string;
  thumbnail: string;
  slug?: string;
}

export interface APIResponse<T> {
  status: boolean;
  data?: T;
  error?: string;
}

export interface Location {
  id: number;
  location: string;
  type?: string;
  coming_soon?: number;
  banner_picture?: string;
  description?: string;
}

export interface Plan {
  id: number;
  name: string;
  slug: string;
  stripe_plan: string;
  price: number;
  frequency: string;
  points: number;
  num_of_guest: number;
  advance_duration: number;
  advance_frequency: string;
  created_at: string;
  updated_at: string | null;
}

export interface Service {
  id: number;
  name: string;
  slug: string;
  description: string;
  frequency: string;
  price: string;
  status: number;
  stripe_plan: string;
  stripe_product: string;
  created_at: string;
  updated_at: string | null;
}

export interface PlansResponse {
  status: boolean;
  data: {
    plans: Plan[];
    services: Service;
  };
}

export interface Reservation {
  id: number;
  date: string;
  location: string;
  studio: string;
  price: number;
  image: string;
  [key: string]: any;
}

export interface GuestInvitation {
  id: number;
  host_uuid: string;
  guest_name: string;
  date: string;
  location: string;
  image: string;
  my_guest_or_by: string;
  [key: string]: any;
}

// Get auth token from localStorage
const getAuthToken = (): string | null => {
  if (typeof window !== "undefined") {
    return localStorage.getItem("auth_token");
  }
  return null;
};

// Set auth token in localStorage
export const setAuthToken = (token: string): void => {
  if (typeof window !== "undefined") {
    localStorage.setItem("auth_token", token);
  }
};

// Remove auth token from localStorage
export const removeAuthToken = (): void => {
  if (typeof window !== "undefined") {
    localStorage.removeItem("auth_token");
    localStorage.removeItem("user_data");
  }
};

// Store user data
export const setUserData = (user: any): void => {
  if (typeof window !== "undefined") {
    localStorage.setItem("user_data", JSON.stringify(user));
  }
};

// Get user data
export const getUserData = (): any => {
  if (typeof window !== "undefined") {
    const userData = localStorage.getItem("user_data");
    return userData ? JSON.parse(userData) : null;
  }
  return null;

  // BYPASS LOGIN FOR TESTING (commented out):
  // return {
  //   fname: 'John',
  //   lname: 'Doe',
  //   email: 'john.doe@example.com',
  //   uuid: 'test-user-123'
  // };
};

// Generic API request function
async function apiRequest<T>(
  endpoint: string,
  options: RequestInit = {}
): Promise<T> {
  try {
    const token = getAuthToken();

    const defaultHeaders: Record<string, string> = {
      "Content-Type": "application/json",
      Accept: "application/json",
    };

    if (token) {
      defaultHeaders.Authorization = `Bearer ${token}`;
    }

    console.log(`Making API request to: ${API_BASE_URL}${endpoint}`);

    const response = await fetch(`${API_BASE_URL}${endpoint}`, {
      ...options,
      headers: {
        ...defaultHeaders,
        ...options.headers,
      },
    });

    console.log(`API Response status: ${response.status}`, response);

    // Check if response is JSON
    const contentType = response.headers.get("content-type");
    if (!contentType || !contentType.includes("application/json")) {
      const textData = await response.text();
      console.error("Non-JSON response received:", textData.substring(0, 500));
      throw new Error(
        `API returned non-JSON response. This might be a network or access issue.`
      );
    }

    const data = await response.json();
    console.log(`API Response data:`, data);

    if (!response.ok) {
      throw new Error(data.error || `HTTP error! status: ${response.status}`);
    }

    return data;
  } catch (error) {
    console.error(`API request failed for ${endpoint}:`, error);
    // Re-throw with more context
    if (error instanceof Error) {
      throw new Error(`API Error: ${error.message}`);
    }
    throw new Error(`Unknown API error occurred`);
  }
}

// Authentication APIs
export const authAPI = {
  // Email/Password Login
  login: async (email: string, password: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/login", {
      method: "POST",
      body: JSON.stringify({ email, password }),
    });
  },

  // Send OTP to mobile number
  sendOTP: async (mobile_number: string): Promise<OTPResponse> => {
    return apiRequest<OTPResponse>("/send-otp", {
      method: "POST",
      body: JSON.stringify({ mobile_number }),
    });
  },

  // Verify OTP and login
  verifyOTP: async (
    mobile_number: string,
    otp_number: string,
    request_type: string = "login",
    additionalData?: {
      fname?: string;
      lname?: string;
      email?: string;
      plan?: string;
      addon_services?: string[];
      password?: string;
      confirm_password?: string;
    }
  ): Promise<AuthResponse> => {
    const payload: any = {
      request_type,
      otp_number,
      mobile_number,
    };

    // Add additional data if provided
    if (additionalData) {
      Object.assign(payload, additionalData);
    }

    return apiRequest<AuthResponse>("/verify-otp", {
      method: "POST",
      body: JSON.stringify(payload),
    });
  },

  // Logout
  logout: async (): Promise<LogoutResponse> => {
    return apiRequest<LogoutResponse>("/logout", {
      method: "POST",
    });
  },

  // Forgot Password - Send OTP to mobile
  forgotPassword: async (email: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/forgot-password", {
      method: "POST",
      body: JSON.stringify({ email }),
    });
  },

  // Reset Password
  resetPassword: async (email: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/reset-password", {
      method: "POST",
      body: JSON.stringify({ email }),
    });
  },

  // Save Profile Picture
  saveProfilePicture: async (
    uuid: string,
    image: string
  ): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/save-profile-picture", {
      method: "POST",
      body: JSON.stringify({ uuid, image }),
    });
  },

  // Save Additional Information
  saveAdditionalInfo: async (data: {
    uuid: string;
    home_city: string;
    creator_interest?: string[];
    profile_link?: string;
    instagram_link?: string;
    facebook_link?: string;
    twitter_link?: string;
    sm_link?: string;
    referred_by?: string;
  }): Promise<AuthResponse> => {
    // Send data as-is without JSON stringification
    return apiRequest<AuthResponse>("/additional-info-save", {
      method: "POST",
      body: JSON.stringify(data),
    });
  },
};

// Locations API
export const locationsAPI = {
  // Get all locations
  getLocations: async (): Promise<APIResponse<Location[]>> => {
    return apiRequest<APIResponse<Location[]>>("/locations", {
      method: "GET",
    });
  },
};

// Studios API
export const studiosAPI = {
  getByLocation: async (locationId: number): Promise<APIResponse<Studio[]>> => {
    return apiRequest<APIResponse<Studio[]>>(`/studios/${locationId}`);
  },
};

// People API
export const peopleAPI = {
  getAll: async (): Promise<APIResponse<Person[]>> => {
    return apiRequest<APIResponse<Person[]>>("/peoples");
  },
};

// Events API
export const eventsAPI = {
  getAll: async (
    filterDate?: string,
    filterLocation?: number
  ): Promise<APIResponse<Event[]>> => {
    const params = new URLSearchParams();
    if (filterDate) params.append("filterDate", filterDate);
    if (filterLocation)
      params.append("filterLocation", filterLocation.toString());

    const queryString = params.toString();
    const endpoint = queryString ? `/events?${queryString}` : "/events";

    return apiRequest<APIResponse<Event[]>>(endpoint);
  },

  getById: async (eventId: number): Promise<APIResponse<Event>> => {
    return apiRequest<APIResponse<Event>>(`/events/${eventId}`);
  },
};

// News API
export const newsAPI = {
  getAll: async (
    filterDate?: string,
    filterTags?: string
  ): Promise<APIResponse<News[]>> => {
    const params = new URLSearchParams();
    if (filterDate) params.append("filterDate", filterDate);
    if (filterTags) params.append("filterTags", filterTags);

    const queryString = params.toString();
    const endpoint = queryString ? `/news?${queryString}` : "/news";

    return apiRequest<APIResponse<News[]>>(endpoint);
  },

  getById: async (newsId: number): Promise<APIResponse<News>> => {
    return apiRequest<APIResponse<News>>(`/news/${newsId}`);
  },

  getBySlug: async (slug: string): Promise<APIResponse<News>> => {
    return apiRequest<APIResponse<News>>(`/news/slug/${slug}`);
  },
};

// Plans API
export const plansAPI = {
  getAll: async (): Promise<PlansResponse> => {
    // Use the specific plans API endpoint with port 444
    const plansUrl = "https://nomadicstudios.net:444/api/plans";
    try {
      const response = await fetch(plansUrl, {
        method: "GET",
        headers: {
          "Content-Type": "application/json",
          Accept: "application/json",
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      return data;
    } catch (error) {
      console.error("Plans API error:", error);
      throw new Error(
        `Plans API Error: ${
          error instanceof Error ? error.message : "Unknown error"
        }`
      );
    }
  },
};

// Upcoming Reservations API
export const reservationsAPI = {
  getUpcoming: async (limit?: number): Promise<APIResponse<Reservation[]>> => {
    const params = new URLSearchParams();
    if (limit) params.append("limit", limit.toString());

    const queryString = params.toString();
    const endpoint = queryString 
      ? `/upcoming-reservations?${queryString}` 
      : "/upcoming-reservations";

    return apiRequest<APIResponse<Reservation[]>>(endpoint);
  },
};

// Guest Invitations API
export const guestInvitationsAPI = {
  getAll: async (limit?: number): Promise<APIResponse<GuestInvitation[]>> => {
    const params = new URLSearchParams();
    if (limit) params.append("limit", limit.toString());

    const queryString = params.toString();
    const endpoint = queryString 
      ? `/guest-invitations?${queryString}` 
      : "/guest-invitations";

    return apiRequest<APIResponse<GuestInvitation[]>>(endpoint);
  },
};

// Auth helper functions
export const isAuthenticated = (): boolean => {
  return getAuthToken() !== null;

  // BYPASS LOGIN FOR TESTING (commented out):
  // return true;
};

export const handleAuthSuccess = (response: AuthResponse): void => {
  if (response.token) {
    setAuthToken(response.token);
    console.log("Auth token set");
  }
  if (response.user) {
    setUserData(response.user);
    console.log("User data set");
  }
  
  // Also store the complete response for account settings compatibility
  if (typeof window !== "undefined") {
    localStorage.setItem("userFullResponse", JSON.stringify(response));
    
    // Call the global saveLoginResponse function if it exists (for account settings)
    if ((window as any).saveLoginResponse) {
      (window as any).saveLoginResponse(response);
    }
  }
};

export const handleLogout = async (): Promise<void> => {
  try {
    await authAPI.logout();
  } catch (error) {
    console.error("Logout error:", error);
  } finally {
    removeAuthToken();
  }

  // BYPASS LOGIN FOR TESTING (commented out):
  // console.log('Logout bypassed for testing');
  // removeAuthToken();
};

// Payment method deletion function
export const deletePaymentMethod = async (pmid: string): Promise<{ status: boolean; message?: string }> => {
  try {
    const token = getAuthToken();
    
    const formData = new FormData();
    formData.append('pmid', pmid);

    const response = await fetch(`${API_BASE_URL}/payment-method-delete`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
      },
      body: formData
    });

    const result = await response.json();
    
    return {
      status: response.ok && result.status,
      message: result.message || (response.ok ? 'Payment method deleted successfully' : 'Failed to delete payment method')
    };
  } catch (error) {
    console.error("Delete payment method error:", error);
    return {
      status: false,
      message: 'An error occurred while deleting the payment method'
    };
  }
};
