import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { useNavigate } from "@remix-run/react";
import { ProtectedRoute } from "~/components/RouteProtection";
import { AppHeader } from "~/components/AppHeader";
import { Button } from "~/components/ui/button";
import { Trash2 } from "lucide-react";

// Stripe types
declare global {
  interface Window {
    Stripe: any;
  }
}

export const meta: MetaFunction = () => {
  return [
    { title: "Add Payment Method - Nomadics Studios" },
    {
      name: "description",
      content: "Add a new payment method to your account.",
    },
  ];
};

const STRIPE_KEY = "pk_test_51NpyhWBfA6yrUxU0IRBFtBsOIk1GXsmQlD1Z8RouELcyISQFBOSfCG5cDohyKBIZPkrHWbSFYcOWdDU2cc9fcjih00CpUxQIAN";

export default function AddPaymentMethod() {
  const navigate = useNavigate();
  const [nameOnCard, setNameOnCard] = useState("");
  const [stripe, setStripe] = useState<any>(null);
  const [elements, setElements] = useState<any>(null);
  const [cardElement, setCardElement] = useState<any>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");
  const [clientSecret, setClientSecret] = useState("");
  
  // Payment methods states
  const [paymentMethods, setPaymentMethods] = useState<any[]>([]);
  const [defaultPaymentMethod, setDefaultPaymentMethod] = useState<any>(null);
  const [isLoadingPaymentMethods, setIsLoadingPaymentMethods] = useState(false);

  // Load Stripe and initialize elements
  useEffect(() => {
    let mounted = true;
    
    const loadStripe = async () => {
      try {
        // Load Stripe script if not already loaded
        if (!window.Stripe) {
          console.log('Loading Stripe script...');
          const script = document.createElement('script');
          script.src = 'https://js.stripe.com/v3/';
          script.async = true;
          document.head.appendChild(script);
          
          await new Promise<void>((resolve, reject) => {
            script.onload = () => {
              console.log('Stripe script loaded successfully');
              resolve();
            };
            script.onerror = () => reject(new Error('Failed to load Stripe'));
          });
        }

        if (!mounted) return;

        console.log('Initializing Stripe with key:', STRIPE_KEY.substring(0, 20) + '...');
        const stripeInstance = window.Stripe(STRIPE_KEY);
        const elementsInstance = stripeInstance.elements();
        
        const cardElementInstance = elementsInstance.create('card', {
          style: {
            base: {
              fontSize: '16px',
              color: '#ffffff',
              backgroundColor: 'transparent',
              fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
              fontWeight: '400',
              lineHeight: '24px',
              padding: '12px 16px',
              '::placeholder': {
                color: 'rgba(255, 255, 255, 0.6)',
              },
              iconColor: '#ffffff',
            },
            invalid: {
              color: '#fa755a',
              iconColor: '#fa755a'
            },
            complete: {
              color: '#ffffff',
            }
          },
          hidePostalCode: true
        });

        if (!mounted) return;

        setStripe(stripeInstance);
        setElements(elementsInstance);
        setCardElement(cardElementInstance);

        // Mount card element after DOM is ready
        setTimeout(() => {
          if (!mounted) return;
          
          const cardContainer = document.getElementById('card-element');
          if (cardContainer) {
            console.log('Mounting Stripe Card Element...');
            cardElementInstance.mount('#card-element');
            
            // Add event listeners for better UX
            cardElementInstance.on('ready', () => {
              console.log('Stripe Card Element is ready and mounted');
            });
            
            cardElementInstance.on('change', (event: any) => {
              if (event.error) {
                setErrorMessage(event.error.message);
              } else {
                setErrorMessage('');
              }
            });
            
            cardElementInstance.on('focus', () => {
              console.log('Card element focused');
            });
          } else {
            console.error('Card element container not found');
          }
        }, 1000);
      } catch (error) {
        console.error('Error loading Stripe:', error);
        setErrorMessage('Failed to load payment system. Please refresh the page.');
      }
    };

    // Create setup intent for saving payment method
    const createSetupIntent = async () => {
      try {
        const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token') || '';
        
        // Check if we have a setup intent endpoint, otherwise use a dummy client secret for demo
        // In your Laravel backend, you should have an endpoint that creates and returns a setup intent
        const response = await fetch('https://nomadicstudios.net:444/api/create-setup-intent', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        });

        if (response.ok) {
          const data = await response.json();
          setClientSecret(data.client_secret);
        } else {
          // Fallback - you'll need to implement create-setup-intent endpoint in your backend
          console.warn('Setup intent endpoint not available. You need to implement /api/create-setup-intent in your backend.');
          // For now, we'll proceed without client secret and handle it differently
          setClientSecret('demo_setup_intent');
        }
      } catch (error) {
        console.error('Failed to create setup intent:', error);
        // Set a demo client secret so the form can still work
        setClientSecret('demo_setup_intent');
      }
    };

    loadStripe();
    createSetupIntent();
    
    // Cleanup function
    return () => {
      mounted = false;
    };
  }, []);

  // Fetch payment methods on mount
  useEffect(() => {
    fetchPaymentMethods();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!stripe || !cardElement) {
      setErrorMessage("Stripe is not loaded properly. Please refresh the page.");
      return;
    }

    if (!nameOnCard.trim()) {
      setErrorMessage("Name is required field.");
      return;
    }

    setIsProcessing(true);
    setErrorMessage("");

    try {
      if (clientSecret && clientSecret !== 'demo_setup_intent') {
        // Use setup intent if available
        const { setupIntent, error } = await stripe.confirmCardSetup(clientSecret, {
          payment_method: {
            card: cardElement,
            billing_details: {
              name: nameOnCard
            }
          }
        });

        if (error) {
          console.error('Stripe error:', error);
          setErrorMessage(error.message);
        } else {
          // Successfully created payment method, now save to API
          await savePaymentMethod(setupIntent.payment_method);
        }
      } else {
        // Fallback: Create payment method directly
        const { paymentMethod, error } = await stripe.createPaymentMethod({
          type: 'card',
          card: cardElement,
          billing_details: {
            name: nameOnCard
          }
        });

        if (error) {
          console.error('Stripe error:', error);
          setErrorMessage(error.message);
        } else {
          // Successfully created payment method, now save to API
          await savePaymentMethod(paymentMethod.id);
        }
      }
    } catch (error) {
      console.error('Payment processing error:', error);
      setErrorMessage("An error occurred while processing your payment method.");
    } finally {
      setIsProcessing(false);
    }
  };

  const savePaymentMethod = async (paymentMethodId: string) => {
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token') || '';
      
      const formData = new FormData();
      formData.append('name', nameOnCard);
      formData.append('token', paymentMethodId);

      const response = await fetch('https://nomadicstudios.net:444/api/add-payment-method', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();

      if (response.ok && result.status) {
        // Refresh the payment methods list
        await fetchPaymentMethods();
        // Clear the form
        setNameOnCard('');
        if (cardElement) {
          cardElement.clear();
        }
        // Show success message
        alert('Payment method added successfully!');
        // Optionally navigate back to account settings
        // navigate("/account-settings?tab=cards");
      } else {
        setErrorMessage(result.message || "Failed to save payment method. Please try again.");
      }
    } catch (error) {
      console.error('API error:', error);
      setErrorMessage("Failed to save payment method. Please try again.");
    }
  };

  // Fetch payment methods
  const fetchPaymentMethods = async () => {
    setIsLoadingPaymentMethods(true);
    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      const response = await fetch('https://nomadicstudios.net:444/api/payment-methods', {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.status) {
        setPaymentMethods(result.payment_methods || []);
        setDefaultPaymentMethod(result.default_payment_method || null);
      } else {
        console.error('Failed to fetch payment methods:', result);
      }
    } catch (error) {
      console.error('Error fetching payment methods:', error);
    } finally {
      setIsLoadingPaymentMethods(false);
    }
  };

  // Delete payment method
  const handleDeletePaymentMethod = async (pmid: string) => {
    if (!confirm('Are you sure you want to delete this payment method?')) {
      return;
    }

    try {
      const token = localStorage.getItem('authToken') || localStorage.getItem('auth_token') || localStorage.getItem('token');
      if (!token) {
        alert('Authentication token not found');
        return;
      }

      const formData = new FormData();
      formData.append('pmid', pmid);

      const response = await fetch('https://nomadicstudios.net:444/api/payment-method-delete', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
        },
        body: formData
      });

      const result = await response.json();

      if (response.ok && result.status) {
        // Refresh payment methods list
        await fetchPaymentMethods();
        alert('Payment method deleted successfully');
      } else {
        alert(result.message || 'Failed to delete payment method. Please try again.');
      }
    } catch (error) {
      console.error('Delete payment method error:', error);
      alert('An error occurred while deleting the payment method.');
    }
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-black">
        <AppHeader theme="dark" />
        
        {/* Hero Section with Solid Black Background */}
        <div className="relative pt-5">
          <div className="absolute inset-0 bg-black"></div>
          <div className="relative z-10">
            <div className="mx-auto w-full px-6" style={{ maxWidth: "1328px" }}>
              <div className="text-center py-20">
                <h1 className="text-4xl lg:text-6xl font-bold text-white mb-2 uppercase tracking-wide">
                  Join global creatives
                </h1>
              </div>
            </div>
          </div>
        </div>

        <main className="bg-black">
          <style>{`
            /* Custom styles for Stripe Elements */
            .StripeElement {
              height: 40px;
              padding: 10px 12px;
              background-color: transparent !important;
              color: white !important;
              border: none;
            }
            
            .StripeElement--focus {
              box-shadow: none;
            }
            
            .StripeElement--invalid {
              border-color: #fa755a;
            }
            
            .StripeElement--webkit-autofill {
              background-color: transparent !important;
            }
            
            #card-element .StripeElement {
              width: 100%;
              height: auto;
              padding: 12px 0;
            }
          `}</style>
          <div className="mx-auto w-full px-6 py-8" style={{ maxWidth: "1328px" }}>
            
            {/* Header Section */}
            <div className="mb-8">
              <button
                onClick={() => navigate(-1)}
                className="text-white/60 hover:text-white transition-colors mb-4 flex items-center gap-2"
              >
                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                </svg>
                <span>Back</span>
              </button>
              <h1 className="text-2xl font-bold text-white mb-2 uppercase tracking-wide">Add payment method</h1>
            </div>

            {/* Payment Form */}
            <div className="max-w-2xl mx-auto">
              <form onSubmit={handleSubmit} className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                
                {/* Error Message */}
                {errorMessage && (
                  <div className="mb-6 p-4 bg-red-500/20 border border-red-500/40 rounded-lg">
                    <p className="text-red-200 text-sm">{errorMessage}</p>
                  </div>
                )}

                {/* Name on Card */}
                <div className="mb-6">
                  <input
                    type="text"
                    placeholder="Name on the card"
                    value={nameOnCard}
                    onChange={(e) => setNameOnCard(e.target.value)}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                  />
                </div>

                {/* Stripe Card Element */}
                <div className="mb-8">
                  <label className="block text-white/60 text-sm mb-2">Card Information</label>
                  <div 
                    id="card-element"
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 focus-within:ring-2 focus-within:ring-white/40 focus-within:border-white/40 backdrop-blur-sm transition-all rounded"
                    style={{
                      minHeight: '56px',
                      display: 'block',
                      backgroundColor: 'rgba(255, 255, 255, 0.1)',
                      position: 'relative'
                    }}
                  >
                    {/* Loading state */}
                    {!cardElement && (
                      <div className="flex items-center justify-center h-full">
                        <span className="text-white/40 text-sm">Loading payment form...</span>
                      </div>
                    )}
                  </div>
                  <p className="text-white/40 text-xs mt-2">
                    Enter your card number, expiry date, and CVC. All fields are in one input.
                  </p>
                </div>

                {/* Submit Button */}
                <Button
                  type="submit"
                  variant="fill-white"
                  size="lg"
                  className="w-full uppercase tracking-wide"
                  arrowRight={true}
                  disabled={isProcessing || !stripe || !clientSecret}
                >
                  {isProcessing ? "Processing..." : "Add your card"}
                </Button>
              </form>

              {/* Existing Payment Methods */}
              {paymentMethods.length > 0 && (
                <div className="mt-8 bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
                  <h3 className="text-white text-sm font-medium mb-6 uppercase tracking-wider">Your Saved Cards</h3>
                  
                  {isLoadingPaymentMethods ? (
                    <div className="text-center py-8">
                      <div className="w-8 h-8 border-2 border-white/40 border-t-white animate-spin rounded-full mx-auto mb-4"></div>
                      <p className="text-sm text-white/60">Loading cards...</p>
                    </div>
                  ) : (
                    <div className="space-y-4">
                      {paymentMethods.map((method) => (
                        <div key={method.id} className="bg-white/5 border border-white/10 rounded-xl p-6 hover:bg-white/10 transition-colors">
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-4">
                              {/* Card Brand Icon */}
                              <div className="w-12 h-12 bg-white/20 rounded-lg flex items-center justify-center">
                                <span className="text-2xl">
                                  {method.card.brand === 'visa' ? '💳' : 
                                   method.card.brand === 'mastercard' ? '💳' : 
                                   method.card.brand === 'amex' ? '💳' : '💳'}
                                </span>
                              </div>
                              
                              <div>
                                <div className="flex items-center gap-2 mb-1">
                                  <p className="text-white font-medium uppercase">
                                    {method.card.brand} •••• {method.card.last4}
                                  </p>
                                  {defaultPaymentMethod?.id === method.id && (
                                    <span className="bg-green-500/20 text-green-400 px-2 py-0.5 rounded text-xs font-medium uppercase">
                                      Default
                                    </span>
                                  )}
                                </div>
                                <p className="text-white/60 text-sm">
                                  {method.billing_details.name || 'No name'}
                                </p>
                                <p className="text-white/40 text-xs mt-1">
                                  Expires {String(method.card.exp_month).padStart(2, '0')}/{method.card.exp_year}
                                </p>
                              </div>
                            </div>
                            
                            <button
                              onClick={() => handleDeletePaymentMethod(method.id)}
                              className="text-white/40 hover:text-red-400 transition-colors p-2"
                              title="Delete payment method"
                            >
                              <Trash2 className="w-5 h-5" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}
            </div>
          </div>
        </main>
      </div>
    </ProtectedRoute>
  );
}