import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState } from "react";
import { useNavigate, useSearchParams } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";

export const meta: MetaFunction = () => {
  return [
    { title: "Select Equipment - Nomadics Studios" },
    {
      name: "description",
      content: "Select equipment you want to rent for your studio session.",
    },
  ];
};

interface Equipment {
  id: string;
  name: string;
  price: string;
  image: string;
  description?: string;
  isRequired?: boolean;
  isSelected?: boolean;
}

export default function BookEquipment() {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const studio = searchParams.get("studio");
  
  const [selectedEquipment, setSelectedEquipment] = useState<string[]>([]);
  const [bookingDuration, setBookingDuration] = useState("6H");
  
  const equipment: Equipment[] = [
    {
      id: "pioneer-cdj3000",
      name: "Pioneer CDJ3000",
      price: "25pts",
      image: "/images/equipment/pioneer-cdj3000.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum...",
      isSelected: true,
    },
    {
      id: "pioneer-djm-v10",
      name: "Pioneer DJM-V10", 
      price: "25pts",
      image: "/images/equipment/pioneer-djm-v10.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum...",
    },
    {
      id: "solid-state-aws924",
      name: "Solid State AWS 924",
      price: "25pts",
      image: "/images/equipment/solid-state-aws924.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum...",
      isRequired: true,
    },
    {
      id: "roland-juno-60",
      name: "Roland JUNO-60",
      price: "25pts", 
      image: "/images/equipment/roland-juno-60.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum...",
    },
    {
      id: "yamaha-dx6",
      name: "Yamaha DX 6",
      price: "25pts",
      image: "/images/equipment/yamaha-dx6.png", 
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum...",
    },
  ];

  const handleEquipmentSelect = (equipmentId: string) => {
    setSelectedEquipment(prev => {
      if (prev.includes(equipmentId)) {
        return prev.filter(id => id !== equipmentId);
      } else {
        return [...prev, equipmentId];
      }
    });
  };

  const handleSkip = () => {
    navigate("/book-support");
  };

  const handleContinue = () => {
    const params = new URLSearchParams();
    if (studio) params.set("studio", studio);
    if (selectedEquipment.length > 0) {
      params.set("equipment", selectedEquipment.join(","));
    }
    navigate(`/book-support?${params.toString()}`);
  };

  // Calculate totals
  const selectedItems = equipment.filter(item => 
    selectedEquipment.includes(item.id) || item.isSelected
  );
  const studioPrice = 25; // Base studio price
  const equipmentPrice = selectedItems.reduce((total, item) => 
    total + parseInt(item.price.replace('pts', '')), 0
  );
  const supportPrice = 25; // Base support price
  const totalPrice = studioPrice + equipmentPrice + supportPrice;

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />
      
      {/* Hero Section */}
<section className="relative bg-black text-white py-16">
          <div className="absolute inset-0 bg-black/20" />
        <div className="relative z-10 max-w-7xl mx-auto px-4">
          <h1 className="text-4xl md:text-5xl font-light mb-4">
            Create something with us!
          </h1>
        </div>
      </section>

      <main className="max-w-7xl mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          
          {/* Left Column - Equipment Selection */}
          <div className="lg:col-span-2">
            <div className="mb-8">
              <h2 className="text-2xl font-light text-black mb-8">
                2. Select equipment you want to rent
              </h2>
            </div>

            <div className="space-y-6">
              {equipment.map((item) => (
                <div
                  key={item.id}
                  className={`border-2 transition-all cursor-pointer ${
                    selectedEquipment.includes(item.id) || item.isSelected
                      ? "border-black bg-gray-50"
                      : "border-gray-200 hover:border-gray-300"
                  }`}
                  onClick={() => handleEquipmentSelect(item.id)}
                >
                  <div className="p-6">
                    <div className="flex items-start gap-6">
                      {/* Checkbox */}
                      <div className="flex-shrink-0 mt-1">
                        <div
                          className={`w-6 h-6 border-2 transition-colors ${
                            selectedEquipment.includes(item.id) || item.isSelected
                              ? "border-black bg-black"
                              : "border-gray-400"
                          }`}
                        >
                          {(selectedEquipment.includes(item.id) || item.isSelected) && (
                            <svg
                              className="w-full h-full text-white p-1"
                              fill="currentColor"
                              viewBox="0 0 20 20"
                            >
                              <path
                                fillRule="evenodd"
                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                clipRule="evenodd"
                              />
                            </svg>
                          )}
                        </div>
                      </div>

                      {/* Content */}
                      <div className="flex-1">
                        <div className="flex items-start justify-between mb-4">
                          <div>
                            <h3 className="text-xl font-medium text-black mb-2">
                              {item.name}
                            </h3>
                            {item.isRequired && (
                              <p className="text-sm text-red-600 mb-2">
                                Requires Solid state engineer
                              </p>
                            )}
                            {item.description && (
                              <p className="text-sm text-gray-600">
                                {item.description}
                              </p>
                            )}
                          </div>
                          <span className="text-lg font-medium text-black">
                            {item.price}
                          </span>
                        </div>
                      </div>

                      {/* Equipment Image */}
                      <div className="flex-shrink-0">
                        <img
                          src={item.image}
                          alt={item.name}
                          className="w-32 h-24 object-cover"
                          onError={(e) => {
                            e.currentTarget.src = "https://via.placeholder.com/128x96/f3f4f6/9ca3af?text=Equipment";
                          }}
                        />
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Right Column - Booking Summary */}
          <div className="lg:col-span-1">
            <div className="bg-gray-50 p-6 sticky top-8">
              <div className="text-center mb-8">
                <div className="text-4xl font-light text-black mb-2">
                  {bookingDuration}
                </div>
              </div>

              <div className="space-y-4 mb-8">
                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Where
                  </label>
                  <input
                    type="text"
                    className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                    placeholder="Bali"
                    defaultValue="Bali"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Start
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="10:30"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    End
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="16:30"
                    />
                  </div>
                </div>
              </div>

              {/* Price Breakdown */}
              <div className="space-y-3 mb-6">
                <div className="flex justify-between text-sm">
                  <span>Studio</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Equipment</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Support</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <hr className="border-gray-300" />
                <div className="flex justify-between font-medium">
                  <span>Total</span>
                  <span>{totalPrice * 6} pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Points left</span>
                  <span>765 pts</span>
                </div>
              </div>

              <div className="space-y-3">
                <Button
                  onClick={handleSkip}
                  variant="outline-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-md"
                >
                  Skip
                </Button>

                <Button
                  onClick={handleContinue}
                  variant="fill-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg"
                >
                  Continue
                </Button>
              </div>
            </div>
          </div>
        </div>
      </main>

      {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}