import type { LoaderFunctionArgs, MetaFunction } from "@remix-run/node";
import { useLoaderData } from "@remix-run/react";
import { PlanYourVisitSection } from "~/components/pages/locations/PlanYourVisitSection";
import { BookingForm } from "~/components/shared/BookingForm";
import {
  StudioHeroSection,
  StudioInfoSection,
  CrewSection,
  EquipmentSection,
} from "~/components/pages/book-studio";
import { getStudioPageData } from "~/config/book-studio";

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadics" },
      { name: "description", content: "The requested studio was not found." },
    ];
  }

  return [
    { title: data.studioPageData.meta.title },
    { name: "description", content: data.studioPageData.meta.description },
  ];
};

export async function loader({ params }: LoaderFunctionArgs) {
  const { studioId } = params;

  if (!studioId) {
    throw new Response("Studio not found", { status: 404 });
  }

  const [location, ...studioTypeParts] = studioId.split("-");
  const studioType = studioTypeParts.join(" ").toUpperCase();

  const studioPageData = getStudioPageData(studioId, location, studioType);

  return Response.json({
    studioId,
    location,
    studioType,
    studioPageData,
  });
}


export default function BookStudio() {
  const { studioPageData } = useLoaderData<typeof loader>();

  return (
    <div className="min-h-screen bg-white">
      <div className="mx-auto max-w-7xl px-4 lg:px-14 py-4 lg:py-10">
        {/* Photo Gallery */}
        <StudioHeroSection data={studioPageData.hero} />

        <div className="grid grid-cols-3 gap-2 mt-8">
          {/* Left column - Studio info (2/3 width) */}
          <div className="col-span-2 space-y-6 pr-8">
            <StudioInfoSection data={studioPageData.hero} />
            <CrewSection data={studioPageData.crew} />
            <EquipmentSection data={studioPageData.equipment} />
          </div>

          {/* Right column - Booking form (1/3 width) */}
          <div>
            <BookingForm />
          </div>
        </div>
      </div>

      <PlanYourVisitSection data={studioPageData.planVisit} />
    </div>
  );
}
