import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState } from "react";
import { useNavigate } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";

export const meta: MetaFunction = () => {
  return [
    { title: "Book Studio - Nomadics Studios" },
    {
      name: "description",
      content: "Select the type of studio that best fits your creative needs.",
    },
  ];
};

interface StudioType {
  id: string;
  name: string;
  price: string;
  image: string;
  description?: string;
  isBooked?: boolean;
}

export default function BookStudio() {
  const navigate = useNavigate();
  const [selectedStudio, setSelectedStudio] = useState<string>("");
  const [bookingDuration, setBookingDuration] = useState("3H");

  const studioTypes: StudioType[] = [
    {
      id: "visual-art",
      name: "Visual Art",
      price: "20pts",
      image: "/images/studios/visual-art.png",
    },
    {
      id: "video",
      name: "Video",
      price: "20pts", 
      image: "/images/studios/video-studio.png",
      description: "Studio is booked at this time please search please select different time. Check Availability",
      isBooked: true,
    },
    {
      id: "music",
      name: "Music", 
      price: "20pts",
      image: "/images/studios/music-studio.png",
      description: "Required minimum time to book a studio is 6hrs",
    },
    {
      id: "fashion",
      name: "Fashion",
      price: "20pts", 
      image: "/images/studios/fashion-studio.png",
    },
  ];

  const handleStudioSelect = (studioId: string) => {
    setSelectedStudio(studioId);
  };

  const handleSkip = () => {
    navigate("/book-equipment");
  };

  const handleContinue = () => {
    if (selectedStudio) {
      navigate(`/book-equipment?studio=${selectedStudio}`);
    }
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />
      
      {/* Hero Section */}
      <section className="relative bg-black text-white py-16">
        <div className="absolute inset-0 bg-black/20" />
        <div className="relative z-10 max-w-7xl mx-auto px-4">
          <h1 className="text-4xl md:text-5xl font-light mb-4">
            Create something with us!
          </h1>
        </div>
      </section>

      <main className="max-w-7xl mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          
          {/* Left Column - Studio Selection */}
          <div className="lg:col-span-2">
            <div className="mb-8">
              <h2 className="text-2xl font-light text-black mb-8">
                1. Select type of studio
              </h2>
            </div>

            <div className="space-y-6">
              {studioTypes.map((studio) => (
                <div
                  key={studio.id}
                  className={`border-2 transition-all cursor-pointer ${
                    selectedStudio === studio.id
                      ? "border-black bg-gray-50"
                      : "border-gray-200 hover:border-gray-300"
                  }`}
                  onClick={() => handleStudioSelect(studio.id)}
                >
                  <div className="p-6">
                    <div className="flex items-start gap-6">
                      {/* Checkbox */}
                      <div className="flex-shrink-0 mt-1">
                        <div
                          className={`w-6 h-6 border-2 transition-colors ${
                            selectedStudio === studio.id
                              ? "border-black bg-black"
                              : "border-gray-400"
                          }`}
                        >
                          {selectedStudio === studio.id && (
                            <svg
                              className="w-full h-full text-white p-1"
                              fill="currentColor"
                              viewBox="0 0 20 20"
                            >
                              <path
                                fillRule="evenodd"
                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                clipRule="evenodd"
                              />
                            </svg>
                          )}
                        </div>
                      </div>

                      {/* Content */}
                      <div className="flex-1">
                        <div className="flex items-start justify-between mb-4">
                          <div>
                            <h3 className="text-xl font-medium text-black mb-2">
                              {studio.name}
                            </h3>
                            {studio.description && (
                              <p className={`text-sm ${
                                studio.isBooked ? "text-red-600" : "text-gray-600"
                              }`}>
                                {studio.description}
                              </p>
                            )}
                          </div>
                          <span className="text-lg font-medium text-black">
                            {studio.price}
                          </span>
                        </div>
                      </div>

                      {/* Studio Image */}
                      <div className="flex-shrink-0">
                        <img
                          src={studio.image}
                          alt={studio.name}
                          className="w-32 h-24 object-cover"
                        />
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Right Column - Booking Summary */}
          <div className="lg:col-span-1">
            <div className="bg-gray-50 p-6 sticky top-8">
              <div className="text-center mb-8">
                <div className="text-4xl font-light text-black mb-2">
                  {bookingDuration}
                </div>
              </div>

              <div className="space-y-4 mb-8">
                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Where
                  </label>
                  <input
                    type="text"
                    className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                    placeholder="Bali"
                    defaultValue="Bali"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Start
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="10:30"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    End
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="13:30"
                    />
                  </div>
                </div>
              </div>

              <div className="space-y-3">
                <Button
                  onClick={handleSkip}
                  variant="outline-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-md"
                >
                  Skip
                </Button>
              <Button
                  onClick={handleContinue}
                  variant="fill-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg"
                  disabled={!selectedStudio}
                >
                  Continue
                </Button>
              </div>
            </div>
          </div>
        </div>
      </main>

      {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}