import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState } from "react";
import { useNavigate, useSearchParams } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";

export const meta: MetaFunction = () => {
  return [
    { title: "Select Additional Support - Nomadics Studios" },
    {
      name: "description",
      content: "Select additional support for your studio session.",
    },
  ];
};

interface SupportPerson {
  id: string;
  name: string;
  role: string;
  price: string;
  image: string;
  description: string;
  specialties: string[];
  isRequired?: boolean;
}

export default function BookSupport() {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const studio = searchParams.get("studio");
  const equipment = searchParams.get("equipment");
  
  const [selectedSupport, setSelectedSupport] = useState<string[]>([]);
  const [selectedLanguage, setSelectedLanguage] = useState("English");
  
  const supportStaff: SupportPerson[] = [
    {
      id: "jason-goodman",
      name: "Jason Goodman",
      role: "Audio Engineer",
      price: "25pts",
      image: "/images/support/jason-goodman.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum. Morbi et eleifend ipsum velit. Duis rutrum magna vestibulum. Quisque vulputate lorem arcu, ut scelerisque mi. Donec sollicitudin nunc mauris, eget rutrum purus imperdiet sit amet. Vestibulum convallis ullamcorper convallis.",
      specialties: ["Post mixer", "Mix designer", "Percival"],
      isRequired: true,
    },
    {
      id: "bring-engineer",
      name: "Bring your own engineer",
      role: "Audio Engineer",
      price: "50 pts",
      image: "/images/support/bring-engineer.png",
      description: "",
      specialties: [],
      isRequired: true,
    },
    {
      id: "allison-lipshutz",
      name: "Allison Lipshutz",
      role: "Mastering Engineer",
      price: "25pts",
      image: "/images/support/allison-lipshutz.png",
      description: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tincidunt ipsum ut aliquet rutrum. Morbi et eleifend ipsum velit. Duis rutrum magna vestibulum. Quisque vulputate lorem arcu, ut scelerisque mi. Donec sollicitudin nunc mauris, eget rutrum purus imperdiet sit amet. Vestibulum convallis ullamcorper convallis.",
      specialties: ["Post mixer", "Mix designer", "Percival"],
    },
  ];

  const languages = ["English", "Spanish", "French", "German", "Portuguese"];

  const handleSupportSelect = (supportId: string) => {
    setSelectedSupport(prev => {
      if (prev.includes(supportId)) {
        return prev.filter(id => id !== supportId);
      } else {
        return [...prev, supportId];
      }
    });
  };

  const handleSkip = () => {
    navigate("/book-confirmation");
  };

  const handleContinue = () => {
    const params = new URLSearchParams();
    if (studio) params.set("studio", studio);
    if (equipment) params.set("equipment", equipment);
    if (selectedSupport.length > 0) {
      params.set("support", selectedSupport.join(","));
    }
    navigate(`/book-confirmation?${params.toString()}`);
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-white text-black">
        <AppHeader theme="dark" />
      
      {/* Hero Section */}
      <section className="relative bg-black text-white py-16">
        <div className="absolute inset-0 bg-black/20" />
        <div className="relative z-10 max-w-7xl mx-auto px-4">
          <h1 className="text-4xl md:text-5xl font-bold mb-4">
            Create something with us!
          </h1>
        </div>
      </section>

      <main className="max-w-7xl mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          
          {/* Left Column - Support Selection */}
          <div className="lg:col-span-2">
            <div className="mb-8 flex items-center justify-between">
              <h2 className="text-2xl font-light text-black">
                3. Select additional support
              </h2>
              
              {/* Language Selector */}
              <div className="flex items-center gap-2">
                <select 
                  value={selectedLanguage}
                  onChange={(e) => setSelectedLanguage(e.target.value)}
                  className="p-2 border border-gray-300 bg-white text-black focus:border-black focus:outline-none"
                >
                  {languages.map(lang => (
                    <option key={lang} value={lang}>{lang}</option>
                  ))}
                </select>
              </div>
            </div>

            <div className="space-y-6">
              {supportStaff.map((person) => (
                <div
                  key={person.id}
                  className={`border-2 transition-all cursor-pointer ${
                    selectedSupport.includes(person.id)
                      ? "border-black bg-gray-50"
                      : "border-gray-200 hover:border-gray-300"
                  }`}
                  onClick={() => handleSupportSelect(person.id)}
                >
                  <div className="p-6">
                    <div className="flex items-start gap-6">
                      {/* Checkbox */}
                      <div className="flex-shrink-0 mt-1">
                        <div
                          className={`w-6 h-6 border-2 transition-colors ${
                            selectedSupport.includes(person.id)
                              ? "border-black bg-black"
                              : "border-gray-400"
                          }`}
                        >
                          {selectedSupport.includes(person.id) && (
                            <svg
                              className="w-full h-full text-white p-1"
                              fill="currentColor"
                              viewBox="0 0 20 20"
                            >
                              <path
                                fillRule="evenodd"
                                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                                clipRule="evenodd"
                              />
                            </svg>
                          )}
                        </div>
                      </div>

                      {/* Content */}
                      <div className="flex-1">
                        <div className="flex items-start justify-between mb-4">
                          <div>
                            <h3 className="text-xl font-medium text-black mb-1">
                              {person.name}
                            </h3>
                            <p className="text-sm text-gray-600 mb-2">
                              {person.role}
                            </p>
                            {person.isRequired && (
                              <p className="text-sm text-red-600 mb-3">
                                Required for Solid state AWS 924
                              </p>
                            )}
                            
                            {person.description && (
                              <div className="mb-4">
                                <h4 className="font-medium text-black mb-2">Description</h4>
                                <p className="text-sm text-gray-600">
                                  {person.description}
                                </p>
                              </div>
                            )}
                            
                            {person.specialties.length > 0 && (
                              <div>
                                <h4 className="font-medium text-black mb-2">Specialties</h4>
                                <ul className="list-disc list-inside text-sm text-gray-600 space-y-1">
                                  {person.specialties.map((specialty, index) => (
                                    <li key={index}>{specialty}</li>
                                  ))}
                                </ul>
                              </div>
                            )}
                          </div>
                          <span className="text-lg font-medium text-black">
                            {person.price}
                          </span>
                        </div>
                      </div>

                      {/* Person Image */}
                      <div className="flex-shrink-0">
                        <img
                          src={person.image}
                          alt={person.name}
                          className="w-32 h-32 object-cover"
                          onError={(e) => {
                            e.currentTarget.src = "https://via.placeholder.com/128x128/e5e7eb/6b7280?text=" + person.name.charAt(0);
                          }}
                        />
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Right Column - Booking Summary */}
          <div className="lg:col-span-1">
            <div className="bg-gray-50 p-6 sticky top-8">
              <div className="text-center mb-8">
                <div className="text-4xl font-light text-black mb-2">
                  6H
                </div>
              </div>

              <div className="space-y-4 mb-8">
                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Where
                  </label>
                  <input
                    type="text"
                    className="w-full p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                    placeholder="Bali"
                    defaultValue="Bali"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    Start
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="10:30"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2 uppercase tracking-wider">
                    End
                  </label>
                  <div className="grid grid-cols-2 gap-2">
                    <input
                      type="date"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="2023-08-21"
                    />
                    <input
                      type="time"
                      className="p-3 border-b-2 border-gray-300 bg-transparent text-black focus:border-black focus:outline-none transition-colors"
                      defaultValue="16:30"
                    />
                  </div>
                </div>
              </div>

              {/* Price Breakdown */}
              <div className="space-y-3 mb-6">
                <div className="flex justify-between text-sm">
                  <span>Studio</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Equipment</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Support</span>
                  <span>25pts x 6 hours = 150 pts</span>
                </div>
                <hr className="border-gray-300" />
                <div className="flex justify-between font-medium">
                  <span>Total</span>
                  <span>450 pts</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Points left</span>
                  <span>775 pts</span>
                </div>
              </div>

              <div className="space-y-3">
                <Button
                  onClick={handleSkip}
                  variant="outline-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-md"
                >
                  Skip
                </Button>

                <Button
                  onClick={handleContinue}
                  variant="fill-black"
                  size="md"
                  className="w-full uppercase tracking-wider font-medium transition-all duration-200 hover:shadow-lg"
                >
                  Continue
                </Button>
              </div>
            </div>
          </div>
        </div>
      </main>

      {/* <AppFooter /> */}
      </div>
    </ProtectedRoute>
  );
}