import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect, useRef } from "react";
import { useNavigate } from "@remix-run/react";
import { ProtectedRoute } from "~/components/RouteProtection";
import { AppHeader } from "~/components/AppHeader";
import { Button } from "~/components/ui/button";
import { Send, Hash, Users, Settings, Plus, Search } from "lucide-react";

export const meta: MetaFunction = () => {
  return [
    { title: "Discuss - Nomadics Studios" },
    {
      name: "description",
      content: "Connect and collaborate with global creatives.",
    },
  ];
};

interface Channel {
  id: string;
  name: string;
  type: "channel" | "dm";
  color: string;
  unread?: number;
}

interface Message {
  id: string;
  user: string;
  avatar: string;
  message: string;
  timestamp: string;
  isCurrentUser?: boolean;
}

export default function Discuss() {
  const navigate = useNavigate();
  const [selectedChannel, setSelectedChannel] = useState("open-channel");
  const [messageInput, setMessageInput] = useState("");
  const [searchQuery, setSearchQuery] = useState("");
  const [showEmojiPicker, setShowEmojiPicker] = useState(false);

  const channels: Channel[] = [
    { id: "open-channel", name: "Open Channel", type: "channel", color: "bg-emerald-500" },
    { id: "music", name: "Music", type: "channel", color: "bg-purple-500" },
    { id: "video", name: "Video", type: "channel", color: "bg-blue-500" },
    { id: "photography", name: "Photography", type: "channel", color: "bg-purple-500" },
    { id: "art", name: "Art", type: "channel", color: "bg-blue-500" },
    { id: "makers", name: "Makers", type: "channel", color: "bg-purple-500" },
    { id: "fashion", name: "Fashion", type: "channel", color: "bg-emerald-500" },
    { id: "promo", name: "Promo", type: "channel", color: "bg-blue-500" },
    { id: "collab", name: "Collab Opportunities", type: "channel", color: "bg-purple-500" },
    { id: "announcements", name: "Announcements", type: "channel", color: "bg-emerald-500" },
  ];

  const messages: Message[] = [
    {
      id: "1",
      user: "You",
      avatar: "/images/profile-avatar.jpg",
      message: "gfhgh",
      timestamp: "2:43 AM, 14th Dec",
      isCurrentUser: true,
    },
  ];

  const handleSendMessage = () => {
    if (messageInput.trim()) {
      // Handle sending message
      console.log("Sending message:", messageInput);
      setMessageInput("");
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  const handleEmojiClick = (emoji: string) => {
    setMessageInput(prev => prev + emoji);
    setShowEmojiPicker(false);
  };

  const commonEmojis = ["😊", "😂", "❤️", "👍", "👎", "😍", "🤔", "😭", "😎", "🔥", "✨", "💯"];
  const emojiPickerRef = useRef<HTMLDivElement>(null);

  // Close emoji picker when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (emojiPickerRef.current && !emojiPickerRef.current.contains(event.target as Node)) {
        setShowEmojiPicker(false);
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const selectedChannelData = channels.find(c => c.id === selectedChannel);

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-black flex flex-col">
        <AppHeader theme="dark" />
        
        <main className="flex-1 flex pt-10">
          {/* Sidebar - Channels */}
          <div className="w-80 bg-white/5 border-r border-white/10 flex flex-col">
            {/* Sidebar Header */}
            <div className="p-6 border-b border-white/10">
              <div className="flex items-center justify-between mb-4">
                <h1 className="text-white font-bold text-lg uppercase tracking-wide">Your Channels</h1>
                {/* <button className="text-white/60 hover:text-white transition-colors">
                  <Plus className="w-5 h-5" />
                </button> */}
              </div>
              
              {/* Search */}
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-white/40" />
                <input
                  type="text"
                  placeholder="Search channels..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-full pl-10 pr-4 py-2 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 text-sm focus:outline-none focus:ring-2 focus:ring-white/40 focus:border-white/40"
                />
              </div>
            </div>

            {/* Channels List */}
            <div className="flex-1 overflow-y-auto">
              <div className="p-4 space-y-2">
                {channels
                  .filter(channel => 
                    channel.name.toLowerCase().includes(searchQuery.toLowerCase())
                  )
                  .map((channel) => (
                    <button
                      key={channel.id}
                      onClick={() => setSelectedChannel(channel.id)}
                      className={`w-full flex items-center gap-3 p-3 rounded-lg text-left transition-colors ${
                        selectedChannel === channel.id
                          ? "bg-white/10 text-white"
                          : "text-white/70 hover:text-white hover:bg-white/5"
                      }`}
                    >
                      <div className={`w-8 h-8 ${channel.color} rounded-full flex items-center justify-center flex-shrink-0`}>
                        {channel.type === "channel" ? (
                          <Hash className="w-4 h-4 text-white" />
                        ) : (
                          <Users className="w-4 h-4 text-white" />
                        )}
                      </div>
                      <span className="font-medium text-sm uppercase tracking-wide">
                        {channel.name}
                      </span>
                      {channel.unread && (
                        <div className="ml-auto bg-red-500 text-white text-xs px-2 py-1 rounded-full">
                          {channel.unread}
                        </div>
                      )}
                    </button>
                  ))}
              </div>
            </div>

          </div>

          {/* Main Chat Area */}
          <div className="flex-1 flex flex-col">
            {/* Chat Header */}
            <div className="p-6 border-b border-white/10 bg-white/5">
              <div className="flex items-center gap-3">
                <div className={`w-8 h-8 ${selectedChannelData?.color} rounded-full flex items-center justify-center`}>
                  <Hash className="w-4 h-4 text-white" />
                </div>
                <div>
                  <h2 className="text-white font-bold text-lg uppercase tracking-wide">
                    {selectedChannelData?.name || "Open Channel"}
                  </h2>
                  <p className="text-white/60 text-sm">
                    Connect and collaborate with global creatives
                  </p>
                </div>
              </div>
            </div>

            {/* Messages Area */}
            <div className="flex-1 overflow-y-auto p-6">
              <div className="space-y-4">
                {messages.length > 0 ? (
                  messages.map((message) => (
                    <div
                      key={message.id}
                      className={`flex gap-3 ${message.isCurrentUser ? "justify-end" : ""}`}
                    >
                      {!message.isCurrentUser && (
                        <img
                          src={message.avatar}
                          alt={message.user}
                          className="w-10 h-10 rounded-full object-cover flex-shrink-0"
                        />
                      )}
                      <div className={`max-w-xs lg:max-w-md ${message.isCurrentUser ? "order-first" : ""}`}>
                        <div className="flex items-center gap-2 mb-1">
                          <span className={`text-sm font-medium ${
                            message.isCurrentUser ? "text-orange-400" : "text-white"
                          }`}>
                            {message.user}
                          </span>
                          <span className="text-xs text-white/60">{message.timestamp}</span>
                        </div>
                        <div className={`p-3 rounded-lg ${
                          message.isCurrentUser
                            ? "bg-white text-black"
                            : "bg-white/10 text-white"
                        }`}>
                          {message.message}
                        </div>
                      </div>
                      {message.isCurrentUser && (
                        <img
                          src={message.avatar}
                          alt={message.user}
                          className="w-10 h-10 rounded-full object-cover flex-shrink-0"
                        />
                      )}
                    </div>
                  ))
                ) : (
                  <div className="flex-1 flex items-center justify-center">
                    <div className="text-center">
                      <div className="w-16 h-16 bg-white/10 rounded-full flex items-center justify-center mx-auto mb-4">
                        <Hash className="w-8 h-8 text-white/40" />
                      </div>
                      <p className="text-white/60 text-sm mb-2">Welcome to #{selectedChannelData?.name || "Open Channel"}!</p>
                      <p className="text-white/40 text-xs">Start the conversation by sending a message.</p>
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Message Input */}
            <div className="p-4 border-t border-white/10 bg-white/5 relative">
              {/* Emoji Picker */}
              {showEmojiPicker && (
                <div 
                  ref={emojiPickerRef}
                  className="absolute bottom-full left-4 mb-2 bg-white border border-gray-200 rounded-lg p-3 shadow-lg z-10"
                >
                  <div className="grid grid-cols-6 gap-2">
                    {commonEmojis.map((emoji, index) => (
                      <button
                        key={index}
                        onClick={() => handleEmojiClick(emoji)}
                        className="text-lg hover:bg-gray-100 p-2 rounded transition-colors"
                      >
                        {emoji}
                      </button>
                    ))}
                  </div>
                </div>
              )}
              
              <div className="flex items-stretch gap-3">
                <button 
                  onClick={() => setShowEmojiPicker(!showEmojiPicker)}
                  className="bg-white hover:bg-white/90 text-black rounded-lg transition-colors flex-shrink-0 px-4 flex items-center justify-center font-medium text-sm uppercase tracking-wide"
                  style={{ minHeight: "40px" }}
                >
                  Emoji
                </button>
                <div className="flex-1 bg-white/10 border border-white/20 rounded-lg focus-within:ring-2 focus-within:ring-white/40 focus-within:border-white/40">
                  <textarea
                    placeholder="Type your message"
                    value={messageInput}
                    onChange={(e) => setMessageInput(e.target.value)}
                    onKeyPress={handleKeyPress}
                    rows={1}
                    className="w-full px-3 py-2 bg-transparent text-white placeholder-white/60 resize-none focus:outline-none text-sm"
                    style={{ minHeight: "40px", maxHeight: "100px" }}
                  />
                </div>
                <button
                  onClick={handleSendMessage}
                  disabled={!messageInput.trim()}
                  className="bg-white hover:bg-white/90 text-black rounded-lg transition-colors flex-shrink-0 px-4 flex items-center justify-center font-medium text-sm uppercase tracking-wide disabled:opacity-50 disabled:cursor-not-allowed"
                  style={{ minHeight: "40px" }}
                >
                  <Send className="w-4 h-4 mr-1" />
                  Send
                </button>
              </div>
            </div>
          </div>
        </main>
      </div>
    </ProtectedRoute>
  );
}