import React, { useState, useEffect } from "react";
import type { MetaFunction } from "@remix-run/node";
import { Link, useParams, useLocation } from "@remix-run/react";
import { Button } from "~/components/ui/button";
import { PublicRoute } from "~/components/RouteProtection";
import { eventsAPI, type Event } from "~/lib/api";
import { useLanguage } from "~/hooks/useLanguage";

// Utility function to strip HTML tags from text
const stripHtmlTags = (html: string) => {
  if (!html) return '';
  // Remove HTML tags using regex
  return html.replace(/<[^>]*>/g, '').trim();
};

export const meta: MetaFunction = ({ params }) => {
  return [
    { title: `${params.eventId || 'Event'} Details - Nomadics Studios` },
    {
      name: "description",
      content: "Join us for an unforgettable experience at Nomadics Studios events.",
    },
  ];
};



export default function EventDetailsPage() {
  const { t } = useLanguage();
  const { eventId } = useParams();
  const location = useLocation();
  const [event, setEvent] = useState<Event | null>(location.state?.event || null);
  const [loading, setLoading] = useState(!location.state?.event);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    // Only fetch if we don't have event data from navigation state
    if (!event && eventId) {
      const fetchEvent = async () => {
        try {
          setLoading(true);
          const response = await eventsAPI.getById(parseInt(eventId));
          
          if (response.status && response.data) {
            setEvent(response.data);
          } else {
            setError(response.error || 'Failed to fetch event details');
          }
        } catch (error) {
          console.error('Error fetching event:', error);
          setError('Failed to fetch event details');
        } finally {
          setLoading(false);
        }
      };

      fetchEvent();
    } else if (!event && !eventId) {
      setError('Event ID not provided');
      setLoading(false);
    }
  }, [eventId, event]);

  // Format date for display
  const formatEventDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'short', 
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  if (loading) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen">
          <div className="flex justify-center items-center h-64">
            <div className="text-gray-500">{t("events.loading", "Loading event...")}</div>
          </div>
        </main>
      </PublicRoute>
    );
  }

  if (error || !event) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen">
          <div className="flex justify-center items-center h-64">
            <div className="text-red-500">{t("events.error", "Error")}: {error || 'Event not found'}</div>
          </div>
        </main>
      </PublicRoute>
    );
  }
  
  return (
    <PublicRoute>
      <div className="relative bg-white min-h-screen">
        {/* Hero Section with Background Image */}
        <section className="relative h-[40vh] sm:h-[50vh] lg:h-[55vh] overflow-hidden">
          {/* Background Image */}
          <div className="absolute inset-0">
            <img
              src={event.thumbnail}
              alt={event.event_title}
              className="h-full w-full object-cover"
              onError={(e) => {
                e.stopPropagation();
                e.currentTarget.src = "/event.png";
              }}
            />
            <div className="absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" />
          </div>

          {/* Hero Content */}
          <div className="relative z-10 h-full flex items-end">
            <div className="mx-auto w-full max-w-[1000px] px-4 sm:px-6 lg:px-8 pb-8 sm:pb-12 lg:pb-16">
              {/* Breadcrumb */}
              <nav className="mb-4 sm:mb-6">
                <Link 
                  to="/events" 
                  className="text-xs sm:text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2"
                >
                  <svg className="w-3 h-3 sm:w-4 sm:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                  </svg>
                  Back to Events
                </Link>
              </nav>

              {/* Title */}
              <h1 className="text-2xl sm:text-3xl lg:text-5xl font-bold text-white mb-4 sm:mb-6 leading-tight max-w-4xl">
                {event.event_title}
              </h1>

              {/* Event Meta Info */}
              <div className="flex flex-col gap-1.5 sm:gap-2 text-white/90">
                <div className="text-sm sm:text-lg font-medium">
                  {event.event_date && event.event_time 
                    ? `${formatEventDate(event.event_date)} at ${event.event_time}`
                    : formatEventDate(event.created_at)
                  }
                </div>
                <div className="text-xs sm:text-base text-white/80">Event #{event.event_id}</div>
              </div>
            </div>
          </div>
        </section>

        {/* Event Details Section */}
        <section className="bg-gray-50 py-8 sm:py-12 lg:py-16">
          <div className="mx-auto w-full max-w-[1000px] px-4 sm:px-6 lg:px-8">
            
            {/* Share Button */}
            <div className="mb-6 sm:mb-8">
              <Button
                variant="outline-black"
                size="sm"
                className="bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300 text-xs sm:text-sm"
              >
                <svg className="w-3 h-3 sm:w-4 sm:h-4 mr-1.5 sm:mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.367 2.684 3 3 0 00-5.367-2.684z" />
                </svg>
                Share
              </Button>
            </div>

            {/* Event Info Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 sm:gap-8 mb-8 sm:mb-12">
              {/* Left Column - Event Details */}
              <div className="space-y-4 sm:space-y-6">
                <div className="flex items-center gap-3">
                  <div className="w-4 h-4 sm:w-5 sm:h-5 flex-shrink-0">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-xs sm:text-sm text-gray-600 mb-1">{event.interested_count} Users Interested / {event.going_count} Going</div>
                  </div>
                </div>

                <div className="flex items-center gap-3">
                  <div className="w-4 h-4 sm:w-5 sm:h-5 flex-shrink-0">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-black font-medium text-sm sm:text-base">
                      {event.event_date && event.event_time 
                        ? `${event.event_date} at ${event.event_time}`
                        : formatEventDate(event.created_at)
                      }
                    </div>
                  </div>
                </div>

                <div className="flex items-center gap-3">
                  <div className="w-4 h-4 sm:w-5 sm:h-5 flex-shrink-0">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                    </svg>
                  </div>
                  <div className="min-w-0 flex-1">
                    <div className="text-black font-medium text-sm sm:text-base break-words">{event.address}</div>
                  </div>
                </div>

                <div className="flex items-center gap-3">
                  <div className="w-4 h-4 sm:w-5 sm:h-5 flex-shrink-0">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-black font-medium text-sm sm:text-base">${event.ticket_price || 'Free'}</div>
                  </div>
                </div>

                <div className="flex items-center gap-3">
                  <div className="w-4 h-4 sm:w-5 sm:h-5 flex-shrink-0">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                    </svg>
                  </div>
                  <div>
                    <div className="text-black font-medium text-sm sm:text-base">
                      {event.event_status === 1 ? 'Active' : 'Inactive'}
                    </div>
                  </div>
                </div>

                {/* Description */}
                <div className="pt-3 sm:pt-4">
                  <p className="text-gray-700 leading-relaxed text-sm sm:text-base">{stripHtmlTags(event.description || '')}</p>
                </div>
              </div>

              {/* Right Column - Participants */}
              <div>
                <h3 className="text-base sm:text-lg font-bold text-black mb-4 sm:mb-6">Participants</h3>
                
                <div className="flex flex-col sm:flex-row gap-3 sm:gap-4 mb-6 sm:mb-8">
                  <Button
                    variant="outline-black"
                    size="md"
                    className="flex-1 bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300 h-10 sm:h-auto text-sm sm:text-base"
                  >
                    <div className="flex items-center gap-1.5 sm:gap-2">
                      <svg className="w-3 h-3 sm:w-4 sm:h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                      </svg>
                      Interested 🔥
                    </div>
                  </Button>
                  
                  <Button
                    variant="fill-black"
                    size="md"
                    className="flex-1 h-10 sm:h-auto text-sm sm:text-base"
                  >
                    <div className="flex items-center gap-1.5 sm:gap-2">
                      {event.going_count} Going
                      <span>Going ➡</span>
                    </div>
                  </Button>
                </div>
              </div>
            </div>

            {/* Address Section */}
            <div className="mb-6 sm:mb-8">
              <h3 className="text-base sm:text-lg font-bold text-black mb-3 sm:mb-4">Address</h3>
              <p className="text-gray-700 text-sm sm:text-base break-words">{event.address}</p>
            </div>

            {/* Map Section */}
            <div className="mb-6 sm:mb-8">
              <div className="w-full h-64 sm:h-80 md:h-96 bg-gray-200 rounded-lg overflow-hidden">
                <iframe
                  src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d387191.33750346623!2d-73.97968099999999!3d40.6974881!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x89c24fa5d33f083b%3A0xc80b8f06e177fe62!2sNew%20York%2C%20NY%2C%20USA!5e0!3m2!1sen!2sus!4v1234567890123"
                  width="100%"
                  height="100%"
                  style={{ border: 0 }}
                  allowFullScreen={true}
                  loading="lazy"
                  referrerPolicy="no-referrer-when-downgrade"
                  className="rounded-lg"
                ></iframe>
              </div>
            </div>
          </div>
        </section>
      </div>
    </PublicRoute>
  );
}