import React, { useState, useEffect } from "react";
import { Link } from "@remix-run/react";
import type { MetaFunction } from "@remix-run/node";
import { eventsAPI, locationsAPI, type Event, type Location } from "~/lib/api";
import { PublicRoute } from "~/components/RouteProtection";
import { useLanguage } from "~/hooks/useLanguage";

export const meta: MetaFunction = () => {
  return [
    { title: "Events - Nomadics Studios" },
    {
      name: "description",
      content: "Discover and join exciting events at Nomadics Studios. Connect with creative professionals and explore new opportunities.",
    },
  ];
};

export default function Events() {
  const { t } = useLanguage();
  const [events, setEvents] = useState<Event[]>([]);
  const [locations, setLocations] = useState<Location[]>([]);
  const [selectedLocation, setSelectedLocation] = useState<number | null>(null);
  const [selectedDate, setSelectedDate] = useState<string>("");
  const [debouncedDate, setDebouncedDate] = useState<string>("");
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Format date for display
  const formatEventDate = (dateString: string) => {
    if (!dateString) {
      return { day: "TBD", month: "DATE" };
    }
    const date = new Date(dateString);
    if (isNaN(date.getTime())) {
      return { day: "TBD", month: "DATE" };
    }
    const day = date.getDate().toString();
    const month = date.toLocaleDateString('en-US', { month: 'short' }).toUpperCase();
    return { day, month };
  };

  // Format time for display
  const formatEventTime = (timeString: string) => {
    return timeString;
  };

  // Generate event slug from title
  const generateSlug = (title: string) => {
    return title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
  };

  // Debounce date filter
  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedDate(selectedDate);
    }, 300); // 300ms delay for date filter

    return () => clearTimeout(timer);
  }, [selectedDate]);

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        
        // Fetch locations and events in parallel
        const [locationsResponse, eventsResponse] = await Promise.all([
          locationsAPI.getLocations(),
          eventsAPI.getAll(debouncedDate, selectedLocation || undefined)
        ]);

        if (locationsResponse.status && locationsResponse.data) {
          // Remove duplicates and add "All Locations" option
          const uniqueLocations = locationsResponse.data.filter((location, index, self) => 
            index === self.findIndex(l => l.id === location.id && l.location === location.location)
          );
          setLocations(uniqueLocations);
        }

        if (eventsResponse.status && eventsResponse.data) {
          setEvents(eventsResponse.data);
        } else {
          setError('Failed to fetch events');
        }
      } catch (err) {
        setError('An error occurred while fetching data');
        console.error('Error fetching data:', err);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, [debouncedDate, selectedLocation]);

  const handleLocationFilter = (locationId: number | null) => {
    setSelectedLocation(locationId);
  };

  const handleDateFilter = (date: string) => {
    setSelectedDate(date);
  };

  if (loading) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen pb-24">
          <div className="max-w-7xl mx-auto pt-8 px-4">
            <div className="flex justify-center items-center h-64">
              <div className="text-gray-500">{t("events.loading", "Loading events...")}</div>
            </div>
          </div>
        </main>
      </PublicRoute>
    );
  }

  if (error) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen pb-24">
          <div className="max-w-7xl mx-auto pt-8 px-4">
            <div className="flex justify-center items-center h-64">
              <div className="text-red-500">{t("events.error", "Error")}: {error}</div>
            </div>
          </div>
        </main>
      </PublicRoute>
    );
  }

  return (
    <PublicRoute>
      <main className="bg-white min-h-screen pb-24">
        {/* Filters */}
        <div className="max-w-[1328px] mx-auto pt-8 px-4 lg:px-14">
          {/* Location Filter - Single Row */}
          <div className="mb-12">
            <label className="block text-body-3 mb-4 uppercase font-normal" style={{ color: '#000000' }}>
              {t("events.filters.location", "LOCATION")}
            </label>
            <div className="flex flex-wrap gap-3">
              <button
                className={`px-6 py-2.5 rounded-full font-normal text-body-3 transition-all ${
                  selectedLocation === null
                    ? "bg-black border-2 border-black"
                    : "bg-white border-2 border-black hover:bg-gray-100"
                }`}
                style={{ color: selectedLocation === null ? '#FFFFFF' : '#000000' }}
                onClick={() => handleLocationFilter(null)}
              >
                {t("events.filters.allLocations", "All Locations")}
              </button>
              {locations.map((location) => (
                <button
                  key={location.id}
                  className={`px-6 py-2.5 rounded-full font-normal text-body-3 transition-all ${
                    selectedLocation === location.id
                      ? "bg-black border-2 border-black"
                      : "bg-white border-2 border-black hover:bg-gray-100"
                  }`}
                  style={{ color: selectedLocation === location.id ? '#FFFFFF' : '#000000' }}
                  onClick={() => handleLocationFilter(location.id)}
                >
                  {location.location}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Events Grid */}
        <div className="max-w-[1328px] mx-auto px-4 lg:px-14">
          {events.length === 0 ? (
            <div className="text-center py-12">
              <div className="text-gray-500 text-body-3">{t("events.noEvents", "No events found matching your filters.")}</div>
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {events.map((event) => {
                const { day, month } = formatEventDate(event.event_date || "");
                const slug = generateSlug(event.event_title);
                
                return (
                  <Link
                    key={event.event_id}
                    to={`/events-details/${event.event_id}`}
                    state={{ event }}
                    className="group block bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300"
                  >
                    {/* Event Image */}
                    <div className="relative overflow-hidden aspect-[426.67/240]">
                      <img
                        src={event.thumbnail || event.picture}
                        alt={event.event_title}
                        className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                        onError={(e) => {
                          e.stopPropagation();
                          const currentSrc = e.currentTarget.src;
                          if (currentSrc === event.thumbnail && event.picture) {
                            e.currentTarget.src = event.picture;
                          } else if (currentSrc === event.picture && event.thumbnail) {
                            e.currentTarget.src = event.thumbnail;
                          } else {
                            e.currentTarget.src = "/event.png";
                          }
                        }}
                      />
                    </div>
                    
                    {/* Event Details */}
                    <div className="flex items-start gap-4 p-4 bg-white">
                      {/* Date Box */}
                      <div className="text-center min-w-[48px] flex-shrink-0">
                        <div className="text-[24px] font-semibold leading-tight" style={{ color: '#000000' }}>
                          {day}
                        </div>
                        <div className="text-[14px] font-normal uppercase" style={{ color: '#000000' }}>
                          {month}
                        </div>
                      </div>
                      
                      {/* Event Info */}
                      <div className="flex-1 min-w-0">
                        <h3 className="text-body-2 font-normal mb-1 line-clamp-2" style={{ color: '#000000' }}>
                          {event.event_title}
                        </h3>
                        <p className="text-body-3 mb-1" style={{ color: '#000000' }}>
                          {formatEventTime(event.event_time || '')}
                        </p>
                        <p className="text-body-3 line-clamp-1" style={{ color: '#000000' }}>
                          {event.address}
                        </p>
                        {(event.interested_count || event.going_count) && (
                          <p className="text-body-3 mt-2" style={{ color: '#000000' }}>
                            {(event.interested_count || 0) + (event.going_count || 0)} interested • {event.going_count || 0} going
                          </p>
                        )}
                      </div>
                    </div>
                  </Link>
                );
              })}
            </div>
          )}
        </div>
      </main>
    </PublicRoute>
  );
}
