import { ProtectedRoute } from "~/components/RouteProtection";
import type { MetaFunction } from "@remix-run/node";
import { useState, useEffect } from "react";
import { Link } from "@remix-run/react";
import { AppHeader } from "~/components/AppHeader";
import { AppFooter } from "~/components/AppFooter";
import { Button } from "~/components/ui/button";
import { ProfileImage } from "~/components/shared";
// import { useLanguage } from "~/config/lang/useLanguage";
import { ArrowRight, ChevronRight } from "lucide-react";
import { newsAPI, eventsAPI, reservationsAPI, guestInvitationsAPI, getUserData } from "~/lib/api";

export const meta: MetaFunction = () => {
  return [
    { title: "Home - Nomadics Studios" },
    {
      name: "description",
      content:
        "Connect creators with premium studio spaces, professional equipment, and a thriving creative community.",
    },
  ];
};

interface Reservation {
  id: string;
  date: string;
  location: string;
  studio: string;
  price: number;
  image: string;
}

interface Invitation {
  id: string;
  name: string;
  date: string;
  location: string;
  image: string;
  myGuestOrBy: string; // "My Guest" or "My By"
  hostUuid: string;
}

interface NewsItem {
  id: string;
  title: string;
  description: string;
  image: string;
  author: string;
  date: string;
  slug: string;
}

interface Event {
  id: string;
  event_id?: number;
  title: string;
  location: string;
  date: string;
  time: string;
  image: string;
  interested_count?: number;
  going_count?: number;
  _originalData?: any;
}

export default function Home() {
  // const { t } = useLanguage();
  const [bookingForm, setBookingForm] = useState({
    where: "",
    start: "",
    startTime: "10:30",
    end: "",
    endTime: "13:30",
  });

  // Generate slug from title (same as news.tsx)
  const generateSlug = (title: string, id: number) => {
    return `${title
      .toLowerCase()
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/(^-|-$)/g, "")}-${id}`;
  };

  // State for API data
  const [news, setNews] = useState<NewsItem[]>([]);
  const [events, setEvents] = useState<Event[]>([]);
  const [reservations, setReservations] = useState<Reservation[]>([]);
  const [invitations, setInvitations] = useState<Invitation[]>([]);
  const [showAllReservations, setShowAllReservations] = useState(false);
  const [showAllInvitations, setShowAllInvitations] = useState(false);
  const [userRole, setUserRole] = useState<string>("");
  const [currentUserUuid, setCurrentUserUuid] = useState<string>("");
  const [loading, setLoading] = useState(true);
  const [newsLoading, setNewsLoading] = useState(true);
  const [eventsLoading, setEventsLoading] = useState(true);
  const [reservationsLoading, setReservationsLoading] = useState(true);
  const [invitationsLoading, setInvitationsLoading] = useState(true);

  // Get user data on mount
  useEffect(() => {
    const userData = getUserData();
    if (userData) {
      setUserRole(userData.role || "");
      setCurrentUserUuid(userData.uuid || "");
    }
  }, []);

  // Fetch reservations data
  useEffect(() => {
    const fetchReservations = async () => {
      try {
        setReservationsLoading(true);
        const limit = showAllReservations ? undefined : 4;
        const response = await reservationsAPI.getUpcoming(limit);

        if (response.status && response.data && Array.isArray(response.data)) {
          const mappedReservations: Reservation[] = response.data.map(
            (item: any) => {
              // Format date from start_date
              const formattedDate = item.start_date
                ? new Date(item.start_date).toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                  })
                : "Date TBD";

              return {
                id: item.id?.toString() || Math.random().toString(),
                date: formattedDate,
                location: item.loc?.location || "Location TBD",
                studio: item.booking_id || "Studio Booking",
                price: item.total_points || 0,
                image: item.loc_landing?.banner_picture
                  ? `https://nomadicstudios.net:444/storage/location/${item.loc_landing.banner_picture}`
                  : "/images/studios/default.png",
              };
            }
          );
          setReservations(mappedReservations);
        } else {
          setReservations([]);
        }
      } catch (error) {
        console.error("Failed to fetch reservations:", error);
        setReservations([]);
      } finally {
        setReservationsLoading(false);
      }
    };

    fetchReservations();
  }, [showAllReservations]);

  // Fetch guest invitations data
  useEffect(() => {
    const fetchInvitations = async () => {
      try {
        setInvitationsLoading(true);
        const limit = showAllInvitations ? undefined : 3;
        const response = await guestInvitationsAPI.getAll(limit);

        if (response.status && response.data && Array.isArray(response.data)) {
          const mappedInvitations: Invitation[] = response.data.map(
            (item: any) => {
              // Determine if this is "My Guest" or "My By"
              const isMyGuest = item.host_uuid === currentUserUuid;
              const myGuestOrBy = isMyGuest ? "My Guest" : "My By";

              // Format date from invitation_date
              const formattedDate = item.invitation_date
                ? new Date(item.invitation_date).toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                    hour: "2-digit",
                    minute: "2-digit",
                  })
                : "Date TBD";

              // Get guest name from guest object or use host info
              const guestName = isMyGuest
                ? `${item.guest?.fname || ""} ${
                    item.guest?.lname || ""
                  }`.trim() || "Guest"
                : `${item.host?.fname || ""} ${
                    item.host?.lname || ""
                  }`.trim() || "Host";

              return {
                id: item.id?.toString() || Math.random().toString(),
                name: guestName,
                date: formattedDate,
                location: item.location?.location || "Location TBD",
                image: "/images/team-member.png", // API doesn't provide image
                myGuestOrBy: myGuestOrBy,
                hostUuid: item.host_uuid || "",
              };
            }
          );
          setInvitations(mappedInvitations);
        } else {
          setInvitations([]);
        }
      } catch (error) {
        console.error("Failed to fetch invitations:", error);
        setInvitations([]);
      } finally {
        setInvitationsLoading(false);
      }
    };

    if (currentUserUuid) {
      fetchInvitations();
    }
  }, [showAllInvitations, currentUserUuid]);

  // Fetch news data
  useEffect(() => {
    const fetchNews = async () => {
      try {
        setNewsLoading(true);
        const newsData = await newsAPI.getAll();

        // Map API data to component interfaces
        const mappedNews: NewsItem[] = (newsData.data || []).map(
          (item: any) => {
            const title = item.name || "No title";
            const id = item.id || Math.random();
            return {
              id: id.toString(),
              title,
              description: item.heading_intro || "No description",
              image:
                item.banner ||
                item.thumbnail ||
                "https://via.placeholder.com/400x240/f3f4f6/9ca3af?text=News+Image",
              author: item.created_by
                ? `${item.created_by.fname} ${item.created_by.lname}`
                : "Unknown",
              date: item.created_at
                ? new Date(item.created_at).toLocaleDateString()
                : new Date().toLocaleDateString(),
              slug: generateSlug(title, id),
            };
          }
        );

        setNews(mappedNews);
      } catch (error) {
        console.error("Failed to fetch news:", error);
        setNews([]);
      } finally {
        setNewsLoading(false);
      }
    };

    fetchNews();
  }, []);

  // Fetch events data
  useEffect(() => {
    const fetchEvents = async () => {
      try {
        setEventsLoading(true);
        const eventsData = await eventsAPI.getAll();

        // Map API data to component interfaces, keeping the original API data
        const mappedEvents: Event[] = (eventsData.data || []).map(
          (item: any) => ({
            id: item.event_id?.toString() || Math.random().toString(),
            event_id: item.event_id,
            title: item.event_title || "No title",
            location: item.address || "Location TBD",
            date: item.event_date
              ? new Date(item.event_date)
                  .toLocaleDateString("en-US", {
                    weekday: "long",
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                  })
                  .toUpperCase()
              : "Date TBD",
            time: item.event_time || "Time TBD",
            image:
              item.thumbnail ||
              item.picture ||
              "https://via.placeholder.com/400x240/1f2937/ffffff?text=Event+Image",
            interested_count: item.interested_count || 0,
            going_count: item.going_count || 0,
            // Keep the original API data for passing to detail page
            _originalData: item,
          })
        );

        setEvents(mappedEvents);
      } catch (error) {
        console.error("Failed to fetch events:", error);
        // Set empty array if API fails
        setEvents([]);
      } finally {
        setEventsLoading(false);
        setLoading(false);
      }
    };

    fetchEvents();
  }, []);

  const handleBookingSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    // Navigate to booking studio selection page
    window.location.href = "/book-studio";
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-black text-white">
        <AppHeader theme="dark" />

        <main className="pt-10">
          {/* Hero Section with Booking Form */}
          <section className="relative h-screen bg-cover bg-center bg-no-repeat">
            {/* Background Image */}
            <div className="absolute inset-0 bg-[#030303]" />
            <img
              src="/images/my-home.png"
              alt=""
              className="absolute inset-0 h-full w-full object-cover"
            />
            <div className="absolute inset-0 bg-black/50" />
            <div className="relative z-10 flex flex-col items-center justify-center h-full px-4">
              {/* Hero Title */}
              <div className="text-center text-white mb-12">
                <h1 className="text-heading-5 lg:text-heading-2 font-light tracking-wider max-w-4xl mx-auto uppercase">
                  Let's create something
                </h1>
              </div>

              {/* Booking Form - Only show if user role is not 'guest' */}
              {userRole !== "guest" && (
                <div className="w-full max-w-7xl mx-auto">
                  <div className="bg-white/95 backdrop-blur-sm p-6 md:p-8 rounded-none shadow-2xl">
                    <form onSubmit={handleBookingSubmit}>
                      <div className="grid grid-cols-1 md:grid-cols-6 gap-4 items-start mb-4">
                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem]"
                            style={{ color: "#000000" }}
                          >
                            Where
                          </label>
                          <select
                            value={bookingForm.where}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                where: e.target.value,
                              })
                            }
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          >
                            <option value="">Location</option>
                            <option value="bali">Bali</option>
                            <option value="bangkok">Bangkok</option>
                            <option value="mexico">Mexico</option>
                            <option value="costa-rica">Costa Rica</option>
                          </select>
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem]"
                            style={{ color: "#000000" }}
                          >
                            Start
                          </label>
                          <input
                            type="date"
                            value={bookingForm.start}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                start: e.target.value,
                              })
                            }
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                            placeholder="dd/mm/yyyy"
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem]"
                            style={{ color: "#000000" }}
                          >
                            Start Time
                          </label>
                          <input
                            type="time"
                            value={bookingForm.startTime}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                startTime: e.target.value,
                              })
                            }
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem]"
                            style={{ color: "#000000" }}
                          >
                            End
                          </label>
                          <input
                            type="date"
                            value={bookingForm.end}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                end: e.target.value,
                              })
                            }
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                            placeholder="dd/mm/yyyy"
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem]"
                            style={{ color: "#000000" }}
                          >
                            End Time
                          </label>
                          <input
                            type="time"
                            value={bookingForm.endTime}
                            onChange={(e) =>
                              setBookingForm({
                                ...bookingForm,
                                endTime: e.target.value,
                              })
                            }
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          />
                        </div>

                        <div className="text-left">
                          <label
                            className="block text-body-3 mb-3 uppercase tracking-wider min-h-[2.5rem] leading-tight"
                            style={{ color: "#000000" }}
                          >
                            Looking for a<br />
                            minimum of
                          </label>
                          <select
                            className="w-full p-4 border-b-2 border-gray-300 bg-transparent text-body-3 focus:border-gray-600 focus:outline-none transition-colors"
                            style={{ color: "#000000" }}
                          >
                            <option value="">Studios</option>
                            <option value="video">Video Studio</option>
                            <option value="music">Music Studio</option>
                            <option value="photo">Photo Studio</option>
                          </select>
                        </div>
                      </div>

                      {/* Booking minimum text */}
                      <div className="mb-4">
                        <p className="text-body-3" style={{ color: "#000000" }}>
                          Booking for a minimum of 2 hours
                        </p>
                      </div>

                      {/* Search Button */}
                      <div className="flex justify-end">
                        <Button
                          type="submit"
                          variant="fill-black"
                          size="md"
                          className="px-8 py-3 text-base font-medium uppercase tracking-wider transition-all duration-200 hover:shadow-lg"
                        >
                          Search
                        </Button>
                      </div>
                    </form>
                  </div>
                </div>
              )}
            </div>
          </section>

          {/* Reservations & Invitations Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-[120px]">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
                {/* Left Column - Reservations */}
                <div>
                  <div className="flex justify-between items-center mb-6">
                    <h2 className="text-heading-5" style={{ color: "#000000" }}>
                      Upcoming reservations
                    </h2>
                    <button
                      onClick={() =>
                        setShowAllReservations(!showAllReservations)
                      }
                      className="flex items-center text-body-3 hover:opacity-80"
                      style={{ color: "#000000" }}
                    >
                      {showAllReservations ? "Show less" : "See all"}{" "}
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </button>
                  </div>

                  {reservationsLoading ? (
                    <div className="space-y-4">
                      {[...Array(3)].map((_, i) => (
                        <div
                          key={i}
                          className="animate-pulse flex items-center gap-4 p-4 border border-gray-200"
                        >
                          <div className="w-16 h-16 bg-gray-200"></div>
                          <div className="flex-1 space-y-2">
                            <div className="h-3 bg-gray-200 w-24"></div>
                            <div className="h-4 bg-gray-200 w-32"></div>
                            <div className="h-3 bg-gray-200 w-28"></div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : reservations.length > 0 ? (
                    <div className="space-y-4">
                      {reservations.map((reservation) => (
                        <div
                          key={reservation.id}
                          className="flex items-center gap-4 p-4 border border-gray-200 hover:border-gray-300 transition-colors"
                        >
                          <img
                            src={reservation.image}
                            alt={reservation.studio}
                            className="w-16 h-16 object-cover"
                          />
                          <div className="flex-1">
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              {reservation.date}
                            </p>
                            <p
                              className="text-body-2"
                              style={{ color: "#000000" }}
                            >
                              {reservation.location}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {reservation.studio}
                            </p>
                          </div>
                          <div className="text-right">
                            <p
                              className="text-body-2"
                              style={{ color: "#000000" }}
                            >
                              {reservation.price} pts
                            </p>
                            <button
                              className="text-body-3 flex items-center hover:opacity-80"
                              style={{ color: "#000000" }}
                            >
                              See details{" "}
                              <ChevronRight className="ml-1 h-3 w-3" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-8">
                      <p className="text-gray-500">No upcoming reservations</p>
                    </div>
                  )}
                </div>

                {/* Right Column - Guest Invitations */}
                <div>
                  <div className="flex justify-between items-center mb-6">
                    <h2 className="text-heading-5" style={{ color: "#000000" }}>
                      Guest invitations
                    </h2>
                    <button
                      onClick={() => setShowAllInvitations(!showAllInvitations)}
                      className="flex items-center text-body-3 hover:opacity-80"
                      style={{ color: "#000000" }}
                    >
                      {showAllInvitations ? "Show less" : "See all"}{" "}
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </button>
                  </div>

                  {invitationsLoading ? (
                    <div className="space-y-4">
                      {[...Array(3)].map((_, i) => (
                        <div
                          key={i}
                          className="animate-pulse flex items-center gap-4 p-4 border border-gray-200"
                        >
                          <div className="w-16 h-16 bg-gray-200 rounded-full"></div>
                          <div className="flex-1 space-y-2">
                            <div className="h-4 bg-gray-200 w-32"></div>
                            <div className="h-3 bg-gray-200 w-24"></div>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : invitations.length > 0 ? (
                    <div className="space-y-4">
                      {invitations.map((invitation) => (
                        <div
                          key={invitation.id}
                          className="flex items-center gap-4 p-4 border border-gray-200 hover:border-gray-300 transition-colors"
                        >
                          <ProfileImage
                            src={invitation.image}
                            alt={invitation.name}
                            className="w-16 h-16 rounded-full object-cover"
                          />
                          <div className="flex-1">
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              Date: {invitation.date}
                            </p>
                            <p
                              className="text-body-3 mb-1"
                              style={{ color: "#000000" }}
                            >
                              Location: {invitation.location}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {invitation.myGuestOrBy}
                            </p>
                          </div>
                          <div className="text-right">
                            <p
                              className="text-body-2 mb-1"
                              style={{ color: "#000000" }}
                            >
                              {invitation.name}
                            </p>
                            <button
                              className="text-body-3 flex items-center mt-1 hover:opacity-80"
                              style={{ color: "#000000" }}
                            >
                              See details{" "}
                              <ChevronRight className="ml-1 h-3 w-3" />
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-8">
                      <p className="text-gray-500">No guest invitations</p>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </section>

          {/* News Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14">
              <div className="flex justify-between items-center mb-8">
                <h2 className="text-heading-4" style={{ color: "#000000" }}>
                  News
                </h2>
                <button
                  onClick={() => (window.location.href = "/news")}
                  className="flex items-center text-body-3 hover:opacity-80 transition-opacity"
                  style={{ color: "#000000" }}
                >
                  See more <ArrowRight className="ml-2 h-4 w-4" />
                </button>
              </div>

              {newsLoading ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {[...Array(6)].map((_, i) => (
                    <div key={i} className="animate-pulse">
                      <div className="w-full h-48 bg-gray-200 mb-4"></div>
                      <div className="h-4 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                      <div className="flex items-center gap-2">
                        <div className="w-5 h-5 bg-gray-200 rounded-full"></div>
                        <div className="h-3 bg-gray-200 w-20"></div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : news.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {news.slice(0, 6).map((item) => {
                    // Format date like event cards
                    const formatCardDate = (dateString: string) => {
                      const date = new Date(dateString);
                      if (isNaN(date.getTime())) {
                        return { day: "TBD", month: "DATE" };
                      }
                      const day = date.getDate().toString();
                      const month = date
                        .toLocaleDateString("en-US", { month: "short" })
                        .toUpperCase();
                      return { day, month };
                    };

                    const { day, month } = formatCardDate(item.date);

                    return (
                      <Link
                        key={item.id}
                        to={`/news-details/${item.slug}`}
                        className="group cursor-pointer bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300 block"
                      >
                        {/* News Image */}
                        <div className="relative overflow-hidden aspect-[426.67/240]">
                          <img
                            src={item.image}
                            alt={item.title}
                            className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                          />
                        </div>

                        {/* News Details */}
                        <div className="flex items-start gap-4 p-4 bg-white">
                          {/* Date Box */}
                          <div className="text-center min-w-[48px] flex-shrink-0">
                            <div
                              className="text-[24px] font-semibold leading-tight"
                              style={{ color: "#000000" }}
                            >
                              {day}
                            </div>
                            <div
                              className="text-[14px] font-normal uppercase"
                              style={{ color: "#000000" }}
                            >
                              {month}
                            </div>
                          </div>

                          {/* News Info */}
                          <div className="flex-1 min-w-0">
                            <h3
                              className="text-body-2 font-normal mb-1 line-clamp-2"
                              style={{ color: "#000000" }}
                            >
                              {item.title}
                            </h3>
                            <p
                              className="text-body-3 mb-1 line-clamp-2"
                              style={{ color: "#000000" }}
                            >
                              {item.description}
                            </p>
                            <p
                              className="text-body-3"
                              style={{ color: "#000000" }}
                            >
                              {item.author}
                            </p>
                          </div>
                        </div>
                      </Link>
                    );
                  })}
                </div>
              ) : (
                <div className="text-center py-12">
                  <p className="text-gray-500">
                    No news articles available at the moment.
                  </p>
                </div>
              )}
            </div>
          </section>

          {/* Upcoming Events Section - White Background */}
          <section className="relative w-full bg-white">
            <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-[120px]">
              <div className="flex justify-between items-center mb-12">
                <h2 className="text-heading-4" style={{ color: "#000000" }}>
                  Upcoming events
                </h2>
                <button
                  onClick={() => (window.location.href = "/events")}
                  className="flex items-center text-body-3 hover:opacity-80 transition-opacity"
                  style={{ color: "#000000" }}
                >
                  See all <ArrowRight className="ml-2 h-4 w-4" />
                </button>
              </div>

              {eventsLoading ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {[...Array(3)].map((_, i) => (
                    <div key={i} className="animate-pulse">
                      <div className="w-full h-48 bg-gray-200 mb-4"></div>
                      <div className="h-4 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                      <div className="h-3 bg-gray-200 mb-2"></div>
                    </div>
                  ))}
                </div>
              ) : events.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {events.slice(0, 3).map((event) => {
                    // Format date like event cards
                    const formatCardDate = (dateString: string) => {
                      const date = new Date(dateString);
                      if (isNaN(date.getTime())) {
                        return { day: "TBD", month: "DATE" };
                      }
                      const day = date.getDate().toString();
                      const month = date
                        .toLocaleDateString("en-US", { month: "short" })
                        .toUpperCase();
                      return { day, month };
                    };

                    const { day, month } = formatCardDate(event.date);

                  return (
                    <Link 
                      key={event.id} 
                      to={`/events-details/${event.event_id || event.id}`}
                      state={{ event: event._originalData }}
                      className="group cursor-pointer bg-white border border-gray-200 overflow-hidden hover:shadow-lg transition-all duration-300 block"
                    >
                      {/* Event Image */}
                      <div className="relative overflow-hidden aspect-[426.67/240]">
                        <img 
                          src={event.image} 
                          alt={event.title}
                          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                        />
                      </div>
                      
                      {/* Event Details */}
                      <div className="flex items-start gap-4 p-4 bg-white">
                        {/* Date Box */}
                        <div className="text-center min-w-[48px] flex-shrink-0">
                          <div className="text-[24px] font-semibold leading-tight" style={{ color: '#000000' }}>
                            {day}
                          </div>
                          <div className="text-[14px] font-normal uppercase" style={{ color: '#000000' }}>
                            {month}
                          </div>
                        </div>
                        
                        {/* Event Info */}
                        <div className="flex-1 min-w-0">
                          <h3 className="text-body-2 font-normal mb-1 line-clamp-2" style={{ color: '#000000' }}>
                            {event.title}
                          </h3>
                          <p className="text-body-3 mb-1" style={{ color: '#000000' }}>
                            {event.time}
                          </p>
                          <p className="text-body-3 line-clamp-1" style={{ color: '#000000' }}>
                            {event.location}
                          </p>
                          {(event.interested_count || event.going_count) && (
                            <p className="text-body-3 mt-2" style={{ color: '#000000' }}>
                              {(event.interested_count || 0) + (event.going_count || 0)} interested • {event.going_count || 0} going
                            </p>
                          )}
                        </div>
                      </div>
                    </Link>
                  );
                })}
              </div>
            ) : (
              <div className="text-center py-12">
                <p className="text-gray-500">No events available at the moment.</p>
              </div>
            )}
          </div>
        </section>
      </main>
      </div>
    </ProtectedRoute>
  );
}
