import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData } from "@remix-run/react";
import {
  WelcomeSection,
  LocationInfo,
  StudiosSection,
  EventsSection,
  PlanYourVisitSection,
} from "~/components/pages/locations";
import { MembershipSection } from "~/components/shared/MembershipSection";
import { PeopleSection } from "~/components/shared/PeopleSection";
import { useLanguage } from "~/hooks/useLanguage";
import en from "~/config/lang/en.json";
import sp from "~/config/lang/sp.json";
const translations = { en, sp };

export async function loader({ params }: LoaderFunctionArgs) {
  // get location from params
  const location = params.location;

  if (!location) {
    throw new Response("Location Not Found", { status: 404 });
  }

  // Fetch API data
  const apiRes = await fetch(
    `https://nomadicstudios.net:444/api/locations/${location}`
  );
  const apiData = await apiRes.json();

  return Response.json({ location, apiData, translations });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Location Not Found | Nomadics" },
      { name: "description", content: "The requested location was not found." },
    ];
  }

  // Use default English for meta tags since we can't access hooks in meta
  const locationData = data.translations.en[data.location];
  
  return [
    { title: locationData?.meta?.title || `${data.location} | Nomadics` },
    { name: "description", content: locationData?.meta?.description || `Explore Nomadics studio in ${data.location}` },
  ];
};

export default function LocationPage() {
  const { location, apiData, translations } = useLoaderData<typeof loader>();
  const { currentLanguage } = useLanguage();
  
  // Get location data based on current language
  const locationData = translations[currentLanguage]?.[location] || translations.en[location];
  
  // Set background image from API
  if (locationData && apiData.data.banner_picture) {
    locationData.welcome.backgroundImage = apiData.data.banner_picture;
  }

  if (!locationData) {
    throw new Response("Location Not Found", { status: 404 });
  }

  return (
    <div>
      <WelcomeSection data={locationData.welcome} />
      <LocationInfo data={locationData.info} />
      <StudiosSection data={apiData.data} />
      {apiData.data.resources?.length > 0 && (
        <PeopleSection
          title={locationData.people.title}
          eyebrow="MEET THE TEAM"
          buttonText="VIEW ALL PEOPLE"
        />
      )}
      {apiData.data.events?.length > 0 && (
        <EventsSection
          title={locationData.events.sectionTitle}
          data={apiData.data.events}
        />
      )}
      <PlanYourVisitSection
        title={locationData.planVisit.title}
        address={apiData.data.address}
        hours={apiData.data.opening_hours}
        phone={apiData.data.phone}
        email={apiData.data.email}
        additionalInfo={apiData.data.additional_info}
        location={apiData.data.location}
      />
      <MembershipSection theme="black" />
    </div>
  );
}
