import { useState, useEffect } from "react";
import { useParams } from "@remix-run/react";
import { peopleAPI, type Person } from "~/lib/api";

export default function ProfilePage() {
  const { uuid } = useParams();
  const [person, setPerson] = useState<Person | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [start, setStart] = useState<string>("2025-08-21T10:30");
  const [end, setEnd] = useState<string>("2025-08-21T13:30");

  // Calculate duration in hours
  const duration =
    (new Date(end).getTime() - new Date(start).getTime()) / (1000 * 60 * 60);

  // Example: 1h = 43 points
  const points = duration > 0 ? Math.round(duration * 43) : 0;

  const handleBooking = () => {
    alert(
      `Booking confirmed!\nStart: ${start}\nEnd: ${end}\nDuration: ${duration}h\nPoints: ${points}`
    );
  };

  useEffect(() => {
    const fetchPersonData = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await peopleAPI.getAll();
        
        if (response.status && response.data) {
          const foundPerson = response.data.find(p => p.uuid === uuid);
          if (foundPerson) {
            setPerson(foundPerson);
          } else {
            setError('Person not found');
          }
        } else {
          setError('Failed to fetch person data');
        }
      } catch (err) {
        console.error('Error fetching person:', err);
        setError(err instanceof Error ? err.message : 'Failed to fetch person data');
      } finally {
        setLoading(false);
      }
    };

    if (uuid) {
      fetchPersonData();
    } else {
      setError('No UUID provided');
      setLoading(false);
    }
  }, [uuid]);

  if (loading) {
    return (
      <div className="min-h-screen bg-white text-black font-sans p-6 md:p-12">
        <div className="flex justify-center items-center h-64">
          <div className="text-gray-500">Loading profile...</div>
        </div>
      </div>
    );
  }

  if (error || !person) {
    return (
      <div className="min-h-screen bg-white text-black font-sans p-6 md:p-12">
        <div className="flex justify-center items-center h-64">
          <div className="text-red-500">Error: {error || 'Person not found'}</div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white text-black font-sans p-6 md:p-12">
      {/* Header */}
      <h1 className="text-4xl font-bold">{person.fname.toUpperCase()} {person.lname.toUpperCase()}</h1>
      <div className="flex gap-2 mt-2">
        <span className="px-3 py-1 rounded-full border text-sm">
          {person.location}
        </span>
        <span className="px-3 py-1 rounded-full border text-sm">
          {person.type}
        </span>
      </div>

      {/* Main Layout */}
      <div className="grid md:grid-cols-2 gap-8 mt-8">
        {/* Left Section */}
        <div>
          <div className="w-full h-[500px] rounded-xl shadow-lg bg-gray-100 flex items-center justify-center">
            <img
              src={person.profile_pic}
              alt={`${person.fname} ${person.lname}`}
              className="max-w-full max-h-full object-contain rounded-xl"
              onError={(e) => {
                e.currentTarget.src = "https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcQOeZjZWEr4oFmJhILQQgTy7-WUX9BmRrAAFw&s";
              }}
            />
          </div>

          <section className="mt-8">
            <h2 className="text-xl font-bold mb-2">ABOUT</h2>
            <p className="text-gray-700 leading-relaxed">
              {person.description}
            </p>
          </section>

          {/* Social Media Links */}
          {(person.instagram_link || person.facebook_link || person.twitter_link || person.profile_link || person.sm_link) && (
            <section className="mt-6">
              <h2 className="text-xl font-bold mb-2">CONNECT</h2>
              <div className="flex gap-3 flex-wrap">
                {person.instagram_link && (
                  <a 
                    href={person.instagram_link} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm transition-colors"
                  >
                    Instagram
                  </a>
                )}
                {person.facebook_link && (
                  <a 
                    href={person.facebook_link} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm transition-colors"
                  >
                    Facebook
                  </a>
                )}
                {person.twitter_link && (
                  <a 
                    href={person.twitter_link} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm transition-colors"
                  >
                    Twitter
                  </a>
                )}
                {person.profile_link && (
                  <a 
                    href={person.profile_link} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm transition-colors"
                  >
                    Portfolio
                  </a>
                )}
                {person.sm_link && (
                  <a 
                    href={person.sm_link} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="px-4 py-2 bg-gray-100 hover:bg-gray-200 rounded-full text-sm transition-colors"
                  >
                    Website
                  </a>
                )}
              </div>
            </section>
          )}

          <section className="mt-6">
            <h2 className="text-xl font-bold mb-2">LANGUAGE SPOKEN</h2>
            <p className="text-gray-700">English</p>
          </section>
        </div>

        {/* Right Section (Booking Widget) */}
        <div className="border rounded-xl p-6 shadow-md h-fit flex flex-col gap-4">
          <div>
            <label className="block text-sm font-medium">Start</label>
            <input
              type="datetime-local"
              value={start}
              onChange={(e) => setStart(e.target.value)}
              className="w-full border rounded-md p-2 mt-1"
            />
          </div>

          <div>
            <label className="block text-sm font-medium">End</label>
            <input
              type="datetime-local"
              value={end}
              onChange={(e) => setEnd(e.target.value)}
              className="w-full border rounded-md p-2 mt-1"
            />
          </div>

          <p className="text-sm text-gray-600">
            Your reservation:{" "}
            {duration > 0 ? `${duration}h • From: ${points} points` : "Invalid"}
          </p>

          <button
            onClick={handleBooking}
            className="bg-black text-white py-3 rounded-md hover:bg-gray-800 transition"
          >
            BOOK NOW →
          </button>
        </div>
      </div>
    </div>
  );
}
