import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData } from "@remix-run/react";
import {
  WelcomeSection,
  StudiosSection,
  PhotoSection,
  BookSection,
} from "~/components/pages/studios";
import { MembershipSection } from "~/components/shared/MembershipSection";
import { PeopleSection } from "~/components/shared/PeopleSection";
import { useLanguage } from "~/hooks/useLanguage";
import en from "~/config/lang/en.json";
import sp from "~/config/lang/sp.json";
const translations = { en, sp };

export async function loader({ params }: LoaderFunctionArgs) {
  const type = params.type;

  if (!type) {
    throw new Response("Studio Type Not Found", { status: 404 });
  }

  // Fetch API data
  const apiRes = await fetch(
    `https://nomadicstudios.net:444/api/studios?type=${type}`
  );
  const apiData = await apiRes.json();
  console.log(apiData);
  
  return Response.json({ type, apiData, translations });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadics" },
      {
        name: "description",
        content: "The requested studio type was not found.",
      },
    ];
  }

  // Use default English for meta tags since we can't access hooks in meta
  const studioData = data.translations.en[data.type];
  
  return [
    { title: studioData?.meta?.title || `${data.type} Studio | Nomadics` },
    { name: "description", content: studioData?.meta?.description || `Explore ${data.type} studios at Nomadics` },
  ];
};

export default function StudioPage() {
  const { type, apiData, translations } = useLoaderData<typeof loader>();
  const { currentLanguage, t } = useLanguage();
  
  // Get studio data based on current language
  const studioData = translations[currentLanguage]?.[type] || translations.en[type];

  if (!studioData) {
    throw new Response("Studio Type Not Found", { status: 404 });
  }

  return (
    <div>
      <WelcomeSection data={studioData.welcome} />
      <StudiosSection data={studioData.studios} />
      <PhotoSection data={studioData.gallery} />
      <BookSection
        title={studioData.book.title}
        description={studioData.book.description}
        data={apiData}
      />
      <PeopleSection 
        title={t('people.ourCrew', 'OUR CREW')}
        eyebrow="MEET THE TEAM"
        buttonText={t('people.viewAll', 'VIEW ALL PEOPLE')}
      />
      <MembershipSection theme="black" />
    </div>
  );
}
