# Design System Documentation

This design system is based on the provided style guide and implements a complete typography and color system using Tailwind CSS v4 with custom utility classes.

## 🎨 Overview

The design system uses **Clash Grotesk Variable** as the primary typeface and implements a sophisticated dark/light color system with multiple opacity variants. All components follow precise specifications from the original style guide.

## 📝 Typography

### Font Family
All typography uses `'Clash Grotesk Variable'` with system fallbacks:
```css
font-family: 'Clash Grotesk Variable', system-ui, sans-serif;
```

### Heading Styles

#### `.text-heading-1`
- **Font Size:** 112px
- **Line Height:** 80px (71%)  
- **Font Weight:** 600 (Semibold)
- **Letter Spacing:** -0.02em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h1 className="text-heading-1">Main Headlines</h1>
```

#### `.text-heading-2`
- **Font Size:** 72px
- **Line Height:** 100%
- **Font Weight:** 500 (Medium)
- **Letter Spacing:** -0.03em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h2 className="text-heading-2">Section Titles</h2>
```

#### `.text-heading-2-bold`
- **Font Size:** 80px
- **Line Height:** 100%
- **Font Weight:** 600 (Semibold)
- **Letter Spacing:** -0.03em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h2 className="text-heading-2-bold">Emphasized Titles</h2>
```

#### `.text-heading-3`
- **Font Size:** 40px
- **Line Height:** 48px (120%)
- **Font Weight:** 500 (Medium)
- **Letter Spacing:** -0.01em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h3 className="text-heading-3">Subsection Headers</h3>
```

#### `.text-heading-4`
- **Font Size:** 32px
- **Line Height:** 40px (125%)
- **Font Weight:** 400 (Normal)
- **Letter Spacing:** -0.01em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h4 className="text-heading-4">Component Titles</h4>
```

#### `.text-heading-5`
- **Font Size:** 24px
- **Line Height:** 32px (133%)
- **Font Weight:** 400 (Normal)
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<h5 className="text-heading-5">Small Headers</h5>
```

### Body Text Styles

#### `.text-body-1`
- **Font Size:** 24px
- **Line Height:** 32px (133%)
- **Font Weight:** 400 (Normal)
- **Color:** #FFFFFF

```jsx
<p className="text-body-1">Large body text for emphasis</p>
```

#### `.text-body-2`
- **Font Size:** 18px
- **Line Height:** 26px (144%)
- **Font Weight:** 400 (Normal)
- **Color:** #FFFFFF

```jsx
<p className="text-body-2">Standard body text</p>
```

#### `.text-body-3`
- **Font Size:** 16px
- **Line Height:** 24px (150%)
- **Font Weight:** 400 (Normal)
- **Color:** #FFFFFF

```jsx
<p className="text-body-3">Small body text and descriptions</p>
```

### Specialized Text

#### `.text-caption`
- **Font Size:** 16px
- **Line Height:** 20px
- **Font Weight:** 400 (Normal)
- **Letter Spacing:** 0.02em
- **Transform:** Uppercase
- **Color:** #FFFFFF

```jsx
<span className="text-caption">Labels and captions</span>
```

### Button Typography

#### `.text-button-big`
- **Font Size:** 18px
- **Line Height:** 22px
- **Font Weight:** 500 (Medium)
- **Letter Spacing:** 0.02em
- **Transform:** Uppercase

#### `.text-button-medium`
- **Font Size:** 16px
- **Line Height:** 20px
- **Font Weight:** 500 (Medium)
- **Letter Spacing:** 0.02em
- **Transform:** Uppercase

#### `.text-button-small`
- **Font Size:** 14px
- **Line Height:** 17px
- **Font Weight:** 500 (Medium)
- **Letter Spacing:** 0.02em
- **Transform:** Uppercase

## 🎯 Color System

### Dark Color Scale
Based on `#121212` with opacity variations:

| Class | Color | Usage |
|-------|-------|--------|
| `.bg-dark-solid` | `#121212` | Primary dark backgrounds |
| `.bg-dark-90` | `rgba(18, 18, 18, 0.9)` | Semi-solid overlays |
| `.bg-dark-80` | `rgba(18, 18, 18, 0.8)` | Modal backgrounds |
| `.bg-dark-70` | `rgba(18, 18, 18, 0.7)` | Hover states |
| `.bg-dark-60` | `rgba(18, 18, 18, 0.6)` | Disabled elements |
| `.bg-dark-50` | `rgba(18, 18, 18, 0.5)` | Subtle overlays |
| `.bg-dark-40` | `rgba(18, 18, 18, 0.4)` | Light overlays |
| `.bg-dark-30` | `rgba(18, 18, 18, 0.3)` | Borders |
| `.bg-dark-20` | `rgba(18, 18, 18, 0.2)` | Subtle borders |
| `.bg-dark-10` | `rgba(18, 18, 18, 0.1)` | Very subtle backgrounds |

### Light Color Scale
Based on `#FFFFFF` with opacity variations:

| Class | Color | Usage |
|-------|-------|--------|
| `.bg-light-solid` | `#FFFFFF` | Primary light backgrounds |
| `.bg-light-90` | `rgba(255, 255, 255, 0.9)` | Semi-solid overlays |
| `.bg-light-80` | `rgba(255, 255, 255, 0.8)` | Modal backgrounds |
| `.bg-light-70` | `rgba(255, 255, 255, 0.7)` | Hover states |
| `.bg-light-60` | `rgba(255, 255, 255, 0.6)` | Disabled elements |
| `.bg-light-50` | `rgba(255, 255, 255, 0.5)` | Subtle overlays |
| `.bg-light-40` | `rgba(255, 255, 255, 0.4)` | Light overlays |
| `.bg-light-30` | `rgba(255, 255, 255, 0.3)` | Borders |
| `.bg-light-20` | `rgba(255, 255, 255, 0.2)` | Subtle borders |
| `.bg-light-10` | `rgba(255, 255, 255, 0.1)` | Very subtle backgrounds |

### Text Colors
Available with same opacity variations:
- `.text-dark-solid` to `.text-dark-10`
- `.text-light-solid` to `.text-light-10`

### Border Colors
Available for key opacity levels:
- `.border-dark-solid`, `.border-dark-30`, `.border-dark-10`
- `.border-light-solid`, `.border-light-30`, `.border-light-10`

## 🔘 Button Component

### Variants

#### Fill Variants
- **`fill-white`**: White background with dark text
- **`fill-black`**: Dark background with light text

#### Outline Variants
- **`outline-white`**: Transparent with white border
- **`outline-black`**: Transparent with dark border

### Sizes
- **`sm`**: 32px height (Small)
- **`md`**: 48px height (Medium) - Default
- **`lg`**: 60px height (Large)

### Button Types
- **`label`**: Text with optional icon - Default
- **`icon-square`**: Square icon-only button
- **`icon-circle`**: Circular icon-only button  

### Usage Examples

```jsx
import { Button } from "~/components/ui/button";

// Basic buttons
<Button variant="fill-white" size="md" buttonType="label">
  Click Me
</Button>

// With icon
<Button variant="fill-black" size="lg" buttonType="label">
  Save Changes
  <SaveIcon />
</Button>

// Icon only
<Button variant="outline-white" size="sm" buttonType="icon-circle">
  <PlusIcon />
</Button>
```

### Button Specifications

#### Small Buttons
- **Height:** 32px
- **Padding:** 12px horizontal
- **Typography:** `.text-button-small`
- **Icon Size:** 16px

#### Medium Buttons
- **Height:** 48px  
- **Padding:** 20px horizontal
- **Typography:** `.text-button-medium`
- **Icon Size:** 20px

#### Large Buttons
- **Height:** 60px
- **Padding:** 24px horizontal  
- **Typography:** `.text-button-big`
- **Icon Size:** 20px

## 📐 Layout Guidelines

### Spacing
Use Tailwind's default spacing scale, with special attention to:
- **Gap between elements:** `gap-2` (8px), `gap-4` (16px), `gap-6` (24px)
- **Section spacing:** `space-y-8` (32px), `space-y-12` (48px), `space-y-16` (64px)
- **Container padding:** `p-6` (24px), `p-8` (32px)

### Backgrounds
- **Primary background:** `bg-dark-solid`
- **Card backgrounds:** `bg-dark-90` or `bg-light-solid`
- **Subtle overlays:** `bg-dark-10` or `bg-light-10`

### Borders
- **Standard borders:** `border border-light-10` or `border border-dark-10`
- **Emphasis borders:** `border border-light-30` or `border border-dark-30`

## 🎨 Color Usage Guidelines

### Hierarchy and Contrast
1. **Primary text:** Use `text-light-solid` on dark backgrounds
2. **Secondary text:** Use `text-light-80` or `text-light-70` for reduced emphasis
3. **Tertiary text:** Use `text-light-60` for captions and labels
4. **Interactive elements:** Use full opacity colors (`text-light-solid`, `bg-light-solid`)

### Accessibility
- Always ensure sufficient contrast ratios
- Test with screen readers and keyboard navigation
- Use semantic HTML elements with appropriate typography classes

## 📱 Responsive Considerations

The typography scales are optimized for desktop. Consider using Tailwind's responsive prefixes for smaller screens:

```jsx
<h1 className="text-heading-2 md:text-heading-1">
  Responsive Heading
</h1>
```

## 🔧 Implementation Notes

### Tailwind CSS v4
This design system is built for Tailwind CSS v4 and uses:
- Custom `@layer utilities` for typography
- CSS custom properties for colors
- No external configuration in `tailwind.config.js`

### Performance
All utility classes are generated at build time and only included if used in the codebase, ensuring optimal bundle size.

### Extensibility
The system can be extended by adding new utility classes in the `@layer utilities` section of `app/tailwind.css`.

## 📋 Quick Reference

### Most Common Combinations

```jsx
// Page title
<h1 className="text-heading-1 mb-8">Page Title</h1>

// Section header
<h2 className="text-heading-3 mb-6">Section Title</h2>

// Body text
<p className="text-body-2 text-light-80 mb-4">
  Standard paragraph text
</p>

// Caption/label
<span className="text-caption text-light-60">
  Form label or caption
</span>

// Card component
<div className="bg-dark-90 border border-light-10 rounded p-6">
  <h3 className="text-heading-4 mb-4">Card Title</h3>
  <p className="text-body-3 text-light-80">Card content</p>
</div>
```

## 🚀 Demo Pages

- **`/design-system`** - Complete typography and color showcase
- **`/buttons`** - Button component demonstrations

This design system provides a solid foundation for building consistent, accessible, and visually appealing interfaces that align with your brand guidelines.