import { ChevronUp } from "lucide-react";
import { Link } from "@remix-run/react";
import { Button } from "./ui/button";
import { LanguageSelector } from "./LanguageSelector";
import { useLanguage } from "~/hooks/useLanguage";

export function AppFooter() {
  const { t, loading } = useLanguage();
  
  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: "smooth" });
  };

  // Social links don't need translation
  const socialLinks = [
    { label: "X", href: "https://twitter.com/nomadicstudios" },
    { label: "Facebook", href: "https://facebook.com/nomadicstudios" },
    { label: "LinkedIn", href: "https://linkedin.com/company/nomadic-studios" },
    { label: "Instagram", href: "https://instagram.com/nomadicstudios" },
  ];

  // Don't render footer content until translations are loaded
  if (loading) {
    return (
      <footer className="bg-black text-white">
        <div className="px-4 lg:px-14 py-6 lg:py-10">
          <div className="text-center">Loading...</div>
        </div>
      </footer>
    );
  }

  return (
    <footer className="bg-black text-white">
      <div className="px-4 lg:px-14 py-6 lg:py-10">
        {/* Desktop Layout */}
        <div className="hidden lg:block">
          <div className="flex items-start justify-between">
            {/* Logo */}
            <div className="flex-shrink-0">
              <Link to="/" className="block">
                <img
                  src="/images/logo-white.png"
                  alt="Nomadic Studios"
                  className="h-12 w-auto"
                />
              </Link>
            </div>

            {/* Navigation Sections */}
            <div className="flex justify-between w-full max-w-[520px]">
              {/* About Links */}
              <div>
                <h3 className="text-heading-5 !text-white/60 mb-6">{t("footer.about.title", "About")}</h3>
                <nav className="space-y-4">
                  <Link
                    to="/about"
                    className="block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]"
                  >
                    {t("footer.about.joinOurTeam", "Join Our Team")}
                  </Link>
                  <Link
                    to="/investors"
                    className="block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]"
                  >
                    {t("footer.about.investors", "Investors")}
                  </Link>
                  <Link
                    to="/partners"
                    className="block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]"
                  >
                    {t("footer.about.partners", "Partners")}
                  </Link>
                </nav>
              </div>

              {/* Social Media */}
              <div>
                <h3 className="text-heading-5 !text-white/60 mb-6">
                  {t("footer.socialMedia.title", "Social Media")}
                </h3>
                <nav className="space-y-4">
                  {socialLinks.map((link) => (
                    <a
                      key={link.href}
                      href={link.href}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]"
                    >
                      {link.label}
                    </a>
                  ))}
                </nav>
              </div>
            </div>
          </div>

          {/* Bottom Row */}
          <div className="flex items-center justify-between mt-16 pt-8 border-t border-white/10">
            {/* Back to Top */}

            <Button
              onClick={scrollToTop}
              variant="outline-white"
              size="md"
              className="text-footer-bottom border-white !text-white hover:bg-white hover:!text-black px-8 py-3"
            >
              {t("footer.backToTop", "Back to Top")}
              <ChevronUp className="ml-2 h-4 w-4" />
            </Button>

            <div className="flex items-center gap-8">
              <p className="text-footer-bottom text-white/60">
                {t("footer.copyright", "©2025 Nomadic Studios")}
              </p>
              <Link
                to="/terms"
                className="text-footer-bottom text-white/60 hover:text-white transition-colors"
              >
                {t("footer.legal.termsConditions", "Terms & Conditions")}
              </Link>
              <Link
                to="/privacy"
                className="text-footer-bottom text-white/60 hover:text-white transition-colors"
              >
                {t("footer.legal.privacyPolicy", "Privacy Policy")}
              </Link>
              <LanguageSelector />
            </div>
          </div>
        </div>

        {/* Mobile Layout */}
        <div className="lg:hidden">
          {/* Logo */}
          <div className="mb-12">
            <Link to="/" className="block w-fit">
              <img
                src="/images/logo-white.png"
                alt="Nomadic Studios"
                className="h-10 w-auto"
              />
            </Link>
          </div>

          {/* About Section */}
          <div className="mb-12">
            <h3 className="text-heading-5 !text-white/60 mb-6">{t("footer.about.title", "About")}</h3>
            <nav className="space-y-6">
              <Link
                to="/about"
                className="block text-white text-lg uppercase tracking-[0.02em]"
              >
                {t("footer.about.joinOurTeam", "Join Our Team")}
              </Link>
              <Link
                to="/investors"
                className="block text-white text-lg uppercase tracking-[0.02em]"
              >
                {t("footer.about.investors", "Investors")}
              </Link>
              <Link
                to="/partners"
                className="block text-white text-lg uppercase tracking-[0.02em]"
              >
                {t("footer.about.partners", "Partners")}
              </Link>
            </nav>
          </div>

          {/* Social Media Section */}
          <div className="mb-12">
            <h3 className="text-heading-5 !text-white/60 mb-6">{t("footer.socialMedia.title", "Social Media")}</h3>
            <nav className="space-y-6">
              {socialLinks.map((link) => (
                <a
                  key={link.href}
                  href={link.href}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="block text-white text-lg uppercase tracking-[0.02em]"
                >
                  {link.label}
                </a>
              ))}
            </nav>
          </div>

          {/* Copyright */}
          <div className="space-y-6 mb-12">
            <Link
              to="/terms"
              className="block text-white/60 text-footer-bottom uppercase tracking-[0.02em]"
            >
              {t("footer.legal.termsConditions", "Terms & Conditions")}
            </Link>
            <Link
              to="/privacy"
              className="block text-white/60 text-footer-bottom uppercase tracking-[0.02em]"
            >
              {t("footer.legal.privacyPolicy", "Privacy Policy")}
            </Link>
            <LanguageSelector className="text-left" />
            <p className="text-footer-bottom text-white/60">
              {t("footer.copyright", "©2025 Nomadic Studios")}
            </p>
          </div>

          {/* Back to Top Button */}
          <div className="flex justify-center">
            <Button
              onClick={scrollToTop}
              variant="outline-white"
              size="lg"
              className="text-footer-bottom border-white !text-white hover:bg-white hover:!text-black px-8 py-4 w-full"
            >
              {t("footer.backToTop", "Back to Top")}
              <ChevronUp className="ml-2 h-5 w-5" />
            </Button>
          </div>
        </div>
      </div>
    </footer>
  );
}
