import React, { useState } from 'react';
import { ChevronUp } from 'lucide-react';
import { useLanguage, type Language } from '~/hooks/useLanguage';

interface LanguageOption {
  code: Language;
  label: string;
  currency: string;
}

const languages: LanguageOption[] = [
  { code: 'en', label: 'ENG', currency: 'USD' },
  { code: 'sp', label: 'SPA', currency: 'EUR' }
];

interface LanguageSelectorProps {
  className?: string;
}

export function LanguageSelector({ className = "" }: LanguageSelectorProps) {
  const { currentLanguage, changeLanguage } = useLanguage();
  const [isOpen, setIsOpen] = useState(false);

  const currentOption = languages.find(lang => lang.code === currentLanguage) || languages[0];

  const handleLanguageChange = (language: Language) => {
    changeLanguage(language);
    setIsOpen(false);
  };

  return (
    <div className={`relative ${className}`}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center gap-2 text-footer-bottom text-white/60 hover:text-white transition-colors cursor-pointer"
      >
        <span>{currentOption.label} ({currentOption.currency})</span>
        <ChevronUp className={`h-3 w-3 text-white/60 transition-transform duration-200 ${isOpen ? 'rotate-0' : 'rotate-180'}`} />
      </button>

      {isOpen && (
        <>
          {/* Overlay */}
          <div 
            className="fixed inset-0 z-40" 
            onClick={() => setIsOpen(false)}
          />
          
          {/* Dropdown */}
          <div className="absolute bottom-full right-0 mb-2 bg-gray-900 border border-white/20 rounded-lg shadow-xl z-50 min-w-[120px] backdrop-blur-sm">
            {languages.map((language) => (
              <button
                key={language.code}
                onClick={() => handleLanguageChange(language.code)}
                className={`w-full text-left px-4 py-3 text-sm transition-colors first:rounded-t-lg last:rounded-b-lg ${
                  currentLanguage === language.code
                    ? 'bg-white/20 text-white font-medium'
                    : 'text-white/70 hover:bg-white/10 hover:text-white'
                }`}
              >
                {language.label} ({language.currency})
              </button>
            ))}
          </div>
        </>
      )}
    </div>
  );
}