
import React, { useState, useEffect } from "react";
import { useNavigate, useLocation } from "@remix-run/react";
import { authAPI, handleAuthSuccess } from "~/lib/api";
import { Button } from "../ui/button";
import { useLanguage } from "~/hooks/useLanguage";

interface CountryCode {
  code: string;
  country: string;
  flag: string;
}

const SignIn: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [phoneFormData, setPhoneFormData] = useState({
    mobileNumber: "",
    keepLoggedIn: true,
  });

  const [selectedCountry, setSelectedCountry] = useState<CountryCode>({
    code: "+1",
    country: "United States",
    flag: "🇺🇸",
  });

  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [successMessage, setSuccessMessage] = useState<string | null>(null);

  // Sorted country codes alphabetically
  const countryCodes: CountryCode[] = [
    { code: "+61", country: "Australia", flag: "🇦🇺" },
    { code: "+43", country: "Austria", flag: "🇦🇹" },
    { code: "+54", country: "Argentina", flag: "🇦🇷" },
    { code: "+32", country: "Belgium", flag: "🇧🇪" },
    { code: "+359", country: "Bulgaria", flag: "🇧🇬" },
    { code: "+55", country: "Brazil", flag: "🇧🇷" },
    { code: "+56", country: "Chile", flag: "🇨🇱" },
    { code: "+86", country: "China", flag: "🇨🇳" },
    { code: "+420", country: "Czech Republic", flag: "🇨🇿" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+20", country: "Egypt", flag: "🇪🇬" },
    { code: "+33", country: "France", flag: "🇫🇷" },
    { code: "+358", country: "Finland", flag: "🇫🇮" },
    { code: "+30", country: "Greece", flag: "🇬🇷" },
    { code: "+49", country: "Germany", flag: "🇩🇪" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+972", country: "Israel", flag: "🇮🇱" },
    { code: "+62", country: "Indonesia", flag: "🇮🇩" },
    { code: "+81", country: "Japan", flag: "🇯🇵" },
    { code: "+254", country: "Kenya", flag: "🇰🇪" },
    { code: "+82", country: "South Korea", flag: "🇰🇷" },
    { code: "+36", country: "Hungary", flag: "🇭🇺" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+39", country: "Italy", flag: "🇮🇹" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+234", country: "Nigeria", flag: "🇳🇬" },
    { code: "+31", country: "Netherlands", flag: "🇳🇱" },
    { code: "+64", country: "New Zealand", flag: "🇳🇿" },
    { code: "+351", country: "Portugal", flag: "🇵🇹" },
    { code: "+48", country: "Poland", flag: "🇵🇱" },
    { code: "+51", country: "Peru", flag: "🇵🇪" },
    { code: "+63", country: "Philippines", flag: "🇵🇭" },
    { code: "+966", country: "Saudi Arabia", flag: "🇸🇦" },
    { code: "+65", country: "Singapore", flag: "🇸🇬" },
    { code: "+46", country: "Sweden", flag: "🇸🇪" },
    { code: "+41", country: "Switzerland", flag: "🇨🇭" },
    { code: "+90", country: "Turkey", flag: "🇹🇷" },
    { code: "+66", country: "Thailand", flag: "🇹🇭" },
    { code: "+27", country: "South Africa", flag: "🇿🇦" },
    { code: "+44", country: "United Kingdom", flag: "🇬🇧" },
    { code: "+1", country: "United States", flag: "🇺🇸" },
    { code: "+971", country: "UAE", flag: "🇦🇪" },
    { code: "+380", country: "Ukraine", flag: "🇺🇦" },
    { code: "+34", country: "Spain", flag: "🇪🇸" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+84", country: "Vietnam", flag: "🇻🇳" },
    { code: "+58", country: "Venezuela", flag: "🇻🇪" },
    { code: "+52", country: "Mexico", flag: "🇲🇽" },
  ];

  useEffect(() => {
    setIsVisible(true);
    
    // Check for success message from password reset
    if (location.state?.message && location.state?.type === 'success') {
      setSuccessMessage(location.state.message);
      // Clear the location state
      window.history.replaceState({}, document.title);
    }
  }, [location.state]);

  // Handle keyboard navigation for country dropdown
  useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      if (isDropdownOpen) {
        const key = e.key.toLowerCase();
        const matchingCountry = countryCodes.find((country) =>
          country.country.toLowerCase().startsWith(key)
        );

        if (matchingCountry) {
          const countryElement = document.querySelector(
            `[data-country="${matchingCountry.country}"]`
          ) as HTMLElement;
          if (countryElement) {
            countryElement.scrollIntoView({ block: "nearest" });
            countryElement.focus();
          }
        }
      }
    };

    if (isDropdownOpen) {
      document.addEventListener("keydown", handleKeyPress);
    }

    return () => {
      document.removeEventListener("keydown", handleKeyPress);
    };
  }, [isDropdownOpen]);

  const handlePhoneInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type, checked } = e.target;
    setPhoneFormData((prev) => ({
      ...prev,
      [name]: type === "checkbox" ? checked : value,
    }));
    if (error) setError(null);
  };

  const handleCountrySelect = (country: CountryCode) => {
    setSelectedCountry(country);
    setIsDropdownOpen(false);
  };

  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);

    try {
      const fullNumber = `${selectedCountry.code}${phoneFormData.mobileNumber}`;
      const response = await authAPI.sendOTP(fullNumber);

      if (response.status) {
        // Navigate to OTP verification page with phone number
        navigate("/otp-verification", {
          state: {
            phoneNumber: fullNumber,
            requestType: "login",
          },
        });
      } else {
        setError(response.error || t("auth.signin.errors.sendOTPFailed", "Failed to send OTP. Please try again."));
      }
    } catch (err) {
      console.error("Send OTP error:", err);
      setError(
        err instanceof Error
          ? err.message
          : t("auth.signin.errors.unexpected", "An unexpected error occurred. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div
          className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60"
          style={{ animationDelay: "0s", animationDuration: "3s" }}
        />
        <div
          className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40"
          style={{ animationDelay: "1s" }}
        />
        <div
          className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30"
          style={{ animationDelay: "2s" }}
        />
        <div
          className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50"
          style={{ animationDelay: "1.5s", animationDuration: "4s" }}
        />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Hero Title */}
        <div
          className={`text-center mb-12 transform transition-all duration-1000 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.signin.welcomeBack", "WELCOME BACK")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.signin.subtitle", "Continue your creative journey")}
          </p>
        </div>

        {/* Login Card */}
        <div
          className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            <h2 className="text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider">
              {t("auth.signin.title", "Sign In")}
            </h2>

            {/* Success Message */}
            {successMessage && (
              <div className="mb-6 p-4 bg-green-500/20 border border-green-500/30 text-green-200 rounded-xl text-sm backdrop-blur-sm">
                {successMessage}
              </div>
            )}

            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            {/* Phone Login Form */}
            <form onSubmit={handleSendOTP} className="space-y-6">
              {/* Mobile Number */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.signin.mobileNumber", "Mobile Number")}
                </label>
                <div className="flex relative">
                  {/* Country Code Dropdown */}
                  <div className="relative">
                    <button
                      type="button"
                      onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                      className="flex items-center px-4 py-4 bg-white/10 border border-white/20 border-r-0 hover:bg-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 backdrop-blur-sm transition-all"
                    >
                      <span className="text-lg mr-2">
                        {selectedCountry.flag}
                      </span>
                      <span className="text-sm text-white/80 mr-1">
                        {selectedCountry.code}
                      </span>
                      <svg
                        className="w-4 h-4 text-white/60"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M19 9l-7 7-7-7"
                        />
                      </svg>
                    </button>

                    {/* Dropdown Menu */}
                    {isDropdownOpen && (
                      <div className="absolute z-20 top-full left-0 w-64 bg-black border border-white/20 shadow-2xl max-h-60 overflow-y-auto">
                        {countryCodes.map((country, index) => (
                          <button
                            key={index}
                            type="button"
                            onClick={() => handleCountrySelect(country)}
                            data-country={country.country}
                            className="w-full flex items-center px-4 py-3 text-left hover:bg-white/20 focus:outline-none focus:bg-white/20 transition-colors"
                          >
                            <span className="text-lg mr-3">{country.flag}</span>
                            <span className="text-sm text-white/80 mr-2">
                              {country.code}
                            </span>
                            <span className="text-sm text-white truncate">
                              {country.country}
                            </span>
                          </button>
                        ))}
                      </div>
                    )}
                  </div>

                  <input
                    type="tel"
                    name="mobileNumber"
                    placeholder={t("auth.signin.phonePlaceholder", "Enter your mobile number")}
                    value={phoneFormData.mobileNumber}
                    onChange={handlePhoneInputChange}
                    className="flex-1 px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                  />
                </div>

                {/* Click outside to close dropdown */}
                {isDropdownOpen && (
                  <div
                    className="fixed inset-0 z-10"
                    onClick={() => setIsDropdownOpen(false)}
                  />
                )}
              </div>

              {/* Keep me logged in */}
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    id="keepLoggedInPhone"
                    name="keepLoggedIn"
                    checked={phoneFormData.keepLoggedIn}
                    onChange={handlePhoneInputChange}
                    className="w-5 h-5 text-orange-500 bg-white/10 border-white/20 rounded focus:ring-orange-400 focus:ring-2"
                  />
                  <label
                    htmlFor="keepLoggedInPhone"
                    className="ml-3 text-sm text-white/80"
                  >
                    {t("auth.signin.keepLoggedIn", "Keep me logged in")}
                  </label>
                </div>
              </div>

              {/* Send OTP button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full"
                disabled={isLoading}
                arrowRight={!isLoading}
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    {t("auth.signin.sendingOTP", "Sending OTP...")}
                  </>
                ) : (
                  t("auth.signin.sendOTP", "Send OTP")
                )}
              </Button>

              {/* Alternative Options */}
              <div className="text-center space-y-4 pt-4">
                <Button
                  type="button"
                  variant="outline-white"
                  size="md"
                  className="w-full"
                  onClick={() => navigate("/login-email")}
                >
                  {t("auth.signin.loginWithEmail", "Login with email instead")}
                </Button>

                <Button
                  type="button"
                  variant="outline-white"
                  size="md"
                  className="w-full"
                  onClick={() => navigate("/reset-password")}
                >
                  {t("auth.signin.forgotPassword", "Forgot Password?")}
                </Button>

                <div className="text-white/60 text-sm">
                  {t("auth.signin.noAccount", "Don't have an account?")}{" "}
                  <Button
                    type="button"
                    variant="outline-white"
                    size="sm"
                    className="ml-2"
                    onClick={() => navigate("/signup")}
                  >
                    {t("auth.signin.signUp", "Sign up")}
                  </Button>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SignIn;
