import React, { useState, useEffect } from "react";
import { Link, useNavigate } from "@remix-run/react";
import { Button } from "../ui/button";
import { useLanguage } from "~/hooks/useLanguage";

interface MembershipTier {
  name: string;
  price: string;
  period: string;
  points: string;
  features: string[];
  buttonText: string;
  isPopular?: boolean;
}

interface AddOnService {
  id: string;
  name: string;
  price: string;
  description: string;
}

const SignupFirst: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [selectedTier, setSelectedTier] = useState<number | null>(null);
  const [selectedAddOns, setSelectedAddOns] = useState<string[]>([]);

  useEffect(() => {
    setIsVisible(true);
  }, []);

  const membershipTiers: MembershipTier[] = [
    {
      name: t("auth.signup.plans.guest.name", "Guest"),
      price: "$0",
      period: t("auth.signup.plans.period", "month"),
      points: "0 Pts",
      features: [
        t("auth.signup.plans.guest.feature1", "Just one day/one by member"), 
        t("auth.signup.plans.guest.feature2", "Zero guests")
      ],
      buttonText: "Choose plan",
    },
    {
      name: t("auth.signup.plans.starvingArtist.name", "Starving Artist"),
      price: "$1149",
      period: t("auth.signup.plans.period", "month"),
      points: "10000 Pts",
      features: [
        t("auth.signup.plans.starvingArtist.feature1", "Book up 1 month in advance"), 
        t("auth.signup.plans.starvingArtist.feature2", "One guests")
      ],
      buttonText: "Choose plan",
    },
    {
      name: t("auth.signup.plans.upAndComing.name", "Up and coming"),
      price: "$1899",
      period: t("auth.signup.plans.period", "month"),
      points: "17500 Pts",
      features: [
        t("auth.signup.plans.upAndComing.feature1", "Book up 1 month in advance"), 
        t("auth.signup.plans.upAndComing.feature2", "Two guests")
      ],
      buttonText: "Choose plan",
      isPopular: true,
    },
    {
      name: t("auth.signup.plans.almostFamous.name", "Almost Famous"),
      price: "$2500",
      period: t("auth.signup.plans.period", "month"),
      points: "2500 Pts",
      features: [
        t("auth.signup.plans.almostFamous.feature1", "Book up 3 month in advance"), 
        t("auth.signup.plans.almostFamous.feature2", "Four guests")
      ],
      buttonText: "Choose plan",
    },
  ];

  const addOnServices: AddOnService[] = [
    {
      id: "test-service",
      name: t('auth.signup.addons.testService.name'),
      price: "$20.00",
      description: t('auth.signup.addons.testService.description'),
    },
    {
      id: "asdfds",
      name: t('auth.signup.addons.premiumSupport.name'),
      price: "$10.00",
      description: t('auth.signup.addons.premiumSupport.description'),
    },
  ];

  const handleAddOnToggle = (serviceId: string) => {
    setSelectedAddOns((prev) =>
      prev.includes(serviceId)
        ? prev.filter((id) => id !== serviceId)
        : [...prev, serviceId]
    );
  };

  const handleTierSelect = (index: number) => {
    setSelectedTier(index);
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-30"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/90 via-[#030303]/70 to-[#030303]/90" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div
          className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60"
          style={{ animationDelay: "0s", animationDuration: "3s" }}
        />
        <div
          className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40"
          style={{ animationDelay: "1s" }}
        />
        <div
          className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30"
          style={{ animationDelay: "2s" }}
        />
        <div
          className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50"
          style={{ animationDelay: "1.5s", animationDuration: "4s" }}
        />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen px-4 lg:px-8 py-12">
        {/* Hero Title */}
        <div
          className={`text-center mb-16 transform transition-all duration-1000 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.signup.joinCommunity", "JOIN THE COMMUNITY")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light mb-8">
            {t("auth.signup.subtitle", "Choose your creative journey")}
          </p>
          <div className="text-white/60 text-sm space-y-1">
            <p>✨ {t("auth.signup.benefits.relocation", "14 days relocation/year")}</p>
            <p>🎬 {t("auth.signup.benefits.studio", "14 days Studio time/year")}</p>
          </div>
        </div>

        {/* Membership Cards */}
        <div
          className={`max-w-7xl mx-auto mb-16 transform transition-all duration-1000 delay-300 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {membershipTiers.map((tier, index) => (
              <div
                key={index}
                className={`relative bg-white/10 backdrop-blur-lg border rounded-2xl p-6 shadow-2xl transition-all duration-300 cursor-pointer transform hover:scale-105 ${
                  tier.isPopular
                    ? "border-orange-400 ring-2 ring-orange-400/50"
                    : "border-white/20 hover:border-orange-400/50"
                } ${
                  selectedTier === index
                    ? "ring-2 ring-orange-400 scale-105"
                    : ""
                }`}
                onClick={() => handleTierSelect(index)}
              >
                {/* Popular Badge */}
                {tier.isPopular && (
                  <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                    <div className="bg-gradient-to-r from-orange-500 to-orange-600 text-white text-xs px-4 py-1 rounded-full uppercase tracking-wider font-semibold">
                      {t("auth.signup.mostPopular", "Most Popular")}
                    </div>
                  </div>
                )}

                {/* Card Content */}
                <div className="text-center">
                  {/* Tier Name */}
                  <h3 className="text-white text-xl font-bold mb-4 uppercase tracking-wider">
                    {tier.name}
                  </h3>

                  {/* Price */}
                  <div className="mb-6">
                    <div className="text-4xl font-bold text-white mb-1">
                      {tier.price}
                    </div>
                    <div className="text-white/60 text-sm">
                      {t("auth.signup.perPeriod", "per")} {tier.period}
                    </div>
                    <div className="text-orange-400 font-semibold mt-2">
                      {tier.points}
                    </div>
                  </div>

                  {/* Features */}
                  <ul className="text-white/80 text-sm space-y-3 mb-8">
                    {tier.features.map((feature, featureIndex) => (
                      <li key={featureIndex} className="flex items-start">
                        <svg
                          className="w-4 h-4 text-orange-400 mr-3 mt-0.5 flex-shrink-0"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        {feature}
                      </li>
                    ))}
                  </ul>

                  {/* Button */}
                  <Button
                    variant={selectedTier === index ? "fill-black" : tier.isPopular ? "fill-white" : "outline-white"}
                    size="md"
                    className="w-full"
                  >
                    {selectedTier === index ? (
                      <span className="flex items-center justify-center">
                        <svg
                          className="w-5 h-5 mr-2"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        {t("auth.signup.selected", "Selected")}
                      </span>
                    ) : (
                      t("auth.signup.choosePlan", "Choose plan")
                    )}
                  </Button>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Add-on Services Section */}
        <div
          className={`max-w-4xl mx-auto mb-16 transform transition-all duration-1000 delay-500 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="text-center mb-8">
            <h2 className="text-3xl lg:text-4xl font-bold text-white uppercase tracking-tight mb-4">
              {t("auth.signup.addons.title", "Select Add-on Services")}
            </h2>
            <p className="text-white/60 text-sm">
              {t("auth.signup.addons.subtitle", "Enhance your membership with optional premium services")}
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {addOnServices.map((service) => (
              <div
                key={service.id}
                className={`bg-white/10 backdrop-blur-lg border rounded-2xl p-6 shadow-2xl transition-all duration-300 cursor-pointer transform hover:scale-105 ${
                  selectedAddOns.includes(service.id)
                    ? "border-orange-400 ring-2 ring-orange-400/50 bg-orange-500/10"
                    : "border-white/20 hover:border-orange-400/50"
                }`}
                onClick={() => handleAddOnToggle(service.id)}
              >
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h3 className="text-white text-xl font-bold mb-2 uppercase tracking-wider">
                      {service.name}
                    </h3>
                    <p className="text-white/70 text-sm mb-4">
                      {service.description}
                    </p>
                  </div>
                  <div className="ml-4">
                    <div className="text-2xl font-bold text-orange-400">
                      {service.price}
                    </div>
                    <div className="text-white/60 text-xs text-right">
                      {t("auth.signup.addons.perMonth", "per month")}
                    </div>
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <div className="flex items-center">
                    <div
                      className={`w-5 h-5 rounded border-2 transition-all duration-200 ${
                        selectedAddOns.includes(service.id)
                          ? "border-orange-400 bg-orange-400"
                          : "border-white/40"
                      }`}
                    >
                      {selectedAddOns.includes(service.id) && (
                        <svg
                          className="w-3 h-3 text-white ml-0.5 mt-0.5"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                      )}
                    </div>
                    <span className="text-white/80 text-sm ml-3">
                      {selectedAddOns.includes(service.id)
                        ? t("auth.signup.addons.selected", "Selected")
                        : t("auth.signup.addons.addToPlan", "Add to plan")}
                    </span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Continue Button */}
        {selectedTier !== null && (
          <div
            className={`text-center mb-16 transform transition-all duration-1000 delay-700 ${
              isVisible
                ? "translate-y-0 opacity-100"
                : "translate-y-10 opacity-0"
            }`}
          >
            <Button
              variant="fill-white"
              size="lg"
              arrowRight={true}
              onClick={() => {
                const selectedTierData = membershipTiers[selectedTier];
                navigate("/signup-details", {
                  state: {
                    selectedTier: selectedTierData,
                    selectedAddOns: selectedAddOns
                      .map((id) =>
                        addOnServices.find((service) => service.id === id)
                      )
                      .filter(Boolean),
                  },
                });
              }}
            >
              {t("auth.signup.subscribe", "Subscribe")}
            </Button>
          </div>
        )}

        {/* Footer Links */}
        <div
          className={`text-center mt-16 transform transition-all duration-1000 delay-500 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="text-white/60 text-sm mb-4">
            {t("auth.signup.haveAccount", "Already have an account?")}{" "}
            <Button
              variant="outline-white"
              size="sm"
              className="ml-2"
              onClick={() => navigate("/login")}
            >
              {t("auth.signup.signIn", "Sign in")}
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SignupFirst;
