import { cn } from "~/lib/utils";

export interface ImageGridItem {
  id: string;
  image: string;
  title: string;
  onClick?: () => void;
}

interface ImageGridProps {
  items: ImageGridItem[];
  className?: string;
  variant?: "single" | "double" | "triple";
  showBorders?: boolean;
}

export function ImageGrid({
  items,
  className,
  variant = "triple",
  showBorders = true,
}: ImageGridProps) {
  return (
    <div
      className={cn(
        showBorders && "border-y border-[rgba(18,18,18,0.2)]",
        className
      )}
    >
      <div className="flex overflow-x-auto gap-0 scrollbar-hide snap-x snap-mandatory">
        {items.map((item, index) => {
          const isLastItem = index === items.length - 1;
          const isFirstItem = index === 0;

          return (
            <div
              key={item.id}
              onClick={item.onClick}
              className={cn(
                "group flex-shrink-0 flex flex-col items-center pt-4 sm:pt-8 lg:pt-12 snap-center",
                item.onClick && "cursor-pointer",
                showBorders &&
                  !isLastItem &&
                  "border-r border-[rgba(18,18,18,0.2)]",
                isFirstItem && "ml-4 sm:ml-6",
                isLastItem && "mr-4 sm:mr-6"
              )}
            >
              <div className="w-[240px] sm:w-[320px] lg:w-[380px] h-[240px] sm:h-[360px] lg:h-[440px] overflow-hidden px-4 sm:px-6 lg:px-8">
                <img
                  src={item.image}
                  alt={item.title}
                  className="w-full h-full object-cover group-hover:scale-101 transition-transform duration-500"
                />
              </div>

              <div className="p-3 sm:p-4 lg:p-6">
                <h3 className="text-[12px] sm:text-[14px] lg:text-[18px] font-medium uppercase tracking-[0.02em] text-[#121212] font-clash-grotesk text-center">
                  {item.title}
                </h3>
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
}
