import { useState } from "react";
import { Button } from "~/components/ui/button";
import { SectionHeader } from "~/components/shared/SectionHeader";
import { PersonCard } from "./components/PersonCard";
import { PeopleToolbar } from "./components/PeopleToolbar";

export interface Person {
  id: string;
  photo: string;
  name: string;
  surname: string;
  role: string;
  location: string;
  href: string;
}

export interface FilterOptions {
  locations: string[];
  roles: string[];
}

interface PeopleSectionProps {
  eyebrow?: string;
  title?: string;
  buttonText?: string;
  peopleData?: Person[];
  filterOptions?: FilterOptions;
}

export function PeopleSection({
  eyebrow,
  title,
  buttonText,
  peopleData = [],
  filterOptions = { locations: [], roles: [] },
}: PeopleSectionProps = {}) {
  const [filters, setFilters] = useState({
    location: "",
    role: "",
  });

  const filteredPeople = peopleData.filter((person) => {
    const locationMatch =
      !filters.location || person.location === filters.location;
    const roleMatch = !filters.role || person.role === filters.role;
    return locationMatch && roleMatch;
  });

  const handleFilterChange = (newFilters: typeof filters) => {
    setFilters(newFilters);
  };

  return (
    <section className="relative w-full bg-white">
      <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-8 lg:py-24">
        {/* Header */}
        <div className="mb-6 lg:mb-16">
          {/* Mobile Header Layout */}
          <div className="lg:hidden">
            {title && (
              <SectionHeader
                layout="vertical"
                eyebrow={eyebrow}
                title={title}
                className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black [&_h2]:text-[24px] [&_h2]:tracking-[-0.02em] mb-4"
              />
            )}
            {buttonText && (
              <Button
                variant="outline-black"
                size="md"
                className="border-black/30 hover:border-black/50 uppercase tracking-wider text-button-small !text-black"
                arrowRight
              >
                {buttonText}
              </Button>
            )}
          </div>

          {/* Desktop Header Layout */}
          <div className="hidden lg:flex lg:items-end lg:justify-between gap-4">
            {title && (
              <SectionHeader
                layout="vertical"
                eyebrow={eyebrow}
                title={title}
                className="flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black"
              />
            )}

            {/* CTA Button */}
            {buttonText && (
              <div className="flex items-center">
                <Button
                  variant="outline-black"
                  size="lg"
                  className="border-black/30 hover:border-black/50 uppercase tracking-wider font-medium !text-black"
                  arrowRight
                >
                  {buttonText}
                </Button>
              </div>
            )}
          </div>
        </div>

        {/* Toolbar - Hidden on mobile */}
        <div className="hidden lg:block">
          <PeopleToolbar
            filters={filters}
            filterOptions={filterOptions}
            onChange={handleFilterChange}
          />
        </div>

        {/* People Grid */}
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 lg:gap-6">
          {filteredPeople.map((person) => (
            <PersonCard key={person.id} {...person} />
          ))}
        </div>
      </div>
    </section>
  );
}
