import { useNavigate } from "@remix-run/react";
import { ImageGrid, type ImageGridItem } from "~/components/shared/ImageGrid";
import type { LocationType } from "~/components/pages/locations/WelcomeSection/WelcomeSection";

export interface Studio {
  id: string;
  image: string;
  title: string;
}

interface StudiosGridProps {
  location?: LocationType;
}

const studioTypes = [
  "VIDEO STUDIO",
  "MUSIC STUDIO", 
  "PHOTOGRAPHY STUDIO",
  "PODCAST STUDIO",
  "DANCE STUDIO",
  "ART STUDIO",
  "FASHION STUDIO",
  "RECORDING STUDIO",
  "LIVESTREAM STUDIO",
  "CREATIVE SUITE"
];

const studioImages = [
  "/images/studios/studio-1.png",
  "/images/studios/music-1.png", 
  "/images/studios/photo-1.png"
];

function generateRandomStudios(location: LocationType): Studio[] {
  const seed = location.split('').reduce((a, b) => a + b.charCodeAt(0), 0);
  const random = (seed: number) => {
    const x = Math.sin(seed) * 10000;
    return x - Math.floor(x);
  };
  
  const numStudios = 3 + Math.floor(random(seed) * 3);
  const studios: Studio[] = [];
  const usedTypes = new Set<string>();
  
  for (let i = 0; i < numStudios; i++) {
    let studioType;
    let attempts = 0;
    do {
      const typeIndex = Math.floor(random(seed + i + attempts) * studioTypes.length);
      studioType = studioTypes[typeIndex];
      attempts++;
    } while (usedTypes.has(studioType) && attempts < 20);
    
    usedTypes.add(studioType);
    
    const imageIndex = Math.floor(random(seed + i + 100) * studioImages.length);
    
    studios.push({
      id: `${location}-${studioType.toLowerCase().replace(/\s+/g, '-')}`,
      image: studioImages[imageIndex],
      title: studioType,
    });
  }
  
  return studios;
}

export function StudiosGrid({ location = "bali" }: StudiosGridProps) {
  const navigate = useNavigate();
  const studiosData = generateRandomStudios(location);
  
  const items: ImageGridItem[] = studiosData.map(studio => ({
    id: studio.id,
    image: studio.image,
    title: studio.title,
    onClick: () => navigate(`/book-studio/${studio.id}`),
  }));

  return <ImageGrid items={items} />;
}