import { useState, useEffect, useCallback } from 'react';

type Language = 'en' | 'sp';

interface LanguageData {
  [key: string]: any;
}

const LANGUAGE_STORAGE_KEY = 'nomadic-language';
const DEFAULT_LANGUAGE: Language = 'en';

export function useLanguage() {
  const [currentLanguage, setCurrentLanguage] = useState<Language>(DEFAULT_LANGUAGE);
  const [translations, setTranslations] = useState<LanguageData>({});
  const [loading, setLoading] = useState(true);

  // Load saved language from localStorage on mount
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedLanguage = localStorage.getItem(LANGUAGE_STORAGE_KEY) as Language;
      if (savedLanguage && (savedLanguage === 'en' || savedLanguage === 'sp')) {
        setCurrentLanguage(savedLanguage);
      }
    }
  }, []);

  // Load translations when language changes
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        setLoading(true);
        const response = await import(`../config/lang/${currentLanguage}.json`);
        setTranslations(response.default || response);
      } catch (error) {
        console.error(`Failed to load translations for ${currentLanguage}:`, error);
        // Fallback to English if current language fails
        if (currentLanguage !== 'en') {
          try {
            const fallback = await import('../config/lang/en.json');
            setTranslations(fallback.default || fallback);
          } catch (fallbackError) {
            console.error('Failed to load fallback English translations:', fallbackError);
          }
        }
      } finally {
        setLoading(false);
      }
    };

    loadTranslations();
  }, [currentLanguage]);

  // Change language function
  const changeLanguage = useCallback((newLanguage: Language) => {
    setCurrentLanguage(newLanguage);
    if (typeof window !== 'undefined') {
      localStorage.setItem(LANGUAGE_STORAGE_KEY, newLanguage);
    }
    // Force re-render by triggering a state update
    setLoading(true);
  }, []);

  // Get translation by key with dot notation support
  const t = useCallback((key: string, defaultValue?: string): string => {
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return defaultValue || key;
      }
    }
    
    return typeof value === 'string' ? value : defaultValue || key;
  }, [translations]);

  // Get nested object by key
  const getSection = useCallback((key: string): any => {
    const keys = key.split('.');
    let value = translations;
    
    for (const k of keys) {
      if (value && typeof value === 'object' && k in value) {
        value = value[k];
      } else {
        return {};
      }
    }
    
    return value || {};
  }, [translations]);

  return {
    currentLanguage,
    changeLanguage,
    t,
    getSection,
    loading,
    translations
  };
}

export type { Language };