// API configuration and utility functions
const API_BASE_URL = "https://nomadicstudios.net:444/api";

// Types for API responses
export interface AuthResponse {
  status: boolean;
  token?: string;
  user?: {
    uuid: string;
    fname: string;
    lname: string;
    email: string;
    role: string;
  };
  expires_in?: number;
  error?: string;
  message?: string;
  mobile_number?: string; // For forgot password response
  uuid?: string; // For signup response
}

export interface LogoutResponse {
  status: boolean;
  message?: string;
  error?: string;
}

export interface OTPResponse {
  status: boolean;
  message?: string;
  error?: string;
}

export interface Studio {
  id: number;
  name: string;
  studio_type: string;
  banner_picture: string;
  description: string;
}

export interface Person {
  uuid: string;
  fname: string;
  lname: string;
  profile_pic: string;
  type: string;
  location: string;
  profile_link: string;
  instagram_link: string;
  facebook_link: string;
  twitter_link: string;
  sm_link: string;
  description: string;
}

export interface Event {
  event_id: number;
  location_id: number;
  user_uuid: string;
  event_type: "online" | "location";
  public_private: "public" | "private";
  stream_url?: string;
  event_title: string;
  event_date: string;
  event_time: string;
  duration?: string;
  address: string;
  ticket_price: number;
  ticket_url?: string;
  thumbnail: string;
  description: string;
  event_status: number;
  created_by: number;
  updated_by?: number;
  created_at: string;
  updated_at: string;
  channel_arn?: string;
  stream_arn?: string;
  playbackurl?: string;
  stream_key?: string;
  interested_count: number;
  going_count: number;
}

export interface APIResponse<T> {
  status: boolean;
  data?: T;
  error?: string;
}

export interface Location {
  id: number;
  location: string;
  type?: string;
  coming_soon?: number;
  banner_picture?: string;
  description?: string;
}

// Get auth token from localStorage
const getAuthToken = (): string | null => {
  if (typeof window !== "undefined") {
    return localStorage.getItem("auth_token");
  }
  return null;
};

// Set auth token in localStorage
export const setAuthToken = (token: string): void => {
  if (typeof window !== "undefined") {
    localStorage.setItem("auth_token", token);
  }
};

// Remove auth token from localStorage
export const removeAuthToken = (): void => {
  if (typeof window !== "undefined") {
    localStorage.removeItem("auth_token");
    localStorage.removeItem("user_data");
  }
};

// Store user data
export const setUserData = (user: any): void => {
  if (typeof window !== "undefined") {
    localStorage.setItem("user_data", JSON.stringify(user));
  }
};

// Get user data
export const getUserData = (): any => {
  if (typeof window !== "undefined") {
    const userData = localStorage.getItem("user_data");
    return userData ? JSON.parse(userData) : null;
  }
  return null;

  // BYPASS LOGIN FOR TESTING (commented out):
  // return {
  //   fname: 'John',
  //   lname: 'Doe',
  //   email: 'john.doe@example.com',
  //   uuid: 'test-user-123'
  // };
};

// Generic API request function
async function apiRequest<T>(
  endpoint: string,
  options: RequestInit = {}
): Promise<T> {
  const token = getAuthToken();

  const defaultHeaders: Record<string, string> = {
    "Content-Type": "application/json",
    Accept: "application/json",
  };

  if (token) {
    defaultHeaders.Authorization = `Bearer ${token}`;
  }

  const response = await fetch(`${API_BASE_URL}${endpoint}`, {
    ...options,
    headers: {
      ...defaultHeaders,
      ...options.headers,
    },
  });

  const data = await response.json();

  if (!response.ok) {
    throw new Error(data.error || `HTTP error! status: ${response.status}`);
  }

  return data;
}

// Authentication APIs
export const authAPI = {
  // Email/Password Login
  login: async (email: string, password: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/login", {
      method: "POST",
      body: JSON.stringify({ email, password }),
    });
  },

  // Send OTP to mobile number
  sendOTP: async (mobile_number: string): Promise<OTPResponse> => {
    return apiRequest<OTPResponse>("/send-otp", {
      method: "POST",
      body: JSON.stringify({ mobile_number }),
    });
  },

  // Verify OTP and login
  verifyOTP: async (
    mobile_number: string,
    otp_number: string,
    request_type: string = "login",
    additionalData?: {
      fname?: string;
      lname?: string;
      email?: string;
      plan?: string;
      addon_services?: string[];
      password?: string;
      confirm_password?: string;
    }
  ): Promise<AuthResponse> => {
    const payload: any = {
      request_type,
      otp_number,
      mobile_number,
    };

    // Add additional data if provided
    if (additionalData) {
      Object.assign(payload, additionalData);
    }

    return apiRequest<AuthResponse>("/verify-otp", {
      method: "POST",
      body: JSON.stringify(payload),
    });
  },

  // Logout
  logout: async (): Promise<LogoutResponse> => {
    return apiRequest<LogoutResponse>("/logout", {
      method: "POST",
    });
  },

  // Forgot Password - Send OTP to mobile
  forgotPassword: async (email: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/forgot-password", {
      method: "POST",
      body: JSON.stringify({ email }),
    });
  },

  // Reset Password
  resetPassword: async (email: string): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/reset-password", {
      method: "POST",
      body: JSON.stringify({ email }),
    });
  },

  // Save Profile Picture
  saveProfilePicture: async (
    uuid: string,
    image: string
  ): Promise<AuthResponse> => {
    return apiRequest<AuthResponse>("/save-profile-picture", {
      method: "POST",
      body: JSON.stringify({ uuid, image }),
    });
  },

  // Save Additional Information
  saveAdditionalInfo: async (data: {
    uuid: string;
    home_city: string;
    creator_interest?: string[];
    profile_link?: string;
    instagram_link?: string;
    facebook_link?: string;
    twitter_link?: string;
    sm_link?: string;
    referred_by?: string;
  }): Promise<AuthResponse> => {
    // Send data as-is without JSON stringification
    return apiRequest<AuthResponse>("/additional-info-save", {
      method: "POST",
      body: JSON.stringify(data),
    });
  },
};

// Locations API
export const locationsAPI = {
  // Get all locations
  getLocations: async (): Promise<APIResponse<Location[]>> => {
    return apiRequest<APIResponse<Location[]>>("/locations", {
      method: "GET",
    });
  },
};

// Studios API
export const studiosAPI = {
  getByLocation: async (locationId: number): Promise<APIResponse<Studio[]>> => {
    return apiRequest<APIResponse<Studio[]>>(`/studios/${locationId}`);
  },
};

// People API
export const peopleAPI = {
  getAll: async (): Promise<APIResponse<Person[]>> => {
    return apiRequest<APIResponse<Person[]>>("/peoples");
  },
};

// Events API
export const eventsAPI = {
  getAll: async (
    filterDate?: string,
    filterLocation?: number
  ): Promise<APIResponse<Event[]>> => {
    const params = new URLSearchParams();
    if (filterDate) params.append("filterDate", filterDate);
    if (filterLocation)
      params.append("filterLocation", filterLocation.toString());

    const queryString = params.toString();
    const endpoint = queryString ? `/events?${queryString}` : "/events";

    return apiRequest<APIResponse<Event[]>>(endpoint);
  },
};

// Auth helper functions
export const isAuthenticated = (): boolean => {
  return getAuthToken() !== null;

  // BYPASS LOGIN FOR TESTING (commented out):
  // return true;
};

export const handleAuthSuccess = (response: AuthResponse): void => {
  if (response.token) {
    setAuthToken(response.token);
    console.log("Auth token set");
  }
  if (response.user) {
    setUserData(response.user);
    console.log("User data set");
  }
};

export const handleLogout = async (): Promise<void> => {
  try {
    await authAPI.logout();
  } catch (error) {
    console.error("Logout error:", error);
  } finally {
    removeAuthToken();
  }

  // BYPASS LOGIN FOR TESTING (commented out):
  // console.log('Logout bypassed for testing');
  // removeAuthToken();
};
