import React, { useState, useEffect } from "react";
import { Link } from "@remix-run/react";
import type { MetaFunction } from "@remix-run/node";
import { eventsAPI, locationsAPI, type Event, type Location } from "~/lib/api";
import { PublicRoute } from "~/components/RouteProtection";
import { useLanguage } from "~/hooks/useLanguage";

export const meta: MetaFunction = () => {
  return [
    { title: "Events - Nomadics Studios" },
    {
      name: "description",
      content: "Discover and join exciting events at Nomadics Studios. Connect with creative professionals and explore new opportunities.",
    },
  ];
};

export default function Events() {
  const { t } = useLanguage();
  const [events, setEvents] = useState<Event[]>([]);
  const [locations, setLocations] = useState<Location[]>([]);
  const [selectedLocation, setSelectedLocation] = useState<number | null>(null);
  const [selectedDate, setSelectedDate] = useState<string>("");
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Format date for display
  const formatEventDate = (dateString: string) => {
    const date = new Date(dateString);
    const day = date.getDate().toString();
    const month = date.toLocaleDateString('en-US', { month: 'short' }).toUpperCase();
    return { day, month };
  };

  // Format time for display
  const formatEventTime = (timeString: string) => {
    return timeString;
  };

  // Generate event slug from title
  const generateSlug = (title: string) => {
    return title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
  };

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        
        // Fetch locations and events in parallel
        const [locationsResponse, eventsResponse] = await Promise.all([
          locationsAPI.getLocations(),
          eventsAPI.getAll(selectedDate, selectedLocation || undefined)
        ]);

        if (locationsResponse.status && locationsResponse.data) {
          // Remove duplicates and add "All Locations" option
          const uniqueLocations = locationsResponse.data.filter((location, index, self) => 
            index === self.findIndex(l => l.id === location.id && l.location === location.location)
          );
          setLocations(uniqueLocations);
        }

        if (eventsResponse.status && eventsResponse.data) {
          setEvents(eventsResponse.data);
        } else {
          setError('Failed to fetch events');
        }
      } catch (err) {
        setError('An error occurred while fetching data');
        console.error('Error fetching data:', err);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, [selectedDate, selectedLocation]);

  const handleLocationFilter = (locationId: number | null) => {
    setSelectedLocation(locationId);
  };

  const handleDateFilter = (date: string) => {
    setSelectedDate(date);
  };

  if (loading) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen pb-24">
          <div className="max-w-7xl mx-auto pt-8 px-4">
            <div className="flex justify-center items-center h-64">
              <div className="text-gray-500">{t("events.loading", "Loading events...")}</div>
            </div>
          </div>
        </main>
      </PublicRoute>
    );
  }

  if (error) {
    return (
      <PublicRoute>
        <main className="bg-white min-h-screen pb-24">
          <div className="max-w-7xl mx-auto pt-8 px-4">
            <div className="flex justify-center items-center h-64">
              <div className="text-red-500">{t("events.error", "Error")}: {error}</div>
            </div>
          </div>
        </main>
      </PublicRoute>
    );
  }

  return (
    <PublicRoute>
      <main className="bg-white min-h-screen pb-24">
        {/* Filters */}
        <div className="max-w-7xl mx-auto pt-8 px-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
            {/* Location Filter */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                {t("events.filters.location", "LOCATION")}
              </label>
              <div className="flex flex-wrap gap-3">
                <button
                  className={`px-5 py-2 rounded-full border font-medium text-sm transition ${
                    selectedLocation === null
                      ? "bg-black text-white"
                      : "bg-white text-black border-black hover:bg-gray-100"
                  }`}
                  onClick={() => handleLocationFilter(null)}
                >
                  {t("events.filters.allLocations", "All Locations")}
                </button>
                {locations.map((location) => (
                  <button
                    key={location.id}
                    className={`px-5 py-2 rounded-full border font-medium text-sm transition ${
                      selectedLocation === location.id
                        ? "bg-black text-white"
                        : "bg-white text-black border-black hover:bg-gray-100"
                    }`}
                    onClick={() => handleLocationFilter(location.id)}
                  >
                    {location.location}
                  </button>
                ))}
              </div>
            </div>

            {/* Date Filter */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                {t("events.filters.date", "DATE")}
              </label>
              <input
                type="date"
                value={selectedDate}
                onChange={(e) => handleDateFilter(e.target.value)}
                className="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-black"
              />
            </div>
          </div>
        </div>

        {/* Events Grid */}
        <div className="max-w-7xl mx-auto px-4">
          {events.length === 0 ? (
            <div className="text-center py-12">
              <div className="text-gray-500">{t("events.noEvents", "No events found matching your filters.")}</div>
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {events.map((event) => {
                const { day, month } = formatEventDate(event.event_date);
                const slug = generateSlug(event.event_title);
                
                return (
                  <Link
                    key={event.event_id}
                    to={`/events-details/${slug}`}
                    className="group block bg-white rounded-lg shadow-sm overflow-hidden hover:shadow-xl transition-all duration-300 transform hover:scale-[1.02] cursor-pointer"
                  >
                    <div className="relative overflow-hidden">
                      <img
                        src={event.thumbnail}
                        alt={event.event_title}
                        className="w-full h-40 object-cover transition-transform duration-500 group-hover:scale-110"
                        onError={(e) => {
                          e.currentTarget.src = "/event.png";
                        }}
                      />
                      <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-300"></div>
                      
                      {/* Event Type Badge */}
                      <div className="absolute top-3 right-3">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                          event.event_type === 'online' 
                            ? 'bg-green-500 text-white' 
                            : 'bg-blue-500 text-white'
                        }`}>
                          {event.event_type}
                        </span>
                      </div>
                      
                      {/* Private Badge */}
                      {event.public_private === 'private' && (
                        <div className="absolute top-3 left-3">
                          <span className="px-2 py-1 text-xs font-medium rounded-full bg-gray-800 text-white">
                            Private
                          </span>
                        </div>
                      )}
                    </div>
                    
                    <div className="flex items-center gap-4 px-4 py-3 group-hover:bg-gray-50 transition-colors duration-300">
                      <div className="text-center min-w-[40px]">
                        <div className="text-lg font-bold text-black group-hover:text-black/80 transition-colors">
                          {day}
                        </div>
                        <div className="text-xs text-gray-500 group-hover:text-gray-600 transition-colors">
                          {month}
                        </div>
                      </div>
                      
                      <div className="flex-1">
                        <div className="font-semibold text-black group-hover:text-black/90 transition-colors">
                          {event.event_title}
                        </div>
                        <div className="text-xs text-gray-500 group-hover:text-gray-600 transition-colors">
                          {formatEventTime(event.event_time)}
                        </div>
                        <div className="text-xs text-gray-500 group-hover:text-gray-600 transition-colors mt-1">
                          {event.address}
                        </div>
                        {event.ticket_price > 0 && (
                          <div className="text-xs font-medium text-green-600 mt-1">
                            ${event.ticket_price}
                          </div>
                        )}
                      </div>
                      
                      <div className="text-center">
                        <div className="text-xs text-gray-500">
                          {event.interested_count + event.going_count} interested
                        </div>
                        <div className="opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                          <svg className="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                          </svg>
                        </div>
                      </div>
                    </div>
                  </Link>
                );
              })}
            </div>
          )}
        </div>
      </main>
    </PublicRoute>
  );
}
