import type { MetaFunction, LoaderFunctionArgs } from "@remix-run/node";
import { useLoaderData } from "@remix-run/react";
import {
  WelcomeSection,
  LocationInfo,
  StudiosSection,
  EventsSection,
  PlanYourVisitSection,
} from "~/components/pages/locations";
import { MembershipSection } from "~/components/shared/MembershipSection";
import { PeopleSection } from "~/components/shared/PeopleSection";
//import { locationsConfig, type LocationType } from "~/config/locations";
import en from "~/config/lang/en.json";
import sp from "~/config/lang/sp.json";
const translations = { en, sp };

export async function loader({ params }: LoaderFunctionArgs) {
  // get location from params
  const location = params.location;

  // get language from location storage
  const lang = "en"; //localStorage.getItem("lang") || "en";

  // get translations based on language
  const languageData = translations[lang];

  const locationData = languageData[location];

  if (!locationData) {
    throw new Response("Location Not Found", { status: 404 });
  }

  const apiRes = await fetch(
    `https://nomadicstudios.net:444/api/locations/${location}`
  );
  const apiData = await apiRes.json();
  locationData.welcome.backgroundImage = apiData.data.banner_picture;

  return Response.json({ location, locationData, apiData: apiData });
}

export const meta: MetaFunction<typeof loader> = ({ data }) => {
  if (!data) {
    return [
      { title: "Location Not Found | Nomadics" },
      { name: "description", content: "The requested location was not found." },
    ];
  }

  return [
    { title: data.locationData.meta.title },
    { name: "description", content: data.locationData.meta.description },
  ];
};

export default function LocationPage() {
  const { locationData, apiData } = useLoaderData<typeof loader>();

  return (
    <div>
      <WelcomeSection data={locationData.welcome} />
      <LocationInfo data={locationData.info} />
      <StudiosSection data={apiData.data} />
      {apiData.data.resources?.length > 0 && (
        <PeopleSection
          title={locationData.people.title}
          peopleData={apiData.data.resources}
          filterOptions={apiData.data.filters}
        />
      )}
      {apiData.data.events?.length > 0 && (
        <EventsSection
          title={locationData.events.sectionTitle}
          data={apiData.data.events}
        />
      )}
      <PlanYourVisitSection
        title={locationData.planVisit.title}
        address={apiData.data.address}
        hours={apiData.data.opening_hours}
        phone={apiData.data.phone}
        email={apiData.data.email}
        additionalInfo={apiData.data.additional_info}
        location={apiData.data.location}
      />
      <MembershipSection theme="black" />
    </div>
  );
}
