import type { MetaFunction } from "@remix-run/node";
import { Link, useParams } from "@remix-run/react";
import { Button } from "~/components/ui/button";
import { PublicRoute } from "~/components/RouteProtection";

// CSS for line clamping
const styles = `
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
`;

export const meta: MetaFunction = ({ params }) => {
  return [
    { title: `${params.slug?.replace(/-/g, ' ')} - Nomadics Studios` },
    {
      name: "description",
      content: "Read the latest news and insights from the creative community at Nomadics Studios.",
    },
  ];
};

// Mock articles data - this would typically come from your API
const newsArticles = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: {
      name: "Akanksha Bhartaure",
      avatar: "/placeholder-avatar.jpg",
      bio: "Travel Writer & Creative Director"
    },
    publishedAt: "5 min ago",
    readTime: "5 min read",
    image: "/bw (1).png",
    slug: "travel-as-a-failure",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>A story about a trip to Barcelona 2</h3>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <p>Travel opens our minds to new possibilities and perspectives that we might never encounter in our daily routines. When we step outside our comfort zones and immerse ourselves in different cultures, we gain invaluable insights that can transform our creative work.</p>
      
      <p>Sometimes, the most valuable lessons come from the moments when things don't go according to plan. Failed connections, missed opportunities, and unexpected detours often become the foundation for our most compelling stories and creative breakthroughs.</p>
    `,
    tags: ["Travel", "Lessons", "Barcelona", "Growth"]
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: {
      name: "Super Admin",
      avatar: "/placeholder-avatar.jpg",
      bio: "Creative Director at Nomadics Studios"
    },
    publishedAt: "2 hours ago",
    readTime: "4 min read",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>The Creative Impact of Travel</h3>
      
      <p>For creators, travel is more than just a break from routine—it's a source of inspiration that can revolutionize their approach to their craft. Whether you're a filmmaker seeking new locations, a photographer hunting for unique lighting, or a musician searching for fresh sounds, travel provides the raw material for creative breakthroughs.</p>
      
      <p>At Nomadics Studios, we've seen firsthand how our community members return from their travels with renewed energy and innovative ideas. The stories they share, the techniques they've learned, and the perspectives they've gained all contribute to the rich tapestry of our creative community.</p>
    `,
    tags: ["Travel", "Creativity", "Inspiration", "Community"]
  }
];

// Function to find article by slug
function findArticleBySlug(slug: string) {
  return newsArticles.find(article => article.slug === slug) || newsArticles[0];
}

// Related articles - get other articles from the main articles array
function getRelatedArticles(currentSlug: string) {
  return newsArticles
    .filter(article => article.slug !== currentSlug)
    .slice(0, 3);
}

export default function NewsDetailsPage() {
  const { slug } = useParams();
  const article = findArticleBySlug(slug || "");
  const relatedArticles = getRelatedArticles(slug || "");
  
  return (
    <PublicRoute>
      <div className="relative bg-white min-h-screen">
        <style dangerouslySetInnerHTML={{ __html: styles }} />
        
        {/* Clear indicator this is the details page */}
        <div className="fixed top-4 right-4 z-50 bg-red-500 text-white px-4 py-2 rounded text-sm font-bold">
          NEWS DETAILS PAGE
        </div>
        {/* Hero Section with Background Image */}
        <section className="relative h-[70vh] lg:h-[80vh] overflow-hidden">
          {/* Background Image */}
          <div className="absolute inset-0">
            <img
              src={article.image}
              alt={article.title}
              className="h-full w-full object-cover"
              onError={(e) => {
                e.currentTarget.src = "/placeholder-news.jpg";
              }}
            />
            <div className="absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" />
          </div>

          {/* Hero Content */}
          <div className="relative z-10 h-full flex items-end">
            <div className="mx-auto w-full max-w-[1000px] px-4 lg:px-8 pb-16 lg:pb-24">
              {/* Breadcrumb */}
              <nav className="mb-6">
                <Link 
                  to="/news" 
                  className="text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                  </svg>
                  Back to News
                </Link>
              </nav>
              
              {/* Debug info - remove this in production */}
              <div className="mb-4 text-white/60 text-sm bg-black/20 p-2 rounded">
                Debug: Slug = "{slug}" | Article = "{article.title}"
              </div>

              {/* Category */}
              <div className="mb-4">
                <span className="inline-block px-4 py-2 bg-white/20 backdrop-blur-sm text-white text-sm font-medium uppercase tracking-wider rounded">
                  {article.category}
                </span>
              </div>

              {/* Title */}
              <h1 className="text-3xl lg:text-5xl font-bold text-white mb-6 leading-tight max-w-4xl">
                {article.title}
              </h1>

              {/* Meta Info */}
              <div className="flex flex-col sm:flex-row sm:items-center gap-4 text-white/90">
                <div className="flex items-center gap-3">
                  <img
                    src={article.author.avatar}
                    alt={article.author.name}
                    className="w-10 h-10 rounded-full object-cover bg-white/20"
                    onError={(e) => {
                      e.currentTarget.src = "/placeholder-avatar.jpg";
                    }}
                  />
                  <div>
                    <div className="font-medium text-white">{article.author.name}</div>
                    <div className="text-sm text-white/70">{article.publishedAt} • {article.readTime}</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* Article Content Section */}
        <section className="bg-white py-16 lg:py-24">
          <div className="mx-auto w-full max-w-[800px] px-4 lg:px-8">
            
            {/* Article Body */}
            <div className="prose prose-lg max-w-none mb-16">
              <div 
                className="article-content 
                  [&_h2]:text-2xl [&_h2]:lg:text-3xl [&_h2]:font-bold [&_h2]:text-black [&_h2]:mb-6 [&_h2]:mt-12 [&_h2]:leading-tight
                  [&_h3]:text-xl [&_h3]:lg:text-2xl [&_h3]:font-bold [&_h3]:text-black [&_h3]:mb-4 [&_h3]:mt-8 [&_h3]:leading-tight
                  [&_p]:text-black/80 [&_p]:text-base [&_p]:lg:text-lg [&_p]:leading-relaxed [&_p]:mb-6
                  [&_img]:rounded-lg [&_img]:shadow-lg [&_img]:my-8"
                dangerouslySetInnerHTML={{ __html: article.content }}
              />
            </div>

            {/* Tags */}
            <div className="flex flex-wrap gap-3 mb-12 pb-8 border-b border-gray-200">
              <span className="text-sm font-medium text-gray-600 mr-2">Tags:</span>
              {article.tags.map((tag: string) => (
                <span
                  key={tag}
                  className="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-full hover:bg-gray-200 transition-colors cursor-pointer"
                >
                  {tag}
                </span>
              ))}
            </div>

            {/* Author Bio */}
            <div className="bg-gray-50 rounded-2xl p-8 mb-12">
              <div className="flex items-start gap-4">
                <img
                  src={article.author.avatar}
                  alt={article.author.name}
                  className="w-16 h-16 rounded-full object-cover bg-gray-200"
                  onError={(e) => {
                    e.currentTarget.src = "/placeholder-avatar.jpg";
                  }}
                />
                <div className="flex-1">
                  <h3 className="text-xl font-bold text-black mb-2">{article.author.name}</h3>
                  <p className="text-gray-600 mb-4">{article.author.bio}</p>
                  <div className="flex gap-3">
                    <Button variant="outline-black" size="sm">Follow</Button>
                    <Button variant="outline-black" size="sm">View Profile</Button>
                  </div>
                </div>
              </div>
            </div>

            {/* Share Section */}
            <div className="text-center mb-12">
              <h3 className="text-xl font-bold text-black mb-4">Share this article</h3>
              <div className="flex justify-center gap-3">
                <Button variant="outline-black" size="md">Twitter</Button>
                <Button variant="outline-black" size="md">LinkedIn</Button>
                <Button variant="outline-black" size="md">Facebook</Button>
                <Button variant="fill-black" size="md" arrowRight>
                  Subscribe
                </Button>
              </div>
            </div>
          </div>
        </section>

        {/* Related News Section */}
        <section className="bg-gray-50">
          <div className="mx-auto w-full max-w-[1200px] px-4 lg:px-8 py-16 lg:py-24">
            <div className="text-center mb-16">
              <h2 className="text-3xl lg:text-4xl font-bold text-black mb-4">
                Related news
              </h2>
              <div className="w-24 h-1 bg-black mx-auto"></div>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12">
              {relatedArticles.map((relatedArticle) => (
                <Link
                  key={relatedArticle.id}
                  to={`/news/${relatedArticle.slug}`}
                  className="group block bg-white shadow-sm hover:shadow-lg transition-all duration-300 overflow-hidden hover:transform hover:scale-[1.02] rounded-lg"
                >
                  <div className="aspect-[4/3] overflow-hidden bg-neutral-100">
                    <img
                      src={relatedArticle.image}
                      alt={relatedArticle.title}
                      className="h-full w-full object-cover transition-transform duration-500 group-hover:scale-105"
                      onError={(e) => {
                        e.currentTarget.src = "/placeholder-news.jpg";
                      }}
                    />
                  </div>
                  <div className="p-6">
                    <div className="inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded mb-3">
                      {relatedArticle.category}
                    </div>
                    <h3 className="text-lg font-bold text-black mb-3 group-hover:text-black/80 transition-colors leading-tight">
                      {relatedArticle.title}
                    </h3>
                    <p className="text-gray-600 text-sm line-clamp-2 mb-4">
                      {relatedArticle.excerpt}
                    </p>
                    <div className="flex items-center gap-2 text-xs text-gray-500">
                      <span>{relatedArticle.author.name}</span>
                      <span>•</span>
                      <span>{relatedArticle.publishedAt}</span>
                    </div>
                  </div>
                </Link>
              ))}
            </div>

            <div className="text-center">
              <Button
                variant="outline-black"
                size="lg"
                arrowRight
                className="text-black"
                onClick={() => window.location.href = '/news'}
              >
                View All News
              </Button>
            </div>
          </div>
        </section>


      </div>
    </PublicRoute>
  );
}