import type { MetaFunction } from "@remix-run/node";
import { Link } from "@remix-run/react";
import { Button } from "~/components/ui/button";
import { SectionHeader } from "~/components/shared/SectionHeader";
import { PublicRoute } from "~/components/RouteProtection";
import { useState, useMemo } from "react";
import { useLanguage } from "~/hooks/useLanguage";

// CSS for line clamping and input styling
const styles = `
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  
  /* Custom date input styling */
  input[type="date"]::-webkit-calendar-picker-indicator {
    opacity: 0;
    position: absolute;
    right: 8px;
    width: 20px;
    height: 20px;
    cursor: pointer;
  }
  
  input[type="date"] {
    color-scheme: light;
  }
`;

export const meta: MetaFunction = () => {
  return [
    { title: "News - Nomadics Studios" },
    {
      name: "description",
      content: "Stay updated with the latest news, insights, and stories from the creative community at Nomadics Studios.",
    },
  ];
};

// Mock data - you can replace this with actual data from your API
const newsArticles = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: "Akanksha Bhartaure",
    publishedAt: "5 min ago",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    image: "/bw (1).png",
    slug: "travel-as-a-failure"
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: "Super Admin",
    publishedAt: "2 hours ago",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective"
  },
  {
    id: 3,
    title: "Creative Studios: The Future of Work",
    category: "Studios",
    author: "John Smith",
    publishedAt: "1 day ago",
    excerpt: "Discover how modern creative studios are reshaping the future of work for artists, filmmakers, and content creators worldwide.",
    image: "/image 210.png",
    slug: "creative-studios-future-of-work"
  },
  {
    id: 4,
    title: "Equipment Innovation in 2024",
    category: "Equipment",
    author: "Tech Team",
    publishedAt: "2 days ago",
    excerpt: "Explore the latest equipment innovations that are transforming creative workflows and production quality in professional studios.",
    image: "/eqi-1.png",
    slug: "equipment-innovation-2024"
  },
  {
    id: 5,
    title: "Building Creative Communities",
    category: "Community",
    author: "Community Team",
    publishedAt: "3 days ago",
    excerpt: "Learn how Nomadics Studios is fostering connections and collaboration among creators from diverse backgrounds and disciplines.",
    image: "/bw.png",
    slug: "building-creative-communities"
  },
  {
    id: 6,
    title: "Sustainable Production Practices",
    category: "Sustainability",
    author: "Green Team",
    publishedAt: "1 week ago",
    excerpt: "Discover our commitment to environmentally conscious production methods and how studios can reduce their carbon footprint.",
    image: "/service-1.png",
    slug: "sustainable-production-practices"
  }
];



interface NewsCardProps {
  article: typeof newsArticles[0];
  featured?: boolean;
}

function NewsCard({ article, featured = false }: NewsCardProps) {
  return (
    <Link 
      to={`/news-details/${article.slug}`}
      className="group block transition-all duration-300 hover:transform hover:scale-[1.02]"
    >
      <article className="bg-white shadow-lg hover:shadow-xl transition-shadow duration-300 overflow-hidden h-full flex flex-col">
        {/* Image */}
        <div className="aspect-[4/3] overflow-hidden bg-neutral-100">
          <img
            src={article.image}
            alt={article.title}
            className="h-full w-full object-cover transition-transform duration-500 group-hover:scale-105"
            onError={(e) => {
              e.currentTarget.src = "/placeholder-news.jpg";
            }}
          />
        </div>

        {/* Content */}
        <div className="p-6 flex-1 flex flex-col">
          {/* Category Tag */}
          <div className="mb-4">
            <span className="inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded">
              {article.category}
            </span>
          </div>

          {/* Title */}
          <h3 className="text-xl font-bold text-black mb-4 group-hover:text-black/80 transition-colors leading-tight">
            {article.title}
          </h3>

          {/* Excerpt */}
          <p className="text-gray-600 text-sm leading-relaxed mb-6 flex-1 line-clamp-3">
            {article.excerpt}
          </p>

          {/* Author & Date */}
          <div className="flex items-center gap-3 pt-4 border-t border-gray-100">
            <div className="flex items-center gap-2">
              <div className="w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center">
                <span className="text-xs font-semibold text-gray-600">
                  {article.author.charAt(0)}
                </span>
              </div>
              <span className="text-sm font-medium text-gray-700">{article.author}</span>
            </div>
            <span className="text-xs text-gray-400">•</span>
            <span className="text-xs text-gray-500">{article.publishedAt}</span>
          </div>
        </div>
      </article>
    </Link>
  );
}

export default function NewsPage() {
  const [selectedDate, setSelectedDate] = useState("");
  const [searchTags, setSearchTags] = useState("");

  // Filter articles based on search criteria
  const filteredArticles = useMemo(() => {
    let filtered = newsArticles;

    // Filter by tags (search in title, category, and excerpt)
    if (searchTags.trim()) {
      const searchTerms = searchTags.toLowerCase().split(" ");
      filtered = filtered.filter(article =>
        searchTerms.some(term =>
          article.title.toLowerCase().includes(term) ||
          article.category.toLowerCase().includes(term) ||
          article.excerpt.toLowerCase().includes(term)
        )
      );
    }

    // Filter by date (mock date filtering - in real app you'd have proper dates)
    if (selectedDate) {
      // For demo purposes, we'll just show recent articles when a date is selected
      filtered = filtered.slice(0, 3);
    }

    return filtered;
  }, [searchTags, selectedDate]);

  return (
    <PublicRoute>
      <div className="relative bg-white min-h-screen">
        <style dangerouslySetInnerHTML={{ __html: styles }} />
        {/* Hero Section with Background Image */}
        <section className="relative h-[30vh] lg:h-[35vh] overflow-hidden">
          {/* Background Image */}
          <div className="absolute inset-0">
            <img
              src="/bw.png"
              alt="Discover our news"
              className="h-full w-full object-cover"
            />
            <div className="absolute inset-0 bg-gradient-to-b from-black/20 via-black/40 to-black/60" />
          </div>

          {/* Hero Content */}
          <div className="relative z-10 h-full flex items-end">
            <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14 pb-12 lg:pb-16">
              <div className="max-w-4xl">
                <h1 className="text-4xl lg:text-6xl font-bold text-white mb-6 uppercase tracking-tight">
                  Discover our news
                </h1>
                
                {/* Search and Filter Bar */}
                <div className="flex flex-col sm:flex-row gap-4 mt-8 mb-4">
                  <div className="flex-1 relative">
                    <input
                      type="date"
                      value={selectedDate}
                      onChange={(e) => setSelectedDate(e.target.value)}
                      placeholder="mm/dd/yyyy"
                      className="w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
                    />
                    <div className="absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
                      <svg className="w-4 h-4 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                      </svg>
                    </div>
                  </div>
                  <div className="flex-1 relative">
                    <input
                      type="text"
                      value={searchTags}
                      onChange={(e) => setSearchTags(e.target.value)}
                      placeholder="Search By Tags"
                      className="w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
                    />
                    <div className="absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none">
                      <svg className="w-4 h-4 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                      </svg>
                    </div>
                  </div>
                </div>

                {/* Search Results Info */}
                {(searchTags || selectedDate) && (
                  <div className="mt-4 text-white/80 text-sm">
                    {filteredArticles.length === 0 
                      ? "No articles found matching your search criteria"
                      : `Showing ${filteredArticles.length} article${filteredArticles.length !== 1 ? 's' : ''}`
                    }
                    {searchTags && (
                      <span className="ml-2">
                        for "{searchTags}"
                      </span>
                    )}
                    {(searchTags || selectedDate) && (
                      <button
                        onClick={() => {
                          setSearchTags("");
                          setSelectedDate("");
                        }}
                        className="ml-3 text-white/60 hover:text-white underline"
                      >
                        Clear filters
                      </button>
                    )}
                  </div>
                )}
              </div>
            </div>
          </div>
        </section>

        {/* News Grid Section */}
        <section className="bg-neutral-50 py-12 lg:py-16">
          <div className="mx-auto w-full max-w-[1328px] px-4 lg:px-14">
            {/* News Grid */}
            {filteredArticles.length > 0 ? (
              <>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-8 mb-16">
                  {filteredArticles.map((article) => (
                    <NewsCard key={article.id} article={article} />
                  ))}
                </div>

                {/* Load More - only show if not filtered and there are more articles */}
                {!searchTags && !selectedDate && filteredArticles.length >= 4 && (
                  <div className="text-center">
                    <Button
                      variant="outline-black"
                      size="lg"
                      arrowRight
                      className="text-black"
                    >
                      Load More Articles
                    </Button>
                  </div>
                )}
              </>
            ) : (
              /* No Results State */
              <div className="text-center py-16">
                <div className="max-w-md mx-auto">
                  <svg className="w-16 h-16 text-gray-300 mx-auto mb-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  <h3 className="text-xl font-semibold text-gray-700 mb-2">No articles found</h3>
                  <p className="text-gray-500 mb-6">
                    Try adjusting your search criteria or browse all articles
                  </p>
                  <Button
                    variant="outline-black"
                    size="md"
                    onClick={() => {
                      setSearchTags("");
                      setSelectedDate("");
                    }}
                  >
                    Show All Articles
                  </Button>
                </div>
              </div>
            )}
          </div>
        </section>


      </div>
    </PublicRoute>
  );
}