import { jsx, jsxs, Fragment } from "react/jsx-runtime";
import { createReadableStreamFromReadable } from "@remix-run/node";
import { RemixServer, useLocation, Link, Outlet, Meta, Links, ScrollRestoration, Scripts, useRouteError, isRouteErrorResponse, useNavigate, useParams, useLoaderData, useSearchParams } from "@remix-run/react";
import { isbot } from "isbot";
import { PassThrough } from "node:stream";
import { renderToPipeableStream } from "react-dom/server";
import { Slot } from "@radix-ui/react-slot";
import { cva } from "class-variance-authority";
import { clsx } from "clsx";
import { twMerge } from "tailwind-merge";
import { ChevronDownIcon, XIcon, MenuIcon, ChevronUp, ChevronDown, ArrowRight, ArrowLeft } from "lucide-react";
import { useState, useEffect, useCallback, useRef, useMemo } from "react";
import * as NavigationMenuPrimitive from "@radix-ui/react-navigation-menu";
import * as SheetPrimitive from "@radix-ui/react-dialog";
import * as AccordionPrimitive from "@radix-ui/react-accordion";
const ABORT_DELAY = 5e3;
function handleRequest(request, responseStatusCode, responseHeaders, remixContext) {
  return isbot(request.headers.get("user-agent") || "") ? handleBotRequest(
    request,
    responseStatusCode,
    responseHeaders,
    remixContext
  ) : handleBrowserRequest(
    request,
    responseStatusCode,
    responseHeaders,
    remixContext
  );
}
function handleBotRequest(request, responseStatusCode, responseHeaders, remixContext) {
  return new Promise((resolve, reject) => {
    let shellRendered = false;
    const { pipe, abort } = renderToPipeableStream(
      /* @__PURE__ */ jsx(
        RemixServer,
        {
          context: remixContext,
          url: request.url,
          abortDelay: ABORT_DELAY
        }
      ),
      {
        onAllReady() {
          shellRendered = true;
          const body = new PassThrough();
          const stream = createReadableStreamFromReadable(body);
          responseHeaders.set("Content-Type", "text/html");
          resolve(
            new Response(stream, {
              headers: responseHeaders,
              status: responseStatusCode
            })
          );
          pipe(body);
        },
        onShellError(error) {
          reject(error);
        },
        onError(error) {
          responseStatusCode = 500;
          if (shellRendered) {
            console.error(error);
          }
        }
      }
    );
    setTimeout(abort, ABORT_DELAY);
  });
}
function handleBrowserRequest(request, responseStatusCode, responseHeaders, remixContext) {
  return new Promise((resolve, reject) => {
    let shellRendered = false;
    const { pipe, abort } = renderToPipeableStream(
      /* @__PURE__ */ jsx(
        RemixServer,
        {
          context: remixContext,
          url: request.url,
          abortDelay: ABORT_DELAY
        }
      ),
      {
        onShellReady() {
          shellRendered = true;
          const body = new PassThrough();
          const stream = createReadableStreamFromReadable(body);
          responseHeaders.set("Content-Type", "text/html");
          resolve(
            new Response(stream, {
              headers: responseHeaders,
              status: responseStatusCode
            })
          );
          pipe(body);
        },
        onShellError(error) {
          reject(error);
        },
        onError(error) {
          responseStatusCode = 500;
          if (shellRendered) {
            console.error(error);
          }
        }
      }
    );
    setTimeout(abort, ABORT_DELAY);
  });
}
const entryServer = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: handleRequest
}, Symbol.toStringTag, { value: "Module" }));
const stylesheet = "/assets/tailwind-DFrBI1z5.css";
function cn(...inputs) {
  return twMerge(clsx(inputs));
}
const ArrowRightIcon = ({ size = 20 }) => /* @__PURE__ */ jsx(
  "svg",
  {
    width: size,
    height: size,
    fill: "none",
    stroke: "currentColor",
    viewBox: "0 0 24 24",
    children: /* @__PURE__ */ jsx(
      "path",
      {
        strokeLinecap: "round",
        strokeLinejoin: "round",
        strokeWidth: 2,
        d: "M13 7l5 5m0 0l-5 5m5-5H6"
      }
    )
  }
);
const buttonVariants = cva(
  "inline-flex items-center justify-center transition-all disabled:pointer-events-none disabled:opacity-50 box-border cursor-pointer uppercase whitespace-nowrap",
  {
    variants: {
      variant: {
        // Fill variants
        "fill-white": "bg-white !text-black border border-white",
        "fill-black": "bg-black !text-white border border-black",
        // Outline variants
        "outline-white": "border !border-white/30 bg-transparent !text-white",
        "outline-black": "border !border-black bg-transparent text-black"
      },
      size: {
        // Small: 32px height, 14px font
        sm: "h-8 text-button-small",
        // Medium: 48px height, 16px font
        md: "h-12 text-button-medium",
        // Large: 60px height, 18px font
        lg: "h-[60px] text-button-big"
      },
      buttonType: {
        // Label buttons (with text)
        label: "",
        // Icon square buttons
        "icon-square": "aspect-square p-0",
        // Icon circle buttons
        "icon-circle": "aspect-square p-0 rounded-full"
      }
    },
    compoundVariants: [
      // Small label buttons - padding based on CSS: 12px
      {
        size: "sm",
        buttonType: "label",
        class: "px-3 gap-1.5"
      },
      // Medium label buttons - padding based on CSS: 20px
      {
        size: "md",
        buttonType: "label",
        class: "px-5 gap-2"
      },
      // Large label buttons - padding based on CSS: 24px
      {
        size: "lg",
        buttonType: "label",
        class: "px-6 gap-2"
      },
      // Small icon buttons
      {
        size: "sm",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-8 h-8"
      },
      // Medium icon buttons
      {
        size: "md",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-12 h-12"
      },
      // Large icon buttons
      {
        size: "lg",
        buttonType: ["icon-square", "icon-circle"],
        class: "w-[60px] h-[60px]"
      },
      // All variants hover states - consistent border animation
      {
        variant: "fill-white",
        class: "hover:text-gray-600"
      },
      {
        variant: "fill-black",
        class: "hover:text-gray-300"
      },
      {
        variant: "outline-white",
        class: "hover:text-gray-300"
      },
      {
        variant: "outline-black",
        class: "hover:text-gray-600"
      }
    ],
    defaultVariants: {
      variant: "fill-white",
      size: "md",
      buttonType: "label"
    }
  }
);
function Button({
  className,
  variant,
  size,
  buttonType,
  asChild = false,
  arrowRight = false,
  responsive = false,
  children,
  ...props
}) {
  const Comp = asChild ? Slot : "button";
  const computedSize = responsive && !size ? void 0 : size;
  const responsiveClasses = responsive && !size ? buttonType === "label" ? "h-12 text-button-medium px-5 gap-2 lg:h-[60px] lg:text-button-big lg:px-6" : "w-12 h-12 lg:w-[60px] lg:h-[60px]" : "";
  const content = arrowRight ? /* @__PURE__ */ jsxs(Fragment, { children: [
    children,
    /* @__PURE__ */ jsx(ArrowRightIcon, { size: responsive && !size ? 20 : computedSize === "sm" ? 16 : computedSize === "lg" ? 24 : 20 })
  ] }) : children;
  return /* @__PURE__ */ jsx(
    Comp,
    {
      "data-slot": "button",
      className: cn(
        buttonVariants({ variant, size: computedSize, buttonType }),
        responsiveClasses,
        className
      ),
      ...props,
      children: content
    }
  );
}
function NavigationMenu({
  className,
  children,
  viewport = true,
  ...props
}) {
  return /* @__PURE__ */ jsxs(
    NavigationMenuPrimitive.Root,
    {
      "data-slot": "navigation-menu",
      "data-viewport": viewport,
      className: cn(
        "group/navigation-menu relative flex max-w-max flex-1 items-center justify-center",
        className
      ),
      ...props,
      children: [
        children,
        viewport && /* @__PURE__ */ jsx(NavigationMenuViewport, {})
      ]
    }
  );
}
function NavigationMenuList({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    NavigationMenuPrimitive.List,
    {
      "data-slot": "navigation-menu-list",
      className: cn(
        "group flex flex-1 list-none items-center justify-center gap-1",
        className
      ),
      ...props
    }
  );
}
function NavigationMenuItem({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    NavigationMenuPrimitive.Item,
    {
      "data-slot": "navigation-menu-item",
      className: cn("relative", className),
      ...props
    }
  );
}
const navigationMenuTriggerStyle = cva(
  "group inline-flex h-9 w-max items-center justify-center rounded-md bg-transparent px-4 py-2 text-body-3 uppercase font-medium tracking-[0.02em] text-white/90 hover:bg-white/10 hover:text-white focus:bg-white/10 focus:text-white disabled:pointer-events-none disabled:opacity-50 data-[state=open]:hover:bg-white/10 data-[state=open]:text-white data-[state=open]:focus:bg-white/10 data-[state=open]:bg-white/5 focus-visible:ring-white/50 outline-none transition-[color,box-shadow] focus-visible:ring-[3px] focus-visible:outline-1"
);
function NavigationMenuTrigger({
  className,
  children,
  ...props
}) {
  return /* @__PURE__ */ jsxs(
    NavigationMenuPrimitive.Trigger,
    {
      "data-slot": "navigation-menu-trigger",
      className: cn(navigationMenuTriggerStyle(), "group", className),
      ...props,
      children: [
        children,
        " ",
        /* @__PURE__ */ jsx(
          ChevronDownIcon,
          {
            className: "relative top-[1px] ml-1 size-3 transition duration-300 group-data-[state=open]:rotate-180",
            "aria-hidden": "true"
          }
        )
      ]
    }
  );
}
function NavigationMenuContent({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    NavigationMenuPrimitive.Content,
    {
      "data-slot": "navigation-menu-content",
      className: cn(
        "data-[motion^=from-]:animate-in data-[motion^=to-]:animate-out data-[motion^=from-]:fade-in data-[motion^=to-]:fade-out data-[motion=from-end]:slide-in-from-right-52 data-[motion=from-start]:slide-in-from-left-52 data-[motion=to-end]:slide-out-to-right-52 data-[motion=to-start]:slide-out-to-left-52 top-0 left-0 w-full p-2 pr-2.5 md:absolute md:w-auto",
        "group-data-[viewport=false]/navigation-menu:bg-black group-data-[viewport=false]/navigation-menu:text-white group-data-[viewport=false]/navigation-menu:data-[state=open]:animate-in group-data-[viewport=false]/navigation-menu:data-[state=closed]:animate-out group-data-[viewport=false]/navigation-menu:data-[state=closed]:zoom-out-95 group-data-[viewport=false]/navigation-menu:data-[state=open]:zoom-in-95 group-data-[viewport=false]/navigation-menu:data-[state=open]:fade-in-0 group-data-[viewport=false]/navigation-menu:data-[state=closed]:fade-out-0 group-data-[viewport=false]/navigation-menu:top-full group-data-[viewport=false]/navigation-menu:mt-1.5 group-data-[viewport=false]/navigation-menu:overflow-hidden group-data-[viewport=false]/navigation-menu:rounded-md group-data-[viewport=false]/navigation-menu:border group-data-[viewport=false]/navigation-menu:border-white/20 group-data-[viewport=false]/navigation-menu:shadow group-data-[viewport=false]/navigation-menu:duration-200 **:data-[slot=navigation-menu-link]:focus:ring-0 **:data-[slot=navigation-menu-link]:focus:outline-none",
        className
      ),
      ...props
    }
  );
}
function NavigationMenuViewport({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    "div",
    {
      className: cn(
        "absolute top-full left-0 isolate z-50 flex justify-center bg-transparent"
      ),
      children: /* @__PURE__ */ jsx(
        NavigationMenuPrimitive.Viewport,
        {
          "data-slot": "navigation-menu-viewport",
          className: cn(
            "origin-top-center bg-black text-white data-[state=open]:animate-in data-[state=closed]:animate-out data-[state=closed]:zoom-out-95 data-[state=open]:zoom-in-90 relative mt-1.5 h-[var(--radix-navigation-menu-viewport-height)] w-full overflow-hidden rounded-md md:w-[var(--radix-navigation-menu-viewport-width)]",
            className
          ),
          ...props
        }
      )
    }
  );
}
function NavigationMenuLink({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    NavigationMenuPrimitive.Link,
    {
      "data-slot": "navigation-menu-link",
      className: cn(
        "data-[active=true]:focus:bg-white/10 data-[active=true]:hover:bg-white/10 data-[active=true]:bg-white/5 data-[active=true]:text-white hover:bg-white/10 hover:text-white focus:bg-white/10 focus:text-white focus-visible:ring-white/50 [&_svg:not([class*='text-'])]:text-white/60 flex flex-col gap-1 rounded-sm p-2 text-body-3 uppercase font-medium tracking-[0.02em] transition-all outline-none focus-visible:ring-[3px] focus-visible:outline-1 [&_svg:not([class*='size-'])]:size-4 text-white/90",
        className
      ),
      ...props
    }
  );
}
function Sheet({ ...props }) {
  return /* @__PURE__ */ jsx(SheetPrimitive.Root, { "data-slot": "sheet", ...props });
}
function SheetTrigger({
  ...props
}) {
  return /* @__PURE__ */ jsx(SheetPrimitive.Trigger, { "data-slot": "sheet-trigger", ...props });
}
function SheetPortal({
  ...props
}) {
  return /* @__PURE__ */ jsx(SheetPrimitive.Portal, { "data-slot": "sheet-portal", ...props });
}
function SheetOverlay({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    SheetPrimitive.Overlay,
    {
      "data-slot": "sheet-overlay",
      className: cn(
        "data-[state=open]:animate-in data-[state=closed]:animate-out data-[state=closed]:fade-out-0 data-[state=open]:fade-in-0 fixed inset-0 z-50 bg-black/50",
        className
      ),
      ...props
    }
  );
}
function SheetContent({
  className,
  children,
  side = "right",
  ...props
}) {
  return /* @__PURE__ */ jsxs(SheetPortal, { children: [
    /* @__PURE__ */ jsx(SheetOverlay, {}),
    /* @__PURE__ */ jsxs(
      SheetPrimitive.Content,
      {
        "data-slot": "sheet-content",
        className: cn(
          "bg-background data-[state=open]:animate-in data-[state=closed]:animate-out fixed z-50 flex flex-col gap-4 shadow-lg transition ease-in-out data-[state=closed]:duration-300 data-[state=open]:duration-500",
          side === "right" && "data-[state=closed]:slide-out-to-right data-[state=open]:slide-in-from-right inset-y-0 right-0 h-full w-3/4 border-l sm:max-w-sm",
          side === "left" && "data-[state=closed]:slide-out-to-left data-[state=open]:slide-in-from-left inset-y-0 left-0 h-full w-3/4 border-r sm:max-w-sm",
          side === "top" && "data-[state=closed]:slide-out-to-top data-[state=open]:slide-in-from-top inset-x-0 top-0 h-auto border-b",
          side === "bottom" && "data-[state=closed]:slide-out-to-bottom data-[state=open]:slide-in-from-bottom inset-x-0 bottom-0 h-auto border-t",
          className
        ),
        ...props,
        children: [
          children,
          /* @__PURE__ */ jsxs(SheetPrimitive.Close, { className: "ring-offset-background focus:ring-ring data-[state=open]:bg-secondary absolute top-4 right-4 rounded-xs opacity-70 transition-opacity hover:opacity-100 focus:ring-2 focus:ring-offset-2 focus:outline-hidden disabled:pointer-events-none", children: [
            /* @__PURE__ */ jsx(XIcon, { className: "size-4" }),
            /* @__PURE__ */ jsx("span", { className: "sr-only", children: "Close" })
          ] })
        ]
      }
    )
  ] });
}
function SheetHeader({ className, ...props }) {
  return /* @__PURE__ */ jsx(
    "div",
    {
      "data-slot": "sheet-header",
      className: cn("flex flex-col gap-1.5 p-4", className),
      ...props
    }
  );
}
const API_BASE_URL = "https://nomadicstudios.net:444/api";
const getAuthToken = () => {
  if (typeof window !== "undefined") {
    return localStorage.getItem("auth_token");
  }
  return null;
};
const setAuthToken = (token) => {
  if (typeof window !== "undefined") {
    localStorage.setItem("auth_token", token);
  }
};
const removeAuthToken = () => {
  if (typeof window !== "undefined") {
    localStorage.removeItem("auth_token");
    localStorage.removeItem("user_data");
  }
};
const setUserData = (user) => {
  if (typeof window !== "undefined") {
    localStorage.setItem("user_data", JSON.stringify(user));
  }
};
const getUserData = () => {
  if (typeof window !== "undefined") {
    const userData = localStorage.getItem("user_data");
    return userData ? JSON.parse(userData) : null;
  }
  return null;
};
async function apiRequest(endpoint, options = {}) {
  const token = getAuthToken();
  const defaultHeaders = {
    "Content-Type": "application/json",
    Accept: "application/json"
  };
  if (token) {
    defaultHeaders.Authorization = `Bearer ${token}`;
  }
  const response = await fetch(`${API_BASE_URL}${endpoint}`, {
    ...options,
    headers: {
      ...defaultHeaders,
      ...options.headers
    }
  });
  const data = await response.json();
  if (!response.ok) {
    throw new Error(data.error || `HTTP error! status: ${response.status}`);
  }
  return data;
}
const authAPI = {
  // Email/Password Login
  login: async (email, password) => {
    return apiRequest("/login", {
      method: "POST",
      body: JSON.stringify({ email, password })
    });
  },
  // Send OTP to mobile number
  sendOTP: async (mobile_number) => {
    return apiRequest("/send-otp", {
      method: "POST",
      body: JSON.stringify({ mobile_number })
    });
  },
  // Verify OTP and login
  verifyOTP: async (mobile_number, otp_number, request_type = "login", additionalData) => {
    const payload = {
      request_type,
      otp_number,
      mobile_number
    };
    if (additionalData) {
      Object.assign(payload, additionalData);
    }
    return apiRequest("/verify-otp", {
      method: "POST",
      body: JSON.stringify(payload)
    });
  },
  // Logout
  logout: async () => {
    return apiRequest("/logout", {
      method: "POST"
    });
  },
  // Forgot Password - Send OTP to mobile
  forgotPassword: async (email) => {
    return apiRequest("/forgot-password", {
      method: "POST",
      body: JSON.stringify({ email })
    });
  },
  // Reset Password
  resetPassword: async (email) => {
    return apiRequest("/reset-password", {
      method: "POST",
      body: JSON.stringify({ email })
    });
  },
  // Save Profile Picture
  saveProfilePicture: async (uuid, image) => {
    return apiRequest("/save-profile-picture", {
      method: "POST",
      body: JSON.stringify({ uuid, image })
    });
  },
  // Save Additional Information
  saveAdditionalInfo: async (data) => {
    return apiRequest("/additional-info-save", {
      method: "POST",
      body: JSON.stringify(data)
    });
  }
};
const locationsAPI = {
  // Get all locations
  getLocations: async () => {
    return apiRequest("/locations", {
      method: "GET"
    });
  }
};
const eventsAPI = {
  getAll: async (filterDate, filterLocation) => {
    const params = new URLSearchParams();
    if (filterDate) params.append("filterDate", filterDate);
    if (filterLocation)
      params.append("filterLocation", filterLocation.toString());
    const queryString = params.toString();
    const endpoint = queryString ? `/events?${queryString}` : "/events";
    return apiRequest(endpoint);
  }
};
const isAuthenticated = () => {
  return getAuthToken() !== null;
};
const handleAuthSuccess = (response) => {
  if (response.token) {
    setAuthToken(response.token);
    console.log("Auth token set");
  }
  if (response.user) {
    setUserData(response.user);
    console.log("User data set");
  }
};
const handleLogout = async () => {
  try {
    await authAPI.logout();
  } catch (error) {
    console.error("Logout error:", error);
  } finally {
    removeAuthToken();
  }
};
const __variableDynamicImportRuntimeHelper = (glob, path, segs) => {
  const v = glob[path];
  if (v) {
    return typeof v === "function" ? v() : Promise.resolve(v);
  }
  return new Promise((_, reject) => {
    (typeof queueMicrotask === "function" ? queueMicrotask : setTimeout)(
      reject.bind(
        null,
        new Error(
          "Unknown variable dynamic import: " + path + (path.split("/").length !== segs ? ". Note that variables only represent file names one level deep." : "")
        )
      )
    );
  });
};
const LANGUAGE_STORAGE_KEY = "nomadic-language";
const DEFAULT_LANGUAGE = "en";
function useLanguage() {
  const [currentLanguage, setCurrentLanguage] = useState(DEFAULT_LANGUAGE);
  const [translations2, setTranslations] = useState({});
  const [loading, setLoading] = useState(true);
  useEffect(() => {
    if (typeof window !== "undefined") {
      const savedLanguage = localStorage.getItem(LANGUAGE_STORAGE_KEY);
      if (savedLanguage && (savedLanguage === "en" || savedLanguage === "sp")) {
        setCurrentLanguage(savedLanguage);
      }
    }
  }, []);
  useEffect(() => {
    const loadTranslations = async () => {
      try {
        setLoading(true);
        const response = await __variableDynamicImportRuntimeHelper(/* @__PURE__ */ Object.assign({ "../config/lang/en.json": () => Promise.resolve().then(() => en$1), "../config/lang/sp.json": () => Promise.resolve().then(() => sp$1) }), `../config/lang/${currentLanguage}.json`, 4);
        setTranslations(response.default || response);
      } catch (error) {
        console.error(`Failed to load translations for ${currentLanguage}:`, error);
        if (currentLanguage !== "en") {
          try {
            const fallback = await Promise.resolve().then(() => en$1);
            setTranslations(fallback.default || fallback);
          } catch (fallbackError) {
            console.error("Failed to load fallback English translations:", fallbackError);
          }
        }
      } finally {
        setLoading(false);
      }
    };
    loadTranslations();
  }, [currentLanguage]);
  const changeLanguage = useCallback((newLanguage) => {
    setCurrentLanguage(newLanguage);
    if (typeof window !== "undefined") {
      localStorage.setItem(LANGUAGE_STORAGE_KEY, newLanguage);
    }
    setLoading(true);
  }, []);
  const t = useCallback((key, defaultValue) => {
    const keys = key.split(".");
    let value = translations2;
    for (const k of keys) {
      if (value && typeof value === "object" && k in value) {
        value = value[k];
      } else {
        return defaultValue || key;
      }
    }
    return typeof value === "string" ? value : defaultValue || key;
  }, [translations2]);
  const getSection = useCallback((key) => {
    const keys = key.split(".");
    let value = translations2;
    for (const k of keys) {
      if (value && typeof value === "object" && k in value) {
        value = value[k];
      } else {
        return {};
      }
    }
    return value || {};
  }, [translations2]);
  return {
    currentLanguage,
    changeLanguage,
    t,
    getSection,
    loading,
    translations: translations2
  };
}
function AppHeader({ theme = "dark" }) {
  const location = useLocation();
  const [isScrolled, setIsScrolled] = useState(false);
  const [isHovered, setIsHovered] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [user, setUser] = useState(null);
  const [showProfileDropdown, setShowProfileDropdown] = useState(false);
  const { t } = useLanguage();
  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 0);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);
  useEffect(() => {
    const checkAuthStatus = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);
      if (authenticated) {
        const userData = getUserData();
        setUser(userData);
      } else {
        setUser(null);
      }
    };
    checkAuthStatus();
    const interval = setInterval(checkAuthStatus, 1e3);
    return () => clearInterval(interval);
  }, [location.pathname]);
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (showProfileDropdown) {
        setShowProfileDropdown(false);
      }
    };
    document.addEventListener("click", handleClickOutside);
    return () => document.removeEventListener("click", handleClickOutside);
  }, [showProfileDropdown]);
  const handleLogoutClick = async () => {
    try {
      await handleLogout();
      setIsLoggedIn(false);
      setUser(null);
      window.location.href = "/";
    } catch (error) {
      console.error("Logout error:", error);
      setIsLoggedIn(false);
      setUser(null);
      window.location.href = "/";
    }
  };
  const navItems = [
    {
      label: t("nav.studios"),
      href: "/studios",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.videoStudios"), href: "/studios/video" },
        { label: t("nav.musicStudios"), href: "/studios/music" },
        { label: t("nav.photoStudios"), href: "/studios/photo" }
      ]
    },
    {
      label: t("nav.locations"),
      href: "/locations",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.bali"), href: "/locations/bali" },
        { label: t("nav.bangkok"), href: "/locations/bangkok" },
        { label: t("nav.mexico"), href: "/locations/mexico" },
        { label: t("nav.costaRica"), href: "/locations/costa-rica" },
        { label: t("nav.newYork"), href: "/locations/new-york" },
        { label: t("nav.mumbai"), href: "/locations/mumbai" }
      ]
    },
    { label: t("nav.people"), href: "/people" },
    { label: t("nav.equipment"), href: "/equipment" },
    { label: t("nav.events"), href: "/events" },
    {
      label: t("nav.services"),
      href: "/services",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.allServices"), href: "/services" },
        { label: t("nav.production"), href: "/services/production" },
        { label: t("nav.postProduction"), href: "/services/post-production" }
      ]
    },
    { label: t("nav.news"), href: "/news" }
  ];
  const isDark = theme === "dark";
  const bgColor = isDark ? "bg-black" : "bg-white";
  const textColor = isDark ? "text-white" : "text-black";
  const borderColor = isDark ? "border-white/20" : "border-black/20";
  const hoverBg = isDark ? "hover:bg-white/10" : "hover:bg-black/10";
  const logoSrc = isDark ? "/images/logo-white.png" : "/images/logo-black.png";
  return /* @__PURE__ */ jsx(
    "header",
    {
      className: `fixed top-0 left-0 right-0 ${bgColor} z-50 transition-opacity duration-300 ${isScrolled && !isHovered ? "opacity-50" : "opacity-100"}`,
      onMouseEnter: () => setIsHovered(true),
      onMouseLeave: () => setIsHovered(false),
      children: /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between px-4 lg:px-14 h-16  lg:h-20 min-w-0", children: [
        /* @__PURE__ */ jsx(Link, { to: "/", className: "block flex-shrink-0", children: /* @__PURE__ */ jsx("img", { src: logoSrc, alt: t("common.logo"), className: "h-10 lg:h-12 w-auto" }) }),
        /* @__PURE__ */ jsx("div", { className: "hidden xl:flex items-center gap-8", children: /* @__PURE__ */ jsx(NavigationMenu, { viewport: true, className: "flex-1", children: /* @__PURE__ */ jsx(NavigationMenuList, { children: navItems.map((item) => {
          var _a;
          return /* @__PURE__ */ jsx(NavigationMenuItem, { children: item.hasDropdown ? /* @__PURE__ */ jsxs(Fragment, { children: [
            /* @__PURE__ */ jsx(NavigationMenuTrigger, { children: item.label }),
            /* @__PURE__ */ jsx(
              NavigationMenuContent,
              {
                className: `${bgColor} border ${borderColor} rounded-md overflow-hidden`,
                children: /* @__PURE__ */ jsx("ul", { className: "grid w-[600px] gap-3 p-4 grid-cols-2", children: (_a = item.dropdownItems) == null ? void 0 : _a.map((dropdownItem) => /* @__PURE__ */ jsx("li", { children: /* @__PURE__ */ jsx(NavigationMenuLink, { asChild: true, children: /* @__PURE__ */ jsx(
                  Link,
                  {
                    to: dropdownItem.href,
                    className: `block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors ${hoverBg} ${isDark ? "hover:text-white focus:text-white text-white/90" : "hover:text-black focus:text-black text-black/90"} ${isDark ? "focus:bg-white/10" : "focus:bg-black/10"}`,
                    children: /* @__PURE__ */ jsx("div", { className: "text-sm font-medium leading-none uppercase tracking-[0.02em]", children: dropdownItem.label })
                  }
                ) }) }, dropdownItem.href)) })
              }
            )
          ] }) : /* @__PURE__ */ jsx(Link, { to: item.href, children: /* @__PURE__ */ jsx(
            NavigationMenuLink,
            {
              className: navigationMenuTriggerStyle(),
              children: item.label
            }
          ) }) }, item.href);
        }) }) }) }),
        /* @__PURE__ */ jsx("div", { className: "xl:hidden", children: /* @__PURE__ */ jsxs(Sheet, { children: [
          /* @__PURE__ */ jsx(SheetTrigger, { asChild: true, children: /* @__PURE__ */ jsxs("button", { className: `${textColor} ${hoverBg} p-2 rounded`, children: [
            /* @__PURE__ */ jsx(MenuIcon, { className: "h-6 w-6" }),
            /* @__PURE__ */ jsx("span", { className: "sr-only", children: t("header.openMenu") })
          ] }) }),
          /* @__PURE__ */ jsx(
            SheetContent,
            {
              side: "right",
              className: `${bgColor} border-l ${borderColor} ${textColor} w-full sm:w-96 overflow-y-auto`,
              children: /* @__PURE__ */ jsxs("div", { className: "px-8 py-12 min-h-full", children: [
                /* @__PURE__ */ jsx(SheetHeader, { className: "p-0 mb-8", children: /* @__PURE__ */ jsx("div", { className: "flex items-center justify-between", children: /* @__PURE__ */ jsx("img", { src: logoSrc, alt: t("common.companyName"), className: "h-10" }) }) }),
                /* @__PURE__ */ jsx(
                  "div",
                  {
                    className: `space-y-4 mb-8 pb-8 border-b ${isDark ? "border-white/10" : "border-black/10"}`,
                    children: !isLoggedIn ? /* @__PURE__ */ jsxs(Fragment, { children: [
                      /* @__PURE__ */ jsx(
                        Button,
                        {
                          size: "sm",
                          asChild: true,
                          variant: isDark ? "outline-white" : "outline-black",
                          className: "w-full h-12 text-base font-medium",
                          children: /* @__PURE__ */ jsx(Link, { to: "/login", children: t("auth.login") })
                        }
                      ),
                      /* @__PURE__ */ jsx(
                        Button,
                        {
                          size: "sm",
                          asChild: true,
                          variant: isDark ? "outline-white" : "outline-black",
                          className: "w-full h-12 text-base font-medium",
                          children: /* @__PURE__ */ jsx(Link, { to: "/signup", children: t("auth.signUp") })
                        }
                      )
                    ] }) : /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
                      /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 p-3 rounded-lg bg-white/5", children: [
                        /* @__PURE__ */ jsx(
                          "img",
                          {
                            src: "/images/profile-avatar.jpg",
                            alt: t("header.profile"),
                            className: "w-12 h-12 rounded-full object-cover"
                          }
                        ),
                        /* @__PURE__ */ jsxs("div", { children: [
                          /* @__PURE__ */ jsx("p", { className: `font-medium ${textColor}`, children: "akanksha bhadauria" }),
                          /* @__PURE__ */ jsx("p", { className: `text-sm ${isDark ? "text-white/60" : "text-black/60"}`, children: "0 pts" })
                        ] })
                      ] }),
                      /* @__PURE__ */ jsxs("div", { className: "flex justify-between items-center gap-4 p-3 rounded-lg bg-white/5", children: [
                        /* @__PURE__ */ jsxs("button", { className: "flex flex-col items-center gap-1", children: [
                          /* @__PURE__ */ jsxs("div", { className: "relative", children: [
                            /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" }) }),
                            /* @__PURE__ */ jsx("div", { className: "absolute -top-1 -right-1 w-2 h-2 bg-red-500 rounded-full" })
                          ] }),
                          /* @__PURE__ */ jsx("span", { className: `text-xs ${isDark ? "text-white/60" : "text-black/60"}`, children: t("header.notifications") })
                        ] }),
                        /* @__PURE__ */ jsxs("button", { className: "flex flex-col items-center gap-1", children: [
                          /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" }) }),
                          /* @__PURE__ */ jsx("span", { className: `text-xs ${isDark ? "text-white/60" : "text-black/60"}`, children: t("header.chat") })
                        ] }),
                        /* @__PURE__ */ jsxs("button", { className: "flex flex-col items-center gap-1", children: [
                          /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" }) }),
                          /* @__PURE__ */ jsx("span", { className: `text-xs ${isDark ? "text-white/60" : "text-black/60"}`, children: t("header.wallet") })
                        ] })
                      ] }),
                      /* @__PURE__ */ jsx(
                        Button,
                        {
                          size: "sm",
                          onClick: handleLogoutClick,
                          variant: isDark ? "outline-white" : "outline-black",
                          className: "w-full h-12 text-base font-medium",
                          children: t("auth.signOut")
                        }
                      )
                    ] })
                  }
                ),
                /* @__PURE__ */ jsx("nav", { className: "space-y-6 pb-10", children: navItems.map((item) => /* @__PURE__ */ jsxs("div", { className: "space-y-3", children: [
                  /* @__PURE__ */ jsx(
                    Link,
                    {
                      to: item.href,
                      className: `block ${textColor} ${isDark ? "hover:text-white/70" : "hover:text-black/70"} transition-colors text-lg font-medium uppercase tracking-wide`,
                      children: item.label
                    }
                  ),
                  item.hasDropdown && item.dropdownItems && /* @__PURE__ */ jsx(
                    "div",
                    {
                      className: `pl-4 space-y-2 border-l ${isDark ? "border-white/10" : "border-black/10"}`,
                      children: item.dropdownItems.map((dropdownItem) => /* @__PURE__ */ jsx(
                        Link,
                        {
                          to: dropdownItem.href,
                          className: `block ${isDark ? "text-white/60 hover:text-white" : "text-black/60 hover:text-black"} transition-colors text-sm uppercase tracking-wide`,
                          children: dropdownItem.label
                        },
                        dropdownItem.href
                      ))
                    }
                  )
                ] }, item.href)) })
              ] })
            }
          )
        ] }) }),
        /* @__PURE__ */ jsx("div", { className: "hidden xl:flex items-center gap-4", children: !isLoggedIn ? /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(
            Button,
            {
              className: isDark ? "!text-white/90" : "!text-black/90",
              asChild: true,
              variant: isDark ? "fill-black" : "outline-black",
              responsive: true,
              children: /* @__PURE__ */ jsx(Link, { to: "/login", children: t("auth.login") })
            }
          ),
          /* @__PURE__ */ jsx(
            Button,
            {
              className: isDark ? "!text-white/90" : "!text-black/90",
              asChild: true,
              variant: isDark ? "fill-black" : "outline-black",
              responsive: true,
              children: /* @__PURE__ */ jsx(Link, { to: "/signup", children: t("auth.signUp") })
            }
          )
        ] }) : /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-6", children: [
          /* @__PURE__ */ jsxs("button", { className: "relative", children: [
            /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" }) }),
            /* @__PURE__ */ jsx("div", { className: "absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full" })
          ] }),
          /* @__PURE__ */ jsx("button", { className: "", children: /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" }) }) }),
          /* @__PURE__ */ jsx("div", { className: `flex items-center gap-2 ${textColor}`, children: /* @__PURE__ */ jsx("span", { className: "text-sm font-medium", children: "0 pts" }) }),
          /* @__PURE__ */ jsx("button", { className: "", children: /* @__PURE__ */ jsx("svg", { className: `w-6 h-6 ${textColor}`, fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" }) }) }),
          /* @__PURE__ */ jsxs("div", { className: "relative", children: [
            /* @__PURE__ */ jsx(
              "button",
              {
                onClick: () => setShowProfileDropdown(!showProfileDropdown),
                className: "w-10 h-10 rounded-full overflow-hidden focus:outline-none focus:ring-2 focus:ring-white/20",
                children: /* @__PURE__ */ jsx(
                  "img",
                  {
                    src: "/images/profile-avatar.jpg",
                    alt: t("header.profile"),
                    className: "w-full h-full object-cover"
                  }
                )
              }
            ),
            showProfileDropdown && /* @__PURE__ */ jsxs("div", { className: `absolute right-0 mt-2 w-64 ${bgColor} border ${borderColor} rounded-lg shadow-lg z-50`, children: [
              /* @__PURE__ */ jsx("div", { className: "px-4 py-3 border-b border-gray-200", children: /* @__PURE__ */ jsx("p", { className: `font-medium text-base ${textColor}`, children: "akanksha bhadauria" }) }),
              /* @__PURE__ */ jsxs("div", { className: "py-1", children: [
                /* @__PURE__ */ jsxs(
                  Link,
                  {
                    to: "/home",
                    className: `px-4 py-3 ${hoverBg} cursor-pointer flex items-center gap-3 ${textColor} hover:${textColor} block`,
                    onClick: () => setShowProfileDropdown(false),
                    children: [
                      /* @__PURE__ */ jsx("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" }) }),
                      /* @__PURE__ */ jsx("span", { children: t("nav.home") })
                    ]
                  }
                ),
                /* @__PURE__ */ jsxs(
                  Link,
                  {
                    to: "/account-settings",
                    className: `px-4 py-3 ${hoverBg} cursor-pointer flex items-center gap-3 ${textColor} hover:${textColor} block`,
                    onClick: () => setShowProfileDropdown(false),
                    children: [
                      /* @__PURE__ */ jsxs("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: [
                        /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" }),
                        /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 12a3 3 0 11-6 0 3 3 0 016 0z" })
                      ] }),
                      /* @__PURE__ */ jsx("span", { children: t("header.accountSettings") })
                    ]
                  }
                ),
                /* @__PURE__ */ jsxs(
                  "button",
                  {
                    onClick: () => {
                      setShowProfileDropdown(false);
                      handleLogoutClick();
                    },
                    className: `w-full text-left px-4 py-3 ${hoverBg} cursor-pointer flex items-center gap-3 ${textColor} hover:${textColor}`,
                    children: [
                      /* @__PURE__ */ jsx("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" }) }),
                      /* @__PURE__ */ jsx("span", { children: t("auth.signOut") })
                    ]
                  }
                )
              ] })
            ] })
          ] })
        ] }) })
      ] })
    }
  );
}
const languages = [
  { code: "en", label: "ENG", currency: "USD" },
  { code: "sp", label: "SPA", currency: "EUR" }
];
function LanguageSelector({ className = "" }) {
  const { currentLanguage, changeLanguage } = useLanguage();
  const [isOpen, setIsOpen] = useState(false);
  const currentOption = languages.find((lang) => lang.code === currentLanguage) || languages[0];
  const handleLanguageChange = (language) => {
    changeLanguage(language);
    setIsOpen(false);
  };
  return /* @__PURE__ */ jsxs("div", { className: `relative ${className}`, children: [
    /* @__PURE__ */ jsxs(
      "button",
      {
        onClick: () => setIsOpen(!isOpen),
        className: "flex items-center gap-2 text-footer-bottom text-white/60 hover:text-white transition-colors cursor-pointer",
        children: [
          /* @__PURE__ */ jsxs("span", { children: [
            currentOption.label,
            " (",
            currentOption.currency,
            ")"
          ] }),
          /* @__PURE__ */ jsx(ChevronUp, { className: `h-3 w-3 text-white/60 transition-transform duration-200 ${isOpen ? "rotate-0" : "rotate-180"}` })
        ]
      }
    ),
    isOpen && /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "fixed inset-0 z-40",
          onClick: () => setIsOpen(false)
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-full right-0 mb-2 bg-gray-900 border border-white/20 rounded-lg shadow-xl z-50 min-w-[120px] backdrop-blur-sm", children: languages.map((language) => /* @__PURE__ */ jsxs(
        "button",
        {
          onClick: () => handleLanguageChange(language.code),
          className: `w-full text-left px-4 py-3 text-sm transition-colors first:rounded-t-lg last:rounded-b-lg ${currentLanguage === language.code ? "bg-white/20 text-white font-medium" : "text-white/70 hover:bg-white/10 hover:text-white"}`,
          children: [
            language.label,
            " (",
            language.currency,
            ")"
          ]
        },
        language.code
      )) })
    ] })
  ] });
}
function AppFooter() {
  const { t, loading } = useLanguage();
  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: "smooth" });
  };
  const socialLinks = [
    { label: "X", href: "https://twitter.com/nomadicstudios" },
    { label: "Facebook", href: "https://facebook.com/nomadicstudios" },
    { label: "LinkedIn", href: "https://linkedin.com/company/nomadic-studios" },
    { label: "Instagram", href: "https://instagram.com/nomadicstudios" }
  ];
  if (loading) {
    return /* @__PURE__ */ jsx("footer", { className: "bg-black text-white", children: /* @__PURE__ */ jsx("div", { className: "px-4 lg:px-14 py-6 lg:py-10", children: /* @__PURE__ */ jsx("div", { className: "text-center", children: "Loading..." }) }) });
  }
  return /* @__PURE__ */ jsx("footer", { className: "bg-black text-white", children: /* @__PURE__ */ jsxs("div", { className: "px-4 lg:px-14 py-6 lg:py-10", children: [
    /* @__PURE__ */ jsxs("div", { className: "hidden lg:block", children: [
      /* @__PURE__ */ jsxs("div", { className: "flex items-start justify-between", children: [
        /* @__PURE__ */ jsx("div", { className: "flex-shrink-0", children: /* @__PURE__ */ jsx(Link, { to: "/", className: "block", children: /* @__PURE__ */ jsx(
          "img",
          {
            src: "/images/logo-white.png",
            alt: "Nomadic Studios",
            className: "h-12 w-auto"
          }
        ) }) }),
        /* @__PURE__ */ jsxs("div", { className: "flex justify-between w-full max-w-[520px]", children: [
          /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("h3", { className: "text-heading-5 !text-white/60 mb-6", children: t("footer.about.title", "About") }),
            /* @__PURE__ */ jsxs("nav", { className: "space-y-4", children: [
              /* @__PURE__ */ jsx(
                Link,
                {
                  to: "/about",
                  className: "block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]",
                  children: t("footer.about.joinOurTeam", "Join Our Team")
                }
              ),
              /* @__PURE__ */ jsx(
                Link,
                {
                  to: "/investors",
                  className: "block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]",
                  children: t("footer.about.investors", "Investors")
                }
              ),
              /* @__PURE__ */ jsx(
                Link,
                {
                  to: "/partners",
                  className: "block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]",
                  children: t("footer.about.partners", "Partners")
                }
              )
            ] })
          ] }),
          /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("h3", { className: "text-heading-5 !text-white/60 mb-6", children: t("footer.socialMedia.title", "Social Media") }),
            /* @__PURE__ */ jsx("nav", { className: "space-y-4", children: socialLinks.map((link) => /* @__PURE__ */ jsx(
              "a",
              {
                href: link.href,
                target: "_blank",
                rel: "noopener noreferrer",
                className: "block text-white hover:text-white/70 transition-colors text-sm uppercase tracking-[0.02em]",
                children: link.label
              },
              link.href
            )) })
          ] })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between mt-16 pt-8 border-t border-white/10", children: [
        /* @__PURE__ */ jsxs(
          Button,
          {
            onClick: scrollToTop,
            variant: "outline-white",
            size: "md",
            className: "text-footer-bottom border-white !text-white hover:bg-white hover:!text-black px-8 py-3",
            children: [
              t("footer.backToTop", "Back to Top"),
              /* @__PURE__ */ jsx(ChevronUp, { className: "ml-2 h-4 w-4" })
            ]
          }
        ),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-8", children: [
          /* @__PURE__ */ jsx("p", { className: "text-footer-bottom text-white/60", children: t("footer.copyright", "©2025 Nomadic Studios") }),
          /* @__PURE__ */ jsx(
            Link,
            {
              to: "/terms",
              className: "text-footer-bottom text-white/60 hover:text-white transition-colors",
              children: t("footer.legal.termsConditions", "Terms & Conditions")
            }
          ),
          /* @__PURE__ */ jsx(
            Link,
            {
              to: "/privacy",
              className: "text-footer-bottom text-white/60 hover:text-white transition-colors",
              children: t("footer.legal.privacyPolicy", "Privacy Policy")
            }
          ),
          /* @__PURE__ */ jsx(LanguageSelector, {})
        ] })
      ] })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "lg:hidden", children: [
      /* @__PURE__ */ jsx("div", { className: "mb-12", children: /* @__PURE__ */ jsx(Link, { to: "/", className: "block w-fit", children: /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/logo-white.png",
          alt: "Nomadic Studios",
          className: "h-10 w-auto"
        }
      ) }) }),
      /* @__PURE__ */ jsxs("div", { className: "mb-12", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-heading-5 !text-white/60 mb-6", children: t("footer.about.title", "About") }),
        /* @__PURE__ */ jsxs("nav", { className: "space-y-6", children: [
          /* @__PURE__ */ jsx(
            Link,
            {
              to: "/about",
              className: "block text-white text-lg uppercase tracking-[0.02em]",
              children: t("footer.about.joinOurTeam", "Join Our Team")
            }
          ),
          /* @__PURE__ */ jsx(
            Link,
            {
              to: "/investors",
              className: "block text-white text-lg uppercase tracking-[0.02em]",
              children: t("footer.about.investors", "Investors")
            }
          ),
          /* @__PURE__ */ jsx(
            Link,
            {
              to: "/partners",
              className: "block text-white text-lg uppercase tracking-[0.02em]",
              children: t("footer.about.partners", "Partners")
            }
          )
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "mb-12", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-heading-5 !text-white/60 mb-6", children: t("footer.socialMedia.title", "Social Media") }),
        /* @__PURE__ */ jsx("nav", { className: "space-y-6", children: socialLinks.map((link) => /* @__PURE__ */ jsx(
          "a",
          {
            href: link.href,
            target: "_blank",
            rel: "noopener noreferrer",
            className: "block text-white text-lg uppercase tracking-[0.02em]",
            children: link.label
          },
          link.href
        )) })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "space-y-6 mb-12", children: [
        /* @__PURE__ */ jsx(
          Link,
          {
            to: "/terms",
            className: "block text-white/60 text-footer-bottom uppercase tracking-[0.02em]",
            children: t("footer.legal.termsConditions", "Terms & Conditions")
          }
        ),
        /* @__PURE__ */ jsx(
          Link,
          {
            to: "/privacy",
            className: "block text-white/60 text-footer-bottom uppercase tracking-[0.02em]",
            children: t("footer.legal.privacyPolicy", "Privacy Policy")
          }
        ),
        /* @__PURE__ */ jsx(LanguageSelector, { className: "text-left" }),
        /* @__PURE__ */ jsx("p", { className: "text-footer-bottom text-white/60", children: t("footer.copyright", "©2025 Nomadic Studios") })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "flex justify-center", children: /* @__PURE__ */ jsxs(
        Button,
        {
          onClick: scrollToTop,
          variant: "outline-white",
          size: "lg",
          className: "text-footer-bottom border-white !text-white hover:bg-white hover:!text-black px-8 py-4 w-full",
          children: [
            t("footer.backToTop", "Back to Top"),
            /* @__PURE__ */ jsx(ChevronUp, { className: "ml-2 h-5 w-5" })
          ]
        }
      ) })
    ] })
  ] }) });
}
const links = () => [
  { rel: "stylesheet", href: stylesheet },
  { rel: "icon", href: "/favicon.ico" },
  { rel: "manifest", href: "/manifest.json" },
  { rel: "apple-touch-icon", href: "/logo192.png" },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Variable.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous"
  },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Bold.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous"
  },
  {
    rel: "preload",
    href: "/fonts/ClashGrotesk-Semibold.woff2",
    as: "font",
    type: "font/woff2",
    crossOrigin: "anonymous"
  }
];
function Layout({ children }) {
  return /* @__PURE__ */ jsxs("html", { lang: "en", children: [
    /* @__PURE__ */ jsxs("head", { children: [
      /* @__PURE__ */ jsx("meta", { charSet: "utf-8" }),
      /* @__PURE__ */ jsx("meta", { name: "viewport", content: "width=device-width, initial-scale=1" }),
      /* @__PURE__ */ jsx(Meta, {}),
      /* @__PURE__ */ jsx(Links, {})
    ] }),
    /* @__PURE__ */ jsxs("body", { children: [
      children,
      /* @__PURE__ */ jsx(ScrollRestoration, {}),
      /* @__PURE__ */ jsx(Scripts, {})
    ] })
  ] });
}
function App() {
  useLocation();
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-black", children: [
    /* @__PURE__ */ jsx(AppHeader, { theme: "dark" }),
    /* @__PURE__ */ jsx("main", { className: "pt-16 lg:pt-20", children: /* @__PURE__ */ jsx(Outlet, {}) }),
    /* @__PURE__ */ jsx(AppFooter, {})
  ] });
}
function ErrorBoundary() {
  const error = useRouteError();
  if (isRouteErrorResponse(error)) {
    return /* @__PURE__ */ jsx("div", { className: "min-h-screen flex items-center justify-center", children: /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
      /* @__PURE__ */ jsxs("h1", { className: "text-4xl font-bold mb-4", children: [
        error.status,
        " ",
        error.statusText
      ] }),
      error.data && /* @__PURE__ */ jsx("p", { className: "text-gray-600", children: error.data })
    ] }) });
  }
  return /* @__PURE__ */ jsx("div", { className: "min-h-screen flex items-center justify-center", children: /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
    /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold mb-4", children: "Oops!" }),
    /* @__PURE__ */ jsx("p", { className: "text-gray-600", children: "Something went wrong." })
  ] }) });
}
const route0 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  ErrorBoundary,
  Layout,
  default: App,
  links
}, Symbol.toStringTag, { value: "Module" }));
function RouteProtection({
  children,
  requireAuth = true
}) {
  const navigate = useNavigate();
  const location = useLocation();
  useEffect(() => {
    const checkAndRedirect = () => {
      const authenticated2 = isAuthenticated();
      if (requireAuth && !authenticated2) {
        navigate("/login");
      } else if (!requireAuth && authenticated2) {
        navigate("/home");
      }
    };
    checkAndRedirect();
    const timeout = setTimeout(checkAndRedirect, 100);
    return () => clearTimeout(timeout);
  }, [requireAuth, navigate, location.pathname]);
  const authenticated = isAuthenticated();
  if (requireAuth && !authenticated) {
    return null;
  }
  if (!requireAuth && authenticated) {
    return null;
  }
  return /* @__PURE__ */ jsx(Fragment, { children });
}
function ProtectedRoute({ children }) {
  return /* @__PURE__ */ jsx(RouteProtection, { requireAuth: true, children });
}
function PublicRoute({ children }) {
  return /* @__PURE__ */ jsx(RouteProtection, { requireAuth: false, children });
}
const meta$k = ({ params }) => {
  return [
    { title: `${params.eventId || "Event"} Details - Nomadics Studios` },
    {
      name: "description",
      content: "Join us for an unforgettable experience at Nomadics Studios events."
    }
  ];
};
const eventsData = [
  {
    id: 1,
    title: "Test event",
    date: "Wednesday, 23 Jul AT 11:50 PM",
    organizer: "Super Admin",
    location: "Testing....",
    points: "100 Points",
    status: "public",
    description: "testing...",
    participants: {
      interested: 1,
      going: 110
    },
    image: "/event.png",
    slug: "test-event"
  },
  {
    id: 2,
    title: "Music event",
    date: "Saturday, 28 Jun AT 12:30 AM",
    organizer: "Super Admin",
    location: "ONLINE",
    points: "20 Points",
    status: "public",
    description: "Embark on a journey with Nomadic Events – where every moment becomes an adventure. Join us for an unforgettable experience filled with laughter, connection, and the thrill of exploring new horizons. It's more than an event; it's a celebration of wanderlust and shared stories. Be part of the Nomadic community and let the extraordinary unfold.",
    participants: {
      interested: 2,
      going: 20
    },
    image: "/bw (2).png",
    slug: "music-event"
  }
];
function findEventBySlug(slug) {
  return eventsData.find((event) => event.slug === slug || event.id.toString() === slug) || eventsData[0];
}
function EventDetailsPage() {
  const { eventId } = useParams();
  const event = findEventBySlug(eventId || "");
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("div", { className: "relative bg-white min-h-screen", children: [
    /* @__PURE__ */ jsxs("section", { className: "relative h-[50vh] lg:h-[55vh] overflow-hidden", children: [
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: event.image,
            alt: event.title,
            className: "h-full w-full object-cover",
            onError: (e) => {
              e.currentTarget.src = "/event.png";
            }
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative z-10 h-full flex items-end", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1000px] px-4 lg:px-8 pb-12 lg:pb-16", children: [
        /* @__PURE__ */ jsx("nav", { className: "mb-6", children: /* @__PURE__ */ jsxs(
          Link,
          {
            to: "/events",
            className: "text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2",
            children: [
              /* @__PURE__ */ jsx("svg", { className: "w-4 h-4", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 19l-7-7 7-7" }) }),
              "Back to Events"
            ]
          }
        ) }),
        /* @__PURE__ */ jsx("h1", { className: "text-3xl lg:text-5xl font-bold text-white mb-6 leading-tight max-w-4xl", children: event.title }),
        /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-2 text-white/90", children: [
          /* @__PURE__ */ jsx("div", { className: "text-lg font-medium", children: event.date }),
          /* @__PURE__ */ jsxs("div", { className: "text-white/80", children: [
            "by ",
            event.organizer
          ] })
        ] })
      ] }) })
    ] }),
    /* @__PURE__ */ jsx("section", { className: "bg-gray-50 py-12 lg:py-16", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1000px] px-4 lg:px-8", children: [
      /* @__PURE__ */ jsx("div", { className: "mb-8", children: /* @__PURE__ */ jsxs(
        Button,
        {
          variant: "outline-black",
          size: "sm",
          className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300",
          children: [
            /* @__PURE__ */ jsx("svg", { className: "w-4 h-4 mr-2", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.367 2.684 3 3 0 00-5.367-2.684z" }) }),
            "Share"
          ]
        }
      ) }),
      /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 md:grid-cols-2 gap-8 mb-12", children: [
        /* @__PURE__ */ jsxs("div", { className: "space-y-6", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex-shrink-0", children: /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 text-gray-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" }) }) }),
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsxs("div", { className: "text-sm text-gray-600 mb-1", children: [
              event.participants.interested,
              " User Interested /"
            ] }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex-shrink-0", children: /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 text-gray-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" }) }) }),
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("div", { className: "text-black font-medium", children: event.date }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex-shrink-0", children: /* @__PURE__ */ jsxs("svg", { className: "w-5 h-5 text-gray-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: [
              /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" }),
              /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 11a3 3 0 11-6 0 3 3 0 016 0z" })
            ] }) }),
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("div", { className: "text-black font-medium", children: event.location }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex-shrink-0", children: /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 text-gray-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" }) }) }),
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("div", { className: "text-black font-medium", children: event.points }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex-shrink-0", children: /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 text-gray-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" }) }) }),
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("div", { className: "text-black font-medium", children: event.status }) })
          ] }),
          /* @__PURE__ */ jsx("div", { className: "pt-4", children: /* @__PURE__ */ jsx("p", { className: "text-gray-700 leading-relaxed", children: event.description }) })
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "text-lg font-bold text-black mb-6", children: "Participants" }),
          /* @__PURE__ */ jsxs("div", { className: "flex gap-4 mb-8", children: [
            /* @__PURE__ */ jsx(
              Button,
              {
                variant: "outline-black",
                size: "md",
                className: "flex-1 bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300",
                children: /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2", children: [
                  /* @__PURE__ */ jsx("svg", { className: "w-4 h-4", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" }) }),
                  "Interested 🔥"
                ] })
              }
            ),
            /* @__PURE__ */ jsx(
              Button,
              {
                variant: "fill-black",
                size: "md",
                className: "flex-1",
                children: /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2", children: [
                  event.participants.going,
                  " Points",
                  /* @__PURE__ */ jsx("span", { children: "Going ➡" })
                ] })
              }
            )
          ] })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "mb-8", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-lg font-bold text-black mb-4", children: "Address" }),
        /* @__PURE__ */ jsx("p", { className: "text-gray-700", children: event.location })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "mb-8", children: /* @__PURE__ */ jsx("div", { className: "w-full h-96 bg-gray-200 rounded-lg overflow-hidden", children: /* @__PURE__ */ jsx(
        "iframe",
        {
          src: "https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d387191.33750346623!2d-73.97968099999999!3d40.6974881!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x89c24fa5d33f083b%3A0xc80b8f06e177fe62!2sNew%20York%2C%20NY%2C%20USA!5e0!3m2!1sen!2sus!4v1234567890123",
          width: "100%",
          height: "100%",
          style: { border: 0 },
          allowFullScreen: true,
          loading: "lazy",
          referrerPolicy: "no-referrer-when-downgrade",
          className: "rounded-lg"
        }
      ) }) })
    ] }) })
  ] }) });
}
const route1 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: EventDetailsPage,
  meta: meta$k
}, Symbol.toStringTag, { value: "Module" }));
const AdditionalInformation = () => {
  var _a, _b;
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const uuid = (_a = location.state) == null ? void 0 : _a.uuid;
  (_b = location.state) == null ? void 0 : _b.userData;
  const [formData, setFormData] = useState({
    home_city: "",
    creator_interest: [],
    profile_link: "",
    instagram_link: "",
    facebook_link: "",
    twitter_link: "",
    sm_link: "",
    referred_by: ""
  });
  const creativeInterests = [
    "Fine Art",
    "Fashion",
    "Music",
    "Photography",
    "Video",
    "Maker Space"
  ];
  const [locations, setLocations] = useState([]);
  const [locationsLoading, setLocationsLoading] = useState(true);
  useEffect(() => {
    if (!uuid) {
      navigate("/signup");
      return;
    }
    setIsVisible(true);
    const fetchLocations = async () => {
      try {
        setLocationsLoading(true);
        const response = await locationsAPI.getLocations();
        if (response.status && response.data) {
          const uniqueLocations = response.data.filter(
            (location2, index, self) => index === self.findIndex((l) => l.id === location2.id && l.location === location2.location)
          );
          setLocations(uniqueLocations);
        } else {
          console.error("Failed to fetch locations:", response.error);
        }
      } catch (error2) {
        console.error("Error fetching locations:", error2);
      } finally {
        setLocationsLoading(false);
      }
    };
    fetchLocations();
  }, [uuid, navigate]);
  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };
  const handleInterestToggle = (interest) => {
    setFormData((prev) => ({
      ...prev,
      creator_interest: prev.creator_interest.includes(interest) ? prev.creator_interest.filter((i) => i !== interest) : [...prev.creator_interest, interest]
    }));
    if (error) setError(null);
  };
  const handleSubmit = async (e) => {
    e.preventDefault();
    if (!formData.home_city.trim()) {
      setError(t("auth.additionalInfo.errors.homeCityRequired", "Home city is required"));
      return;
    }
    setIsLoading(true);
    setError(null);
    try {
      const response = await authAPI.saveAdditionalInfo({
        uuid,
        home_city: formData.home_city,
        // Send location ID as selected
        creator_interest: formData.creator_interest,
        profile_link: formData.profile_link || void 0,
        instagram_link: formData.instagram_link || void 0,
        facebook_link: formData.facebook_link || void 0,
        twitter_link: formData.twitter_link || void 0,
        sm_link: formData.sm_link || void 0,
        referred_by: formData.referred_by || void 0
      });
      if (response.status) {
        navigate("/login", {
          state: {
            message: t("auth.additionalInfo.success", "Account created successfully! Please login to continue."),
            type: "success"
          }
        });
      } else {
        setError(response.error || t("auth.additionalInfo.errors.saveFailed", "Failed to save information. Please try again."));
      }
    } catch (err) {
      console.error("Additional info save error:", err);
      setError(err instanceof Error ? err.message : t("auth.additionalInfo.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx("style", { children: `
        select option {
          background-color: #1f2937 !important;
          color: white !important;
          padding: 8px !important;
        }
        select option:hover {
          background-color: #374151 !important;
        }
        select option:checked {
          background-color: #f97316 !important;
        }
      ` }),
    /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "/images/home/welcome.png",
            alt: "",
            className: "absolute inset-0 h-full w-full object-cover opacity-40"
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
        /* @__PURE__ */ jsx("div", { className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60", style: { animationDelay: "0s", animationDuration: "3s" } }),
        /* @__PURE__ */ jsx("div", { className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40", style: { animationDelay: "1s" } }),
        /* @__PURE__ */ jsx("div", { className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30", style: { animationDelay: "2s" } }),
        /* @__PURE__ */ jsx("div", { className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50", style: { animationDelay: "1.5s", animationDuration: "4s" } })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0 py-12", children: [
        /* @__PURE__ */ jsx("div", { className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.additionalInfo.title", "We would like to know you more") }) }),
        /* @__PURE__ */ jsx("div", { className: `w-full max-w-2xl transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
          error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
          /* @__PURE__ */ jsxs("form", { onSubmit: handleSubmit, className: "space-y-6", children: [
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsxs("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: [
                t("auth.additionalInfo.homeCity", "What is your home city?"),
                " *"
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "relative", children: [
                /* @__PURE__ */ jsxs(
                  "select",
                  {
                    name: "home_city",
                    value: formData.home_city,
                    onChange: handleInputChange,
                    className: "w-full px-4 py-4 bg-gray-800/90 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all appearance-none cursor-pointer hover:bg-gray-700/90",
                    required: true,
                    disabled: isLoading || locationsLoading,
                    style: {
                      colorScheme: "dark"
                    },
                    children: [
                      /* @__PURE__ */ jsx("option", { value: "", className: "bg-gray-800 text-gray-400", children: locationsLoading ? t("auth.additionalInfo.loadingLocations", "Loading locations...") : t("auth.additionalInfo.selectLocation", "Select a location") }),
                      locations.map((location2) => /* @__PURE__ */ jsx(
                        "option",
                        {
                          value: location2.id.toString(),
                          className: "bg-gray-800 text-white py-2",
                          children: location2.location
                        },
                        `${location2.id}-${location2.location}`
                      ))
                    ]
                  }
                ),
                /* @__PURE__ */ jsx("svg", { className: "absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-white/60 pointer-events-none", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M19 9l-7 7-7-7" }) })
              ] })
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.additionalInfo.creativeInterests", "What are you creating?") }),
              /* @__PURE__ */ jsx("div", { className: "grid grid-cols-2 gap-4", children: creativeInterests.map((interest) => /* @__PURE__ */ jsxs(
                "div",
                {
                  className: `flex items-center p-4 rounded-xl border-2 cursor-pointer transition-all ${formData.creator_interest.includes(interest) ? "border-orange-400 bg-orange-500/20" : "border-white/20 bg-white/5 hover:bg-white/10"}`,
                  onClick: () => handleInterestToggle(interest),
                  children: [
                    /* @__PURE__ */ jsx("div", { className: `w-5 h-5 rounded border-2 transition-all duration-200 ${formData.creator_interest.includes(interest) ? "border-orange-400 bg-orange-400" : "border-white/40"}`, children: formData.creator_interest.includes(interest) && /* @__PURE__ */ jsx("svg", { className: "w-3 h-3 text-white ml-0.5 mt-0.5", fill: "currentColor", viewBox: "0 0 20 20", children: /* @__PURE__ */ jsx("path", { fillRule: "evenodd", d: "M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z", clipRule: "evenodd" }) }) }),
                    /* @__PURE__ */ jsx("span", { className: "text-white ml-3 font-medium", children: interest })
                  ]
                },
                interest
              )) })
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.additionalInfo.profileLink", "Link to profile") }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "url",
                  name: "profile_link",
                  placeholder: t("auth.additionalInfo.linkPlaceholder", "Link"),
                  value: formData.profile_link,
                  onChange: handleInputChange,
                  className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  disabled: isLoading
                }
              )
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.additionalInfo.socialMedia", "Link to social media") }),
              /* @__PURE__ */ jsxs("div", { className: "space-y-3", children: [
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "url",
                    name: "instagram_link",
                    placeholder: t("auth.additionalInfo.instagramLink", "Link to Instagram"),
                    value: formData.instagram_link,
                    onChange: handleInputChange,
                    className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                    disabled: isLoading
                  }
                ),
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "url",
                    name: "facebook_link",
                    placeholder: t("auth.additionalInfo.facebookLink", "Link to Facebook"),
                    value: formData.facebook_link,
                    onChange: handleInputChange,
                    className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                    disabled: isLoading
                  }
                ),
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "url",
                    name: "twitter_link",
                    placeholder: t("auth.additionalInfo.twitterLink", "Link to Twitter"),
                    value: formData.twitter_link,
                    onChange: handleInputChange,
                    className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                    disabled: isLoading
                  }
                ),
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "url",
                    name: "sm_link",
                    placeholder: t("auth.additionalInfo.smLink", "Link to SM"),
                    value: formData.sm_link,
                    onChange: handleInputChange,
                    className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                    disabled: isLoading
                  }
                )
              ] })
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.additionalInfo.referredBy", "Please provide the name of the member who referred you") }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "text",
                  name: "referred_by",
                  placeholder: t("auth.additionalInfo.namePlaceholder", "Name"),
                  value: formData.referred_by,
                  onChange: handleInputChange,
                  className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  disabled: isLoading
                }
              )
            ] }),
            /* @__PURE__ */ jsx("div", { className: "pt-6", children: /* @__PURE__ */ jsx(
              Button,
              {
                type: "submit",
                variant: "fill-white",
                size: "lg",
                className: "w-full",
                disabled: isLoading,
                arrowRight: !isLoading,
                children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                  /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                  t("auth.additionalInfo.saving", "Saving...")
                ] }) : t("auth.additionalInfo.save", "Save")
              }
            ) })
          ] })
        ] }) })
      ] })
    ] })
  ] });
};
function AdditionalInformationRoute() {
  return /* @__PURE__ */ jsx(AdditionalInformation, {});
}
const route2 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: AdditionalInformationRoute
}, Symbol.toStringTag, { value: "Module" }));
const UploadProfilePicture = () => {
  var _a, _b;
  const navigate = useNavigate();
  const location = useLocation();
  const fileInputRef = useRef(null);
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedImage, setSelectedImage] = useState(null);
  const [imageFile, setImageFile] = useState(null);
  const { t } = useLanguage();
  const uuid = (_a = location.state) == null ? void 0 : _a.uuid;
  const userData = (_b = location.state) == null ? void 0 : _b.userData;
  useEffect(() => {
    if (!uuid) {
      navigate("/signup");
      return;
    }
    setIsVisible(true);
  }, [uuid, navigate]);
  const handleImageSelect = (e) => {
    var _a2;
    const file = (_a2 = e.target.files) == null ? void 0 : _a2[0];
    if (!file) return;
    if (!file.type.startsWith("image/")) {
      setError(t("uploadProfile.invalidFileType"));
      return;
    }
    if (file.size > 5 * 1024 * 1024) {
      setError(t("uploadProfile.fileSizeLimit"));
      return;
    }
    setImageFile(file);
    setError(null);
    const reader = new FileReader();
    reader.onload = (e2) => {
      var _a3;
      setSelectedImage((_a3 = e2.target) == null ? void 0 : _a3.result);
    };
    reader.readAsDataURL(file);
  };
  const handleUpload = async () => {
    if (!imageFile || !uuid) return;
    setIsLoading(true);
    setError(null);
    try {
      const reader = new FileReader();
      reader.onload = async (e) => {
        var _a2;
        const base64Image = (_a2 = e.target) == null ? void 0 : _a2.result;
        try {
          const response = await authAPI.saveProfilePicture(uuid, base64Image);
          if (response.status) {
            navigate("/additional-information", {
              state: {
                uuid,
                userData
              }
            });
          } else {
            setError(response.error || t("uploadProfile.uploadFailed"));
          }
        } catch (err) {
          console.error("Profile picture upload error:", err);
          setError(err instanceof Error ? err.message : t("uploadProfile.unexpectedError"));
        } finally {
          setIsLoading(false);
        }
      };
      reader.readAsDataURL(imageFile);
    } catch (err) {
      console.error("File processing error:", err);
      setError(t("uploadProfile.processingError"));
      setIsLoading(false);
    }
  };
  const handleSkip = () => {
    navigate("/additional-information", {
      state: {
        uuid,
        userData
      }
    });
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx("div", { className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60", style: { animationDelay: "0s", animationDuration: "3s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40", style: { animationDelay: "1s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30", style: { animationDelay: "2s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50", style: { animationDelay: "1.5s", animationDuration: "4s" } })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs("div", { className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: [
        /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("uploadProfile.title") }),
        /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("uploadProfile.subtitle") })
      ] }),
      /* @__PURE__ */ jsx("div", { className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
        /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider", children: t("uploadProfile.profilePicture") }),
        error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
        /* @__PURE__ */ jsx("div", { className: "mb-8 text-center", children: /* @__PURE__ */ jsxs("div", { className: "relative inline-block", children: [
          /* @__PURE__ */ jsx("div", { className: "w-32 h-32 mx-auto rounded-full overflow-hidden border-4 border-white/20 bg-white/10 flex items-center justify-center", children: selectedImage ? /* @__PURE__ */ jsx(
            "img",
            {
              src: selectedImage,
              alt: t("uploadProfile.profilePreview"),
              className: "w-full h-full object-cover"
            }
          ) : /* @__PURE__ */ jsx("svg", { className: "w-16 h-16 text-white/40", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" }) }) }),
          selectedImage && /* @__PURE__ */ jsx(
            "button",
            {
              onClick: () => {
                setSelectedImage(null);
                setImageFile(null);
                if (fileInputRef.current) {
                  fileInputRef.current.value = "";
                }
              },
              className: "absolute -top-2 -right-2 bg-red-500 hover:bg-red-600 text-white rounded-full p-2 transition-colors",
              children: /* @__PURE__ */ jsx("svg", { className: "w-4 h-4", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M6 18L18 6M6 6l12 12" }) })
            }
          )
        ] }) }),
        /* @__PURE__ */ jsxs("div", { className: "space-y-6", children: [
          /* @__PURE__ */ jsx(
            "input",
            {
              ref: fileInputRef,
              type: "file",
              accept: "image/*",
              onChange: handleImageSelect,
              className: "hidden",
              disabled: isLoading
            }
          ),
          !selectedImage && /* @__PURE__ */ jsxs(
            Button,
            {
              type: "button",
              variant: "outline-white",
              size: "lg",
              className: "w-full",
              onClick: () => {
                var _a2;
                return (_a2 = fileInputRef.current) == null ? void 0 : _a2.click();
              },
              disabled: isLoading,
              children: [
                /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 mr-3", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M12 6v6m0 0v6m0-6h6m-6 0H6" }) }),
                t("uploadProfile.chooseImage")
              ]
            }
          ),
          selectedImage && /* @__PURE__ */ jsx(
            Button,
            {
              type: "button",
              variant: "fill-white",
              size: "lg",
              className: "w-full",
              onClick: handleUpload,
              disabled: isLoading,
              arrowRight: !isLoading,
              children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                t("uploadProfile.uploading")
              ] }) : t("uploadProfile.uploadAndContinue")
            }
          ),
          /* @__PURE__ */ jsx(
            Button,
            {
              type: "button",
              variant: "outline-white",
              size: "md",
              className: "w-full",
              onClick: handleSkip,
              disabled: isLoading,
              children: t("uploadProfile.skipForNow")
            }
          ),
          /* @__PURE__ */ jsxs("div", { className: "text-center text-white/60 text-xs space-y-1", children: [
            /* @__PURE__ */ jsx("p", { children: t("uploadProfile.acceptedFormats") }),
            /* @__PURE__ */ jsx("p", { children: t("uploadProfile.maxFileSize") })
          ] })
        ] })
      ] }) })
    ] })
  ] });
};
function UploadProfilePictureRoute() {
  return /* @__PURE__ */ jsx(UploadProfilePicture, {});
}
const route3 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: UploadProfilePictureRoute
}, Symbol.toStringTag, { value: "Module" }));
function SectionHeader({
  title,
  eyebrow,
  className = "",
  layout = "horizontal"
}) {
  console.log(layout, title);
  const layoutClasses = layout === "vertical" ? "flex flex-col gap-3 lg:gap-4" : "flex flex-col lg:flex-row lg:items-end gap-3 lg:gap-6";
  return /* @__PURE__ */ jsxs("div", { className: `${layoutClasses} ${className}`, children: [
    eyebrow && /* @__PURE__ */ jsx("div", { className: "flex items-center gap-2", children: /* @__PURE__ */ jsxs("span", { className: "text-button-small lg:text-base leading-4 lg:leading-5 tracking-[0.02em] uppercase text-neutral-900/60 whitespace-nowrap", children: [
      "[",
      eyebrow,
      "]"
    ] }) }),
    /* @__PURE__ */ jsx(
      "h2",
      {
        className: cn(
          "text-heading-5 lg:text-[68px] leading-[100%] font-medium tracking-[-0.03em] uppercase text-neutral-900 ",
          layout === "horizontal" ? "ml-6" : "ml-0",
          eyebrow ? "" : "!ml-0",
          className
        ),
        children: title
      }
    )
  ] });
}
function LocationIcon() {
  return /* @__PURE__ */ jsx("div", { className: "w-5 h-5 flex items-center justify-center", children: /* @__PURE__ */ jsx("svg", { width: "15", height: "21", viewBox: "0 0 15 21", fill: "none", xmlns: "http://www.w3.org/2000/svg", children: /* @__PURE__ */ jsx("path", { d: "M13.125 8.26562C13.125 5.17969 10.5859 2.64062 7.5 2.64062C4.375 2.64062 1.875 5.17969 1.875 8.26562C1.875 8.77344 2.03125 9.51562 2.46094 10.5312C2.85156 11.4688 3.4375 12.5625 4.10156 13.6562C5.23438 15.4531 6.52344 17.1719 7.5 18.4219C8.4375 17.1719 9.72656 15.4531 10.8594 13.6562C11.5234 12.5625 12.1094 11.4688 12.5 10.5312C12.9297 9.51562 13.125 8.77344 13.125 8.26562ZM15 8.26562C15 11.7031 10.4297 17.7578 8.39844 20.2969C7.92969 20.8828 7.03125 20.8828 6.5625 20.2969C4.57031 17.7578 0 11.7031 0 8.26562C0 4.125 3.35938 0.765625 7.5 0.765625C11.6406 0.765625 15 4.125 15 8.26562ZM8.75 8.26562C8.75 7.60156 8.16406 7.01562 7.5 7.01562C6.79688 7.01562 6.25 7.60156 6.25 8.26562C6.25 8.96875 6.79688 9.51562 7.5 9.51562C8.16406 9.51562 8.75 8.96875 8.75 8.26562ZM4.375 8.26562C4.375 7.17188 4.96094 6.15625 5.9375 5.57031C6.875 5.02344 8.08594 5.02344 9.0625 5.57031C10 6.15625 10.625 7.17188 10.625 8.26562C10.625 9.39844 10 10.4141 9.0625 11C8.08594 11.5469 6.875 11.5469 5.9375 11C4.96094 10.4141 4.375 9.39844 4.375 8.26562Z", fill: "#121212", fillOpacity: "0.4" }) }) });
}
function ClockIcon() {
  return /* @__PURE__ */ jsx("svg", { width: "20", height: "22", viewBox: "0 0 20 22", fill: "none", xmlns: "http://www.w3.org/2000/svg", children: /* @__PURE__ */ jsx("path", { d: "M18.125 10.7656C18.125 7.875 16.5625 5.21875 14.0625 3.73438C11.5234 2.28906 8.4375 2.28906 5.9375 3.73438C3.39844 5.21875 1.875 7.875 1.875 10.7656C1.875 13.6953 3.39844 16.3516 5.9375 17.8359C8.4375 19.2812 11.5234 19.2812 14.0625 17.8359C16.5625 16.3516 18.125 13.6953 18.125 10.7656ZM0 10.7656C0 7.21094 1.875 3.92969 5 2.13281C8.08594 0.335938 11.875 0.335938 15 2.13281C18.0859 3.92969 20 7.21094 20 10.7656C20 14.3594 18.0859 17.6406 15 19.4375C11.875 21.2344 8.08594 21.2344 5 19.4375C1.875 17.6406 0 14.3594 0 10.7656ZM9.0625 5.45312C9.0625 4.94531 9.45312 4.51562 10 4.51562C10.5078 4.51562 10.9375 4.94531 10.9375 5.45312V10.2969L14.2578 12.4844C14.6875 12.7969 14.8047 13.3828 14.5312 13.8125C14.2188 14.2422 13.6328 14.3594 13.2031 14.0469L9.45312 11.5469C9.21875 11.3906 9.0625 11.0781 9.0625 10.7656V5.45312Z", fill: "#121212", fillOpacity: "0.4" }) });
}
function PhoneIcon() {
  return /* @__PURE__ */ jsx("svg", { width: "21", height: "21", viewBox: "0 0 21 21", fill: "none", xmlns: "http://www.w3.org/2000/svg", children: /* @__PURE__ */ jsx("path", { d: "M14.6484 11.5469L19.0234 13.4219C19.7266 13.6953 20.1172 14.4375 19.9609 15.1797L19.0234 19.5547C18.8672 20.2578 18.2031 20.8047 17.5 20.8047C17.2266 20.8047 16.9922 20.7656 16.7578 20.7656C16.3672 20.7656 15.9766 20.7266 15.625 20.6875C6.83594 19.75 0 12.3281 0 3.26562C0 2.5625 0.507812 1.89844 1.21094 1.74219L5.58594 0.804688C6.32812 0.648438 7.07031 1.03906 7.34375 1.74219L9.21875 6.11719C9.49219 6.74219 9.33594 7.48438 8.78906 7.91406L7.1875 9.24219C8.24219 11.0391 9.72656 12.5234 11.5234 13.5781L12.8516 11.9766C13.2812 11.4297 14.0234 11.2734 14.6484 11.5469ZM17.2266 18.8906L18.0469 15.0234L14.1406 13.3438L13.0078 14.75C12.4219 15.4531 11.4062 15.6484 10.5859 15.1797C8.51562 13.9688 6.79688 12.25 5.58594 10.1797C5.11719 9.35938 5.3125 8.34375 6.01562 7.75781L7.42188 6.625L5.74219 2.71875L1.875 3.53906C1.99219 11.9766 8.78906 18.7734 17.2266 18.8906Z", fill: "#121212", fillOpacity: "0.4" }) });
}
function EmailIcon() {
  return /* @__PURE__ */ jsx("svg", { width: "20", height: "16", viewBox: "0 0 20 16", fill: "none", xmlns: "http://www.w3.org/2000/svg", children: /* @__PURE__ */ jsx("path", { d: "M2.5 2.14062C2.14844 2.14062 1.875 2.45312 1.875 2.76562V3.66406L8.59375 9.17188C9.41406 9.83594 10.5469 9.83594 11.3672 9.17188L18.125 3.66406V2.76562C18.125 2.45312 17.8125 2.14062 17.5 2.14062H2.5ZM1.875 6.08594V12.7656C1.875 13.1172 2.14844 13.3906 2.5 13.3906H17.5C17.8125 13.3906 18.125 13.1172 18.125 12.7656V6.08594L12.5781 10.6172C11.0547 11.8672 8.90625 11.8672 7.42188 10.6172L1.875 6.08594ZM0 2.76562C0 1.39844 1.09375 0.265625 2.5 0.265625H17.5C18.8672 0.265625 20 1.39844 20 2.76562V12.7656C20 14.1719 18.8672 15.2656 17.5 15.2656H2.5C1.09375 15.2656 0 14.1719 0 12.7656V2.76562Z", fill: "rgba(18, 18, 18, 0.4)" }) });
}
function PlanYourVisitSection({
  address,
  hours,
  phone,
  email,
  additionalInfo,
  title,
  location
}) {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full px-4 lg:px-14 py-14 lg:py-24", children: [
    title && /* @__PURE__ */ jsx(SectionHeader, { title, className: "mb-4" }),
    /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8 lg:gap-12 mb-12 lg:mb-16", children: [
      /* @__PURE__ */ jsxs("div", { className: "border-r border-black/10 pr-8 lg:pr-12", children: [
        /* @__PURE__ */ jsx("div", { className: "flex items-center gap-3 mb-4", children: /* @__PURE__ */ jsx("h3", { className: "text-plan-header text-black/60", children: "ADDRESS" }) }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-start gap-3", children: [
          /* @__PURE__ */ jsx("div", { className: "mt-1", children: /* @__PURE__ */ jsx(LocationIcon, {}) }),
          /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black", children: address })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "border-r border-black/10 pr-8 lg:pr-12", children: [
        /* @__PURE__ */ jsx("div", { className: "flex items-center gap-3 mb-4", children: /* @__PURE__ */ jsx("h3", { className: "text-plan-header text-black/60", children: "OPENING HOURS" }) }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
          /* @__PURE__ */ jsx(ClockIcon, {}),
          /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black", children: hours })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "border-r border-black/10 pr-8 lg:pr-12", children: [
        /* @__PURE__ */ jsx("div", { className: "flex items-center gap-3 mb-4", children: /* @__PURE__ */ jsx("h3", { className: "text-plan-header text-black/60", children: "CONTACT" }) }),
        /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
            /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(PhoneIcon, {}) }),
            /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black", children: phone })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-start gap-3", children: [
            /* @__PURE__ */ jsx("div", { className: "mt-1", children: /* @__PURE__ */ jsx(EmailIcon, {}) }),
            /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black", children: email })
          ] })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx("div", { className: "flex items-center gap-3 mb-4", children: /* @__PURE__ */ jsx("h3", { className: "text-plan-header text-black/60", children: "ADDITIONAL INFORMATION" }) }),
        /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black", children: additionalInfo })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "mapouter", style: { width: "100%", height: "400px" }, children: /* @__PURE__ */ jsx(
      "iframe",
      {
        className: "gmap_iframe",
        width: "100%",
        height: "400",
        frameBorder: 0,
        scrolling: "no",
        marginHeight: 0,
        marginWidth: 0,
        src: `https://maps.google.com/maps?width=1324&height=400&hl=en&q=${encodeURIComponent(
          location
        )}&t=&z=14&ie=UTF8&iwloc=B&output=embed`
      }
    ) })
  ] }) });
}
function BookingForm({
  defaultStartTime = "2025-08-21T10:30",
  defaultEndTime = "2025-08-21T13:30",
  duration = "3h",
  points = "130 points"
}) {
  return /* @__PURE__ */ jsx("div", { className: "bg-white border border-gray-200  p-6", children: /* @__PURE__ */ jsxs("form", { className: "space-y-2", children: [
    /* @__PURE__ */ jsxs("div", { children: [
      /* @__PURE__ */ jsxs("label", { className: "flex items-center text-sm font-medium text-gray-700 mb-2", children: [
        /* @__PURE__ */ jsx(
          "svg",
          {
            className: "w-4 h-4 mr-2",
            fill: "currentColor",
            viewBox: "0 0 20 20",
            children: /* @__PURE__ */ jsx(
              "path",
              {
                fillRule: "evenodd",
                d: "M6 2a1 1 0 00-1 1v1H4a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2h-1V3a1 1 0 10-2 0v1H7V3a1 1 0 00-1-1zm0 5a1 1 0 000 2h8a1 1 0 100-2H6z",
                clipRule: "evenodd"
              }
            )
          }
        ),
        "START"
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx(
        "input",
        {
          type: "datetime-local",
          className: "w-full px-4 py-3 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-black !text-black",
          defaultValue: defaultStartTime
        }
      ) })
    ] }),
    /* @__PURE__ */ jsxs("div", { children: [
      /* @__PURE__ */ jsxs("label", { className: "flex items-center text-sm font-medium text-gray-700 mb-2", children: [
        /* @__PURE__ */ jsx(
          "svg",
          {
            className: "w-4 h-4 mr-2",
            fill: "currentColor",
            viewBox: "0 0 20 20",
            children: /* @__PURE__ */ jsx(
              "path",
              {
                fillRule: "evenodd",
                d: "M6 2a1 1 0 00-1 1v1H4a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2h-1V3a1 1 0 10-2 0v1H7V3a1 1 0 00-1-1zm0 5a1 1 0 000 2h8a1 1 0 100-2H6z",
                clipRule: "evenodd"
              }
            )
          }
        ),
        "END"
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx(
        "input",
        {
          type: "datetime-local",
          className: "w-full px-4 py-3 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-black !text-black",
          defaultValue: defaultEndTime
        }
      ) })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "text-sm text-gray-600 py-2", children: [
      "Your reservation: ",
      duration,
      " • From: ",
      points
    ] }),
    /* @__PURE__ */ jsx(
      Button,
      {
        type: "submit",
        className: "w-full",
        variant: "fill-black",
        size: "lg",
        arrowRight: true,
        children: "BOOK NOW"
      }
    )
  ] }) });
}
function PhotoGallery({
  images,
  className,
  showAllButtonText = "SHOW ALL PHOTOS"
}) {
  var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j;
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const visibleImages = images.slice(0, 3);
  const openModal = (index = 0) => {
    setCurrentImageIndex(index);
    setIsModalOpen(true);
  };
  const closeModal = () => {
    setIsModalOpen(false);
  };
  const goToPrevious = useCallback(() => {
    setCurrentImageIndex((prev) => (prev - 1 + images.length) % images.length);
  }, [images.length]);
  const goToNext = useCallback(() => {
    setCurrentImageIndex((prev) => (prev + 1) % images.length);
  }, [images.length]);
  useEffect(() => {
    const handleKeyDown = (e) => {
      if (!isModalOpen) return;
      switch (e.key) {
        case "Escape":
          closeModal();
          break;
        case "ArrowLeft":
          goToPrevious();
          break;
        case "ArrowRight":
          goToNext();
          break;
      }
    };
    window.addEventListener("keydown", handleKeyDown);
    return () => window.removeEventListener("keydown", handleKeyDown);
  }, [isModalOpen, goToPrevious, goToNext]);
  useEffect(() => {
    if (isModalOpen) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }
    return () => {
      document.body.style.overflow = "";
    };
  }, [isModalOpen]);
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsxs("div", { className: cn("grid grid-cols-3 gap-2", className), children: [
      /* @__PURE__ */ jsx("div", { className: "col-span-2 relative group", children: /* @__PURE__ */ jsx(
        "img",
        {
          src: (_a = visibleImages[0]) == null ? void 0 : _a.src,
          alt: (_b = visibleImages[0]) == null ? void 0 : _b.alt,
          className: "w-full h-[400px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]",
          onClick: () => openModal(0)
        }
      ) }),
      visibleImages.length > 1 && /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
        /* @__PURE__ */ jsx("div", { className: "relative group", children: /* @__PURE__ */ jsx(
          "img",
          {
            src: (_c = visibleImages[1]) == null ? void 0 : _c.src,
            alt: (_d = visibleImages[1]) == null ? void 0 : _d.alt,
            className: "w-full h-[196px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]",
            onClick: () => openModal(1)
          }
        ) }),
        visibleImages[2] && /* @__PURE__ */ jsxs("div", { className: "relative group", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: visibleImages[2].src,
              alt: visibleImages[2].alt,
              className: "w-full h-[196px] object-cover cursor-pointer transition-transform duration-300 group-hover:scale-[1.01]",
              onClick: () => openModal(2)
            }
          ),
          images.length > 3 && /* @__PURE__ */ jsx(
            Button,
            {
              onClick: () => openModal(0),
              className: "absolute bottom-4 right-4 shadow-sm",
              variant: "fill-white",
              size: "sm",
              children: showAllButtonText
            }
          )
        ] })
      ] })
    ] }),
    isModalOpen && /* @__PURE__ */ jsxs("div", { className: "fixed inset-0 z-50 bg-black/95 flex items-center justify-center", children: [
      /* @__PURE__ */ jsx(
        "button",
        {
          onClick: closeModal,
          className: "absolute top-4 right-4 text-white/80 hover:text-white p-2 transition-colors z-10",
          "aria-label": "Close",
          children: /* @__PURE__ */ jsx(
            "svg",
            {
              className: "w-8 h-8",
              fill: "none",
              stroke: "currentColor",
              viewBox: "0 0 24 24",
              children: /* @__PURE__ */ jsx(
                "path",
                {
                  strokeLinecap: "round",
                  strokeLinejoin: "round",
                  strokeWidth: 2,
                  d: "M6 18L18 6M6 6l12 12"
                }
              )
            }
          )
        }
      ),
      /* @__PURE__ */ jsx(
        "button",
        {
          onClick: goToPrevious,
          className: "absolute left-4 top-1/2 -translate-y-1/2 text-white/80 hover:text-white p-2 transition-colors",
          "aria-label": "Previous image",
          children: /* @__PURE__ */ jsx(
            "svg",
            {
              className: "w-8 h-8",
              fill: "none",
              stroke: "currentColor",
              viewBox: "0 0 24 24",
              children: /* @__PURE__ */ jsx(
                "path",
                {
                  strokeLinecap: "round",
                  strokeLinejoin: "round",
                  strokeWidth: 2,
                  d: "M15 19l-7-7 7-7"
                }
              )
            }
          )
        }
      ),
      /* @__PURE__ */ jsx(
        "button",
        {
          onClick: goToNext,
          className: "absolute right-4 top-1/2 -translate-y-1/2 text-white/80 hover:text-white p-2 transition-colors",
          "aria-label": "Next image",
          children: /* @__PURE__ */ jsx(
            "svg",
            {
              className: "w-8 h-8",
              fill: "none",
              stroke: "currentColor",
              viewBox: "0 0 24 24",
              children: /* @__PURE__ */ jsx(
                "path",
                {
                  strokeLinecap: "round",
                  strokeLinejoin: "round",
                  strokeWidth: 2,
                  d: "M9 5l7 7-7 7"
                }
              )
            }
          )
        }
      ),
      /* @__PURE__ */ jsxs("div", { className: "relative max-w-[90vw] max-h-[90vh] flex flex-col items-center", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: (_e = images[currentImageIndex]) == null ? void 0 : _e.src,
            alt: (_f = images[currentImageIndex]) == null ? void 0 : _f.alt,
            className: "max-w-full max-h-[70vh] object-contain"
          }
        ),
        (((_g = images[currentImageIndex]) == null ? void 0 : _g.title) || ((_h = images[currentImageIndex]) == null ? void 0 : _h.description)) && /* @__PURE__ */ jsxs("div", { className: "mt-4 text-center text-white", children: [
          ((_i = images[currentImageIndex]) == null ? void 0 : _i.title) && /* @__PURE__ */ jsx("h3", { className: "text-xl font-medium mb-2", children: images[currentImageIndex].title }),
          ((_j = images[currentImageIndex]) == null ? void 0 : _j.description) && /* @__PURE__ */ jsx("p", { className: "text-white/80", children: images[currentImageIndex].description })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "mt-4 text-white/60 text-sm", children: [
          currentImageIndex + 1,
          " / ",
          images.length
        ] }),
        /* @__PURE__ */ jsx("div", { className: "mt-6 flex gap-2 overflow-x-auto max-w-full px-4", children: images.map((image, index) => /* @__PURE__ */ jsx(
          "button",
          {
            onClick: () => setCurrentImageIndex(index),
            className: cn(
              "flex-shrink-0 transition-all duration-200",
              index === currentImageIndex ? "ring-2 ring-white opacity-100" : "opacity-60 hover:opacity-80"
            ),
            children: /* @__PURE__ */ jsx(
              "img",
              {
                src: image.src,
                alt: image.alt,
                className: "w-20 h-20 object-cover"
              }
            )
          },
          image.id
        )) })
      ] })
    ] })
  ] });
}
function Tag({ children, className }) {
  return /* @__PURE__ */ jsx(
    "span",
    {
      className: cn(
        "inline-flex items-center h-9 px-4 rounded-full border border-dark-solid bg-transparent text-button-medium text-black transition-colors hover:bg-dark-10",
        className
      ),
      children
    }
  );
}
function StudioHeroSection({ data }) {
  return /* @__PURE__ */ jsx(PhotoGallery, { images: data.images });
}
function StudioInfoSection({ data }) {
  return /* @__PURE__ */ jsxs("div", { className: "space-y-6", children: [
    /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-2", children: [
      /* @__PURE__ */ jsx(Tag, { children: "NOMADIC SATELLITE" }),
      /* @__PURE__ */ jsx(Tag, { children: data.studioType }),
      /* @__PURE__ */ jsxs(Tag, { children: [
        data.location.toUpperCase(),
        ", ",
        data.country
      ] })
    ] }),
    /* @__PURE__ */ jsxs("h1", { className: "text-stat-number font-bold text-gray-900 leading-tight", children: [
      "PROFESSIONAL ",
      data.studioType,
      " FOR CREATIVES"
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "flex items-start space-x-2 text-gray-600", children: [
      /* @__PURE__ */ jsx(LocationIcon, {}),
      /* @__PURE__ */ jsx("p", { className: "text-sm", children: data.address })
    ] }),
    /* @__PURE__ */ jsx("p", { className: "text-gray-700 leading-relaxed", children: data.description })
  ] });
}
function PersonCard({
  photo: photo2,
  name,
  surname,
  role,
  location,
  href
}) {
  return /* @__PURE__ */ jsxs(
    "a",
    {
      href,
      className: "group block overflow-hidden transition-all duration-300 hover:scale-[1.02]",
      children: [
        /* @__PURE__ */ jsx("div", { className: "aspect-[3/4] overflow-hidden bg-neutral-100 mb-2 lg:mb-4", children: /* @__PURE__ */ jsx(
          "img",
          {
            src: photo2,
            alt: `${name} ${surname} - ${role}`,
            className: "h-full w-full object-cover transition-transform duration-500 group-hover:scale-105 grayscale group-hover:grayscale-0",
            loading: "lazy"
          }
        ) }),
        /* @__PURE__ */ jsxs("div", { className: "space-y-0.5 lg:space-y-1", children: [
          /* @__PURE__ */ jsxs("h3", { className: "text-body-3 lg:text-heading-5 tracking-[-0.01em] !text-black uppercase leading-tight", children: [
            name,
            " ",
            surname
          ] }),
          /* @__PURE__ */ jsx("p", { className: "text-button-small lg:text-body-3 !text-black/60 capitalize", children: role })
        ] })
      ]
    }
  );
}
function PeopleToolbar({
  filters,
  filterOptions: filterOptions2,
  onChange
}) {
  const handleLocationChange = (location) => {
    onChange({
      ...filters,
      location: filters.location === location ? "" : location
    });
  };
  const handleRoleChange = (role) => {
    onChange({
      ...filters,
      role: filters.role === role ? "" : role
    });
  };
  return /* @__PURE__ */ jsxs("div", { className: "mb-8 lg:mb-12 flex flex-col sm:flex-row gap-4", children: [
    /* @__PURE__ */ jsxs("div", { className: "relative", children: [
      /* @__PURE__ */ jsxs(
        "select",
        {
          value: filters.location,
          onChange: (e) => handleLocationChange(e.target.value),
          className: "appearance-none bg-transparent border border-black/20 rounded-none px-4 py-3 pr-10 text-button-small uppercase tracking-wider text-black focus:outline-none focus:border-black/40 min-w-[200px]",
          children: [
            /* @__PURE__ */ jsx("option", { value: "", children: "ALL LOCATIONS" }),
            filterOptions2.locations.map((location) => /* @__PURE__ */ jsx("option", { value: location, children: location.toUpperCase() }, location))
          ]
        }
      ),
      /* @__PURE__ */ jsx(ChevronDown, { className: "absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-black/60 pointer-events-none" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative", children: [
      /* @__PURE__ */ jsxs(
        "select",
        {
          value: filters.role,
          onChange: (e) => handleRoleChange(e.target.value),
          className: "appearance-none bg-transparent border border-black/20 rounded-none px-4 py-3 pr-10 text-button-small uppercase tracking-wider text-black focus:outline-none focus:border-black/40 min-w-[200px]",
          children: [
            /* @__PURE__ */ jsx("option", { value: "", children: "ALL ROLES" }),
            filterOptions2.roles.map((role) => /* @__PURE__ */ jsx("option", { value: role, children: role.toUpperCase() }, role))
          ]
        }
      ),
      /* @__PURE__ */ jsx(ChevronDown, { className: "absolute right-3 top-1/2 -translate-y-1/2 h-4 w-4 text-black/60 pointer-events-none" })
    ] }),
    (filters.location || filters.role) && /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2 text-body-3 text-black/60", children: [
      /* @__PURE__ */ jsx("span", { children: "Active filters:" }),
      filters.location && /* @__PURE__ */ jsx("span", { className: "px-2 py-1 bg-black/10 text-black rounded text-button-small uppercase", children: filters.location }),
      filters.role && /* @__PURE__ */ jsx("span", { className: "px-2 py-1 bg-black/10 text-black rounded text-button-small uppercase", children: filters.role }),
      /* @__PURE__ */ jsx(
        "button",
        {
          onClick: () => onChange({ location: "", role: "" }),
          className: "text-button-small text-black/40 hover:text-black underline uppercase",
          children: "Clear all"
        }
      )
    ] })
  ] });
}
function CrewSection({ data }) {
  const [filters, setFilters] = useState({
    location: "",
    role: ""
  });
  const filteredPeople = data.people.filter((person) => {
    const locationMatch = !filters.location || person.location === filters.location;
    const roleMatch = !filters.role || person.role === filters.role;
    return locationMatch && roleMatch;
  });
  const handleFilterChange = (newFilters) => {
    setFilters(newFilters);
  };
  return /* @__PURE__ */ jsxs("div", { className: "mt-8 space-y-8", children: [
    /* @__PURE__ */ jsx("h2", { className: "text-heading-3 !text-neutral-900", children: data.title }),
    /* @__PURE__ */ jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsx("div", { className: "lg:block", children: /* @__PURE__ */ jsx(
      PeopleToolbar,
      {
        filters,
        filterOptions: data.filterOptions,
        onChange: handleFilterChange
      }
    ) }) }),
    /* @__PURE__ */ jsx("div", { className: "grid grid-cols-2 lg:grid-cols-4 gap-3 lg:gap-6", children: filteredPeople.map((person) => /* @__PURE__ */ jsx(PersonCard, { ...person }, person.id)) })
  ] });
}
function EquipmentSection$1({ data }) {
  return /* @__PURE__ */ jsxs("div", { className: "mt-12", children: [
    /* @__PURE__ */ jsx("h2", { className: "text-heading-5 lg:text-[68px] leading-[100%] font-medium tracking-[-0.03em] uppercase text-neutral-900 mb-6", children: data.title }),
    /* @__PURE__ */ jsx("div", { className: "border border-gray-300", children: data.items.map((item, index) => /* @__PURE__ */ jsxs(
      "a",
      {
        href: item.href,
        className: `flex hover:bg-gray-50 transition-colors ${index < data.items.length - 1 ? "border-b border-gray-300" : ""}`,
        children: [
          /* @__PURE__ */ jsx("div", { className: "w-16 h-16 sm:w-20 sm:h-20 lg:w-24 lg:h-24 border-r border-gray-300 p-2", children: /* @__PURE__ */ jsx(
            "img",
            {
              src: item.image,
              alt: item.name,
              className: "w-full h-full object-cover"
            }
          ) }),
          /* @__PURE__ */ jsx("div", { className: "flex-1 p-3 sm:p-4 flex items-center", children: /* @__PURE__ */ jsx("span", { className: "text-sm sm:text-base text-gray-900 font-medium", children: item.name }) })
        ]
      },
      item.id
    )) })
  ] });
}
const locationAddresses = {
  bali: "Jl. Raya Ubud No.88, Ubud, Kecamatan Ubud, Kabupaten Gianyar, Bali 80571, Indonesia",
  bangkok: "123 Sukhumvit Rd, Khlong Toei, Bangkok 10110, Thailand",
  mexico: "Av. Revolución 123, Roma Norte, Ciudad de México, Mexico",
  "costa-rica": "Avenida Central 456, San José, Costa Rica",
  "new-york": "123 Broadway, New York, NY 10001, USA",
  mumbai: "456 Linking Road, Bandra West, Mumbai 400050, India"
};
const peopleData = [
  {
    id: "1",
    photo: "/images/home/people/person-1.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "New York",
    href: "/people/person-1"
  },
  {
    id: "2",
    photo: "/images/home/people/person-2.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bangkok",
    href: "/people/person-2"
  },
  {
    id: "3",
    photo: "/images/home/people/person-3.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Costa Rica",
    href: "/people/person-3"
  },
  {
    id: "4",
    photo: "/images/home/people/person-4.png",
    name: "NAME",
    surname: "SURNAME",
    role: "Profession",
    location: "Bali",
    href: "/people/person-4"
  }
];
const filterOptions = {
  locations: ["New York", "Bangkok", "Costa Rica", "Bali"],
  roles: ["Profession", "Creative Director", "Sound Engineer"]
};
const studioImages = [
  {
    id: "1",
    src: "/images/studios/studio-1.png",
    alt: "Studio main view",
    title: "Main Studio Space",
    description: "Professional setup with state-of-the-art equipment"
  },
  {
    id: "2",
    src: "/images/studios/music-1.png",
    alt: "Studio equipment view",
    title: "Recording Setup",
    description: "High-quality audio and video recording capabilities"
  },
  {
    id: "3",
    src: "/images/studios/photo-1.png",
    alt: "Studio lighting setup",
    title: "Lighting & Backdrop",
    description: "Professional lighting with customizable backdrops"
  },
  {
    id: "4",
    src: "/images/studios/studio-2.png",
    alt: "Studio workspace",
    title: "Creative Workspace",
    description: "Comfortable workspace for editing and post-production"
  },
  {
    id: "5",
    src: "/images/studios/music-2.png",
    alt: "Studio control room",
    title: "Control Room",
    description: "Advanced control room with mixing capabilities"
  },
  {
    id: "6",
    src: "/images/studios/studio-3.png",
    alt: "Studio lounge area",
    title: "Lounge Area",
    description: "Relaxation space for breaks and client meetings"
  }
];
const equipmentItems = [
  {
    id: "dj-equipment",
    name: "DJ EQUIPMENT",
    image: "/images/equipment/dj-equipment.png",
    href: "/equipment/dj-equipment"
  },
  {
    id: "cameras",
    name: "CAMERAS",
    image: "/images/equipment/cameras.png",
    href: "/equipment/cameras"
  },
  {
    id: "light",
    name: "LIGHT",
    image: "/images/equipment/light.png",
    href: "/equipment/light"
  }
];
const countryMap = {
  bali: "INDONESIA",
  bangkok: "THAILAND",
  mexico: "MEXICO",
  "costa-rica": "COSTA RICA",
  "new-york": "USA",
  mumbai: "INDIA"
};
function getStudioPageData(studioId, location, studioType) {
  const country = countryMap[location] || "UNKNOWN";
  const address = locationAddresses[location] || "Location address";
  const studioImagesWithAlt = studioImages.map((image) => ({
    ...image,
    alt: `${studioType} ${image.alt.split(" ").slice(1).join(" ")}`
  }));
  const description = `${location.charAt(0).toUpperCase() + location.slice(1)} is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.`;
  return {
    meta: {
      title: `Book ${studioType} Studio | Nomadics`,
      description: `Book your ${studioType.toLowerCase()} studio session at Nomadics`
    },
    hero: {
      studioType,
      location,
      country,
      address,
      description,
      images: studioImagesWithAlt
    },
    crew: {
      title: "OUR CREW",
      people: peopleData,
      filterOptions
    },
    equipment: {
      title: "EQUIPMENT",
      items: equipmentItems
    },
    planVisit: {
      address: "123 Linking Road, Bandra West, Mumbai 400050, India",
      hours: "7:00 AM - 11:00 PM",
      phone: "+91 22 1234 5678",
      email: "mumbai@nomadics.com",
      additionalInfo: "Near Bandra railway station..."
    }
  };
}
const meta$j = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadics" },
      { name: "description", content: "The requested studio was not found." }
    ];
  }
  return [
    { title: data.studioPageData.meta.title },
    { name: "description", content: data.studioPageData.meta.description }
  ];
};
async function loader$3({ params }) {
  const { studioId } = params;
  if (!studioId) {
    throw new Response("Studio not found", { status: 404 });
  }
  const [location, ...studioTypeParts] = studioId.split("-");
  const studioType = studioTypeParts.join(" ").toUpperCase();
  const studioPageData = getStudioPageData(studioId, location, studioType);
  return Response.json({
    studioId,
    location,
    studioType,
    studioPageData
  });
}
function BookStudio() {
  const { studioPageData } = useLoaderData();
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-white", children: [
    /* @__PURE__ */ jsxs("div", { className: "mx-auto max-w-7xl px-4 lg:px-14 py-4 lg:py-10", children: [
      /* @__PURE__ */ jsx(StudioHeroSection, { data: studioPageData.hero }),
      /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-3 gap-2 mt-8", children: [
        /* @__PURE__ */ jsxs("div", { className: "col-span-2 space-y-6 pr-8", children: [
          /* @__PURE__ */ jsx(StudioInfoSection, { data: studioPageData.hero }),
          /* @__PURE__ */ jsx(CrewSection, { data: studioPageData.crew }),
          /* @__PURE__ */ jsx(EquipmentSection$1, { data: studioPageData.equipment })
        ] }),
        /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(BookingForm, {}) })
      ] })
    ] }),
    /* @__PURE__ */ jsx(PlanYourVisitSection, { data: studioPageData.planVisit })
  ] });
}
const route4 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: BookStudio,
  loader: loader$3,
  meta: meta$j
}, Symbol.toStringTag, { value: "Module" }));
function HeroSection$1({
  backgroundImage,
  title,
  subtitle,
  children
}) {
  return /* @__PURE__ */ jsxs("section", { className: "relative w-full h-[calc(100vh-4rem)] lg:h-[calc(100vh-5rem)] flex flex-col", children: [
    /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-[#030303]" }),
    /* @__PURE__ */ jsx(
      "img",
      {
        src: backgroundImage,
        alt: "",
        className: "absolute inset-0 h-full w-full object-cover"
      }
    ),
    /* @__PURE__ */ jsx("div", { className: "absolute left-1/2 top-1/2 -translate-x-1/2 -translate-y-1/2 w-full flex flex-col items-center justify-center gap-4 px-4 lg:px-0", children: /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
      /* @__PURE__ */ jsx(
        "h1",
        {
          className: "text-heading-2-bold lg:text-heading-1 uppercase text-white mb-4",
          dangerouslySetInnerHTML: { __html: title }
        }
      ),
      subtitle && /* @__PURE__ */ jsx(Fragment, { children: typeof subtitle === "string" ? /* @__PURE__ */ jsx("p", { className: "text-center text-body-2 lg:text-body-1 text-white/80 max-w-[600px] mx-auto", children: subtitle }) : /* @__PURE__ */ jsx(Fragment, { children: subtitle.mobile && /* @__PURE__ */ jsx("p", { className: "lg:hidden text-center text-body-2 text-white/80 max-w-[600px] mx-auto", children: subtitle.mobile }) }) })
    ] }) }),
    children && /* @__PURE__ */ jsx("div", { className: "relative flex-1 flex flex-col items-center justify-end px-4 lg:px-14 pb-8 lg:pb-12", children: /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-center w-full max-w-[1328px]", children: [
      children,
      subtitle && typeof subtitle !== "string" && "desktop" in subtitle && /* @__PURE__ */ jsx("p", { className: "hidden lg:block text-right text-body-1 text-white/80 max-w-[600px] ml-auto", children: subtitle.desktop })
    ] }) })
  ] });
}
function HeroActions({ actions }) {
  return /* @__PURE__ */ jsx("div", { className: "flex flex-col sm:flex-row items-center gap-4", children: actions.map((action, index) => {
    if (action.onClick) {
      return /* @__PURE__ */ jsxs(
        Button,
        {
          variant: action.variant,
          size: "lg",
          className: "w-full sm:w-auto",
          onClick: action.onClick,
          children: [
            action.label,
            action.icon && /* @__PURE__ */ jsx(ArrowRight, { className: "h-5 w-5" })
          ]
        },
        index
      );
    }
    return /* @__PURE__ */ jsx(
      Button,
      {
        asChild: true,
        variant: action.variant,
        size: "lg",
        className: "w-full sm:w-auto",
        children: /* @__PURE__ */ jsxs(Link, { to: action.href, children: [
          action.label,
          action.icon && /* @__PURE__ */ jsx(ArrowRight, { className: "h-5 w-5" })
        ] })
      },
      index
    );
  }) });
}
function WelcomeSection$2({ data }) {
  const handleScrollToStudios = () => {
    const studiosSection = document.getElementById("studios-section");
    if (studiosSection) {
      studiosSection.scrollIntoView({ behavior: "smooth" });
    }
  };
  const heroActions2 = [
    {
      label: "Book a studio",
      onClick: handleScrollToStudios,
      variant: "fill-white",
      icon: true
    }
  ];
  return /* @__PURE__ */ jsx(
    HeroSection$1,
    {
      backgroundImage: data.backgroundImage,
      title: data.title,
      subtitle: data.subtitle,
      children: /* @__PURE__ */ jsx(HeroActions, { actions: heroActions2 })
    }
  );
}
function LocationInfo({ data }) {
  return /* @__PURE__ */ jsx("section", { className: "w-full bg-white py-12 lg:py-20", children: /* @__PURE__ */ jsxs("div", { className: "max-w-7xl mx-auto px-4 lg:px-14", children: [
    /* @__PURE__ */ jsx(SectionHeader, { eyebrow: "ABOUT", title: data.title }),
    /* @__PURE__ */ jsx("div", { className: "my-12 lg:my-16", children: /* @__PURE__ */ jsx("p", { className: "text-base lg:text-lg text-neutral-900 leading-relaxed max-w-5xl", children: data.description }) }),
    /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 lg:grid-cols-3 gap-6 lg:gap-8", children: data.stats.map((stat, index) => /* @__PURE__ */ jsxs(
      "div",
      {
        className: "border border-body-80 rounded-[999px] px-6 lg:px-8 py-6 lg:py-12 text-center flex justify-center items-center gap-[12px]",
        children: [
          /* @__PURE__ */ jsx("div", { className: "text-stat-number text-neutral-900", children: stat.number }),
          /* @__PURE__ */ jsx("div", { className: "text-stat-label text-neutral-600 whitespace-pre-line", children: stat.label })
        ]
      },
      index
    )) })
  ] }) });
}
function ImageGrid({
  items,
  className,
  variant = "triple",
  showBorders = true
}) {
  return /* @__PURE__ */ jsx(
    "div",
    {
      className: cn(
        showBorders && "border-y border-[rgba(18,18,18,0.2)]",
        className
      ),
      children: /* @__PURE__ */ jsx("div", { className: "flex overflow-x-auto gap-0 scrollbar-hide snap-x snap-mandatory", children: items.map((item, index) => {
        const isLastItem = index === items.length - 1;
        const isFirstItem = index === 0;
        return /* @__PURE__ */ jsxs(
          "div",
          {
            onClick: item.onClick,
            className: cn(
              "group flex-shrink-0 flex flex-col items-center pt-4 sm:pt-8 lg:pt-12 snap-center",
              item.onClick && "cursor-pointer",
              showBorders && !isLastItem && "border-r border-[rgba(18,18,18,0.2)]",
              isFirstItem && "ml-4 sm:ml-6",
              isLastItem && "mr-4 sm:mr-6"
            ),
            children: [
              /* @__PURE__ */ jsx("div", { className: "w-[240px] sm:w-[320px] lg:w-[380px] h-[240px] sm:h-[360px] lg:h-[440px] overflow-hidden px-4 sm:px-6 lg:px-8", children: /* @__PURE__ */ jsx(
                "img",
                {
                  src: item.image,
                  alt: item.title,
                  className: "w-full h-full object-cover group-hover:scale-101 transition-transform duration-500"
                }
              ) }),
              /* @__PURE__ */ jsx("div", { className: "p-3 sm:p-4 lg:p-6", children: /* @__PURE__ */ jsx("h3", { className: "text-[12px] sm:text-[14px] lg:text-[18px] font-medium uppercase tracking-[0.02em] text-[#121212] font-clash-grotesk text-center", children: item.title }) })
            ]
          },
          item.id
        );
      }) })
    }
  );
}
function StudiosSection$2({ data }) {
  const navigate = useNavigate();
  console.log("StudiosSection data:", data);
  const items = data.studios.map((studio) => ({
    id: studio.id,
    image: studio.image,
    title: studio.title,
    onClick: () => navigate(`/book-studio/${studio.id}`)
  }));
  return /* @__PURE__ */ jsx("section", { id: "studios-section", className: "relative w-full bg-white", children: /* @__PURE__ */ jsx("div", { className: "mx-auto w-full py-14 lg:py-24", children: /* @__PURE__ */ jsx(ImageGrid, { items }) }) });
}
function EventCard({ image, date, time, title, href }) {
  return /* @__PURE__ */ jsx("a", { href, className: "group block transition-all duration-300", children: /* @__PURE__ */ jsxs("div", { className: "flex flex-col", children: [
    /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden rounded-lg bg-neutral-100 mb-4", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: image,
        alt: title,
        className: "h-full w-full object-cover transition-transform duration-500 group-hover:scale-105",
        loading: "lazy"
      }
    ) }),
    /* @__PURE__ */ jsxs("div", { className: "flex items-center h-[80px] ", children: [
      /* @__PURE__ */ jsxs("div", { className: "flex flex-col items-center justify-center px-6 py-4 border border-black/10 bg-gray-50/50", children: [
        /* @__PURE__ */ jsx("div", { className: "text-heading-5 leading-none !text-black", children: date.day }),
        /* @__PURE__ */ jsx("div", { className: "text-body-3 uppercase !text-black/60 mt-1", children: date.month })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "flex-1 px-6 py-4", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-heading-5 !text-black uppercase tracking-wider", children: title }),
        /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-black/50 mt-1", children: time })
      ] })
    ] })
  ] }) });
}
function EventsGrid({ data }) {
  return /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 lg:grid-cols-3 gap-6", children: data.map((event) => /* @__PURE__ */ jsx(EventCard, { ...event }, event.id)) });
}
function EventsSection$1({ title, data }) {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-24", children: [
    /* @__PURE__ */ jsx(SectionHeader, { title, className: "mb-4" }),
    /* @__PURE__ */ jsx(EventsGrid, { data })
  ] }) });
}
function MembershipSection({ theme = "white" }) {
  const isWhiteTheme = theme === "white";
  return /* @__PURE__ */ jsxs(
    "section",
    {
      className: cn(
        "relative w-full overflow-hidden",
        isWhiteTheme ? "bg-white" : "bg-black"
      ),
      children: [
        /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
          /* @__PURE__ */ jsx(
            "div",
            {
              className: cn(
                "absolute top-[-100px] right-[-100px] w-[400px] h-[400px] border rounded-full",
                isWhiteTheme ? "border-black/10" : "border-white/10"
              )
            }
          ),
          /* @__PURE__ */ jsx(
            "div",
            {
              className: cn(
                "absolute bottom-[-150px] left-[-150px] w-[500px] h-[500px] border rounded-full",
                isWhiteTheme ? "border-black/10" : "border-white/10"
              )
            }
          ),
          /* @__PURE__ */ jsx(
            "div",
            {
              className: cn(
                "absolute top-[200px] left-[50%] transform -translate-x-1/2 w-[300px] h-[300px] border rounded-full opacity-50",
                isWhiteTheme ? "border-black/10" : "border-white/10"
              )
            }
          )
        ] }),
        /* @__PURE__ */ jsx("div", { className: "relative mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-[120px]", children: /* @__PURE__ */ jsxs("div", { className: "flex flex-col items-center text-center gap-6 lg:gap-10", children: [
          /* @__PURE__ */ jsxs("div", { className: "max-w-[950px]", children: [
            /* @__PURE__ */ jsx(
              "h2",
              {
                className: cn(
                  "text-heading-2-bold uppercase mb-6",
                  isWhiteTheme ? "!text-black" : "!text-white"
                ),
                children: "CONNECT, COLLABORATE AND GROW"
              }
            ),
            /* @__PURE__ */ jsx(
              "p",
              {
                className: cn(
                  "text-body-2 lg:text-body-1 max-w-[600px] mx-auto",
                  isWhiteTheme ? "!text-black/80" : "!text-white/80"
                ),
                children: "Be part of something bigger. Nomadic Studios is more than just a place to work - it's a vibrant network of creatives who inspire and support each other."
              }
            )
          ] }),
          /* @__PURE__ */ jsx("div", { className: "flex justify-center", children: /* @__PURE__ */ jsx(Button, { asChild: true, variant: isWhiteTheme ? "fill-black" : "fill-white", size: "lg", children: /* @__PURE__ */ jsxs(Link, { to: "/membership/apply", children: [
            "APPLY FOR MEMBERSHIP",
            /* @__PURE__ */ jsx(ArrowRight, { className: "h-5 w-5" })
          ] }) }) })
        ] }) })
      ]
    }
  );
}
function PeopleSection({
  eyebrow,
  title,
  buttonText,
  peopleData: peopleData2 = [],
  filterOptions: filterOptions2 = { locations: [], roles: [] }
} = {}) {
  const [filters, setFilters] = useState({
    location: "",
    role: ""
  });
  const filteredPeople = peopleData2.filter((person) => {
    const locationMatch = !filters.location || person.location === filters.location;
    const roleMatch = !filters.role || person.role === filters.role;
    return locationMatch && roleMatch;
  });
  const handleFilterChange = (newFilters) => {
    setFilters(newFilters);
  };
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-8 lg:py-24", children: [
    /* @__PURE__ */ jsxs("div", { className: "mb-6 lg:mb-16", children: [
      /* @__PURE__ */ jsxs("div", { className: "lg:hidden", children: [
        title && /* @__PURE__ */ jsx(
          SectionHeader,
          {
            layout: "vertical",
            eyebrow,
            title,
            className: "flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black [&_h2]:text-[24px] [&_h2]:tracking-[-0.02em] mb-4"
          }
        ),
        buttonText && /* @__PURE__ */ jsx(
          Button,
          {
            variant: "outline-black",
            size: "md",
            className: "border-black/30 hover:border-black/50 uppercase tracking-wider text-button-small !text-black",
            arrowRight: true,
            children: buttonText
          }
        )
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "hidden lg:flex lg:items-end lg:justify-between gap-4", children: [
        title && /* @__PURE__ */ jsx(
          SectionHeader,
          {
            layout: "vertical",
            eyebrow,
            title,
            className: "flex-1 max-w-[991px] [&_span]:text-black/60 [&_h2]:text-black"
          }
        ),
        buttonText && /* @__PURE__ */ jsx("div", { className: "flex items-center", children: /* @__PURE__ */ jsx(
          Button,
          {
            variant: "outline-black",
            size: "lg",
            className: "border-black/30 hover:border-black/50 uppercase tracking-wider font-medium !text-black",
            arrowRight: true,
            children: buttonText
          }
        ) })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "hidden lg:block", children: /* @__PURE__ */ jsx(
      PeopleToolbar,
      {
        filters,
        filterOptions: filterOptions2,
        onChange: handleFilterChange
      }
    ) }),
    /* @__PURE__ */ jsx("div", { className: "grid grid-cols-2 lg:grid-cols-4 gap-3 lg:gap-6", children: filteredPeople.map((person) => /* @__PURE__ */ jsx(PersonCard, { ...person }, person.id)) })
  ] }) });
}
const bali$1 = { "meta": { "title": "Bali Location | Nomadics", "description": "Nomadics Bali - Creative hub with video and music studios in the heart of Indonesia" }, "welcome": { "title": "BALI, CREATIVE HUB", "subtitle": "Welcome to Bali, your creative hub. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "BALI IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "3", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "12", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "5", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN BALI", "studios": [{ "id": "bali-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "bali-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "bali-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN BALI" }, "planVisit": { "title": "" }, "people": { "title": "MEET THE CREATIVES OF BALI" } };
const bangkok$1 = { "meta": { "title": "Bangkok Location | Nomadics", "description": "Nomadics Bangkok - Innovation center with video and music studios in Thailand's creative capital" }, "welcome": { "title": "BANGKOK, INNOVATION CENTER", "subtitle": "Bangkok is a city of innovation, culture, and creativity. Our studios provide the perfect environment for creators to thrive in Southeast Asia's most dynamic city.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "BANGKOK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "4", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "18", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "3", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN BANGKOK", "studios": [{ "id": "bangkok-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "bangkok-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "bangkok-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "bangkok-livestream-studio", "image": "/images/studios/studio-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN BANGKOK" }, "planVisit": { "address": "123 Sukhumvit Rd, Khlong Toei, Bangkok 10110, Thailand", "hours": "9:00 AM - 9:00 PM", "phone": "+66 2 123 4567", "email": "bangkok@nomadics.com", "additionalInfo": "Located near BTS Skytrain stations..." }, "people": { "title": "MEET THE CREATIVES OF BANGKOK" } };
const mexico$1 = { "meta": { "title": "Mexico Location | Nomadics", "description": "Nomadics Mexico - Artistic heart with photography and music studios in Mexico City" }, "welcome": { "title": "MEXICO, ARTISTIC HEART", "subtitle": "Mexico's rich cultural heritage meets modern creative technology. Our studios offer a unique blend of traditional artistry and cutting-edge production facilities.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / MUSIC STUDIO" }, "info": { "title": "MEXICO IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "5", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "25", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "10", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN MEXICO", "studios": [{ "id": "mexico-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "mexico-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "mexico-art-studio", "image": "/images/studios/studio-1.png", "title": "ART STUDIO" }, { "id": "mexico-dance-studio", "image": "/images/studios/photo-1.png", "title": "DANCE STUDIO" }, { "id": "mexico-recording-studio", "image": "/images/studios/music-1.png", "title": "RECORDING STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN MEXICO" }, "planVisit": { "address": "Av. Insurgentes Sur 123, Roma Norte, CDMX 06700, Mexico", "hours": "9:00 AM - 8:00 PM", "phone": "+52 55 1234 5678", "email": "mexico@nomadics.com", "additionalInfo": "Easy access via Metro Line 1..." }, "people": { "title": "MEET THE CREATIVES OF MEXICO" } };
const mumbai$1 = { "meta": { "title": "Mumbai Location | Nomadics", "description": "Nomadics Mumbai - Cultural fusion with music, art, and fashion studios in India's entertainment capital" }, "welcome": { "title": "MUMBAI, CULTURAL FUSION", "subtitle": "Mumbai's vibrant energy fuels creativity. Our studios here blend Bollywood's magic with international production standards.", "backgroundImage": "/images/home/locations/bali.png", "meta": "MUSIC STUDIO / ART STUDIO / FASHION STUDIO" }, "info": { "title": "MUMBAI IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "6", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "22", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "7", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN MUMBAI", "studios": [{ "id": "mumbai-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "mumbai-art-studio", "image": "/images/studios/studio-1.png", "title": "ART STUDIO" }, { "id": "mumbai-fashion-studio", "image": "/images/studios/photo-1.png", "title": "FASHION STUDIO" }, { "id": "mumbai-recording-studio", "image": "/images/studios/music-1.png", "title": "RECORDING STUDIO" }, { "id": "mumbai-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "mumbai-video-studio", "image": "/images/studios/photo-1.png", "title": "VIDEO STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN MUMBAI" }, "planVisit": { "address": "123 Linking Road, Bandra West, Mumbai 400050, India", "hours": "7:00 AM - 11:00 PM", "phone": "+91 22 1234 5678", "email": "mumbai@nomadics.com", "additionalInfo": "Near Bandra railway station..." }, "people": { "title": "MEET THE CREATIVES OF MUMBAI" } };
const video = { "meta": { "title": "Video Studios | Nomadics", "description": "Professional video studios for filming, livestreaming, and editing" }, "welcome": { "title": "STUDIOS TO CREATE STORIES", "subtitle": "Professional video studios for filming, livestreaming, and editing", "backgroundImage": "https://nomadicstudios.net:444/images/fe/studio-video-sec-one.png", "ctaLabel": "Book a video studio", "ctaHref": "/studios/video/book" }, "studios": [{ "title": "A Space Designed for Video Creators", "description": "Whether you're filming a commercial, live-streaming a podcast, or editing a short film, our studios provide the space and tools you need to deliver professional results. Unlike other generic rental spaces, we're built specifically for creators who demand both flexibility and quality.", "image": "https://nomadicstudios.net:444/images/fe/studio-video-sec-four-1.png", "imagePosition": "right" }, { "title": "Professional Lighting & Backdrops", "description": "Our video studios feature comprehensive lighting grids, green screens, and customizable backdrops. From key lights to fill lights, diffusers to reflectors, create the perfect lighting setup for any production style or mood.", "image": "https://nomadicstudios.net:444/images/fe/studio-video-sec-four-2.png", "imagePosition": "left" }, { "title": "Live Streaming & Post-Production", "description": "Equipped with high-speed internet, streaming encoders, and powerful editing workstations. Whether you're broadcasting live or perfecting your final cut, our studios provide the technical infrastructure for seamless production workflows.", "image": "https://nomadicstudios.net:444/images/fe/studio-video-sec-four-3.png", "imagePosition": "right" }], "book": { "title": "Book a Video Studio", "description": "Choose the perfect space for your video production needs" }, "gallery": { "title": "STUDIO GALLERY", "description": "Explore our diverse range of creative spaces designed for every type of production", "image1": "https://nomadicstudios.net:444/images/fe/studio-video-sec-two-1.png", "title1": "Video Production", "subtitle1": "Professional filming spaces", "image2": "https://nomadicstudios.net:444/images/fe/studio-video-sec-two-2.png", "title2": "Music Recording", "subtitle2": "Acoustically treated rooms", "image3": "https://nomadicstudios.net:444/images/fe/studio-video-sec-two-3.png", "title3": "Photo Studio", "image4": "https://nomadicstudios.net:444/images/fe/studio-video-sec-two-4.png", "title4": "Creative Space", "image5": "https://nomadicstudios.net:444/images/fe/studio-video-sec-two-5.png", "title5": "Production Suite", "image6": "https://nomadicstudios.net:444/images/fe/studio-video-sec-four-1.png", "title6": "Multi-Purpose Studios", "subtitle6": "Flexible spaces that adapt to your creative needs" } };
const music = { "meta": { "title": "Music Studios | Nomadics", "description": "Professional music studios with treated rooms, pro interfaces, and backline on request" }, "welcome": { "title": "SPACES TO MAKE YOUR MUSIC HEARD", "subtitle": "Acoustically treated rooms with pro interfaces and backline on request", "backgroundImage": "/images/studios/music-1.png", "ctaLabel": "Book a music studio", "ctaHref": "/studios/music/book" }, "studios": [{ "title": "Professional Recording Environment", "description": "Our music studios feature acoustically treated rooms designed for optimal sound quality. From vocal booths to live rooms, every space is engineered to capture the perfect take. Professional-grade soundproofing ensures your sessions remain private and distraction-free.", "image": "/images/studios/music-1.png", "imagePosition": "right" }, { "title": "Industry-Standard Equipment", "description": "Record with confidence using our curated selection of industry-standard interfaces, microphones, and monitoring systems. From vintage analog gear to cutting-edge digital processors, we provide the tools that professional producers and engineers trust.", "image": "/images/studios/music-2.png", "imagePosition": "left" }, { "title": "Backline and Instruments Available", "description": "Need instruments? We've got you covered. Our studios offer a selection of quality instruments and backline equipment available on request. From drum kits to keyboards, guitars to synthesizers, expand your sonic palette without the hassle of transport.", "image": "/images/studios/music-3.png", "imagePosition": "right" }], "book": { "title": "Book a Music Studio", "description": "Find the ideal acoustic environment for your recording session", "items": [{ "id": "recording-room", "image": "/images/studios/music-1.png", "title": "RECORDING ROOM" }, { "id": "mixing-suite", "image": "/images/studios/music-2.png", "title": "MIXING SUITE" }, { "id": "live-room", "image": "/images/studios/music-3.png", "title": "LIVE ROOM" }] } };
const photo = { "meta": { "title": "Photo Studios | Nomadics", "description": "Photography studios equipped for everything from product shoots to portrait sessions" }, "welcome": { "title": "CAPTURE YOUR VISION", "subtitle": "Photography studios equipped for everything from product shoots to portraits", "backgroundImage": "/images/studios/photo-1.png", "ctaLabel": "Book a photo studio", "ctaHref": "/studios/photo/book" }, "studios": [{ "title": "Versatile Shooting Spaces", "description": "Our photography studios adapt to your creative vision. From intimate portrait sessions to large-scale product shoots, each space offers the flexibility you need. Customizable backdrops, moveable walls, and modular sets let you create the perfect environment for any project.", "image": "/images/studios/photo-1.png", "imagePosition": "right" }, { "title": "Professional Lighting Systems", "description": "Light your subjects perfectly with our comprehensive lighting setups. Each studio includes professional strobe systems, continuous LED panels, and a full range of modifiers. From softboxes to beauty dishes, create any mood or style with our extensive lighting equipment.", "image": "/images/studios/photo-2.png", "imagePosition": "left" }, { "title": "Props and Backgrounds", "description": "Access our curated collection of backgrounds and props to enhance your shoots. Choose from seamless paper rolls in various colors, textured backdrops, and themed props. Whether you're shooting fashion, products, or portraits, we have the elements to complete your vision.", "image": "/images/studios/photo-1.png", "imagePosition": "right" }], "book": { "title": "Book a Photo Studio", "description": "Select the perfect backdrop for your photography project", "items": [{ "id": "daylight-studio", "image": "/images/studios/photo-1.png", "title": "DAYLIGHT STUDIO" }, { "id": "blackout-studio", "image": "/images/studios/photo-2.png", "title": "BLACKOUT STUDIO" }, { "id": "infinity-cove", "image": "/images/studios/photo-1.png", "title": "INFINITY COVE" }] } };
const auth$1 = { "login": "Login", "signUp": "Sign Up", "signOut": "Sign Out", "signIn": "Sign In", "signin": { "welcomeBack": "WELCOME BACK", "subtitle": "Continue your creative journey", "title": "Sign In", "mobileNumber": "Mobile Number", "phonePlaceholder": "Enter your mobile number", "keepLoggedIn": "Keep me logged in", "sendOTP": "Send OTP", "sendingOTP": "Sending OTP...", "loginWithEmail": "Login with email instead", "forgotPassword": "Forgot Password?", "noAccount": "Don't have an account?", "signUp": "Sign up", "errors": { "sendOTPFailed": "Failed to send OTP. Please try again.", "unexpected": "An unexpected error occurred. Please try again." } }, "signinemail": { "title": "SIGN IN", "subtitle": "Access your creative workspace", "emailLogin": "Email Login", "emailAddress": "Email Address", "emailPlaceholder": "Enter your email", "password": "Password", "passwordPlaceholder": "Enter your password", "keepLoggedIn": "Keep me logged in", "forgotPassword": "Forgot Password?", "signIn": "Sign In", "signingIn": "Signing in...", "loginWithPhone": "Login with phone instead", "noAccount": "Don't have an account?", "signUp": "Sign up", "errors": { "loginFailed": "Login failed. Please try again.", "unexpected": "An unexpected error occurred. Please try again." } }, "signup": { "joinCommunity": "JOIN THE COMMUNITY", "subtitle": "Choose your creative journey", "benefits": { "relocation": "14 days relocation/year", "studio": "14 days Studio time/year" }, "mostPopular": "Most Popular", "perPeriod": "per", "selected": "Selected", "choosePlan": "Choose plan", "addons": { "title": "Select Add-on Services", "subtitle": "Enhance your membership with optional premium services", "perMonth": "per month", "selected": "Selected", "addToPlan": "Add to plan", "testService": { "name": "Test Service", "description": "Additional testing and quality assurance services" }, "premiumSupport": { "name": "Premium Support", "description": "24/7 priority support and consultation services" } }, "subscribe": "Subscribe", "haveAccount": "Already have an account?", "signIn": "Sign in", "plans": { "period": "month", "guest": { "name": "Guest", "feature1": "Just one day/one by member", "feature2": "Zero guests" }, "starvingArtist": { "name": "Starving Artist", "feature1": "Book up 1 month in advance", "feature2": "One guests" }, "upAndComing": { "name": "Up and coming", "feature1": "Book up 1 month in advance", "feature2": "Two guests" }, "almostFamous": { "name": "Almost Famous", "feature1": "Book up 3 month in advance", "feature2": "Four guests" } } }, "otp": { "title": "CONFIRM YOUR PHONE", "subtitle": "Please input 6-digit code we just sent to", "resendTimer": "Didn't get an SMS? Resend in", "seconds": "seconds", "resendCode": "Resend code", "verifying": "Verifying...", "submit": "Submit", "wrongNumber": "Wrong phone number?", "backToLogin": "Login", "errors": { "incompleteCode": "Please enter the complete 6-digit code", "invalidCode": "Invalid verification code. Please try again.", "verificationFailed": "Verification failed. Please try again.", "resendFailed": "Failed to resend code. Please try again." } }, "additionalInfo": { "title": "We would like to know you more", "homeCity": "What is your home city?", "loadingLocations": "Loading locations...", "selectLocation": "Select a location", "creativeInterests": "What are you creating?", "profileLink": "Link to profile", "linkPlaceholder": "Link", "socialMedia": "Link to social media", "instagramLink": "Link to Instagram", "facebookLink": "Link to Facebook", "twitterLink": "Link to Twitter", "smLink": "Link to SM", "referredBy": "Please provide the name of the member who referred you", "namePlaceholder": "Name", "save": "Save", "saving": "Saving...", "success": "Account created successfully! Please login to continue.", "errors": { "homeCityRequired": "Home city is required", "saveFailed": "Failed to save information. Please try again.", "unexpected": "An unexpected error occurred. Please try again." } }, "resetPassword": { "title": "RESET PASSWORD", "subtitle": "Enter your email to receive OTP on your mobile", "cardTitle": "Reset Password", "emailAddress": "Email Address", "emailPlaceholder": "Please enter your email", "sendOTP": "Send OTP to Mobile", "sending": "Sending...", "backToLogin": "Back to Login", "errors": { "sendFailed": "Failed to send OTP. Please try again.", "unexpected": "An unexpected error occurred. Please try again." } }, "resetPasswordOtp": { "title": "RESET PASSWORD", "subtitle": "Enter the OTP sent to your mobile and set new password", "cardTitle": "Verify & Reset", "enterOtp": "Enter 6-digit OTP", "sentTo": "Sent to", "registeredNumber": "your registered mobile number", "resendOtp": "Resend OTP", "resendIn": "Resend OTP in", "newPassword": "New Password", "newPasswordPlaceholder": "Enter new password", "confirmPassword": "Confirm New Password", "confirmPasswordPlaceholder": "Confirm new password", "resetPasswordButton": "Reset Password", "resetting": "Resetting Password...", "backToReset": "Back to Reset Password", "success": "Password reset successfully! Please login with your new password.", "errors": { "incompleteOtp": "Please enter the complete 6-digit OTP", "passwordMismatch": "Passwords do not match", "passwordTooShort": "Password must be at least 8 characters long", "resetFailed": "Failed to reset password. Please try again.", "resendFailed": "Failed to resend OTP", "unexpected": "An unexpected error occurred. Please try again." } }, "uploadProfile": { "title": "ADD PROFILE PICTURE", "subtitle": "Let's make your profile shine", "profilePicture": "Profile Picture", "profilePreview": "Profile preview", "chooseImage": "Choose Image", "uploading": "Uploading...", "uploadAndContinue": "Upload & Continue", "skipForNow": "Skip for now", "acceptedFormats": "Accepted formats: JPG, PNG, GIF", "maxFileSize": "Maximum file size: 5MB", "invalidFileType": "Please select a valid image file", "fileSizeLimit": "Image size should be less than 5MB", "uploadFailed": "Failed to upload profile picture. Please try again.", "unexpectedError": "An unexpected error occurred. Please try again.", "processingError": "Failed to process image. Please try again." } };
const signupSecond$1 = { "mobileNumber": "Mobile Number", "mobileNumberPlaceholder": "Enter your mobile number", "termsAgreement": "By clicking 'Complete Signup', you agree to our", "termsOfService": "Terms of Service", "privacyPolicy": "Privacy Policy", "smsVerification": "We'll send SMS verification to your mobile number.", "creatingAccount": "Creating Account...", "completeSignup": "Complete Signup", "alreadyHaveAccount": "Already have an account?" };
const uploadProfile$1 = { "title": "ADD PROFILE PICTURE", "subtitle": "Let's make your profile shine", "profilePicture": "Profile Picture", "profilePreview": "Profile preview", "chooseImage": "Choose Image", "uploading": "Uploading...", "uploadAndContinue": "Upload & Continue", "skipForNow": "Skip for now", "acceptedFormats": "Accepted formats: JPG, PNG, GIF", "maxFileSize": "Maximum file size: 5MB", "invalidFileType": "Please select a valid image file", "fileSizeLimit": "Image size should be less than 5MB", "uploadFailed": "Failed to upload profile picture. Please try again.", "unexpectedError": "An unexpected error occurred. Please try again.", "processingError": "Failed to process image. Please try again." };
const common$1 = { "and": "and", "signIn": "Sign In", "logo": "Logo", "companyName": "Nomadics" };
const nav$1 = { "studios": "Studios", "videoStudios": "Video Studios", "musicStudios": "Music Studios", "photoStudios": "Photo Studios", "locations": "Locations", "bali": "Bali", "bangkok": "Bangkok", "mexico": "Mexico", "costaRica": "Costa Rica", "newYork": "New York", "mumbai": "Mumbai", "people": "People", "equipment": "Equipment", "events": "Events", "services": "Services", "allServices": "All Services", "production": "Production", "postProduction": "Post-Production", "news": "News", "home": "Home" };
const header$1 = { "notifications": "Notifications", "chat": "Chat", "wallet": "Wallet", "profile": "Profile", "accountSettings": "Account Settings", "openMenu": "Open menu" };
const equipment$1 = { "hero": { "title": "THE TOOLS YOU\nNEED TO CREATE", "description": "At Nomadic Studios, we know the right tools can make all the difference. That's why we provide state-of-the-art, standardized equipment across all our locations, ensuring you're familiar with the gear no matter where your creativity takes you. From video production to music recording, fashion design to prototyping, our tools are designed to support creators at every level." }, "consistentGear": { "title": "CONSISTENT GEAR, WHEREVER YOU ARE", "description": "Familiarity breeds creativity. Our studios feature standardized equipment, so you'll know exactly how to use the tools you need, no matter which location you visit. Focus on your craft—not on learning new setups.", "imageAlt": "Professional Studio Setup" }, "interconnected": { "title": "INTERCONNECTED FOR SEAMLESS COLLABORATION", "description": "Our private, high-speed network connects all Nomadic Studios locations, allowing you to securely transfer your digital work between studios or collaborate with team members in other cities. Your creativity doesn't stop at city limits, and neither do we.", "imageAlt": "High-Speed Network" }, "stockedEssentials": { "title": "STOCKED WITH ESSENTIALS", "description": "Get started quickly with our in-studio stock of samples, materials, and consumables. Whether you need fabric for fashion, paints for art, or cables for music production, we've got you covered with the essentials.", "imageAlt": "Stocked Essentials" }, "expertGuidance": { "title": "EXPERT GUIDANCE ON-SITE", "description": "Every studio is managed by professionals who know the gear inside and out. Whether you're learning a new tool or troubleshooting an issue, our team is here to ensure your workflow stays smooth and productive.", "imageAlt": "Expert Guidance Dashboard" }, "specializedRentals": { "title": "SPECIALIZED RENTALS FOR UNIQUE NEEDS", "description": "Need uncommon equipment or planning a project offsite? Take advantage of our rental services to access specialized tools, event production gear, or anything else you might need to bring your vision to life.", "imageAlt": "Specialized Equipment Rentals" }, "connectCollaborate": { "title": "CONNECT, COLLABORATE\nAND GROW", "description": "Be part of something bigger. Nomadic Studios is more than just a place to work - it's a vibrant network of creatives who inspire and support each other." }, "bookStudio": "BOOK A STUDIO", "applyForMembership": "APPLY FOR MEMBERSHIP", "copyright": "© 2025 Nomadic Studios. All rights reserved." };
const footer$1 = { "about": { "title": "About", "joinOurTeam": "Join Our Team", "investors": "Investors", "partners": "Partners" }, "socialMedia": { "title": "Social Media" }, "legal": { "termsConditions": "Terms & Conditions", "privacyPolicy": "Privacy Policy" }, "backToTop": "Back to Top", "copyright": "©2025 Nomadic Studios" };
const en = {
  bali: bali$1,
  bangkok: bangkok$1,
  mexico: mexico$1,
  "costa-rica": { "meta": { "title": "Costa Rica Location | Nomadics", "description": "Nomadics Costa Rica - Natural inspiration with video and music studios surrounded by beauty" }, "welcome": { "title": "COSTA RICA, NATURAL INSPIRATION", "subtitle": "Surrounded by natural beauty, our Costa Rica location provides a serene environment for focused creative work and inspired productions.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "COSTA RICA IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "3", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "15", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "8", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN COSTA RICA", "studios": [{ "id": "costa-rica-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "costa-rica-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "costa-rica-creative-suite", "image": "/images/studios/photo-1.png", "title": "CREATIVE SUITE" }] }, "events": { "sectionTitle": "EVENTS IN COSTA RICA" }, "planVisit": { "address": "Avenida Central, San José, Costa Rica", "hours": "8:00 AM - 7:00 PM", "phone": "+506 1234 5678", "email": "costarica@nomadics.com", "additionalInfo": "Downtown location near main attractions..." }, "people": { "title": "MEET THE CREATIVES OF COSTA RICA" } },
  "new-york": { "meta": { "title": "New York Location | Nomadics", "description": "Nomadics New York - Global epicenter with photography and fashion studios in Manhattan" }, "welcome": { "title": "NEW YORK, GLOBAL EPICENTER", "subtitle": "In the heart of the world's creative capital, our New York studios offer unparalleled resources for fashion, photography, and artistic innovation.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / FASHION STUDIO" }, "info": { "title": "NEW YORK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "8", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "35", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "2", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN NEW YORK", "studios": [{ "id": "new-york-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "new-york-fashion-studio", "image": "/images/studios/studio-1.png", "title": "FASHION STUDIO" }, { "id": "new-york-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "new-york-art-studio", "image": "/images/studios/photo-1.png", "title": "ART STUDIO" }, { "id": "new-york-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "new-york-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "new-york-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "new-york-livestream-studio", "image": "/images/studios/music-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN NEW YORK" }, "planVisit": { "address": "123 Broadway, Manhattan, New York, NY 10013, USA", "hours": "24/7 Access for Members", "phone": "+1 212 123 4567", "email": "newyork@nomadics.com", "additionalInfo": "Accessible via multiple subway lines..." }, "people": { "title": "MEET THE CREATIVES OF NEW YORK" } },
  mumbai: mumbai$1,
  "costa-rica1": { "meta": { "title": "Costa Rica Location | Nomadics", "description": "Nomadics Costa Rica - Natural inspiration with video and music studios surrounded by beauty" }, "welcome": { "title": "COSTA RICA, NATURAL INSPIRATION", "subtitle": "Surrounded by natural beauty, our Costa Rica location provides a serene environment for focused creative work and inspired productions.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "COSTA RICA IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "3", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "15", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "8", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN COSTA RICA", "studios": [{ "id": "costa-rica-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "costa-rica-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "costa-rica-creative-suite", "image": "/images/studios/photo-1.png", "title": "CREATIVE SUITE" }] }, "events": { "sectionTitle": "EVENTS IN COSTA RICA" }, "planVisit": { "address": "Avenida Central, San José, Costa Rica", "hours": "8:00 AM - 7:00 PM", "phone": "+506 1234 5678", "email": "costarica@nomadics.com", "additionalInfo": "Downtown location near main attractions..." }, "people": { "title": "MEET THE CREATIVES OF COSTA RICA" } },
  "new-york1": { "meta": { "title": "New York Location | Nomadics", "description": "Nomadics New York - Global epicenter with photography and fashion studios in Manhattan" }, "welcome": { "title": "NEW YORK, GLOBAL EPICENTER", "subtitle": "In the heart of the world's creative capital, our New York studios offer unparalleled resources for fashion, photography, and artistic innovation.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / FASHION STUDIO" }, "info": { "title": "NEW YORK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "8", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "35", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "2", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN NEW YORK", "studios": [{ "id": "new-york-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "new-york-fashion-studio", "image": "/images/studios/studio-1.png", "title": "FASHION STUDIO" }, { "id": "new-york-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "new-york-art-studio", "image": "/images/studios/photo-1.png", "title": "ART STUDIO" }, { "id": "new-york-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "new-york-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "new-york-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "new-york-livestream-studio", "image": "/images/studios/music-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN NEW YORK" }, "planVisit": { "address": "123 Broadway, Manhattan, New York, NY 10013, USA", "hours": "24/7 Access for Members", "phone": "+1 212 123 4567", "email": "newyork@nomadics.com", "additionalInfo": "Accessible via multiple subway lines..." }, "people": { "title": "MEET THE CREATIVES OF NEW YORK" } },
  video,
  music,
  photo,
  auth: auth$1,
  signupSecond: signupSecond$1,
  uploadProfile: uploadProfile$1,
  common: common$1,
  nav: nav$1,
  header: header$1,
  equipment: equipment$1,
  footer: footer$1
};
const en$1 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  auth: auth$1,
  bali: bali$1,
  bangkok: bangkok$1,
  common: common$1,
  default: en,
  equipment: equipment$1,
  footer: footer$1,
  header: header$1,
  mexico: mexico$1,
  mumbai: mumbai$1,
  music,
  nav: nav$1,
  photo,
  signupSecond: signupSecond$1,
  uploadProfile: uploadProfile$1,
  video
}, Symbol.toStringTag, { value: "Module" }));
const bali = { "meta": { "title": "Ubicación Bali | Nomadics", "description": "Nomadics Bali - Centro creativo con estudios de video y música en el corazón de Indonesia" }, "welcome": { "title": "BALI, CENTRO CREATIVO", "subtitle": "Bienvenido a Bali, tu centro creativo. Nuestro espacio aquí está diseñado para empoderar a los creadores de diversas disciplinas, brindando las herramientas, recursos e inspiración necesarios para llevar tu trabajo al siguiente nivel.", "backgroundImage": "/images/home/locations/bali.png", "meta": "ESTUDIO DE VIDEO / ESTUDIO DE MÚSICA" }, "info": { "title": "BALI ES UNA CIUDAD DE INNOVACIÓN, CULTURA Y CREATIVIDAD", "description": "Nomadic Studios se enorgullece de estar en el corazón de ello. Nuestro espacio aquí está diseñado para empoderar a los creadores de diversas disciplinas, brindando las herramientas, recursos e inspiración necesarios para llevar tu trabajo al siguiente nivel. Ya seas un profesional local en busca de un estudio versátil o un creador viajero que busca un lugar para llamar su hogar creativo, aquí encontrarás todo lo que necesitas.", "stats": [{ "number": "3", "label": "ESPACIOS DE ESTUDIO\nDISPONIBLES" }, { "number": "12", "label": "PROFESIONALES\nCOLABORANDO\nMENSUALMENTE" }, { "number": "5", "label": "MINUTOS DEL\nCENTRO DE LA CIUDAD" }] }, "studios": { "sectionTitle": "ESTUDIOS PARA CREAR EN BALI", "studios": [{ "id": "bali-video-studio", "image": "/images/studios/studio-1.png", "title": "ESTUDIO DE VIDEO" }, { "id": "bali-music-studio", "image": "/images/studios/music-1.png", "title": "ESTUDIO DE MÚSICA" }, { "id": "bali-photography-studio", "image": "/images/studios/photo-1.png", "title": "ESTUDIO DE FOTOGRAFÍA" }] }, "events": { "sectionTitle": "EVENTOS EN BALI" }, "planVisit": { "address": "Jl. Raya Ubud No.88, Ubud, Kecamatan Ubud, Kabupaten Gianyar, Bali 80571, Indonesia", "hours": "6:00 AM - 11:00 PM", "phone": "+91 8786564378", "email": "bali@sativehub@gmail.com", "additionalInfo": "Puedes llegar en autobús desde..." }, "people": { "title": "CONOCE A LOS CREATIVOS DE BALI" } };
const bangkok = { "meta": { "title": "Bangkok Location | Nomadics", "description": "Nomadics Bangkok - Innovation center with video and music studios in Thailand's creative capital" }, "welcome": { "title": "BANGKOK, INNOVATION CENTER", "subtitle": "Bangkok is a city of innovation, culture, and creativity. Our studios provide the perfect environment for creators to thrive in Southeast Asia's most dynamic city.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "BANGKOK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "4", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "18", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "3", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN BANGKOK", "studios": [{ "id": "bangkok-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "bangkok-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "bangkok-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "bangkok-livestream-studio", "image": "/images/studios/studio-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN BANGKOK" }, "planVisit": { "address": "123 Sukhumvit Rd, Khlong Toei, Bangkok 10110, Thailand", "hours": "9:00 AM - 9:00 PM", "phone": "+66 2 123 4567", "email": "bangkok@nomadics.com", "additionalInfo": "Located near BTS Skytrain stations..." }, "people": { "title": "MEET THE CREATIVES OF BANGKOK" } };
const mexico = { "meta": { "title": "Mexico Location | Nomadics", "description": "Nomadics Mexico - Artistic heart with photography and music studios in Mexico City" }, "welcome": { "title": "MEXICO, ARTISTIC HEART", "subtitle": "Mexico's rich cultural heritage meets modern creative technology. Our studios offer a unique blend of traditional artistry and cutting-edge production facilities.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / MUSIC STUDIO" }, "info": { "title": "MEXICO IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "5", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "25", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "10", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN MEXICO", "studios": [{ "id": "mexico-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "mexico-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "mexico-art-studio", "image": "/images/studios/studio-1.png", "title": "ART STUDIO" }, { "id": "mexico-dance-studio", "image": "/images/studios/photo-1.png", "title": "DANCE STUDIO" }, { "id": "mexico-recording-studio", "image": "/images/studios/music-1.png", "title": "RECORDING STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN MEXICO" }, "planVisit": { "address": "Av. Insurgentes Sur 123, Roma Norte, CDMX 06700, Mexico", "hours": "9:00 AM - 8:00 PM", "phone": "+52 55 1234 5678", "email": "mexico@nomadics.com", "additionalInfo": "Easy access via Metro Line 1..." }, "people": { "title": "MEET THE CREATIVES OF MEXICO" } };
const mumbai = { "meta": { "title": "Mumbai Location | Nomadics", "description": "Nomadics Mumbai - Cultural fusion with music, art, and fashion studios in India's entertainment capital" }, "welcome": { "title": "MUMBAI, CULTURAL FUSION", "subtitle": "Mumbai's vibrant energy fuels creativity. Our studios here blend Bollywood's magic with international production standards.", "backgroundImage": "/images/home/locations/bali.png", "meta": "MUSIC STUDIO / ART STUDIO / FASHION STUDIO" }, "info": { "title": "MUMBAI IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "6", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "22", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "7", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN MUMBAI", "studios": [{ "id": "mumbai-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "mumbai-art-studio", "image": "/images/studios/studio-1.png", "title": "ART STUDIO" }, { "id": "mumbai-fashion-studio", "image": "/images/studios/photo-1.png", "title": "FASHION STUDIO" }, { "id": "mumbai-recording-studio", "image": "/images/studios/music-1.png", "title": "RECORDING STUDIO" }, { "id": "mumbai-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "mumbai-video-studio", "image": "/images/studios/photo-1.png", "title": "VIDEO STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN MUMBAI" }, "planVisit": { "address": "123 Linking Road, Bandra West, Mumbai 400050, India", "hours": "7:00 AM - 11:00 PM", "phone": "+91 22 1234 5678", "email": "mumbai@nomadics.com", "additionalInfo": "Near Bandra railway station..." }, "people": { "title": "MEET THE CREATIVES OF MUMBAI" } };
const auth = { "login": "Iniciar Sesión", "signUp": "Registrarse", "signOut": "Cerrar Sesión", "signIn": "Iniciar Sesión", "signin": { "welcomeBack": "BIENVENIDO DE VUELTA", "subtitle": "Continúa tu viaje creativo", "title": "Iniciar Sesión", "mobileNumber": "Número de Teléfono", "phonePlaceholder": "Ingresa tu número de teléfono", "keepLoggedIn": "Mantenerme conectado", "sendOTP": "Enviar OTP", "sendingOTP": "Enviando OTP...", "loginWithEmail": "Iniciar sesión con email en su lugar", "forgotPassword": "¿Olvidaste tu contraseña?", "noAccount": "¿No tienes una cuenta?", "signUp": "Registrarse", "errors": { "sendOTPFailed": "Error al enviar OTP. Inténtalo de nuevo.", "unexpected": "Ocurrió un error inesperado. Inténtalo de nuevo." } }, "signinemail": { "title": "INICIAR SESIÓN", "subtitle": "Accede a tu espacio creativo", "emailLogin": "Inicio de Sesión con Email", "emailAddress": "Dirección de Email", "emailPlaceholder": "Ingresa tu email", "password": "Contraseña", "passwordPlaceholder": "Ingresa tu contraseña", "keepLoggedIn": "Mantenerme conectado", "forgotPassword": "¿Olvidaste tu contraseña?", "signIn": "Iniciar Sesión", "signingIn": "Iniciando sesión...", "loginWithPhone": "Iniciar sesión con teléfono en su lugar", "noAccount": "¿No tienes una cuenta?", "signUp": "Registrarse", "errors": { "loginFailed": "Error al iniciar sesión. Inténtalo de nuevo.", "unexpected": "Ocurrió un error inesperado. Inténtalo de nuevo." } }, "signup": { "joinCommunity": "ÚNETE A LA COMUNIDAD", "subtitle": "Elige tu viaje creativo", "benefits": { "relocation": "14 días de reubicación/año", "studio": "14 días de tiempo de estudio/año" }, "mostPopular": "Más Popular", "perPeriod": "por", "selected": "Seleccionado", "choosePlan": "Elegir plan", "addons": { "title": "Seleccionar Servicios Adicionales", "subtitle": "Mejora tu membresía con servicios premium opcionales", "perMonth": "por mes", "selected": "Seleccionado", "addToPlan": "Agregar al plan", "testService": { "name": "Servicio de Prueba", "description": "Servicios adicionales de prueba y garantía de calidad" }, "premiumSupport": { "name": "Soporte Premium", "description": "Servicios de soporte prioritario 24/7 y consultoría" } }, "subscribe": "Suscribirse", "haveAccount": "¿Ya tienes una cuenta?", "signIn": "Iniciar sesión", "plans": { "period": "mes", "guest": { "name": "Invitado", "feature1": "Solo un día/uno por miembro", "feature2": "Cero invitados" }, "starvingArtist": { "name": "Artista Hambriento", "feature1": "Reservar hasta 1 mes por adelantado", "feature2": "Un invitado" }, "upAndComing": { "name": "Prometedor", "feature1": "Reservar hasta 1 mes por adelantado", "feature2": "Dos invitados" }, "almostFamous": { "name": "Casi Famoso", "feature1": "Reservar hasta 3 meses por adelantado", "feature2": "Cuatro invitados" } } }, "otp": { "title": "CONFIRMA TU TELÉFONO", "subtitle": "Por favor ingresa el código de 6 dígitos que enviamos a", "resendTimer": "¿No recibiste SMS? Reenviar en", "seconds": "segundos", "resendCode": "Reenviar código", "verifying": "Verificando...", "submit": "Enviar", "wrongNumber": "¿Número de teléfono incorrecto?", "backToLogin": "Iniciar sesión", "errors": { "incompleteCode": "Por favor ingresa el código completo de 6 dígitos", "invalidCode": "Código de verificación inválido. Inténtalo de nuevo.", "verificationFailed": "Verificación fallida. Inténtalo de nuevo.", "resendFailed": "Error al reenviar código. Inténtalo de nuevo." } }, "additionalInfo": { "title": "Nos gustaría conocerte más", "homeCity": "¿Cuál es tu ciudad de origen?", "loadingLocations": "Cargando ubicaciones...", "selectLocation": "Seleccionar una ubicación", "creativeInterests": "¿Qué estás creando?", "profileLink": "Enlace al perfil", "linkPlaceholder": "Enlace", "socialMedia": "Enlace a redes sociales", "instagramLink": "Enlace a Instagram", "facebookLink": "Enlace a Facebook", "twitterLink": "Enlace a Twitter", "smLink": "Enlace a SM", "referredBy": "Por favor proporciona el nombre del miembro que te refirió", "namePlaceholder": "Nombre", "save": "Guardar", "saving": "Guardando...", "success": "¡Cuenta creada exitosamente! Por favor inicia sesión para continuar.", "errors": { "homeCityRequired": "La ciudad de origen es requerida", "saveFailed": "Error al guardar información. Inténtalo de nuevo.", "unexpected": "Ocurrió un error inesperado. Inténtalo de nuevo." } }, "resetPassword": { "title": "RESTABLECER CONTRASEÑA", "subtitle": "Ingresa tu email para recibir OTP en tu móvil", "cardTitle": "Restablecer Contraseña", "emailAddress": "Dirección de Email", "emailPlaceholder": "Por favor ingresa tu email", "sendOTP": "Enviar OTP al Móvil", "sending": "Enviando...", "backToLogin": "Volver al Inicio de Sesión", "errors": { "sendFailed": "Error al enviar OTP. Inténtalo de nuevo.", "unexpected": "Ocurrió un error inesperado. Inténtalo de nuevo." } }, "resetPasswordOtp": { "title": "RESTABLECER CONTRASEÑA", "subtitle": "Ingresa el OTP enviado a tu móvil y establece una nueva contraseña", "cardTitle": "Verificar y Restablecer", "enterOtp": "Ingresa OTP de 6 dígitos", "sentTo": "Enviado a", "registeredNumber": "tu número móvil registrado", "resendOtp": "Reenviar OTP", "resendIn": "Reenviar OTP en", "newPassword": "Nueva Contraseña", "newPasswordPlaceholder": "Ingresa nueva contraseña", "confirmPassword": "Confirmar Nueva Contraseña", "confirmPasswordPlaceholder": "Confirmar nueva contraseña", "resetPasswordButton": "Restablecer Contraseña", "resetting": "Restableciendo Contraseña...", "backToReset": "Volver a Restablecer Contraseña", "success": "¡Contraseña restablecida exitosamente! Por favor inicia sesión con tu nueva contraseña.", "errors": { "incompleteOtp": "Por favor ingresa el OTP completo de 6 dígitos", "passwordMismatch": "Las contraseñas no coinciden", "passwordTooShort": "La contraseña debe tener al menos 8 caracteres", "resetFailed": "Error al restablecer contraseña. Inténtalo de nuevo.", "resendFailed": "Error al reenviar OTP", "unexpected": "Ocurrió un error inesperado. Inténtalo de nuevo." } }, "uploadProfile": { "title": "AGREGAR FOTO DE PERFIL", "subtitle": "Haz que tu perfil brille", "profilePicture": "Foto de Perfil", "profilePreview": "Vista previa del perfil", "chooseImage": "Elegir Imagen", "uploading": "Subiendo...", "uploadAndContinue": "Subir y Continuar", "skipForNow": "Omitir por ahora", "acceptedFormats": "Formatos aceptados: JPG, PNG, GIF", "maxFileSize": "Tamaño máximo de archivo: 5MB", "invalidFileType": "Por favor selecciona un archivo de imagen válido", "fileSizeLimit": "El tamaño de la imagen debe ser menor a 5MB", "uploadFailed": "Error al subir la foto de perfil. Inténtalo de nuevo.", "unexpectedError": "Ocurrió un error inesperado. Inténtalo de nuevo.", "processingError": "Error al procesar la imagen. Inténtalo de nuevo." }, "nav": { "studios": "Estudios", "videoStudios": "Estudios de Video", "musicStudios": "Estudios de Música", "photoStudios": "Estudios de Fotografía", "locations": "Ubicaciones", "bali": "Bali", "bangkok": "Bangkok", "mexico": "México", "costaRica": "Costa Rica", "newYork": "Nueva York", "mumbai": "Mumbái", "people": "Personas", "equipment": "Equipamiento", "events": "Eventos", "services": "Servicios", "allServices": "Todos los Servicios", "production": "Producción", "postProduction": "Post-Producción", "news": "Noticias", "home": "Inicio" }, "header": { "notifications": "Notificaciones", "chat": "Chat", "wallet": "Cartera", "profile": "Perfil", "accountSettings": "Configuración de Cuenta", "openMenu": "Abrir menú" }, "equipment": { "hero": { "title": "LAS HERRAMIENTAS QUE\nNECESITAS PARA CREAR", "description": "En Nomadic Studios, sabemos que las herramientas correctas pueden marcar toda la diferencia. Por eso proporcionamos equipos estandarizados de última generación en todas nuestras ubicaciones, asegurando que estés familiarizado con el equipo sin importar a dónde te lleve tu creatividad. Desde producción de video hasta grabación musical, diseño de moda hasta prototipado, nuestras herramientas están diseñadas para apoyar a creadores de todos los niveles." }, "consistentGear": { "title": "EQUIPO CONSISTENTE, DONDE SEA QUE ESTÉS", "description": "La familiaridad genera creatividad. Nuestros estudios cuentan con equipos estandarizados, así sabrás exactamente cómo usar las herramientas que necesitas, sin importar qué ubicación visites. Enfócate en tu arte, no en aprender nuevas configuraciones.", "imageAlt": "Configuración de Estudio Profesional" }, "interconnected": { "title": "INTERCONECTADO PARA COLABORACIÓN PERFECTA", "description": "Nuestra red privada de alta velocidad conecta todas las ubicaciones de Nomadic Studios, permitiéndote transferir de forma segura tu trabajo digital entre estudios o colaborar con miembros del equipo en otras ciudades. Tu creatividad no se detiene en los límites de la ciudad, y nosotros tampoco.", "imageAlt": "Red de Alta Velocidad" }, "stockedEssentials": { "title": "ABASTECIDO CON ESENCIALES", "description": "Comienza rápidamente con nuestro stock de muestras, materiales y consumibles en el estudio. Ya sea que necesites tela para moda, pinturas para arte, o cables para producción musical, te tenemos cubierto con lo esencial.", "imageAlt": "Esenciales Abastecidos" }, "expertGuidance": { "title": "GUÍA EXPERTA EN SITIO", "description": "Cada estudio es administrado por profesionales que conocen el equipo al dedillo. Ya sea que estés aprendiendo una nueva herramienta o solucionando un problema, nuestro equipo está aquí para asegurar que tu flujo de trabajo se mantenga fluido y productivo.", "imageAlt": "Panel de Guía Experta" }, "specializedRentals": { "title": "ALQUILERES ESPECIALIZADOS PARA NECESIDADES ÚNICAS", "description": "¿Necesitas equipo poco común o planeas un proyecto fuera del sitio? Aprovecha nuestros servicios de alquiler para acceder a herramientas especializadas, equipo de producción de eventos, o cualquier otra cosa que necesites para dar vida a tu visión.", "imageAlt": "Alquileres de Equipo Especializado" }, "connectCollaborate": { "title": "CONECTA, COLABORA\nY CRECE", "description": "Sé parte de algo más grande. Nomadic Studios es más que solo un lugar para trabajar - es una red vibrante de creativos que se inspiran y se apoyan mutuamente." }, "bookStudio": "RESERVAR UN ESTUDIO", "applyForMembership": "APLICAR PARA MEMBRESÍA", "copyright": "© 2025 Nomadic Studios. Todos los derechos reservados." } };
const signupSecond = { "mobileNumber": "Número Móvil", "mobileNumberPlaceholder": "Ingresa tu número móvil", "termsAgreement": "Al hacer clic en 'Completar Registro', aceptas nuestros", "termsOfService": "Términos de Servicio", "privacyPolicy": "Política de Privacidad", "smsVerification": "Enviaremos verificación SMS a tu número móvil.", "creatingAccount": "Creando Cuenta...", "completeSignup": "Completar Registro", "alreadyHaveAccount": "¿Ya tienes una cuenta?" };
const uploadProfile = { "title": "Sube Tu Foto de Perfil", "instructions": "Ayúdanos a conocerte subiendo tu foto de perfil", "dragDropText": "Arrastra y suelta tus archivos aquí, o", "browseFiles": "busca archivos", "supportedFormats": "Formatos soportados: PNG, JPG, GIF hasta 10MB", "uploading": "Subiendo...", "skipForNow": "Omitir por Ahora", "continue": "Continuar" };
const common = { "and": "y", "signIn": "Iniciar Sesión", "login": "Iniciar Sesión", "signUp": "Registrarse", "signOut": "Cerrar Sesión", "logo": "Logo", "companyName": "Nomadics" };
const nav = { "studios": "Estudios", "videoStudios": "Estudios de Video", "musicStudios": "Estudios de Música", "photoStudios": "Estudios de Fotografía", "locations": "Ubicaciones", "bali": "Bali", "bangkok": "Bangkok", "mexico": "México", "costaRica": "Costa Rica", "newYork": "Nueva York", "mumbai": "Mumbái", "people": "Personas", "equipment": "Equipamiento", "events": "Eventos", "services": "Servicios", "allServices": "Todos los Servicios", "production": "Producción", "postProduction": "Post-Producción", "news": "Noticias", "home": "Inicio" };
const header = { "notifications": "Notificaciones", "chat": "Chat", "wallet": "Cartera", "profile": "Perfil", "accountSettings": "Configuración de Cuenta", "openMenu": "Abrir menú" };
const equipment = { "hero": { "title": "LAS HERRAMIENTAS\nQUE NECESITAS PARA CREAR", "description": "En Nomadic Studios, sabemos que las herramientas adecuadas pueden marcar la diferencia. Por eso proporcionamos equipos estandarizados de última generación en todas nuestras ubicaciones, asegurando que estés familiarizado con el equipo sin importar a dónde te lleve tu creatividad. Desde producción de video hasta grabación musical, diseño de moda hasta prototipos, nuestras herramientas están diseñadas para apoyar a los creadores en todos los niveles." }, "consistentGear": { "title": "EQUIPO CONSISTENTE, DONDE QUIERA QUE ESTÉS", "description": "La familiaridad genera creatividad. Nuestros estudios cuentan con equipos estandarizados, por lo que sabrás exactamente cómo usar las herramientas que necesitas, sin importar qué ubicación visites. Concéntrate en tu arte, no en aprender nuevas configuraciones.", "imageAlt": "Configuración de Estudio Profesional" }, "interconnected": { "title": "INTERCONECTADO PARA COLABORACIÓN PERFECTA", "description": "Nuestra red privada de alta velocidad conecta todas las ubicaciones de Nomadic Studios, permitiéndote transferir de forma segura tu trabajo digital entre estudios o colaborar con miembros del equipo en otras ciudades. Tu creatividad no se detiene en los límites de la ciudad, y nosotros tampoco.", "imageAlt": "Red de Alta Velocidad" }, "stockedEssentials": { "title": "ABASTECIDO CON ESENCIALES", "description": "Comienza rápidamente con nuestro stock en estudio de muestras, materiales y consumibles. Ya sea que necesites tela para moda, pinturas para arte, o cables para producción musical, te tenemos cubierto con lo esencial.", "imageAlt": "Esenciales en Stock" }, "expertGuidance": { "title": "ORIENTACIÓN EXPERTA EN SITIO", "description": "Cada estudio es gestionado por profesionales que conocen el equipo por dentro y por fuera. Ya sea que estés aprendiendo una nueva herramienta o solucionando un problema, nuestro equipo está aquí para asegurar que tu flujo de trabajo se mantenga fluido y productivo.", "imageAlt": "Panel de Orientación Experta" }, "specializedRentals": { "title": "ALQUILERES ESPECIALIZADOS PARA NECESIDADES ÚNICAS", "description": "¿Necesitas equipo poco común o planeas un proyecto fuera del sitio? Aprovecha nuestros servicios de alquiler para acceder a herramientas especializadas, equipo de producción de eventos, o cualquier otra cosa que puedas necesitar para dar vida a tu visión.", "imageAlt": "Alquileres de Equipo Especializado" }, "connectCollaborate": { "title": "CONECTA, COLABORA\nY CRECE", "description": "Sé parte de algo más grande. Nomadic Studios es más que solo un lugar para trabajar - es una red vibrante de creativos que se inspiran y se apoyan mutuamente." }, "bookStudio": "RESERVAR UN ESTUDIO", "applyForMembership": "APLICAR PARA MEMBRESÍA", "copyright": "© 2025 Nomadic Studios. Todos los derechos reservados." };
const footer = { "about": { "title": "Acerca de", "joinOurTeam": "Únete a Nuestro Equipo", "investors": "Inversores", "partners": "Socios" }, "socialMedia": { "title": "Redes Sociales" }, "legal": { "termsConditions": "Términos y Condiciones", "privacyPolicy": "Política de Privacidad" }, "backToTop": "Volver Arriba", "copyright": "©2025 Nomadic Studios" };
const sp = {
  bali,
  bangkok,
  mexico,
  "costa-rica": { "meta": { "title": "Costa Rica Location | Nomadics", "description": "Nomadics Costa Rica - Natural inspiration with video and music studios surrounded by beauty" }, "welcome": { "title": "COSTA RICA, NATURAL INSPIRATION", "subtitle": "Surrounded by natural beauty, our Costa Rica location provides a serene environment for focused creative work and inspired productions.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "COSTA RICA IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "3", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "15", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "8", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN COSTA RICA", "studios": [{ "id": "costa-rica-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "costa-rica-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "costa-rica-creative-suite", "image": "/images/studios/photo-1.png", "title": "CREATIVE SUITE" }] }, "events": { "sectionTitle": "EVENTS IN COSTA RICA" }, "planVisit": { "address": "Avenida Central, San José, Costa Rica", "hours": "8:00 AM - 7:00 PM", "phone": "+506 1234 5678", "email": "costarica@nomadics.com", "additionalInfo": "Downtown location near main attractions..." }, "people": { "title": "MEET THE CREATIVES OF COSTA RICA" } },
  "new-york": { "meta": { "title": "New York Location | Nomadics", "description": "Nomadics New York - Global epicenter with photography and fashion studios in Manhattan" }, "welcome": { "title": "NEW YORK, GLOBAL EPICENTER", "subtitle": "In the heart of the world's creative capital, our New York studios offer unparalleled resources for fashion, photography, and artistic innovation.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / FASHION STUDIO" }, "info": { "title": "NEW YORK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "8", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "35", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "2", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN NEW YORK", "studios": [{ "id": "new-york-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "new-york-fashion-studio", "image": "/images/studios/studio-1.png", "title": "FASHION STUDIO" }, { "id": "new-york-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "new-york-art-studio", "image": "/images/studios/photo-1.png", "title": "ART STUDIO" }, { "id": "new-york-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "new-york-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "new-york-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "new-york-livestream-studio", "image": "/images/studios/music-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN NEW YORK" }, "planVisit": { "address": "123 Broadway, Manhattan, New York, NY 10013, USA", "hours": "24/7 Access for Members", "phone": "+1 212 123 4567", "email": "newyork@nomadics.com", "additionalInfo": "Accessible via multiple subway lines..." }, "people": { "title": "MEET THE CREATIVES OF NEW YORK" } },
  mumbai,
  "costa-rica1": { "meta": { "title": "Costa Rica Location | Nomadics", "description": "Nomadics Costa Rica - Natural inspiration with video and music studios surrounded by beauty" }, "welcome": { "title": "COSTA RICA, NATURAL INSPIRATION", "subtitle": "Surrounded by natural beauty, our Costa Rica location provides a serene environment for focused creative work and inspired productions.", "backgroundImage": "/images/home/locations/bali.png", "meta": "VIDEO STUDIO / MUSIC STUDIO" }, "info": { "title": "COSTA RICA IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "3", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "15", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "8", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN COSTA RICA", "studios": [{ "id": "costa-rica-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "costa-rica-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "costa-rica-creative-suite", "image": "/images/studios/photo-1.png", "title": "CREATIVE SUITE" }] }, "events": { "sectionTitle": "EVENTS IN COSTA RICA" }, "planVisit": { "address": "Avenida Central, San José, Costa Rica", "hours": "8:00 AM - 7:00 PM", "phone": "+506 1234 5678", "email": "costarica@nomadics.com", "additionalInfo": "Downtown location near main attractions..." }, "people": { "title": "MEET THE CREATIVES OF COSTA RICA" } },
  "new-york1": { "meta": { "title": "New York Location | Nomadics", "description": "Nomadics New York - Global epicenter with photography and fashion studios in Manhattan" }, "welcome": { "title": "NEW YORK, GLOBAL EPICENTER", "subtitle": "In the heart of the world's creative capital, our New York studios offer unparalleled resources for fashion, photography, and artistic innovation.", "backgroundImage": "/images/home/locations/bali.png", "meta": "PHOTOGRAPHY STUDIO / FASHION STUDIO" }, "info": { "title": "NEW YORK IS A CITY OF INNOVATION, CULTURE AND CREATIVITY", "description": "Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here.", "stats": [{ "number": "8", "label": "STUDIO SPACES\nAVAILABLE" }, { "number": "35", "label": "PROFESSIONALS\nCOLLABORATING\nMONTHLY" }, { "number": "2", "label": "MINUTES FROM\nTHE CITY CENTER" }] }, "studios": { "sectionTitle": "STUDIOS TO CREATE IN NEW YORK", "studios": [{ "id": "new-york-photography-studio", "image": "/images/studios/photo-1.png", "title": "PHOTOGRAPHY STUDIO" }, { "id": "new-york-fashion-studio", "image": "/images/studios/studio-1.png", "title": "FASHION STUDIO" }, { "id": "new-york-video-studio", "image": "/images/studios/studio-1.png", "title": "VIDEO STUDIO" }, { "id": "new-york-art-studio", "image": "/images/studios/photo-1.png", "title": "ART STUDIO" }, { "id": "new-york-music-studio", "image": "/images/studios/music-1.png", "title": "MUSIC STUDIO" }, { "id": "new-york-dance-studio", "image": "/images/studios/studio-1.png", "title": "DANCE STUDIO" }, { "id": "new-york-podcast-studio", "image": "/images/studios/photo-1.png", "title": "PODCAST STUDIO" }, { "id": "new-york-livestream-studio", "image": "/images/studios/music-1.png", "title": "LIVESTREAM STUDIO" }] }, "events": { "sectionTitle": "EVENTS IN NEW YORK" }, "planVisit": { "address": "123 Broadway, Manhattan, New York, NY 10013, USA", "hours": "24/7 Access for Members", "phone": "+1 212 123 4567", "email": "newyork@nomadics.com", "additionalInfo": "Accessible via multiple subway lines..." }, "people": { "title": "CONOCE A LOS CREATIVOS DE NUEVA YORK" } },
  auth,
  signupSecond,
  uploadProfile,
  common,
  nav,
  header,
  equipment,
  footer
};
const sp$1 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  auth,
  bali,
  bangkok,
  common,
  default: sp,
  equipment,
  footer,
  header,
  mexico,
  mumbai,
  nav,
  signupSecond,
  uploadProfile
}, Symbol.toStringTag, { value: "Module" }));
const translations$1 = { en, sp };
async function loader$2({ params }) {
  const location = params.location;
  const lang = "en";
  const languageData = translations$1[lang];
  const locationData = languageData[location];
  if (!locationData) {
    throw new Response("Location Not Found", { status: 404 });
  }
  const apiRes = await fetch(
    `https://nomadicstudios.net:444/api/locations/${location}`
  );
  const apiData = await apiRes.json();
  locationData.welcome.backgroundImage = apiData.data.banner_picture;
  return Response.json({ location, locationData, apiData });
}
const meta$i = ({ data }) => {
  if (!data) {
    return [
      { title: "Location Not Found | Nomadics" },
      { name: "description", content: "The requested location was not found." }
    ];
  }
  return [
    { title: data.locationData.meta.title },
    { name: "description", content: data.locationData.meta.description }
  ];
};
function LocationPage() {
  var _a, _b;
  const { locationData, apiData } = useLoaderData();
  return /* @__PURE__ */ jsxs("div", { children: [
    /* @__PURE__ */ jsx(WelcomeSection$2, { data: locationData.welcome }),
    /* @__PURE__ */ jsx(LocationInfo, { data: locationData.info }),
    /* @__PURE__ */ jsx(StudiosSection$2, { data: apiData.data }),
    ((_a = apiData.data.resources) == null ? void 0 : _a.length) > 0 && /* @__PURE__ */ jsx(
      PeopleSection,
      {
        title: locationData.people.title,
        peopleData: apiData.data.resources,
        filterOptions: apiData.data.filters
      }
    ),
    ((_b = apiData.data.events) == null ? void 0 : _b.length) > 0 && /* @__PURE__ */ jsx(
      EventsSection$1,
      {
        title: locationData.events.sectionTitle,
        data: apiData.data.events
      }
    ),
    /* @__PURE__ */ jsx(
      PlanYourVisitSection,
      {
        title: locationData.planVisit.title,
        address: apiData.data.address,
        hours: apiData.data.opening_hours,
        phone: apiData.data.phone,
        email: apiData.data.email,
        additionalInfo: apiData.data.additional_info,
        location: apiData.data.location
      }
    ),
    /* @__PURE__ */ jsx(MembershipSection, { theme: "black" })
  ] });
}
const route5 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: LocationPage,
  loader: loader$2,
  meta: meta$i
}, Symbol.toStringTag, { value: "Module" }));
const styles$2 = `
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
`;
const meta$h = ({ params }) => {
  var _a;
  return [
    { title: `${(_a = params.slug) == null ? void 0 : _a.replace(/-/g, " ")} - Nomadics Studios` },
    {
      name: "description",
      content: "Read the latest news and insights from the creative community at Nomadics Studios."
    }
  ];
};
const newsArticles$2 = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: {
      name: "Akanksha Bhartaure",
      avatar: "/team-member.png",
      bio: "Travel Writer & Creative Director"
    },
    publishedAt: "5 min ago",
    readTime: "5 min read",
    image: "/bw (1).png",
    slug: "travel-as-a-failure",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>A story about a trip to Barcelona 2</h3>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <p>Travel opens our minds to new possibilities and perspectives that we might never encounter in our daily routines. When we step outside our comfort zones and immerse ourselves in different cultures, we gain invaluable insights that can transform our creative work.</p>
      
      <p>Sometimes, the most valuable lessons come from the moments when things don't go according to plan. Failed connections, missed opportunities, and unexpected detours often become the foundation for our most compelling stories and creative breakthroughs.</p>
    `,
    tags: ["Travel", "Lessons", "Barcelona", "Growth"]
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: {
      name: "Super Admin",
      avatar: "/team-member.png",
      bio: "Creative Director at Nomadics Studios"
    },
    publishedAt: "2 hours ago",
    readTime: "4 min read",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>The Creative Impact of Travel</h3>
      
      <p>For creators, travel is more than just a break from routine—it's a source of inspiration that can revolutionize their approach to their craft. Whether you're a filmmaker seeking new locations, a photographer hunting for unique lighting, or a musician searching for fresh sounds, travel provides the raw material for creative breakthroughs.</p>
      
      <p>At Nomadics Studios, we've seen firsthand how our community members return from their travels with renewed energy and innovative ideas. The stories they share, the techniques they've learned, and the perspectives they've gained all contribute to the rich tapestry of our creative community.</p>
    `,
    tags: ["Travel", "Creativity", "Inspiration", "Community"]
  }
];
function findArticleBySlug$1(slug) {
  return newsArticles$2.find((article) => article.slug === slug) || newsArticles$2[0];
}
function getRelatedArticles$1(currentSlug) {
  return newsArticles$2.filter((article) => article.slug !== currentSlug).slice(0, 3);
}
function NewsDetailsPage$1() {
  const { slug } = useParams();
  const article = findArticleBySlug$1(slug || "");
  const relatedArticles = getRelatedArticles$1(slug || "");
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("div", { className: "relative bg-white min-h-screen", children: [
    /* @__PURE__ */ jsx("style", { dangerouslySetInnerHTML: { __html: styles$2 } }),
    /* @__PURE__ */ jsxs("section", { className: "relative h-[50vh] lg:h-[55vh] overflow-hidden", children: [
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: article.image,
            alt: article.title,
            className: "h-full w-full object-cover",
            onError: (e) => {
              e.currentTarget.src = "/image.png";
            }
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative z-10 h-full flex items-end", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1000px] px-4 lg:px-8 pb-12 lg:pb-16", children: [
        /* @__PURE__ */ jsx("nav", { className: "mb-6", children: /* @__PURE__ */ jsxs(
          Link,
          {
            to: "/news",
            className: "text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2",
            children: [
              /* @__PURE__ */ jsx("svg", { className: "w-4 h-4", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 19l-7-7 7-7" }) }),
              "Back to News"
            ]
          }
        ) }),
        /* @__PURE__ */ jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsx("span", { className: "inline-block px-4 py-2 bg-white/20 backdrop-blur-sm text-white text-sm font-medium uppercase tracking-wider rounded", children: article.category }) }),
        /* @__PURE__ */ jsx("h1", { className: "text-3xl lg:text-5xl font-bold text-white mb-6 leading-tight max-w-4xl", children: article.title }),
        /* @__PURE__ */ jsx("div", { className: "flex flex-col sm:flex-row sm:items-center gap-4 text-white/90", children: /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: article.author.avatar,
              alt: article.author.name,
              className: "w-10 h-10 rounded-full object-cover bg-white/20",
              onError: (e) => {
                e.currentTarget.src = "/team-member.png";
              }
            }
          ),
          /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("div", { className: "font-medium text-white", children: article.author.name }),
            /* @__PURE__ */ jsxs("div", { className: "text-sm text-white/70", children: [
              article.publishedAt,
              " • ",
              article.readTime
            ] })
          ] })
        ] }) })
      ] }) })
    ] }),
    /* @__PURE__ */ jsx("section", { className: "bg-white py-12 lg:py-16", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[800px] px-4 lg:px-8", children: [
      /* @__PURE__ */ jsx("div", { className: "prose prose-lg max-w-none mb-16", children: /* @__PURE__ */ jsx(
        "div",
        {
          className: "article-content \n                  [&_h2]:text-2xl [&_h2]:lg:text-3xl [&_h2]:font-bold [&_h2]:text-black [&_h2]:mb-6 [&_h2]:mt-12 [&_h2]:leading-tight\n                  [&_h3]:text-xl [&_h3]:lg:text-2xl [&_h3]:font-bold [&_h3]:text-black [&_h3]:mb-4 [&_h3]:mt-8 [&_h3]:leading-tight\n                  [&_p]:text-black/80 [&_p]:text-base [&_p]:lg:text-lg [&_p]:leading-relaxed [&_p]:mb-6\n                  [&_img]:rounded-lg [&_img]:shadow-lg [&_img]:my-8",
          dangerouslySetInnerHTML: { __html: article.content }
        }
      ) }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-3 mb-12 pb-8 border-b border-gray-200", children: [
        /* @__PURE__ */ jsx("span", { className: "text-sm font-medium text-gray-600 mr-2", children: "Tags:" }),
        article.tags.map((tag) => /* @__PURE__ */ jsx(
          "span",
          {
            className: "px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-full hover:bg-gray-200 transition-colors cursor-pointer",
            children: tag
          },
          tag
        ))
      ] }),
      /* @__PURE__ */ jsx("div", { className: "bg-gray-50 rounded-2xl p-8 mb-12", children: /* @__PURE__ */ jsxs("div", { className: "flex items-start gap-4", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: article.author.avatar,
            alt: article.author.name,
            className: "w-16 h-16 rounded-full object-cover bg-gray-200",
            onError: (e) => {
              e.currentTarget.src = "/team-member.png";
            }
          }
        ),
        /* @__PURE__ */ jsxs("div", { className: "flex-1", children: [
          /* @__PURE__ */ jsx("h3", { className: "text-xl font-bold text-black mb-2", children: article.author.name }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 mb-4", children: article.author.bio }),
          /* @__PURE__ */ jsxs("div", { className: "flex gap-3", children: [
            /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "sm", className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300", children: "Follow" }),
            /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "sm", className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300", children: "View Profile" })
          ] })
        ] })
      ] }) }),
      /* @__PURE__ */ jsxs("div", { className: "text-center mb-12", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-xl font-bold text-black mb-4", children: "Share this article" }),
        /* @__PURE__ */ jsxs("div", { className: "flex justify-center gap-3", children: [
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300", children: "Twitter" }),
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300", children: "LinkedIn" }),
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", className: "bg-white text-black border-black hover:bg-black hover:text-white transition-colors duration-300", children: "Facebook" }),
          /* @__PURE__ */ jsx(Button, { variant: "fill-black", size: "md", arrowRight: true, children: "Subscribe" })
        ] })
      ] })
    ] }) }),
    /* @__PURE__ */ jsx("section", { className: "bg-gray-50", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1200px] px-4 lg:px-8 py-12 lg:py-16", children: [
      /* @__PURE__ */ jsxs("div", { className: "text-center mb-16", children: [
        /* @__PURE__ */ jsx("h2", { className: "text-3xl lg:text-4xl font-bold text-black mb-4", children: "Related news" }),
        /* @__PURE__ */ jsx("div", { className: "w-24 h-1 bg-black mx-auto" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12", children: relatedArticles.map((relatedArticle) => /* @__PURE__ */ jsxs(
        Link,
        {
          to: `/news-details/${relatedArticle.slug}`,
          className: "group block bg-white shadow-sm hover:shadow-lg transition-all duration-300 overflow-hidden hover:transform hover:scale-[1.02] rounded-lg",
          children: [
            /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden bg-neutral-100", children: /* @__PURE__ */ jsx(
              "img",
              {
                src: relatedArticle.image,
                alt: relatedArticle.title,
                className: "h-full w-full object-cover transition-transform duration-500 group-hover:scale-105",
                onError: (e) => {
                  e.currentTarget.src = "/image.png";
                }
              }
            ) }),
            /* @__PURE__ */ jsxs("div", { className: "p-6", children: [
              /* @__PURE__ */ jsx("div", { className: "inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded mb-3", children: relatedArticle.category }),
              /* @__PURE__ */ jsx("h3", { className: "text-lg font-bold text-black mb-3 group-hover:text-black/80 transition-colors leading-tight", children: relatedArticle.title }),
              /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-sm line-clamp-2 mb-4", children: relatedArticle.excerpt }),
              /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2 text-xs text-gray-500", children: [
                /* @__PURE__ */ jsx("span", { children: relatedArticle.author.name }),
                /* @__PURE__ */ jsx("span", { children: "•" }),
                /* @__PURE__ */ jsx("span", { children: relatedArticle.publishedAt })
              ] })
            ] })
          ]
        },
        relatedArticle.id
      )) }),
      /* @__PURE__ */ jsx("div", { className: "text-center", children: /* @__PURE__ */ jsx(
        Button,
        {
          variant: "outline-black",
          size: "lg",
          arrowRight: true,
          className: "text-black",
          onClick: () => window.location.href = "/news",
          children: "View All News"
        }
      ) })
    ] }) })
  ] }) });
}
const route6 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: NewsDetailsPage$1,
  meta: meta$h
}, Symbol.toStringTag, { value: "Module" }));
const ResetPasswordOtp = () => {
  var _a, _b, _c;
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [otp, setOtp] = useState(["", "", "", "", "", ""]);
  const [passwords, setPasswords] = useState({
    password: "",
    confirmPassword: ""
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [resendTimer, setResendTimer] = useState(30);
  const [canResend, setCanResend] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const email = ((_a = location.state) == null ? void 0 : _a.email) || "";
  const mobile_number = ((_b = location.state) == null ? void 0 : _b.mobile_number) || "";
  const requestType = ((_c = location.state) == null ? void 0 : _c.requestType) || "reset-password";
  useEffect(() => {
    if (!email || !mobile_number) {
      navigate("/reset-password");
      return;
    }
    setIsVisible(true);
    const timer = setInterval(() => {
      setResendTimer((prev) => {
        if (prev <= 1) {
          setCanResend(true);
          clearInterval(timer);
          return 0;
        }
        return prev - 1;
      });
    }, 1e3);
    return () => clearInterval(timer);
  }, [email, mobile_number, navigate]);
  const handleOtpChange = (index, value) => {
    if (value.length > 1) return;
    const newOtp = [...otp];
    newOtp[index] = value;
    setOtp(newOtp);
    if (value && index < 5) {
      const nextInput = document.getElementById(`otp-${index + 1}`);
      nextInput == null ? void 0 : nextInput.focus();
    }
    if (error) setError(null);
  };
  const handleOtpKeyDown = (index, e) => {
    if (e.key === "Backspace" && !otp[index] && index > 0) {
      const prevInput = document.getElementById(`otp-${index - 1}`);
      prevInput == null ? void 0 : prevInput.focus();
    }
  };
  const handlePasswordChange = (e) => {
    const { name, value } = e.target;
    setPasswords((prev) => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };
  const handleSubmit = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    const otpString = otp.join("");
    if (otpString.length !== 6) {
      setError(t("auth.resetPasswordOtp.errors.incompleteOtp", "Please enter the complete 6-digit OTP"));
      setIsLoading(false);
      return;
    }
    if (passwords.password !== passwords.confirmPassword) {
      setError(t("auth.resetPasswordOtp.errors.passwordMismatch", "Passwords do not match"));
      setIsLoading(false);
      return;
    }
    if (passwords.password.length < 8) {
      setError(t("auth.resetPasswordOtp.errors.passwordTooShort", "Password must be at least 8 characters long"));
      setIsLoading(false);
      return;
    }
    try {
      const response = await authAPI.verifyOTP(
        mobile_number,
        otpString,
        requestType,
        {
          password: passwords.password,
          confirm_password: passwords.confirmPassword
        }
      );
      if (response.status) {
        navigate("/login", {
          state: {
            message: t("auth.resetPasswordOtp.success", "Password reset successfully! Please login with your new password."),
            type: "success"
          }
        });
      } else {
        setError(response.error || t("auth.resetPasswordOtp.errors.resetFailed", "Failed to reset password. Please try again."));
      }
    } catch (err) {
      console.error("Reset password OTP verification error:", err);
      setError(err instanceof Error ? err.message : t("auth.resetPasswordOtp.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };
  const handleResendOTP = async () => {
    if (!canResend) return;
    try {
      setIsLoading(true);
      const response = await authAPI.forgotPassword(email);
      if (response.status) {
        setCanResend(false);
        setResendTimer(30);
        setError(null);
        const timer = setInterval(() => {
          setResendTimer((prev) => {
            if (prev <= 1) {
              setCanResend(true);
              clearInterval(timer);
              return 0;
            }
            return prev - 1;
          });
        }, 1e3);
        setError(null);
      } else {
        setError(response.error || t("auth.resetPasswordOtp.errors.resendFailed", "Failed to resend OTP"));
      }
    } catch (err) {
      setError(t("auth.resetPasswordOtp.errors.resendFailed", "Failed to resend OTP"));
    } finally {
      setIsLoading(false);
    }
  };
  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, "0")}`;
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx("div", { className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60", style: { animationDelay: "0s", animationDuration: "3s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40", style: { animationDelay: "1s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30", style: { animationDelay: "2s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50", style: { animationDelay: "1.5s", animationDuration: "4s" } })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs("div", { className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: [
        /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.resetPasswordOtp.title", "RESET PASSWORD") }),
        /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("auth.resetPasswordOtp.subtitle", "Enter the OTP sent to your mobile and set new password") })
      ] }),
      /* @__PURE__ */ jsx("div", { className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
        /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider", children: t("auth.resetPasswordOtp.cardTitle", "Verify & Reset") }),
        error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
        /* @__PURE__ */ jsxs("form", { onSubmit: handleSubmit, className: "space-y-6", children: [
          /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
            /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
              /* @__PURE__ */ jsx("p", { className: "text-white/80 text-sm mb-2", children: t("auth.resetPasswordOtp.enterOtp", "Enter 6-digit OTP") }),
              /* @__PURE__ */ jsxs("p", { className: "text-white/60 text-xs", children: [
                t("auth.resetPasswordOtp.sentTo", "Sent to"),
                " ",
                mobile_number ? mobile_number.replace(/(\d{2})(\d{4})(\d{4})/, "$1****$3") : t("auth.resetPasswordOtp.registeredNumber", "your registered mobile number")
              ] })
            ] }),
            /* @__PURE__ */ jsx("div", { className: "flex justify-center space-x-3", children: otp.map((digit, index) => /* @__PURE__ */ jsx(
              "input",
              {
                id: `otp-${index}`,
                type: "text",
                inputMode: "numeric",
                pattern: "[0-9]*",
                maxLength: 1,
                value: digit,
                onChange: (e) => handleOtpChange(index, e.target.value),
                onKeyDown: (e) => handleOtpKeyDown(index, e),
                className: "w-12 h-12 text-center text-xl font-bold bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all",
                disabled: isLoading
              },
              index
            )) }),
            /* @__PURE__ */ jsx("div", { className: "text-center", children: canResend ? /* @__PURE__ */ jsx(
              Button,
              {
                type: "button",
                variant: "outline-white",
                size: "sm",
                onClick: handleResendOTP,
                disabled: isLoading,
                children: t("auth.resetPasswordOtp.resendOtp", "Resend OTP")
              }
            ) : /* @__PURE__ */ jsxs("p", { className: "text-white/60 text-sm", children: [
              t("auth.resetPasswordOtp.resendIn", "Resend OTP in"),
              " ",
              formatTime(resendTimer)
            ] }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
            /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.resetPasswordOtp.newPassword", "New Password") }),
            /* @__PURE__ */ jsxs("div", { className: "relative", children: [
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: showPassword ? "text" : "password",
                  name: "password",
                  placeholder: t("auth.resetPasswordOtp.newPasswordPlaceholder", "Enter new password"),
                  value: passwords.password,
                  onChange: handlePasswordChange,
                  className: "w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading,
                  minLength: 8
                }
              ),
              /* @__PURE__ */ jsx(
                "button",
                {
                  type: "button",
                  onClick: () => setShowPassword(!showPassword),
                  className: "absolute right-3 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors",
                  children: showPassword ? /* @__PURE__ */ jsx("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21" }) }) : /* @__PURE__ */ jsxs("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: [
                    /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 12a3 3 0 11-6 0 3 3 0 016 0z" }),
                    /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" })
                  ] })
                }
              )
            ] })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
            /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.resetPasswordOtp.confirmPassword", "Confirm New Password") }),
            /* @__PURE__ */ jsxs("div", { className: "relative", children: [
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: showConfirmPassword ? "text" : "password",
                  name: "confirmPassword",
                  placeholder: t("auth.resetPasswordOtp.confirmPasswordPlaceholder", "Confirm new password"),
                  value: passwords.confirmPassword,
                  onChange: handlePasswordChange,
                  className: "w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading,
                  minLength: 8
                }
              ),
              /* @__PURE__ */ jsx(
                "button",
                {
                  type: "button",
                  onClick: () => setShowConfirmPassword(!showConfirmPassword),
                  className: "absolute right-3 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors",
                  children: showConfirmPassword ? /* @__PURE__ */ jsx("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21" }) }) : /* @__PURE__ */ jsxs("svg", { className: "w-5 h-5", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: [
                    /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 12a3 3 0 11-6 0 3 3 0 016 0z" }),
                    /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" })
                  ] })
                }
              )
            ] })
          ] }),
          /* @__PURE__ */ jsx(
            Button,
            {
              type: "submit",
              variant: "fill-white",
              size: "lg",
              className: "w-full",
              disabled: isLoading,
              arrowRight: !isLoading,
              children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                t("auth.resetPasswordOtp.resetting", "Resetting Password...")
              ] }) : t("auth.resetPasswordOtp.resetPasswordButton", "Reset Password")
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "text-center pt-4", children: /* @__PURE__ */ jsx(
            Button,
            {
              type: "button",
              variant: "outline-white",
              size: "sm",
              onClick: () => navigate("/reset-password"),
              children: t("auth.resetPasswordOtp.backToReset", "Back to Reset Password")
            }
          ) })
        ] })
      ] }) })
    ] })
  ] });
};
function ResetPasswordOtpRoute() {
  return /* @__PURE__ */ jsx(ResetPasswordOtp, {});
}
const route7 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: ResetPasswordOtpRoute
}, Symbol.toStringTag, { value: "Module" }));
const meta$g = () => {
  return [
    { title: "Account Settings - Nomadics Studios" },
    {
      name: "description",
      content: "Manage your account settings and preferences."
    }
  ];
};
function AccountSettings() {
  return /* @__PURE__ */ jsx(ProtectedRoute, { children: /* @__PURE__ */ jsx("div", { className: "min-h-screen flex items-center justify-center bg-black text-white", children: /* @__PURE__ */ jsxs("div", { className: "text-center px-4", children: [
    /* @__PURE__ */ jsx("div", { className: "mb-8", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/images/logo-white.png",
        alt: "Nomadics Studios",
        className: "h-16 lg:h-20 w-auto mx-auto mb-8"
      }
    ) }),
    /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold mb-6 tracking-wider", children: "ACCOUNT SETTINGS" }),
    /* @__PURE__ */ jsxs("p", { className: "text-lg lg:text-xl text-white/80 mb-8 max-w-2xl mx-auto leading-relaxed", children: [
      "Account settings page is under development.",
      /* @__PURE__ */ jsx("br", {}),
      "You can manage your profile, preferences, and more here soon."
    ] }),
    /* @__PURE__ */ jsx("div", { className: "flex flex-col sm:flex-row gap-4 justify-center items-center", children: /* @__PURE__ */ jsx("div", { className: "text-sm text-white/60 uppercase tracking-widest", children: "Coming Soon" }) })
  ] }) }) });
}
const route8 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: AccountSettings,
  meta: meta$g
}, Symbol.toStringTag, { value: "Module" }));
const OtpVerification = () => {
  var _a, _b, _c;
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [otp, setOtp] = useState(["", "", "", "", "", ""]);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [resendTimer, setResendTimer] = useState(30);
  const [canResend, setCanResend] = useState(false);
  const phoneNumber = ((_a = location.state) == null ? void 0 : _a.phoneNumber) || "+1 XXX XXX XXXX";
  const requestType = ((_b = location.state) == null ? void 0 : _b.requestType) || "login";
  const signupData = (_c = location.state) == null ? void 0 : _c.signupData;
  useEffect(() => {
    setIsVisible(true);
    const timer = setInterval(() => {
      setResendTimer((prev) => {
        if (prev <= 1) {
          setCanResend(true);
          clearInterval(timer);
          return 0;
        }
        return prev - 1;
      });
    }, 1e3);
    return () => clearInterval(timer);
  }, []);
  const handleOtpChange = (index, value) => {
    if (value.length > 1) return;
    const newOtp = [...otp];
    newOtp[index] = value;
    setOtp(newOtp);
    if (value && index < 5) {
      const nextInput = document.getElementById(`otp-${index + 1}`);
      nextInput == null ? void 0 : nextInput.focus();
    }
    if (error) setError(null);
  };
  const handleKeyDown = (index, e) => {
    if (e.key === "Backspace" && !otp[index] && index > 0) {
      const prevInput = document.getElementById(`otp-${index - 1}`);
      prevInput == null ? void 0 : prevInput.focus();
    }
  };
  const handleSubmit = async (e) => {
    e.preventDefault();
    const otpString = otp.join("");
    if (otpString.length !== 6) {
      setError(t("auth.otp.errors.incompleteCode", "Please enter the complete 6-digit code"));
      return;
    }
    setIsLoading(true);
    setError(null);
    try {
      let response;
      if (requestType === "signup" && signupData) {
        response = await authAPI.verifyOTP(
          phoneNumber,
          otpString,
          requestType,
          signupData
        );
      } else {
        response = await authAPI.verifyOTP(phoneNumber, otpString, requestType);
      }
      if (response.status) {
        if (requestType === "signup") {
          navigate("/upload-profile-picture", {
            state: {
              uuid: response.uuid,
              userData: signupData
            }
          });
        } else {
          handleAuthSuccess(response);
          setTimeout(() => {
            window.location.href = "/home";
          }, 100);
        }
      } else {
        setError(
          response.error || t("auth.otp.errors.invalidCode", "Invalid verification code. Please try again.")
        );
      }
    } catch (err) {
      console.error("OTP verification error:", err);
      setError(
        err instanceof Error ? err.message : t("auth.otp.errors.verificationFailed", "Verification failed. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };
  const handleResend = async () => {
    if (!canResend) return;
    setIsLoading(true);
    setError(null);
    try {
      const response = await authAPI.sendOTP(phoneNumber);
      if (response.status) {
        setResendTimer(30);
        setCanResend(false);
        setOtp(["", "", "", "", "", ""]);
        const timer = setInterval(() => {
          setResendTimer((prev) => {
            if (prev <= 1) {
              setCanResend(true);
              clearInterval(timer);
              return 0;
            }
            return prev - 1;
          });
        }, 1e3);
      } else {
        setError(response.error || t("auth.otp.errors.resendFailed", "Failed to resend code. Please try again."));
      }
    } catch (err) {
      console.error("Resend OTP error:", err);
      setError(
        err instanceof Error ? err.message : t("auth.otp.errors.resendFailed", "Failed to resend code. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60",
          style: { animationDelay: "0s", animationDuration: "3s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40",
          style: { animationDelay: "1s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30",
          style: { animationDelay: "2s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50",
          style: { animationDelay: "1.5s", animationDuration: "4s" }
        }
      )
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs(
        "div",
        {
          className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: [
            /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.otp.title", "CONFIRM YOUR PHONE") }),
            /* @__PURE__ */ jsxs("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: [
              t("auth.otp.subtitle", "Please input 6-digit code we just sent to"),
              " ",
              phoneNumber
            ] })
          ]
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
            error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
            /* @__PURE__ */ jsxs("form", { onSubmit: handleSubmit, className: "space-y-8", children: [
              /* @__PURE__ */ jsx("div", { className: "flex justify-center space-x-3", children: otp.map((digit, index) => /* @__PURE__ */ jsx(
                "input",
                {
                  id: `otp-${index}`,
                  type: "text",
                  inputMode: "numeric",
                  maxLength: 1,
                  value: digit,
                  onChange: (e) => handleOtpChange(index, e.target.value),
                  onKeyDown: (e) => handleKeyDown(index, e),
                  className: "w-12 h-14 text-center text-2xl font-bold bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all",
                  disabled: isLoading,
                  autoComplete: "off"
                },
                index
              )) }),
              /* @__PURE__ */ jsx("div", { className: "text-center", children: !canResend ? /* @__PURE__ */ jsxs("p", { className: "text-white/60 text-sm", children: [
                t("auth.otp.resendTimer", "Didn't get an SMS? Resend in"),
                " ",
                resendTimer,
                " ",
                t("auth.otp.seconds", "seconds")
              ] }) : /* @__PURE__ */ jsx(
                Button,
                {
                  type: "button",
                  variant: "outline-white",
                  size: "sm",
                  onClick: handleResend,
                  disabled: isLoading,
                  children: t("auth.otp.resendCode", "Resend code")
                }
              ) }),
              /* @__PURE__ */ jsx(
                Button,
                {
                  type: "submit",
                  variant: "fill-white",
                  size: "lg",
                  className: "w-full",
                  disabled: isLoading || otp.some((digit) => !digit),
                  arrowRight: !isLoading,
                  children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                    /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                    t("auth.otp.verifying", "Verifying...")
                  ] }) : t("auth.otp.submit", "Submit")
                }
              ),
              /* @__PURE__ */ jsxs("div", { className: "text-center text-white/60 text-sm pt-4", children: [
                t("auth.otp.wrongNumber", "Wrong phone number?"),
                " ",
                /* @__PURE__ */ jsx(
                  Button,
                  {
                    type: "button",
                    variant: "outline-white",
                    size: "sm",
                    className: "ml-2",
                    onClick: () => navigate("/login"),
                    children: t("auth.otp.backToLogin", "Login")
                  }
                )
              ] })
            ] })
          ] })
        }
      )
    ] })
  ] });
};
const meta$f = () => {
  return [
    { title: "OTP Verification - Nomadic Studios" },
    { name: "description", content: "Verify your phone number to complete registration" }
  ];
};
function OtpVerificationPage() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(OtpVerification, {}) });
}
const route9 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: OtpVerificationPage,
  meta: meta$f
}, Symbol.toStringTag, { value: "Module" }));
function loader$1() {
  return new Response("", {
    status: 200,
    headers: {
      "Content-Type": "application/javascript"
    }
  });
}
const route10 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  loader: loader$1
}, Symbol.toStringTag, { value: "Module" }));
function DanielAnders$1() {
  const [start, setStart] = useState("2025-08-21T10:30");
  const [end, setEnd] = useState("2025-08-21T13:30");
  const duration = (new Date(end).getTime() - new Date(start).getTime()) / (1e3 * 60 * 60);
  const points = duration > 0 ? Math.round(duration * 43) : 0;
  const handleBooking = () => {
    alert(
      `Booking confirmed!
Start: ${start}
End: ${end}
Duration: ${duration}h
Points: ${points}`
    );
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-white text-black font-sans p-6 md:p-12", children: [
    /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold", children: "DANIEL ANDERS" }),
    /* @__PURE__ */ jsxs("div", { className: "flex gap-2 mt-2", children: [
      /* @__PURE__ */ jsx("span", { className: "px-3 py-1 rounded-full border text-sm", children: "Bali, Indonesia" }),
      /* @__PURE__ */ jsx("span", { className: "px-3 py-1 rounded-full border text-sm", children: "Audio Engineer" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "grid md:grid-cols-2 gap-8 mt-8", children: [
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcQOeZjZWEr4oFmJhILQQgTy7-WUX9BmRrAAFw&s",
            alt: "Daniel Anders",
            className: "rounded-xl shadow-lg w-full h-auto"
          }
        ),
        /* @__PURE__ */ jsxs("section", { className: "mt-8", children: [
          /* @__PURE__ */ jsx("h2", { className: "text-xl font-bold mb-2", children: "ABOUT" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-700 leading-relaxed", children: "Bali is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here." })
        ] }),
        /* @__PURE__ */ jsxs("section", { className: "mt-6", children: [
          /* @__PURE__ */ jsx("h2", { className: "text-xl font-bold mb-2", children: "LANGUAGE SPOKEN" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-700", children: "English" })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "border rounded-xl p-6 shadow-md h-fit flex flex-col gap-4", children: [
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium", children: "Start" }),
          /* @__PURE__ */ jsx(
            "input",
            {
              type: "datetime-local",
              value: start,
              onChange: (e) => setStart(e.target.value),
              className: "w-full border rounded-md p-2 mt-1"
            }
          )
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium", children: "End" }),
          /* @__PURE__ */ jsx(
            "input",
            {
              type: "datetime-local",
              value: end,
              onChange: (e) => setEnd(e.target.value),
              className: "w-full border rounded-md p-2 mt-1"
            }
          )
        ] }),
        /* @__PURE__ */ jsxs("p", { className: "text-sm text-gray-600", children: [
          "Your reservation:",
          " ",
          duration > 0 ? `${duration}h • From: ${points} points` : "Invalid"
        ] }),
        /* @__PURE__ */ jsx(
          "button",
          {
            onClick: handleBooking,
            className: "bg-black text-white py-3 rounded-md hover:bg-gray-800 transition",
            children: "BOOK NOW →"
          }
        )
      ] })
    ] })
  ] });
}
const route11 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: DanielAnders$1
}, Symbol.toStringTag, { value: "Module" }));
const ResetPassword = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [formData, setFormData] = useState({
    email: ""
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(false);
  useEffect(() => {
    setIsVisible(true);
  }, []);
  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };
  const handleSubmit = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    try {
      const response = await authAPI.forgotPassword(formData.email);
      if (response.status) {
        navigate("/reset-password-otp", {
          state: {
            email: formData.email,
            mobile_number: response.mobile_number || "",
            requestType: "reset-password"
          }
        });
      } else {
        setError(response.error || t("auth.resetPassword.errors.sendFailed", "Failed to send OTP. Please try again."));
      }
    } catch (err) {
      console.error("Forgot password error:", err);
      setError(err instanceof Error ? err.message : t("auth.resetPassword.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx("div", { className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60", style: { animationDelay: "0s", animationDuration: "3s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40", style: { animationDelay: "1s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30", style: { animationDelay: "2s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50", style: { animationDelay: "1.5s", animationDuration: "4s" } })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs("div", { className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: [
        /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.resetPassword.title", "RESET PASSWORD") }),
        /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("auth.resetPassword.subtitle", "Enter your email to receive OTP on your mobile") })
      ] }),
      /* @__PURE__ */ jsx("div", { className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
        /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider", children: t("auth.resetPassword.cardTitle", "Reset Password") }),
        success && /* @__PURE__ */ jsxs("div", { className: "mb-6 p-4 bg-green-500/20 border border-green-500/30 text-green-200 rounded-xl text-sm backdrop-blur-sm text-center", children: [
          /* @__PURE__ */ jsx("div", { className: "mb-2", children: "✅ OTP sent to your mobile!" }),
          /* @__PURE__ */ jsx("div", { children: "Check your mobile for the verification code." })
        ] }),
        error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
        !success ? (
          /* Reset Password Form */
          /* @__PURE__ */ jsxs("form", { onSubmit: handleSubmit, className: "space-y-6", children: [
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.resetPassword.emailAddress", "Email Address") }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "email",
                  name: "email",
                  placeholder: t("auth.resetPassword.emailPlaceholder", "Please enter your email"),
                  value: formData.email,
                  onChange: handleInputChange,
                  className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading
                }
              )
            ] }),
            /* @__PURE__ */ jsx(
              Button,
              {
                type: "submit",
                variant: "fill-white",
                size: "lg",
                className: "w-full",
                disabled: isLoading,
                arrowRight: !isLoading,
                children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                  /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                  t("auth.resetPassword.sending", "Sending...")
                ] }) : t("auth.resetPassword.sendOTP", "Send OTP to Mobile")
              }
            ),
            /* @__PURE__ */ jsx("div", { className: "text-center pt-4", children: /* @__PURE__ */ jsx(
              Button,
              {
                type: "button",
                variant: "outline-white",
                size: "sm",
                onClick: () => navigate("/login"),
                children: t("auth.resetPassword.backToLogin", "Back to Login")
              }
            ) })
          ] })
        ) : (
          /* Success State */
          /* @__PURE__ */ jsxs("div", { className: "text-center space-y-6", children: [
            /* @__PURE__ */ jsx("div", { className: "text-6xl mb-4", children: "📧" }),
            /* @__PURE__ */ jsx("p", { className: "text-white/80 mb-6", children: "We've sent password reset instructions to your email address." }),
            /* @__PURE__ */ jsx(
              Button,
              {
                variant: "fill-white",
                size: "lg",
                className: "w-full",
                onClick: () => navigate("/login"),
                arrowRight: true,
                children: "Back to Login"
              }
            )
          ] })
        )
      ] }) })
    ] })
  ] });
};
const meta$e = () => {
  return [
    { title: "Reset Password - Nomadic Studios" },
    { name: "description", content: "Reset your password for Nomadic Studios" }
  ];
};
function ResetPasswordPage() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(ResetPassword, {}) });
}
const route12 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: ResetPasswordPage,
  meta: meta$e
}, Symbol.toStringTag, { value: "Module" }));
const SignIn = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [phoneFormData, setPhoneFormData] = useState({
    mobileNumber: "",
    keepLoggedIn: true
  });
  const [selectedCountry, setSelectedCountry] = useState({
    code: "+1",
    country: "United States",
    flag: "🇺🇸"
  });
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [successMessage, setSuccessMessage] = useState(null);
  const countryCodes = [
    { code: "+61", country: "Australia", flag: "🇦🇺" },
    { code: "+43", country: "Austria", flag: "🇦🇹" },
    { code: "+54", country: "Argentina", flag: "🇦🇷" },
    { code: "+32", country: "Belgium", flag: "🇧🇪" },
    { code: "+359", country: "Bulgaria", flag: "🇧🇬" },
    { code: "+55", country: "Brazil", flag: "🇧🇷" },
    { code: "+56", country: "Chile", flag: "🇨🇱" },
    { code: "+86", country: "China", flag: "🇨🇳" },
    { code: "+420", country: "Czech Republic", flag: "🇨🇿" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+20", country: "Egypt", flag: "🇪🇬" },
    { code: "+33", country: "France", flag: "🇫🇷" },
    { code: "+358", country: "Finland", flag: "🇫🇮" },
    { code: "+30", country: "Greece", flag: "🇬🇷" },
    { code: "+49", country: "Germany", flag: "🇩🇪" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+972", country: "Israel", flag: "🇮🇱" },
    { code: "+62", country: "Indonesia", flag: "🇮🇩" },
    { code: "+81", country: "Japan", flag: "🇯🇵" },
    { code: "+254", country: "Kenya", flag: "🇰🇪" },
    { code: "+82", country: "South Korea", flag: "🇰🇷" },
    { code: "+36", country: "Hungary", flag: "🇭🇺" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+39", country: "Italy", flag: "🇮🇹" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+234", country: "Nigeria", flag: "🇳🇬" },
    { code: "+31", country: "Netherlands", flag: "🇳🇱" },
    { code: "+64", country: "New Zealand", flag: "🇳🇿" },
    { code: "+351", country: "Portugal", flag: "🇵🇹" },
    { code: "+48", country: "Poland", flag: "🇵🇱" },
    { code: "+51", country: "Peru", flag: "🇵🇪" },
    { code: "+63", country: "Philippines", flag: "🇵🇭" },
    { code: "+966", country: "Saudi Arabia", flag: "🇸🇦" },
    { code: "+65", country: "Singapore", flag: "🇸🇬" },
    { code: "+46", country: "Sweden", flag: "🇸🇪" },
    { code: "+41", country: "Switzerland", flag: "🇨🇭" },
    { code: "+90", country: "Turkey", flag: "🇹🇷" },
    { code: "+66", country: "Thailand", flag: "🇹🇭" },
    { code: "+27", country: "South Africa", flag: "🇿🇦" },
    { code: "+44", country: "United Kingdom", flag: "🇬🇧" },
    { code: "+1", country: "United States", flag: "🇺🇸" },
    { code: "+971", country: "UAE", flag: "🇦🇪" },
    { code: "+380", country: "Ukraine", flag: "🇺🇦" },
    { code: "+34", country: "Spain", flag: "🇪🇸" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+84", country: "Vietnam", flag: "🇻🇳" },
    { code: "+58", country: "Venezuela", flag: "🇻🇪" },
    { code: "+52", country: "Mexico", flag: "🇲🇽" }
  ];
  useEffect(() => {
    var _a, _b;
    setIsVisible(true);
    if (((_a = location.state) == null ? void 0 : _a.message) && ((_b = location.state) == null ? void 0 : _b.type) === "success") {
      setSuccessMessage(location.state.message);
      window.history.replaceState({}, document.title);
    }
  }, [location.state]);
  useEffect(() => {
    const handleKeyPress = (e) => {
      if (isDropdownOpen) {
        const key = e.key.toLowerCase();
        const matchingCountry = countryCodes.find(
          (country) => country.country.toLowerCase().startsWith(key)
        );
        if (matchingCountry) {
          const countryElement = document.querySelector(
            `[data-country="${matchingCountry.country}"]`
          );
          if (countryElement) {
            countryElement.scrollIntoView({ block: "nearest" });
            countryElement.focus();
          }
        }
      }
    };
    if (isDropdownOpen) {
      document.addEventListener("keydown", handleKeyPress);
    }
    return () => {
      document.removeEventListener("keydown", handleKeyPress);
    };
  }, [isDropdownOpen]);
  const handlePhoneInputChange = (e) => {
    const { name, value, type, checked } = e.target;
    setPhoneFormData((prev) => ({
      ...prev,
      [name]: type === "checkbox" ? checked : value
    }));
    if (error) setError(null);
  };
  const handleCountrySelect = (country) => {
    setSelectedCountry(country);
    setIsDropdownOpen(false);
  };
  const handleSendOTP = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    try {
      const fullNumber = `${selectedCountry.code}${phoneFormData.mobileNumber}`;
      const response = await authAPI.sendOTP(fullNumber);
      if (response.status) {
        navigate("/otp-verification", {
          state: {
            phoneNumber: fullNumber,
            requestType: "login"
          }
        });
      } else {
        setError(response.error || t("auth.signin.errors.sendOTPFailed", "Failed to send OTP. Please try again."));
      }
    } catch (err) {
      console.error("Send OTP error:", err);
      setError(
        err instanceof Error ? err.message : t("auth.signin.errors.unexpected", "An unexpected error occurred. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60",
          style: { animationDelay: "0s", animationDuration: "3s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40",
          style: { animationDelay: "1s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30",
          style: { animationDelay: "2s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50",
          style: { animationDelay: "1.5s", animationDuration: "4s" }
        }
      )
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs(
        "div",
        {
          className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: [
            /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.signin.welcomeBack", "WELCOME BACK") }),
            /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("auth.signin.subtitle", "Continue your creative journey") })
          ]
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
            /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider", children: t("auth.signin.title", "Sign In") }),
            successMessage && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-green-500/20 border border-green-500/30 text-green-200 rounded-xl text-sm backdrop-blur-sm", children: successMessage }),
            error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
            /* @__PURE__ */ jsxs("form", { onSubmit: handleSendOTP, className: "space-y-6", children: [
              /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
                /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signin.mobileNumber", "Mobile Number") }),
                /* @__PURE__ */ jsxs("div", { className: "flex relative", children: [
                  /* @__PURE__ */ jsxs("div", { className: "relative", children: [
                    /* @__PURE__ */ jsxs(
                      "button",
                      {
                        type: "button",
                        onClick: () => setIsDropdownOpen(!isDropdownOpen),
                        className: "flex items-center px-4 py-4 bg-white/10 border border-white/20 border-r-0 hover:bg-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 backdrop-blur-sm transition-all",
                        children: [
                          /* @__PURE__ */ jsx("span", { className: "text-lg mr-2", children: selectedCountry.flag }),
                          /* @__PURE__ */ jsx("span", { className: "text-sm text-white/80 mr-1", children: selectedCountry.code }),
                          /* @__PURE__ */ jsx(
                            "svg",
                            {
                              className: "w-4 h-4 text-white/60",
                              fill: "none",
                              stroke: "currentColor",
                              viewBox: "0 0 24 24",
                              children: /* @__PURE__ */ jsx(
                                "path",
                                {
                                  strokeLinecap: "round",
                                  strokeLinejoin: "round",
                                  strokeWidth: 2,
                                  d: "M19 9l-7 7-7-7"
                                }
                              )
                            }
                          )
                        ]
                      }
                    ),
                    isDropdownOpen && /* @__PURE__ */ jsx("div", { className: "absolute z-20 top-full left-0 w-64 bg-black border border-white/20 shadow-2xl max-h-60 overflow-y-auto", children: countryCodes.map((country, index) => /* @__PURE__ */ jsxs(
                      "button",
                      {
                        type: "button",
                        onClick: () => handleCountrySelect(country),
                        "data-country": country.country,
                        className: "w-full flex items-center px-4 py-3 text-left hover:bg-white/20 focus:outline-none focus:bg-white/20 transition-colors",
                        children: [
                          /* @__PURE__ */ jsx("span", { className: "text-lg mr-3", children: country.flag }),
                          /* @__PURE__ */ jsx("span", { className: "text-sm text-white/80 mr-2", children: country.code }),
                          /* @__PURE__ */ jsx("span", { className: "text-sm text-white truncate", children: country.country })
                        ]
                      },
                      index
                    )) })
                  ] }),
                  /* @__PURE__ */ jsx(
                    "input",
                    {
                      type: "tel",
                      name: "mobileNumber",
                      placeholder: t("auth.signin.phonePlaceholder", "Enter your mobile number"),
                      value: phoneFormData.mobileNumber,
                      onChange: handlePhoneInputChange,
                      className: "flex-1 px-4 py-4 bg-white/10 border border-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                      required: true
                    }
                  )
                ] }),
                isDropdownOpen && /* @__PURE__ */ jsx(
                  "div",
                  {
                    className: "fixed inset-0 z-10",
                    onClick: () => setIsDropdownOpen(false)
                  }
                )
              ] }),
              /* @__PURE__ */ jsx("div", { className: "flex items-center justify-between", children: /* @__PURE__ */ jsxs("div", { className: "flex items-center", children: [
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "checkbox",
                    id: "keepLoggedInPhone",
                    name: "keepLoggedIn",
                    checked: phoneFormData.keepLoggedIn,
                    onChange: handlePhoneInputChange,
                    className: "w-5 h-5 text-orange-500 bg-white/10 border-white/20 rounded focus:ring-orange-400 focus:ring-2"
                  }
                ),
                /* @__PURE__ */ jsx(
                  "label",
                  {
                    htmlFor: "keepLoggedInPhone",
                    className: "ml-3 text-sm text-white/80",
                    children: t("auth.signin.keepLoggedIn", "Keep me logged in")
                  }
                )
              ] }) }),
              /* @__PURE__ */ jsx(
                Button,
                {
                  type: "submit",
                  variant: "fill-white",
                  size: "lg",
                  className: "w-full",
                  disabled: isLoading,
                  arrowRight: !isLoading,
                  children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                    /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                    t("auth.signin.sendingOTP", "Sending OTP...")
                  ] }) : t("auth.signin.sendOTP", "Send OTP")
                }
              ),
              /* @__PURE__ */ jsxs("div", { className: "text-center space-y-4 pt-4", children: [
                /* @__PURE__ */ jsx(
                  Button,
                  {
                    type: "button",
                    variant: "outline-white",
                    size: "md",
                    className: "w-full",
                    onClick: () => navigate("/login-email"),
                    children: t("auth.signin.loginWithEmail", "Login with email instead")
                  }
                ),
                /* @__PURE__ */ jsx(
                  Button,
                  {
                    type: "button",
                    variant: "outline-white",
                    size: "md",
                    className: "w-full",
                    onClick: () => navigate("/reset-password"),
                    children: t("auth.signin.forgotPassword", "Forgot Password?")
                  }
                ),
                /* @__PURE__ */ jsxs("div", { className: "text-white/60 text-sm", children: [
                  t("auth.signin.noAccount", "Don't have an account?"),
                  " ",
                  /* @__PURE__ */ jsx(
                    Button,
                    {
                      type: "button",
                      variant: "outline-white",
                      size: "sm",
                      className: "ml-2",
                      onClick: () => navigate("/signup"),
                      children: t("auth.signin.signUp", "Sign up")
                    }
                  )
                ] })
              ] })
            ] })
          ] })
        }
      )
    ] })
  ] });
};
const SignInEmail = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [emailFormData, setEmailFormData] = useState({
    email: "",
    password: "",
    keepLoggedIn: true
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  useEffect(() => {
    setIsVisible(true);
  }, []);
  const handleEmailInputChange = (e) => {
    const { name, value, type, checked } = e.target;
    setEmailFormData((prev) => ({
      ...prev,
      [name]: type === "checkbox" ? checked : value
    }));
    if (error) setError(null);
  };
  const handleEmailSubmit = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    try {
      const response = await authAPI.login(
        emailFormData.email,
        emailFormData.password
      );
      if (response.status && response.token) {
        handleAuthSuccess(response);
        setTimeout(() => {
          window.location.href = "/home";
        }, 100);
      } else {
        setError(response.error || t("auth.signinemail.errors.loginFailed", "Login failed. Please try again."));
      }
    } catch (err) {
      console.error("Login error:", err);
      setError(
        err instanceof Error ? err.message : t("auth.signinemail.errors.unexpected", "An unexpected error occurred. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60",
          style: { animationDelay: "0s", animationDuration: "3s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40",
          style: { animationDelay: "1s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30",
          style: { animationDelay: "2s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50",
          style: { animationDelay: "1.5s", animationDuration: "4s" }
        }
      )
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0", children: [
      /* @__PURE__ */ jsxs(
        "div",
        {
          className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: [
            /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.signinemail.title", "SIGN IN") }),
            /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("auth.signinemail.subtitle", "Access your creative workspace") })
          ]
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: `w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
            /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider", children: t("auth.signinemail.emailLogin", "Email Login") }),
            error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
            /* @__PURE__ */ jsxs("form", { onSubmit: handleEmailSubmit, className: "space-y-6", children: [
              /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
                /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signinemail.emailAddress", "Email Address") }),
                /* @__PURE__ */ jsx(
                  "input",
                  {
                    type: "email",
                    name: "email",
                    placeholder: t("auth.signinemail.emailPlaceholder", "Enter your email"),
                    value: emailFormData.email,
                    onChange: handleEmailInputChange,
                    className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                    required: true,
                    disabled: isLoading
                  }
                )
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
                /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signinemail.password", "Password") }),
                /* @__PURE__ */ jsxs("div", { className: "relative", children: [
                  /* @__PURE__ */ jsx(
                    "input",
                    {
                      type: showPassword ? "text" : "password",
                      name: "password",
                      placeholder: t("auth.signinemail.passwordPlaceholder", "Enter your password"),
                      value: emailFormData.password,
                      onChange: handleEmailInputChange,
                      className: "w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                      required: true,
                      disabled: isLoading
                    }
                  ),
                  /* @__PURE__ */ jsx(
                    "button",
                    {
                      type: "button",
                      onClick: () => setShowPassword(!showPassword),
                      className: "absolute right-4 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors",
                      disabled: isLoading,
                      children: showPassword ? /* @__PURE__ */ jsx(
                        "svg",
                        {
                          className: "w-5 h-5",
                          fill: "none",
                          stroke: "currentColor",
                          viewBox: "0 0 24 24",
                          children: /* @__PURE__ */ jsx(
                            "path",
                            {
                              strokeLinecap: "round",
                              strokeLinejoin: "round",
                              strokeWidth: 2,
                              d: "M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                            }
                          )
                        }
                      ) : /* @__PURE__ */ jsxs(
                        "svg",
                        {
                          className: "w-5 h-5",
                          fill: "none",
                          stroke: "currentColor",
                          viewBox: "0 0 24 24",
                          children: [
                            /* @__PURE__ */ jsx(
                              "path",
                              {
                                strokeLinecap: "round",
                                strokeLinejoin: "round",
                                strokeWidth: 2,
                                d: "M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                              }
                            ),
                            /* @__PURE__ */ jsx(
                              "path",
                              {
                                strokeLinecap: "round",
                                strokeLinejoin: "round",
                                strokeWidth: 2,
                                d: "M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                              }
                            )
                          ]
                        }
                      )
                    }
                  )
                ] })
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between", children: [
                /* @__PURE__ */ jsxs("div", { className: "flex items-center", children: [
                  /* @__PURE__ */ jsx(
                    "input",
                    {
                      type: "checkbox",
                      id: "keepLoggedInEmail",
                      name: "keepLoggedIn",
                      checked: emailFormData.keepLoggedIn,
                      onChange: handleEmailInputChange,
                      className: "w-5 h-5 text-orange-500 bg-white/10 border-white/20 rounded focus:ring-orange-400 focus:ring-2"
                    }
                  ),
                  /* @__PURE__ */ jsx(
                    "label",
                    {
                      htmlFor: "keepLoggedInEmail",
                      className: "ml-3 text-sm text-white/80",
                      children: t("auth.signinemail.keepLoggedIn", "Keep me logged in")
                    }
                  )
                ] }),
                /* @__PURE__ */ jsx(
                  Button,
                  {
                    type: "button",
                    variant: "outline-white",
                    size: "sm",
                    onClick: () => navigate("/reset-password"),
                    children: t("auth.signinemail.forgotPassword", "Forgot Password?")
                  }
                )
              ] }),
              "              ",
              /* @__PURE__ */ jsx(
                Button,
                {
                  type: "submit",
                  variant: "fill-white",
                  size: "lg",
                  className: "w-full",
                  disabled: isLoading,
                  arrowRight: !isLoading,
                  children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                    /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                    t("auth.signinemail.signingIn", "Signing in...")
                  ] }) : t("auth.signinemail.signIn", "Sign In")
                }
              ),
              /* @__PURE__ */ jsxs("div", { className: "text-center space-y-4 pt-4", children: [
                /* @__PURE__ */ jsx(
                  Button,
                  {
                    type: "button",
                    variant: "outline-white",
                    size: "md",
                    className: "w-full",
                    onClick: () => navigate("/login"),
                    children: t("auth.signinemail.loginWithPhone", "Login with phone instead")
                  }
                ),
                /* @__PURE__ */ jsxs("div", { className: "text-white/60 text-sm", children: [
                  t("auth.signinemail.noAccount", "Don't have an account?"),
                  " ",
                  /* @__PURE__ */ jsx(
                    Button,
                    {
                      type: "button",
                      variant: "outline-white",
                      size: "sm",
                      className: "ml-2",
                      onClick: () => navigate("/signup"),
                      children: t("auth.signinemail.signUp", "Sign up")
                    }
                  )
                ] })
              ] })
            ] })
          ] })
        }
      )
    ] })
  ] });
};
const SignupFirst = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const [isVisible, setIsVisible] = useState(false);
  const [selectedTier, setSelectedTier] = useState(null);
  const [selectedAddOns, setSelectedAddOns] = useState([]);
  useEffect(() => {
    setIsVisible(true);
  }, []);
  const membershipTiers = [
    {
      name: t("auth.signup.plans.guest.name", "Guest"),
      price: "$0",
      period: t("auth.signup.plans.period", "month"),
      points: "0 Pts",
      features: [
        t("auth.signup.plans.guest.feature1", "Just one day/one by member"),
        t("auth.signup.plans.guest.feature2", "Zero guests")
      ],
      buttonText: "Choose plan"
    },
    {
      name: t("auth.signup.plans.starvingArtist.name", "Starving Artist"),
      price: "$1149",
      period: t("auth.signup.plans.period", "month"),
      points: "10000 Pts",
      features: [
        t("auth.signup.plans.starvingArtist.feature1", "Book up 1 month in advance"),
        t("auth.signup.plans.starvingArtist.feature2", "One guests")
      ],
      buttonText: "Choose plan"
    },
    {
      name: t("auth.signup.plans.upAndComing.name", "Up and coming"),
      price: "$1899",
      period: t("auth.signup.plans.period", "month"),
      points: "17500 Pts",
      features: [
        t("auth.signup.plans.upAndComing.feature1", "Book up 1 month in advance"),
        t("auth.signup.plans.upAndComing.feature2", "Two guests")
      ],
      buttonText: "Choose plan",
      isPopular: true
    },
    {
      name: t("auth.signup.plans.almostFamous.name", "Almost Famous"),
      price: "$2500",
      period: t("auth.signup.plans.period", "month"),
      points: "2500 Pts",
      features: [
        t("auth.signup.plans.almostFamous.feature1", "Book up 3 month in advance"),
        t("auth.signup.plans.almostFamous.feature2", "Four guests")
      ],
      buttonText: "Choose plan"
    }
  ];
  const addOnServices = [
    {
      id: "test-service",
      name: t("auth.signup.addons.testService.name"),
      price: "$20.00",
      description: t("auth.signup.addons.testService.description")
    },
    {
      id: "asdfds",
      name: t("auth.signup.addons.premiumSupport.name"),
      price: "$10.00",
      description: t("auth.signup.addons.premiumSupport.description")
    }
  ];
  const handleAddOnToggle = (serviceId) => {
    setSelectedAddOns(
      (prev) => prev.includes(serviceId) ? prev.filter((id) => id !== serviceId) : [...prev, serviceId]
    );
  };
  const handleTierSelect = (index) => {
    setSelectedTier(index);
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-30"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/90 via-[#030303]/70 to-[#030303]/90" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60",
          style: { animationDelay: "0s", animationDuration: "3s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40",
          style: { animationDelay: "1s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30",
          style: { animationDelay: "2s" }
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50",
          style: { animationDelay: "1.5s", animationDuration: "4s" }
        }
      )
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen px-4 lg:px-8 py-12", children: [
      /* @__PURE__ */ jsxs(
        "div",
        {
          className: `text-center mb-16 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: [
            /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.signup.joinCommunity", "JOIN THE COMMUNITY") }),
            /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light mb-8", children: t("auth.signup.subtitle", "Choose your creative journey") }),
            /* @__PURE__ */ jsxs("div", { className: "text-white/60 text-sm space-y-1", children: [
              /* @__PURE__ */ jsxs("p", { children: [
                "✨ ",
                t("auth.signup.benefits.relocation", "14 days relocation/year")
              ] }),
              /* @__PURE__ */ jsxs("p", { children: [
                "🎬 ",
                t("auth.signup.benefits.studio", "14 days Studio time/year")
              ] })
            ] })
          ]
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: `max-w-7xl mx-auto mb-16 transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6", children: membershipTiers.map((tier, index) => /* @__PURE__ */ jsxs(
            "div",
            {
              className: `relative bg-white/10 backdrop-blur-lg border rounded-2xl p-6 shadow-2xl transition-all duration-300 cursor-pointer transform hover:scale-105 ${tier.isPopular ? "border-orange-400 ring-2 ring-orange-400/50" : "border-white/20 hover:border-orange-400/50"} ${selectedTier === index ? "ring-2 ring-orange-400 scale-105" : ""}`,
              onClick: () => handleTierSelect(index),
              children: [
                tier.isPopular && /* @__PURE__ */ jsx("div", { className: "absolute -top-3 left-1/2 transform -translate-x-1/2", children: /* @__PURE__ */ jsx("div", { className: "bg-gradient-to-r from-orange-500 to-orange-600 text-white text-xs px-4 py-1 rounded-full uppercase tracking-wider font-semibold", children: t("auth.signup.mostPopular", "Most Popular") }) }),
                /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
                  /* @__PURE__ */ jsx("h3", { className: "text-white text-xl font-bold mb-4 uppercase tracking-wider", children: tier.name }),
                  /* @__PURE__ */ jsxs("div", { className: "mb-6", children: [
                    /* @__PURE__ */ jsx("div", { className: "text-4xl font-bold text-white mb-1", children: tier.price }),
                    /* @__PURE__ */ jsxs("div", { className: "text-white/60 text-sm", children: [
                      t("auth.signup.perPeriod", "per"),
                      " ",
                      tier.period
                    ] }),
                    /* @__PURE__ */ jsx("div", { className: "text-orange-400 font-semibold mt-2", children: tier.points })
                  ] }),
                  /* @__PURE__ */ jsx("ul", { className: "text-white/80 text-sm space-y-3 mb-8", children: tier.features.map((feature, featureIndex) => /* @__PURE__ */ jsxs("li", { className: "flex items-start", children: [
                    /* @__PURE__ */ jsx(
                      "svg",
                      {
                        className: "w-4 h-4 text-orange-400 mr-3 mt-0.5 flex-shrink-0",
                        fill: "currentColor",
                        viewBox: "0 0 20 20",
                        children: /* @__PURE__ */ jsx(
                          "path",
                          {
                            fillRule: "evenodd",
                            d: "M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z",
                            clipRule: "evenodd"
                          }
                        )
                      }
                    ),
                    feature
                  ] }, featureIndex)) }),
                  /* @__PURE__ */ jsx(
                    Button,
                    {
                      variant: selectedTier === index ? "fill-black" : tier.isPopular ? "fill-white" : "outline-white",
                      size: "md",
                      className: "w-full",
                      children: selectedTier === index ? /* @__PURE__ */ jsxs("span", { className: "flex items-center justify-center", children: [
                        /* @__PURE__ */ jsx(
                          "svg",
                          {
                            className: "w-5 h-5 mr-2",
                            fill: "currentColor",
                            viewBox: "0 0 20 20",
                            children: /* @__PURE__ */ jsx(
                              "path",
                              {
                                fillRule: "evenodd",
                                d: "M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z",
                                clipRule: "evenodd"
                              }
                            )
                          }
                        ),
                        t("auth.signup.selected", "Selected")
                      ] }) : t("auth.signup.choosePlan", "Choose plan")
                    }
                  )
                ] })
              ]
            },
            index
          )) })
        }
      ),
      /* @__PURE__ */ jsxs(
        "div",
        {
          className: `max-w-4xl mx-auto mb-16 transform transition-all duration-1000 delay-500 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: [
            /* @__PURE__ */ jsxs("div", { className: "text-center mb-8", children: [
              /* @__PURE__ */ jsx("h2", { className: "text-3xl lg:text-4xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.signup.addons.title", "Select Add-on Services") }),
              /* @__PURE__ */ jsx("p", { className: "text-white/60 text-sm", children: t("auth.signup.addons.subtitle", "Enhance your membership with optional premium services") })
            ] }),
            /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 md:grid-cols-2 gap-6", children: addOnServices.map((service) => /* @__PURE__ */ jsxs(
              "div",
              {
                className: `bg-white/10 backdrop-blur-lg border rounded-2xl p-6 shadow-2xl transition-all duration-300 cursor-pointer transform hover:scale-105 ${selectedAddOns.includes(service.id) ? "border-orange-400 ring-2 ring-orange-400/50 bg-orange-500/10" : "border-white/20 hover:border-orange-400/50"}`,
                onClick: () => handleAddOnToggle(service.id),
                children: [
                  /* @__PURE__ */ jsxs("div", { className: "flex items-start justify-between mb-4", children: [
                    /* @__PURE__ */ jsxs("div", { className: "flex-1", children: [
                      /* @__PURE__ */ jsx("h3", { className: "text-white text-xl font-bold mb-2 uppercase tracking-wider", children: service.name }),
                      /* @__PURE__ */ jsx("p", { className: "text-white/70 text-sm mb-4", children: service.description })
                    ] }),
                    /* @__PURE__ */ jsxs("div", { className: "ml-4", children: [
                      /* @__PURE__ */ jsx("div", { className: "text-2xl font-bold text-orange-400", children: service.price }),
                      /* @__PURE__ */ jsx("div", { className: "text-white/60 text-xs text-right", children: t("auth.signup.addons.perMonth", "per month") })
                    ] })
                  ] }),
                  /* @__PURE__ */ jsx("div", { className: "flex items-center justify-between", children: /* @__PURE__ */ jsxs("div", { className: "flex items-center", children: [
                    /* @__PURE__ */ jsx(
                      "div",
                      {
                        className: `w-5 h-5 rounded border-2 transition-all duration-200 ${selectedAddOns.includes(service.id) ? "border-orange-400 bg-orange-400" : "border-white/40"}`,
                        children: selectedAddOns.includes(service.id) && /* @__PURE__ */ jsx(
                          "svg",
                          {
                            className: "w-3 h-3 text-white ml-0.5 mt-0.5",
                            fill: "currentColor",
                            viewBox: "0 0 20 20",
                            children: /* @__PURE__ */ jsx(
                              "path",
                              {
                                fillRule: "evenodd",
                                d: "M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z",
                                clipRule: "evenodd"
                              }
                            )
                          }
                        )
                      }
                    ),
                    /* @__PURE__ */ jsx("span", { className: "text-white/80 text-sm ml-3", children: selectedAddOns.includes(service.id) ? t("auth.signup.addons.selected", "Selected") : t("auth.signup.addons.addToPlan", "Add to plan") })
                  ] }) })
                ]
              },
              service.id
            )) })
          ]
        }
      ),
      selectedTier !== null && /* @__PURE__ */ jsx(
        "div",
        {
          className: `text-center mb-16 transform transition-all duration-1000 delay-700 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsx(
            Button,
            {
              variant: "fill-white",
              size: "lg",
              arrowRight: true,
              onClick: () => {
                const selectedTierData = membershipTiers[selectedTier];
                navigate("/signup-details", {
                  state: {
                    selectedTier: selectedTierData,
                    selectedAddOns: selectedAddOns.map(
                      (id) => addOnServices.find((service) => service.id === id)
                    ).filter(Boolean)
                  }
                });
              },
              children: t("auth.signup.subscribe", "Subscribe")
            }
          )
        }
      ),
      /* @__PURE__ */ jsx(
        "div",
        {
          className: `text-center mt-16 transform transition-all duration-1000 delay-500 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`,
          children: /* @__PURE__ */ jsxs("div", { className: "text-white/60 text-sm mb-4", children: [
            t("auth.signup.haveAccount", "Already have an account?"),
            " ",
            /* @__PURE__ */ jsx(
              Button,
              {
                variant: "outline-white",
                size: "sm",
                className: "ml-2",
                onClick: () => navigate("/login"),
                children: t("auth.signup.signIn", "Sign in")
              }
            )
          ] })
        }
      )
    ] })
  ] });
};
const SignupSecond = () => {
  var _a, _b;
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [searchParams] = useSearchParams();
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const selectedTierData = (_a = location.state) == null ? void 0 : _a.selectedTier;
  const selectedAddOnsData = ((_b = location.state) == null ? void 0 : _b.selectedAddOns) || [];
  const selectedPlan = (selectedTierData == null ? void 0 : selectedTierData.name) || searchParams.get("plan") || "Guest";
  const selectedPrice = (selectedTierData == null ? void 0 : selectedTierData.price) || searchParams.get("price") || "$0";
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "",
    email: "",
    mobileNumber: ""
  });
  const [selectedCountry, setSelectedCountry] = useState({
    code: "+1",
    country: "United States",
    flag: "🇺🇸"
  });
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const countryCodes = [
    { code: "+1", country: "United States", flag: "🇺🇸" },
    { code: "+1", country: "Canada", flag: "🇨🇦" },
    { code: "+44", country: "United Kingdom", flag: "🇬🇧" },
    { code: "+49", country: "Germany", flag: "🇩🇪" },
    { code: "+33", country: "France", flag: "🇫🇷" },
    { code: "+39", country: "Italy", flag: "🇮🇹" },
    { code: "+34", country: "Spain", flag: "🇪🇸" },
    { code: "+31", country: "Netherlands", flag: "🇳🇱" },
    { code: "+41", country: "Switzerland", flag: "🇨🇭" },
    { code: "+43", country: "Austria", flag: "🇦🇹" },
    { code: "+32", country: "Belgium", flag: "🇧🇪" },
    { code: "+45", country: "Denmark", flag: "🇩🇰" },
    { code: "+46", country: "Sweden", flag: "🇸🇪" },
    { code: "+47", country: "Norway", flag: "🇳🇴" },
    { code: "+358", country: "Finland", flag: "🇫🇮" },
    { code: "+351", country: "Portugal", flag: "🇵🇹" },
    { code: "+30", country: "Greece", flag: "🇬🇷" },
    { code: "+353", country: "Ireland", flag: "🇮🇪" },
    { code: "+48", country: "Poland", flag: "🇵🇱" },
    { code: "+420", country: "Czech Republic", flag: "🇨🇿" },
    { code: "+36", country: "Hungary", flag: "🇭🇺" },
    { code: "+40", country: "Romania", flag: "🇷🇴" },
    { code: "+359", country: "Bulgaria", flag: "🇧🇬" },
    { code: "+385", country: "Croatia", flag: "🇭🇷" },
    { code: "+91", country: "India", flag: "🇮🇳" },
    { code: "+86", country: "China", flag: "🇨🇳" },
    { code: "+81", country: "Japan", flag: "🇯🇵" },
    { code: "+82", country: "South Korea", flag: "🇰🇷" },
    { code: "+65", country: "Singapore", flag: "🇸🇬" },
    { code: "+60", country: "Malaysia", flag: "🇲🇾" },
    { code: "+66", country: "Thailand", flag: "🇹🇭" },
    { code: "+84", country: "Vietnam", flag: "🇻🇳" },
    { code: "+63", country: "Philippines", flag: "🇵🇭" },
    { code: "+62", country: "Indonesia", flag: "🇮🇩" },
    { code: "+61", country: "Australia", flag: "🇦🇺" },
    { code: "+64", country: "New Zealand", flag: "🇳🇿" },
    { code: "+27", country: "South Africa", flag: "🇿🇦" },
    { code: "+20", country: "Egypt", flag: "🇪🇬" },
    { code: "+234", country: "Nigeria", flag: "🇳🇬" },
    { code: "+254", country: "Kenya", flag: "🇰🇪" },
    { code: "+55", country: "Brazil", flag: "🇧🇷" },
    { code: "+52", country: "Mexico", flag: "🇲🇽" },
    { code: "+54", country: "Argentina", flag: "🇦🇷" },
    { code: "+56", country: "Chile", flag: "🇨🇱" },
    { code: "+57", country: "Colombia", flag: "🇨🇴" },
    { code: "+51", country: "Peru", flag: "🇵🇪" },
    { code: "+58", country: "Venezuela", flag: "🇻🇪" },
    { code: "+7", country: "Russia", flag: "🇷🇺" },
    { code: "+380", country: "Ukraine", flag: "🇺🇦" },
    { code: "+90", country: "Turkey", flag: "🇹🇷" },
    { code: "+972", country: "Israel", flag: "🇮🇱" },
    { code: "+971", country: "UAE", flag: "🇦🇪" },
    { code: "+966", country: "Saudi Arabia", flag: "🇸🇦" }
  ];
  useEffect(() => {
    var _a2;
    setIsVisible(true);
    if (selectedPlan === "Guest" && !((_a2 = location.state) == null ? void 0 : _a2.selectedTier)) ;
  }, [selectedPlan, location.state]);
  useEffect(() => {
    const handleKeyPress = (e) => {
      if (isDropdownOpen) {
        const key = e.key.toLowerCase();
        const matchingCountry = countryCodes.find(
          (country) => country.country.toLowerCase().startsWith(key)
        );
        if (matchingCountry) {
          const countryElement = document.querySelector(
            `[data-country="${matchingCountry.country}"]`
          );
          if (countryElement) {
            countryElement.scrollIntoView({ block: "nearest" });
            countryElement.focus();
          }
        }
      }
    };
    if (isDropdownOpen) {
      document.addEventListener("keydown", handleKeyPress);
    }
    return () => {
      document.removeEventListener("keydown", handleKeyPress);
    };
  }, [isDropdownOpen]);
  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };
  const handleCountrySelect = (country) => {
    setSelectedCountry(country);
    setIsDropdownOpen(false);
    if (error) setError(null);
  };
  const handleSubmit = async (e) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);
    try {
      const fullPhoneNumber = `${selectedCountry.code}${formData.mobileNumber}`;
      const otpResponse = await authAPI.sendOTP(fullPhoneNumber);
      if (otpResponse.status) {
        navigate("/otp-verification", {
          state: {
            phoneNumber: fullPhoneNumber,
            requestType: "signup",
            signupData: {
              fname: formData.firstName,
              lname: formData.lastName,
              email: formData.email,
              mobile_number: fullPhoneNumber,
              plan: selectedPlan,
              addon_services: selectedAddOnsData.map((addon) => addon.id).join(",")
            }
          }
        });
      } else {
        setError(otpResponse.error || t("auth.signupSecond.errors.otpFailed", "Failed to send OTP. Please try again."));
      }
    } catch (error2) {
      console.error("Signup error:", error2);
      setError(error2 instanceof Error ? error2.message : t("auth.signupSecond.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-[#030303] relative overflow-hidden", children: [
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/welcome.png",
          alt: "",
          className: "absolute inset-0 h-full w-full object-cover opacity-40"
        }
      ),
      /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute inset-0 overflow-hidden pointer-events-none", children: [
      /* @__PURE__ */ jsx("div", { className: "absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60", style: { animationDelay: "0s", animationDuration: "3s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40", style: { animationDelay: "1s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30", style: { animationDelay: "2s" } }),
      /* @__PURE__ */ jsx("div", { className: "absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50", style: { animationDelay: "1.5s", animationDuration: "4s" } })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0 py-12", children: [
      /* @__PURE__ */ jsxs("div", { className: `text-center mb-12 transform transition-all duration-1000 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: [
        /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4", children: t("auth.signupSecond.title", "COMPLETE SIGNUP") }),
        /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-white/80 font-light", children: t("auth.signupSecond.subtitle", "You're one step away from joining the community") })
      ] }),
      /* @__PURE__ */ jsx("div", { className: `w-full max-w-lg transform transition-all duration-1000 delay-300 ${isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"}`, children: /* @__PURE__ */ jsxs("div", { className: "bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl", children: [
        /* @__PURE__ */ jsxs("div", { className: "mb-8 text-center", children: [
          /* @__PURE__ */ jsx("div", { className: "inline-flex items-center bg-orange-500/20 border border-orange-400/30 rounded-xl px-6 py-3 mb-4", children: /* @__PURE__ */ jsxs("div", { className: "text-orange-400 text-sm font-semibold uppercase tracking-wider", children: [
            t("auth.signupSecond.selectedPlan", "Selected Plan"),
            ": ",
            selectedPlan
          ] }) }),
          /* @__PURE__ */ jsxs("div", { className: "text-white/80 text-lg font-medium", children: [
            selectedPrice,
            "/",
            t("auth.signupSecond.month", "month")
          ] }),
          /* @__PURE__ */ jsxs(
            Link,
            {
              to: "/signup",
              className: "text-orange-400 hover:text-orange-300 text-sm font-medium transition-colors mt-2 inline-block",
              children: [
                t("auth.signupSecond.changePlan", "Change plan"),
                " →"
              ]
            }
          )
        ] }),
        error && /* @__PURE__ */ jsx("div", { className: "mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm", children: error }),
        /* @__PURE__ */ jsxs("form", { onSubmit: handleSubmit, className: "space-y-6", children: [
          /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-2 gap-4", children: [
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signupSecond.firstName", "First Name") }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "text",
                  name: "firstName",
                  placeholder: t("auth.signupSecond.firstNamePlaceholder", "John"),
                  value: formData.firstName,
                  onChange: handleInputChange,
                  className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading
                }
              )
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
              /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signupSecond.lastName", "Last Name") }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "text",
                  name: "lastName",
                  placeholder: t("auth.signupSecond.lastNamePlaceholder", "Doe"),
                  value: formData.lastName,
                  onChange: handleInputChange,
                  className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading
                }
              )
            ] })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
            /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("auth.signupSecond.emailAddress", "Email Address") }),
            /* @__PURE__ */ jsx(
              "input",
              {
                type: "email",
                name: "email",
                placeholder: t("auth.signupSecond.emailPlaceholder", "john.doe@example.com"),
                value: formData.email,
                onChange: handleInputChange,
                className: "w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                required: true,
                disabled: isLoading
              }
            )
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "space-y-2", children: [
            /* @__PURE__ */ jsx("label", { className: "text-white/80 text-sm font-medium uppercase tracking-wider", children: t("signupSecond.mobileNumber") }),
            /* @__PURE__ */ jsxs("div", { className: "flex relative", children: [
              /* @__PURE__ */ jsxs("div", { className: "relative", children: [
                /* @__PURE__ */ jsxs(
                  "button",
                  {
                    type: "button",
                    onClick: () => setIsDropdownOpen(!isDropdownOpen),
                    disabled: isLoading,
                    className: "flex items-center px-4 py-4 bg-white/10 border border-white/20 border-r-0 rounded-l-xl hover:bg-white/20 focus:outline-none focus:ring-2 focus:ring-orange-400 backdrop-blur-sm transition-all disabled:opacity-50",
                    children: [
                      /* @__PURE__ */ jsx("span", { className: "text-lg mr-2", children: selectedCountry.flag }),
                      /* @__PURE__ */ jsx("span", { className: "text-sm text-white/80 mr-1", children: selectedCountry.code }),
                      /* @__PURE__ */ jsx("svg", { className: "w-4 h-4 text-white/60", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M19 9l-7 7-7-7" }) })
                    ]
                  }
                ),
                isDropdownOpen && /* @__PURE__ */ jsx("div", { className: "absolute z-20 top-full left-0 w-64 bg-black border border-white/20 rounded-xl shadow-2xl max-h-60 overflow-y-auto", children: countryCodes.map((country, index) => /* @__PURE__ */ jsxs(
                  "button",
                  {
                    type: "button",
                    onClick: () => handleCountrySelect(country),
                    "data-country": country.country,
                    className: "w-full flex items-center px-4 py-3 text-left hover:bg-white/20 focus:outline-none focus:bg-white/20 transition-colors",
                    children: [
                      /* @__PURE__ */ jsx("span", { className: "text-lg mr-3", children: country.flag }),
                      /* @__PURE__ */ jsx("span", { className: "text-sm text-white/80 mr-2", children: country.code }),
                      /* @__PURE__ */ jsx("span", { className: "text-sm text-white truncate", children: country.country })
                    ]
                  },
                  index
                )) })
              ] }),
              /* @__PURE__ */ jsx(
                "input",
                {
                  type: "tel",
                  name: "mobileNumber",
                  placeholder: t("signupSecond.mobileNumberPlaceholder"),
                  value: formData.mobileNumber,
                  onChange: handleInputChange,
                  className: "flex-1 px-4 py-4 bg-white/10 border border-white/20 rounded-r-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all",
                  required: true,
                  disabled: isLoading
                }
              )
            ] }),
            isDropdownOpen && /* @__PURE__ */ jsx(
              "div",
              {
                className: "fixed inset-0 z-10",
                onClick: () => setIsDropdownOpen(false)
              }
            )
          ] }),
          /* @__PURE__ */ jsx("div", { className: "bg-white/5 rounded-xl p-4 border border-white/10", children: /* @__PURE__ */ jsxs("div", { className: "text-xs text-white/70 space-y-2", children: [
            /* @__PURE__ */ jsxs("p", { children: [
              t("signupSecond.termsAgreement"),
              " ",
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-orange-400 hover:text-orange-300 transition-colors", children: t("signupSecond.termsOfService") }),
              " ",
              t("common.and"),
              " ",
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-orange-400 hover:text-orange-300 transition-colors", children: t("signupSecond.privacyPolicy") }),
              "."
            ] }),
            /* @__PURE__ */ jsx("p", { children: t("signupSecond.smsVerification") })
          ] }) }),
          /* @__PURE__ */ jsx(
            Button,
            {
              type: "submit",
              variant: "fill-white",
              size: "lg",
              className: "w-full",
              disabled: isLoading,
              arrowRight: !isLoading,
              children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
                /* @__PURE__ */ jsx("div", { className: "animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3" }),
                t("signupSecond.creatingAccount")
              ] }) : t("signupSecond.completeSignup")
            }
          ),
          /* @__PURE__ */ jsxs("div", { className: "text-center text-white/60 text-sm pt-4", children: [
            t("signupSecond.alreadyHaveAccount"),
            " ",
            /* @__PURE__ */ jsx(
              Button,
              {
                type: "button",
                variant: "outline-white",
                size: "sm",
                className: "ml-2",
                onClick: () => navigate("/login"),
                children: t("common.signIn")
              }
            )
          ] })
        ] })
      ] }) })
    ] })
  ] });
};
const meta$d = () => {
  return [
    { title: "Complete Signup | Nomadics" },
    { name: "description", content: "Complete your registration" }
  ];
};
function SignupDetails() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(SignupSecond, {}) });
}
const route13 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: SignupDetails,
  meta: meta$d
}, Symbol.toStringTag, { value: "Module" }));
const meta$c = () => {
  return [
    { title: "Equipment | Nomadics" },
    { name: "description", content: "Professional equipment for rent" }
  ];
};
function Equipment$1() {
  return /* @__PURE__ */ jsxs("div", { className: "container mx-auto px-4 py-8", children: [
    /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold mb-6", children: "Equipment" }),
    /* @__PURE__ */ jsx("p", { className: "text-lg text-muted-foreground", children: "Browse our professional equipment available for rent." })
  ] });
}
const route14 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Equipment$1,
  meta: meta$c
}, Symbol.toStringTag, { value: "Module" }));
function DanielAnders() {
  const [start, setStart] = useState("2025-08-21T10:30");
  const [end, setEnd] = useState("2025-08-21T13:30");
  const duration = (new Date(end).getTime() - new Date(start).getTime()) / (1e3 * 60 * 60);
  const points = duration > 0 ? Math.round(duration * 43) : 0;
  const handleBooking = () => {
    alert(
      `Booking confirmed!
Start: ${start}
End: ${end}
Duration: ${duration}h
Points: ${points}`
    );
  };
  return /* @__PURE__ */ jsxs("div", { className: "min-h-screen bg-white text-black font-sans p-6 md:p-12", children: [
    /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold", children: "DANIEL ANDERS" }),
    /* @__PURE__ */ jsxs("div", { className: "flex gap-2 mt-2", children: [
      /* @__PURE__ */ jsx("span", { className: "px-3 py-1 rounded-full border text-sm", children: "Bali, Indonesia" }),
      /* @__PURE__ */ jsx("span", { className: "px-3 py-1 rounded-full border text-sm", children: "Audio Engineer" })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "grid md:grid-cols-2 gap-8 mt-8", children: [
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcQOeZjZWEr4oFmJhILQQgTy7-WUX9BmRrAAFw&s",
            alt: "Daniel Anders",
            className: "rounded-xl shadow-lg w-full h-auto"
          }
        ),
        /* @__PURE__ */ jsxs("section", { className: "mt-8", children: [
          /* @__PURE__ */ jsx("h2", { className: "text-xl font-bold mb-2", children: "ABOUT" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-700 leading-relaxed", children: "Bali is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here." })
        ] }),
        /* @__PURE__ */ jsxs("section", { className: "mt-6", children: [
          /* @__PURE__ */ jsx("h2", { className: "text-xl font-bold mb-2", children: "LANGUAGE SPOKEN" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-700", children: "English" })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "border rounded-xl p-6 shadow-md h-fit flex flex-col gap-4", children: [
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium", children: "Start" }),
          /* @__PURE__ */ jsx(
            "input",
            {
              type: "datetime-local",
              value: start,
              onChange: (e) => setStart(e.target.value),
              className: "w-full border rounded-md p-2 mt-1"
            }
          )
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium", children: "End" }),
          /* @__PURE__ */ jsx(
            "input",
            {
              type: "datetime-local",
              value: end,
              onChange: (e) => setEnd(e.target.value),
              className: "w-full border rounded-md p-2 mt-1"
            }
          )
        ] }),
        /* @__PURE__ */ jsxs("p", { className: "text-sm text-gray-600", children: [
          "Your reservation:",
          " ",
          duration > 0 ? `${duration}h • From: ${points} points` : "Invalid"
        ] }),
        /* @__PURE__ */ jsx(
          "button",
          {
            onClick: handleBooking,
            className: "bg-black text-white py-3 rounded-md hover:bg-gray-800 transition",
            children: "BOOK NOW →"
          }
        )
      ] })
    ] })
  ] });
}
const route15 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: DanielAnders
}, Symbol.toStringTag, { value: "Module" }));
function PhotoSection({ data }) {
  return /* @__PURE__ */ jsx("section", { className: "w-full px-4 lg:px-14 py-12 lg:py-24 bg-gradient-to-b from-white to-[#F5F4F0]", children: /* @__PURE__ */ jsxs("div", { className: "max-w-7xl mx-auto", children: [
    /* @__PURE__ */ jsxs("div", { className: "text-center mb-12", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-3xl lg:text-5xl font-medium uppercase tracking-tight text-[#121212] font-clash-grotesk mb-4", children: data.title }),
      /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-[rgba(18,18,18,0.7)] max-w-3xl mx-auto", children: data.description })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "grid gap-6", children: [
      /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 lg:grid-cols-2 gap-6", children: [
        /* @__PURE__ */ jsxs("div", { className: "group relative h-[300px] lg:h-[400px] overflow-hidden rounded-xl shadow-lg", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: data.image1,
              alt: "Video production studio",
              className: "w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-6", children: /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("h3", { className: "text-white text-xl font-medium uppercase", children: data.title1 }),
            /* @__PURE__ */ jsx("p", { className: "text-white/80 text-sm mt-1", children: data.subtitle1 })
          ] }) })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "group relative h-[300px] lg:h-[400px] overflow-hidden rounded-xl shadow-lg", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: data.image2,
              alt: "Music recording studio",
              className: "w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-6", children: /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("h3", { className: "text-white text-xl font-medium uppercase", children: data.title2 }),
            /* @__PURE__ */ jsx("p", { className: "text-white/80 text-sm mt-1", children: data.subtitle2 })
          ] }) })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 md:grid-cols-3 gap-6", children: [
        /* @__PURE__ */ jsxs("div", { className: "group relative h-[250px] overflow-hidden rounded-xl shadow-lg", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: data.image3,
              alt: "Photography studio",
              className: "w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-4", children: /* @__PURE__ */ jsx("h3", { className: "text-white text-lg font-medium uppercase", children: data.title3 }) })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "group relative h-[250px] overflow-hidden rounded-xl shadow-lg", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: data.image4,
              alt: "Creative workspace",
              className: "w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-4", children: /* @__PURE__ */ jsx("h3", { className: "text-white text-lg font-medium uppercase", children: data.title4 }) })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "group relative h-[250px] overflow-hidden rounded-xl shadow-lg", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: data.image5,
              alt: "Production suite",
              className: "w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
            }
          ),
          /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-4", children: /* @__PURE__ */ jsx("h3", { className: "text-white text-lg font-medium uppercase", children: data.title5 }) })
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "group relative h-[250px] lg:h-[350px] overflow-hidden rounded-xl shadow-lg", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: data.image6,
            alt: "Multi-purpose studio space",
            className: "w-full h-full object-cover group-hover:scale-105 transition-transform duration-700"
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500 flex items-end p-6", children: /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "text-white text-2xl font-medium uppercase", children: data.title6 }),
          /* @__PURE__ */ jsx("p", { className: "text-white/80 text-base mt-2", children: data.subtitle6 })
        ] }) })
      ] })
    ] })
  ] }) });
}
function StudioRow({ item }) {
  return /* @__PURE__ */ jsx("div", { className: "flex flex-col lg:flex-row lg:justify-end lg:items-center w-full  border-t border-[rgba(18,18,18,0.2)]", children: item.imagePosition === "right" ? /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx("div", { className: "lg:hidden w-full p-4", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: item.image,
        alt: item.title,
        className: "w-full h-full object-cover"
      }
    ) }),
    /* @__PURE__ */ jsxs("div", { className: "flex flex-col items-start px-4 lg:px-14 py-8 lg:py-0 gap-4 lg:flex-1 lg:self-center", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-[24px] lg:text-[40px] leading-[32px] lg:leading-[48px] font-medium uppercase tracking-[-0.01em] text-[#121212] font-clash-grotesk", children: item.title }),
      /* @__PURE__ */ jsx("p", { className: "text-[16px] lg:text-[24px] leading-[24px] lg:leading-[32px] font-normal text-[rgba(18,18,18,0.8)] font-clash-grotesk", children: item.description })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "hidden lg:flex flex-row items-center px-14 gap-2 w-[612px] h-[551px] border-l border-[rgba(18,18,18,0.2)]", children: /* @__PURE__ */ jsx("div", { className: "flex flex-col items-start p-10 gap-2 w-[500px] h-[375px] flex-1", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: item.image,
        alt: item.title,
        className: "w-full h-full object-cover"
      }
    ) }) })
  ] }) : /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx("div", { className: "lg:hidden w-full h-[300px] p-4", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: item.image,
        alt: item.title,
        className: "w-full h-full object-cover"
      }
    ) }),
    /* @__PURE__ */ jsx("div", { className: "hidden lg:flex flex-row items-center px-14 gap-2 w-[612px] h-[551px] border-r border-[rgba(18,18,18,0.2)]", children: /* @__PURE__ */ jsx("div", { className: "flex flex-col items-start p-10 gap-2 w-[500px] h-[375px] flex-1", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: item.image,
        alt: item.title,
        className: "w-full h-full object-cover"
      }
    ) }) }),
    /* @__PURE__ */ jsxs("div", { className: "flex flex-col items-start px-4 lg:px-14 py-8 lg:py-0 gap-4 lg:flex-1  lg:self-center", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-[24px] lg:text-[40px] leading-[32px] lg:leading-[48px] font-medium uppercase tracking-[-0.01em] text-[#121212] font-clash-grotesk", children: item.title }),
      /* @__PURE__ */ jsx("p", { className: "text-[16px] lg:text-[24px] leading-[24px] lg:leading-[32px] font-normal text-[rgba(18,18,18,0.8)] font-clash-grotesk", children: item.description })
    ] })
  ] }) });
}
function StudiosSection$1({ data }) {
  return /* @__PURE__ */ jsx("section", { id: "studios", className: "w-full bg-white", children: data.map((item, index) => /* @__PURE__ */ jsx(StudioRow, { item }, index)) });
}
function WelcomeSection$1({ data }) {
  const heroActions2 = [
    {
      label: data.ctaLabel,
      href: data.ctaHref,
      variant: "fill-white",
      icon: true
    }
  ];
  return /* @__PURE__ */ jsx(
    HeroSection$1,
    {
      backgroundImage: data.backgroundImage,
      title: data.title,
      subtitle: data.subtitle,
      children: /* @__PURE__ */ jsx(HeroActions, { actions: heroActions2 })
    }
  );
}
function BookSection({ title, description, data }) {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsxs("div", { className: "w-full py-12 sm:py-16 lg:py-24", children: [
    /* @__PURE__ */ jsxs("div", { className: "text-center mb-12 px-4 sm:px-6", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-3xl lg:text-5xl font-medium uppercase tracking-tight text-[#121212] font-clash-grotesk mb-4", children: title }),
      /* @__PURE__ */ jsx("p", { className: "text-lg lg:text-xl text-[rgba(18,18,18,0.7)] max-w-2xl mx-auto mb-8", children: description })
    ] }),
    /* @__PURE__ */ jsx(ImageGrid, { items: data.items })
  ] }) });
}
const translations = { en, sp };
async function loader({ params }) {
  const type = params.type;
  const lang = "en";
  const languageData = translations[lang];
  const studioData = languageData[type];
  if (!studioData) {
    throw new Response("Studio Not Found", { status: 404 });
  }
  const apiRes = await fetch(
    `https://nomadicstudios.net:444/api/studios?type=${type}`
  );
  const apiData = await apiRes.json();
  console.log(apiData);
  return Response.json({ type, studioData, apiData });
}
const meta$b = ({ data }) => {
  if (!data) {
    return [
      { title: "Studio Not Found | Nomadics" },
      {
        name: "description",
        content: "The requested studio type was not found."
      }
    ];
  }
  return [
    { title: data.studioData.meta.title },
    { name: "description", content: data.studioData.meta.description }
  ];
};
function StudioPage$1() {
  const { studioData, apiData } = useLoaderData();
  return /* @__PURE__ */ jsxs("div", { children: [
    /* @__PURE__ */ jsx(WelcomeSection$1, { data: studioData.welcome }),
    /* @__PURE__ */ jsx(StudiosSection$1, { data: studioData.studios }),
    /* @__PURE__ */ jsx(PhotoSection, { data: studioData.gallery }),
    /* @__PURE__ */ jsx(
      BookSection,
      {
        title: studioData.book.title,
        description: studioData.book.description,
        data: apiData
      }
    ),
    /* @__PURE__ */ jsx(PeopleSection, { title: "OUR CREW" }),
    /* @__PURE__ */ jsx(MembershipSection, { theme: "black" })
  ] });
}
const route16 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: StudioPage$1,
  loader,
  meta: meta$b
}, Symbol.toStringTag, { value: "Module" }));
const meta$a = () => {
  return [
    { title: "Login with Email | Nomadics" },
    { name: "description", content: "Sign in with your email and password" }
  ];
};
function LoginWithEmail() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(SignInEmail, {}) });
}
const route17 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: LoginWithEmail,
  meta: meta$a
}, Symbol.toStringTag, { value: "Module" }));
const styles$1 = `
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
`;
const meta$9 = ({ params }) => {
  var _a;
  return [
    { title: `${(_a = params.slug) == null ? void 0 : _a.replace(/-/g, " ")} - Nomadics Studios` },
    {
      name: "description",
      content: "Read the latest news and insights from the creative community at Nomadics Studios."
    }
  ];
};
const newsArticles$1 = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: {
      name: "Akanksha Bhartaure",
      avatar: "/placeholder-avatar.jpg",
      bio: "Travel Writer & Creative Director"
    },
    publishedAt: "5 min ago",
    readTime: "5 min read",
    image: "/bw (1).png",
    slug: "travel-as-a-failure",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>A story about a trip to Barcelona 2</h3>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <p>Travel opens our minds to new possibilities and perspectives that we might never encounter in our daily routines. When we step outside our comfort zones and immerse ourselves in different cultures, we gain invaluable insights that can transform our creative work.</p>
      
      <p>Sometimes, the most valuable lessons come from the moments when things don't go according to plan. Failed connections, missed opportunities, and unexpected detours often become the foundation for our most compelling stories and creative breakthroughs.</p>
    `,
    tags: ["Travel", "Lessons", "Barcelona", "Growth"]
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: {
      name: "Super Admin",
      avatar: "/placeholder-avatar.jpg",
      bio: "Creative Director at Nomadics Studios"
    },
    publishedAt: "2 hours ago",
    readTime: "4 min read",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    content: `
      <h2>A story about travel</h2>
      
      <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Praesent tristique leo vitae purus lacinia, ut euismod nulla viverra. Duis rutrum fringilla vestibulum. Quisque gravida, tortor eget accumsan aliquet.</p>
      
      <h3>The Creative Impact of Travel</h3>
      
      <p>For creators, travel is more than just a break from routine—it's a source of inspiration that can revolutionize their approach to their craft. Whether you're a filmmaker seeking new locations, a photographer hunting for unique lighting, or a musician searching for fresh sounds, travel provides the raw material for creative breakthroughs.</p>
      
      <p>At Nomadics Studios, we've seen firsthand how our community members return from their travels with renewed energy and innovative ideas. The stories they share, the techniques they've learned, and the perspectives they've gained all contribute to the rich tapestry of our creative community.</p>
    `,
    tags: ["Travel", "Creativity", "Inspiration", "Community"]
  }
];
function findArticleBySlug(slug) {
  return newsArticles$1.find((article) => article.slug === slug) || newsArticles$1[0];
}
function getRelatedArticles(currentSlug) {
  return newsArticles$1.filter((article) => article.slug !== currentSlug).slice(0, 3);
}
function NewsDetailsPage() {
  const { slug } = useParams();
  const article = findArticleBySlug(slug || "");
  const relatedArticles = getRelatedArticles(slug || "");
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("div", { className: "relative bg-white min-h-screen", children: [
    /* @__PURE__ */ jsx("style", { dangerouslySetInnerHTML: { __html: styles$1 } }),
    /* @__PURE__ */ jsx("div", { className: "fixed top-4 right-4 z-50 bg-red-500 text-white px-4 py-2 rounded text-sm font-bold", children: "NEWS DETAILS PAGE" }),
    /* @__PURE__ */ jsxs("section", { className: "relative h-[70vh] lg:h-[80vh] overflow-hidden", children: [
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: article.image,
            alt: article.title,
            className: "h-full w-full object-cover",
            onError: (e) => {
              e.currentTarget.src = "/placeholder-news.jpg";
            }
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-b from-black/30 via-black/50 to-black/70" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative z-10 h-full flex items-end", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1000px] px-4 lg:px-8 pb-16 lg:pb-24", children: [
        /* @__PURE__ */ jsx("nav", { className: "mb-6", children: /* @__PURE__ */ jsxs(
          Link,
          {
            to: "/news",
            className: "text-sm text-white/80 hover:text-white transition-colors uppercase tracking-wider flex items-center gap-2",
            children: [
              /* @__PURE__ */ jsx("svg", { className: "w-4 h-4", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M15 19l-7-7 7-7" }) }),
              "Back to News"
            ]
          }
        ) }),
        /* @__PURE__ */ jsxs("div", { className: "mb-4 text-white/60 text-sm bg-black/20 p-2 rounded", children: [
          'Debug: Slug = "',
          slug,
          '" | Article = "',
          article.title,
          '"'
        ] }),
        /* @__PURE__ */ jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsx("span", { className: "inline-block px-4 py-2 bg-white/20 backdrop-blur-sm text-white text-sm font-medium uppercase tracking-wider rounded", children: article.category }) }),
        /* @__PURE__ */ jsx("h1", { className: "text-3xl lg:text-5xl font-bold text-white mb-6 leading-tight max-w-4xl", children: article.title }),
        /* @__PURE__ */ jsx("div", { className: "flex flex-col sm:flex-row sm:items-center gap-4 text-white/90", children: /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: article.author.avatar,
              alt: article.author.name,
              className: "w-10 h-10 rounded-full object-cover bg-white/20",
              onError: (e) => {
                e.currentTarget.src = "/placeholder-avatar.jpg";
              }
            }
          ),
          /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("div", { className: "font-medium text-white", children: article.author.name }),
            /* @__PURE__ */ jsxs("div", { className: "text-sm text-white/70", children: [
              article.publishedAt,
              " • ",
              article.readTime
            ] })
          ] })
        ] }) })
      ] }) })
    ] }),
    /* @__PURE__ */ jsx("section", { className: "bg-white py-16 lg:py-24", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[800px] px-4 lg:px-8", children: [
      /* @__PURE__ */ jsx("div", { className: "prose prose-lg max-w-none mb-16", children: /* @__PURE__ */ jsx(
        "div",
        {
          className: "article-content \n                  [&_h2]:text-2xl [&_h2]:lg:text-3xl [&_h2]:font-bold [&_h2]:text-black [&_h2]:mb-6 [&_h2]:mt-12 [&_h2]:leading-tight\n                  [&_h3]:text-xl [&_h3]:lg:text-2xl [&_h3]:font-bold [&_h3]:text-black [&_h3]:mb-4 [&_h3]:mt-8 [&_h3]:leading-tight\n                  [&_p]:text-black/80 [&_p]:text-base [&_p]:lg:text-lg [&_p]:leading-relaxed [&_p]:mb-6\n                  [&_img]:rounded-lg [&_img]:shadow-lg [&_img]:my-8",
          dangerouslySetInnerHTML: { __html: article.content }
        }
      ) }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-3 mb-12 pb-8 border-b border-gray-200", children: [
        /* @__PURE__ */ jsx("span", { className: "text-sm font-medium text-gray-600 mr-2", children: "Tags:" }),
        article.tags.map((tag) => /* @__PURE__ */ jsx(
          "span",
          {
            className: "px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-full hover:bg-gray-200 transition-colors cursor-pointer",
            children: tag
          },
          tag
        ))
      ] }),
      /* @__PURE__ */ jsx("div", { className: "bg-gray-50 rounded-2xl p-8 mb-12", children: /* @__PURE__ */ jsxs("div", { className: "flex items-start gap-4", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: article.author.avatar,
            alt: article.author.name,
            className: "w-16 h-16 rounded-full object-cover bg-gray-200",
            onError: (e) => {
              e.currentTarget.src = "/placeholder-avatar.jpg";
            }
          }
        ),
        /* @__PURE__ */ jsxs("div", { className: "flex-1", children: [
          /* @__PURE__ */ jsx("h3", { className: "text-xl font-bold text-black mb-2", children: article.author.name }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 mb-4", children: article.author.bio }),
          /* @__PURE__ */ jsxs("div", { className: "flex gap-3", children: [
            /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "sm", children: "Follow" }),
            /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "sm", children: "View Profile" })
          ] })
        ] })
      ] }) }),
      /* @__PURE__ */ jsxs("div", { className: "text-center mb-12", children: [
        /* @__PURE__ */ jsx("h3", { className: "text-xl font-bold text-black mb-4", children: "Share this article" }),
        /* @__PURE__ */ jsxs("div", { className: "flex justify-center gap-3", children: [
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", children: "Twitter" }),
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", children: "LinkedIn" }),
          /* @__PURE__ */ jsx(Button, { variant: "outline-black", size: "md", children: "Facebook" }),
          /* @__PURE__ */ jsx(Button, { variant: "fill-black", size: "md", arrowRight: true, children: "Subscribe" })
        ] })
      ] })
    ] }) }),
    /* @__PURE__ */ jsx("section", { className: "bg-gray-50", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1200px] px-4 lg:px-8 py-16 lg:py-24", children: [
      /* @__PURE__ */ jsxs("div", { className: "text-center mb-16", children: [
        /* @__PURE__ */ jsx("h2", { className: "text-3xl lg:text-4xl font-bold text-black mb-4", children: "Related news" }),
        /* @__PURE__ */ jsx("div", { className: "w-24 h-1 bg-black mx-auto" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12", children: relatedArticles.map((relatedArticle) => /* @__PURE__ */ jsxs(
        Link,
        {
          to: `/news/${relatedArticle.slug}`,
          className: "group block bg-white shadow-sm hover:shadow-lg transition-all duration-300 overflow-hidden hover:transform hover:scale-[1.02] rounded-lg",
          children: [
            /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden bg-neutral-100", children: /* @__PURE__ */ jsx(
              "img",
              {
                src: relatedArticle.image,
                alt: relatedArticle.title,
                className: "h-full w-full object-cover transition-transform duration-500 group-hover:scale-105",
                onError: (e) => {
                  e.currentTarget.src = "/placeholder-news.jpg";
                }
              }
            ) }),
            /* @__PURE__ */ jsxs("div", { className: "p-6", children: [
              /* @__PURE__ */ jsx("div", { className: "inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded mb-3", children: relatedArticle.category }),
              /* @__PURE__ */ jsx("h3", { className: "text-lg font-bold text-black mb-3 group-hover:text-black/80 transition-colors leading-tight", children: relatedArticle.title }),
              /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-sm line-clamp-2 mb-4", children: relatedArticle.excerpt }),
              /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2 text-xs text-gray-500", children: [
                /* @__PURE__ */ jsx("span", { children: relatedArticle.author.name }),
                /* @__PURE__ */ jsx("span", { children: "•" }),
                /* @__PURE__ */ jsx("span", { children: relatedArticle.publishedAt })
              ] })
            ] })
          ]
        },
        relatedArticle.id
      )) }),
      /* @__PURE__ */ jsx("div", { className: "text-center", children: /* @__PURE__ */ jsx(
        Button,
        {
          variant: "outline-black",
          size: "lg",
          arrowRight: true,
          className: "text-black",
          onClick: () => window.location.href = "/news",
          children: "View All News"
        }
      ) })
    ] }) })
  ] }) });
}
const route18 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: NewsDetailsPage,
  meta: meta$9
}, Symbol.toStringTag, { value: "Module" }));
function StudioPage() {
  const [start, setStart] = useState("2025-08-21T10:30");
  const [end, setEnd] = useState("2025-08-21T13:30");
  (new Date(end).getTime() - new Date(start).getTime()) / (1e3 * 60 * 60);
  return /* @__PURE__ */ jsxs("div", { className: "bg-white text-black font-sans min-h-screen", children: [
    /* @__PURE__ */ jsx("section", { className: "max-w-7xl mx-auto px-4 pt-10 grid md:grid-cols-1 gap-10", children: /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-2 grid-rows-2 gap-6", children: [
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/image 209.png",
          alt: "Studio Gallery 1",
          className: "col-span-1 row-span-2 rounded-xl object-cover w-full h-[440px] md:h-[700px]"
        }
      ),
      /* @__PURE__ */ jsx(
        "img",
        {
          src: "/image 210.png",
          alt: "Studio Gallery 2",
          className: "col-span-1 row-span-1 rounded-xl object-cover w-full h-[210px] md:h-[340px]"
        }
      ),
      /* @__PURE__ */ jsxs("div", { className: "relative col-span-1 row-span-1", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "/image 211.png",
            alt: "Studio Gallery 3",
            className: "rounded-xl object-cover w-full h-[210px] md:h-[340px]"
          }
        ),
        /* @__PURE__ */ jsx("button", { className: "absolute right-4 bottom-4 bg-white text-black px-4 py-2 rounded shadow border font-medium", children: "SHOW ALL PHOTOS" })
      ] })
    ] }) }) }),
    /* @__PURE__ */ jsxs("section", { className: "max-w-7xl mx-auto px-4 py-10 mt-12 flex flex-col md:flex-row gap-12 items-start", children: [
      /* @__PURE__ */ jsxs("div", { className: "flex-1 min-w-0", children: [
        /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-4 mb-8", children: [
          /* @__PURE__ */ jsx("span", { className: "px-6 py-2 rounded-full border text-lg font-medium", children: "NOMADIC SATELLITE" }),
          /* @__PURE__ */ jsx("span", { className: "px-6 py-2 rounded-full border text-lg font-medium", children: "MUSIC STUDIO" }),
          /* @__PURE__ */ jsx("span", { className: "px-6 py-2 rounded-full border text-lg font-medium", children: "BALI, INDONESIA" })
        ] }),
        /* @__PURE__ */ jsxs("h1", { className: "text-6xl font-black mb-6 leading-tight tracking-tight", children: [
          "PROFESSIONAL VIDEO",
          /* @__PURE__ */ jsx("br", {}),
          "STUDIO FOR CREATIVES"
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 mb-4 text-gray-800 text-xl font-semibold", children: [
          /* @__PURE__ */ jsxs(
            "svg",
            {
              xmlns: "http://www.w3.org/2000/svg",
              fill: "none",
              viewBox: "0 0 24 24",
              strokeWidth: 1.5,
              stroke: "currentColor",
              className: "w-6 h-6",
              children: [
                /* @__PURE__ */ jsx(
                  "path",
                  {
                    strokeLinecap: "round",
                    strokeLinejoin: "round",
                    d: "M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0z"
                  }
                ),
                /* @__PURE__ */ jsx(
                  "path",
                  {
                    strokeLinecap: "round",
                    strokeLinejoin: "round",
                    d: "M4.5 20.25a8.25 8.25 0 1115 0v.75a.75.75 0 01-.75.75H5.25a.75.75 0 01-.75-.75v-.75z"
                  }
                )
              ]
            }
          ),
          /* @__PURE__ */ jsx("span", { children: "Jl. Raya Ubud No.88, Ubud, Kecamatan Ubud, Kabupaten Gianyar, Bali 80571, Indonesia" })
        ] }),
        /* @__PURE__ */ jsx("p", { className: "text-gray-700 mt-6 text-2xl leading-relaxed max-w-3xl", children: "Bali is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need here." })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "flex-shrink-0 w-full md:w-[520px]", children: /* @__PURE__ */ jsxs("div", { className: "border rounded-xl p-8 shadow-md flex flex-col gap-6", children: [
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-lg font-medium mb-2 text-gray-700", children: "START" }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between border-b pb-4", children: [
            /* @__PURE__ */ jsx("span", { className: "text-xl font-semibold", children: "AUG 21, 2025, 10:30 AM" }),
            /* @__PURE__ */ jsx(
              "svg",
              {
                className: "w-6 h-6 text-gray-400",
                fill: "none",
                stroke: "currentColor",
                viewBox: "0 0 24 24",
                children: /* @__PURE__ */ jsx(
                  "path",
                  {
                    strokeLinecap: "round",
                    strokeLinejoin: "round",
                    strokeWidth: 2,
                    d: "M8 10h.01M12 10h.01M16 10h.01M9 16h6"
                  }
                )
              }
            )
          ] })
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("label", { className: "block text-lg font-medium mb-2 text-gray-700", children: "END" }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between border-b pb-4", children: [
            /* @__PURE__ */ jsx("span", { className: "text-xl font-semibold", children: "AUG 21, 2025, 01:30 PM" }),
            /* @__PURE__ */ jsx(
              "svg",
              {
                className: "w-6 h-6 text-gray-400",
                fill: "none",
                stroke: "currentColor",
                viewBox: "0 0 24 24",
                children: /* @__PURE__ */ jsx(
                  "path",
                  {
                    strokeLinecap: "round",
                    strokeLinejoin: "round",
                    strokeWidth: 2,
                    d: "M8 10h.01M12 10h.01M16 10h.01M9 16h6"
                  }
                )
              }
            )
          ] })
        ] }),
        /* @__PURE__ */ jsx("div", { className: "py-4 text-gray-600 text-lg border-b", children: "Your reservation: 3h   •   From: 130 points" }),
        /* @__PURE__ */ jsxs("button", { className: "bg-black text-white py-5 rounded-md hover:bg-gray-800 transition font-bold text-xl flex items-center justify-center gap-2", children: [
          "BOOK NOW ",
          /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
        ] })
      ] }) })
    ] }),
    /* @__PURE__ */ jsxs("section", { className: "max-w-7xl mx-auto px-6 py-12", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-3xl font-bold mb-6", children: "OUR CREW" }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-2 mb-8", children: [
        /* @__PURE__ */ jsx("button", { className: "px-5 py-2 rounded-full border font-medium text-sm bg-black text-white", children: "ALL CREW" }),
        /* @__PURE__ */ jsx("button", { className: "px-5 py-2 rounded-full border font-medium text-sm", children: "AUDIO ENGINEER" }),
        /* @__PURE__ */ jsx("button", { className: "px-5 py-2 rounded-full border font-medium text-sm", children: "AUDIO TECHNICIAN" }),
        /* @__PURE__ */ jsx("button", { className: "px-5 py-2 rounded-full border font-medium text-sm", children: "MORE" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-8", children: [1, 2, 3, 4, 5, 6].map((i) => /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "./bw.png",
            alt: "Crew",
            className: "rounded-xl w-full h-64 object-cover mb-2"
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "text-lg font-semibold", children: "NAME SURNAME" }),
        /* @__PURE__ */ jsx("div", { className: "text-gray-500 text-sm", children: "Profession" })
      ] }, i)) })
    ] }),
    /* @__PURE__ */ jsxs("section", { className: "max-w-7xl mx-auto px-6 py-12", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-5xl font-black mb-10 tracking-tight", children: "EQUIPMENT" }),
      /* @__PURE__ */ jsx("div", { className: "border-2 border-black rounded-xl overflow-hidden mb-12", children: /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 gap-0", children: [
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-6 border-b border-black p-3", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: "/Frame 1000003571.png",
              alt: "DJ Equipment",
              className: "w-20 h-20 object-cover"
            }
          ),
          /* @__PURE__ */ jsx("span", { className: "text-2xl font-medium tracking-tight", children: "DJ EQUIPMENT" })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-6 border-b border-black p-3", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: "/Frame 1000003571 (1).png",
              alt: "Cameras",
              className: "w-20 h-20 object-cover"
            }
          ),
          /* @__PURE__ */ jsx("span", { className: "text-2xl font-medium tracking-tight", children: "CAMERAS" })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-6 p-3", children: [
          /* @__PURE__ */ jsx(
            "img",
            {
              src: "/Frame 1000003571 (2).png",
              alt: "Light",
              className: "w-20 h-20 object-cover"
            }
          ),
          /* @__PURE__ */ jsx("span", { className: "text-2xl font-medium tracking-tight", children: "LIGHT" })
        ] })
      ] }) }),
      /* @__PURE__ */ jsxs("div", { className: "grid md:grid-cols-4 gap-8 mb-8", children: [
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "font-bold mb-2 text-gray-700", children: "ADDRESS" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-lg", children: "Jl. Raya Ubud No.88, Ubud, Kecamatan Ubud, Kabupaten Gianyar, Bali 80571, Indonesia" })
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "font-bold mb-2 text-gray-700", children: "OPENING HOURS" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-lg", children: "6:00 AM - 11:00 PM" })
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "font-bold mb-2 text-gray-700", children: "CONTACT" }),
          /* @__PURE__ */ jsxs("p", { className: "text-gray-600 text-lg", children: [
            "+91 87896546378",
            /* @__PURE__ */ jsx("br", {}),
            "balicreativehub@gmail.com"
          ] })
        ] }),
        /* @__PURE__ */ jsxs("div", { children: [
          /* @__PURE__ */ jsx("h3", { className: "font-bold mb-2 text-gray-700", children: "ADDITIONAL INFORMATION" }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-lg", children: "You can go to us by bus from..." })
        ] })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "w-full h-96 rounded-xl overflow-hidden border", children: /* @__PURE__ */ jsx("img", { src: "/map.png", alt: "Map", className: "w-full h-full object-cover" }) })
    ] })
  ] });
}
const route19 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: StudioPage
}, Symbol.toStringTag, { value: "Module" }));
const meta$8 = () => {
  return [
    { title: "Equipment | Nomadics" },
    { name: "description", content: "Professional equipment for rent" }
  ];
};
const HeroSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-black text-white min-h-screen relative px-8 py-16", children: /* @__PURE__ */ jsxs("div", { className: "max-w-7xl mx-auto h-full min-h-screen flex flex-col justify-center relative", children: [
    /* @__PURE__ */ jsx("div", { className: "absolute top-1/4 left-8 transform -translate-y-1/2", children: /* @__PURE__ */ jsx("h1", { className: "text-4xl md:text-6xl lg:text-7xl xl:text-8xl font-black leading-[0.8] tracking-tight text-left", children: t("equipment.hero.title") }) }),
    /* @__PURE__ */ jsx("div", { className: "absolute bottom-40 right-8 max-w-xs md:max-w-sm lg:max-w-md", children: /* @__PURE__ */ jsx("p", { className: "text-lg md:text-sm lg:text-base leading-relaxed text-gray-300 text-left", children: t("equipment.hero.description") }) })
  ] }) });
};
const ConsistentGearSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-white py-20 px-4", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto", children: /* @__PURE__ */ jsxs("div", { className: "grid lg:grid-cols-2 gap-12 items-center", children: [
    /* @__PURE__ */ jsxs("div", { className: "order-2 lg:order-1", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-4xl md:text-5xl font-bold text-black mb-6", children: t("equipment.consistentGear.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 mb-8 leading-relaxed", children: t("equipment.consistentGear.description") }),
      /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-8 py-3 bg-white border-2 border-black text-black font-semibold hover:bg-black hover:text-white transition-colors duration-300", children: [
        t("equipment.bookStudio"),
        /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "order-1 lg:order-2", children: /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx("div", { className: "bg-gray-900 rounded-lg overflow-hidden shadow-2xl", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/eqi-1.png",
        alt: t("equipment.consistentGear.imageAlt"),
        className: "w-full h-full object-cover aspect-video"
      }
    ) }) }) })
  ] }) }) });
};
const InterconnectedSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-gray-50 py-20 px-4", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto", children: /* @__PURE__ */ jsxs("div", { className: "grid lg:grid-cols-2 gap-12 items-center", children: [
    /* @__PURE__ */ jsx("div", { className: "order-1", children: /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx("div", { className: "bg-black rounded-lg overflow-hidden shadow-2xl", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/equ-2.png",
        alt: t("equipment.interconnected.imageAlt"),
        className: "w-full h-full object-cover aspect-video"
      }
    ) }) }) }),
    /* @__PURE__ */ jsxs("div", { className: "order-2", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-4xl md:text-5xl font-bold text-black mb-6", children: t("equipment.interconnected.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 mb-8 leading-relaxed", children: t("equipment.interconnected.description") }),
      /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-8 py-3 bg-white border-2 border-black text-black font-semibold hover:bg-black hover:text-white transition-colors duration-300", children: [
        t("equipment.bookStudio"),
        /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
      ] })
    ] })
  ] }) }) });
};
const StockedEssentialsSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-white py-20 px-4", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto", children: /* @__PURE__ */ jsxs("div", { className: "grid lg:grid-cols-2 gap-12 items-center", children: [
    /* @__PURE__ */ jsxs("div", { className: "order-2 lg:order-1", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-4xl md:text-5xl font-bold text-black mb-6", children: t("equipment.stockedEssentials.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 mb-8 leading-relaxed", children: t("equipment.stockedEssentials.description") }),
      /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-8 py-3 bg-white border-2 border-black text-black font-semibold hover:bg-black hover:text-white transition-colors duration-300", children: [
        t("equipment.bookStudio"),
        /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "order-1 lg:order-2", children: /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx("div", { className: "bg-gray-900 rounded-lg overflow-hidden shadow-2xl", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/eqi-3.png",
        alt: t("equipment.stockedEssentials.imageAlt"),
        className: "w-full h-full object-cover aspect-video"
      }
    ) }) }) })
  ] }) }) });
};
const ExpertGuidanceSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-gray-50 py-20 px-4", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto", children: /* @__PURE__ */ jsxs("div", { className: "grid lg:grid-cols-2 gap-12 items-center", children: [
    /* @__PURE__ */ jsx("div", { className: "order-1", children: /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx("div", { className: "bg-black rounded-lg overflow-hidden shadow-2xl", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/equ-4.png",
        alt: t("equipment.expertGuidance.imageAlt"),
        className: "w-full h-full object-cover aspect-video"
      }
    ) }) }) }),
    /* @__PURE__ */ jsxs("div", { className: "order-2", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-4xl md:text-5xl font-bold text-black mb-6", children: t("equipment.expertGuidance.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 mb-8 leading-relaxed", children: t("equipment.expertGuidance.description") }),
      /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-8 py-3 bg-white border-2 border-black text-black font-semibold hover:bg-black hover:text-white transition-colors duration-300", children: [
        t("equipment.bookStudio"),
        /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
      ] })
    ] })
  ] }) }) });
};
const SpecializedRentalsSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-white py-20 px-4", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto", children: /* @__PURE__ */ jsxs("div", { className: "grid lg:grid-cols-2 gap-12 items-center", children: [
    /* @__PURE__ */ jsxs("div", { className: "order-2 lg:order-1", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-4xl md:text-5xl font-bold text-black mb-6", children: t("equipment.specializedRentals.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 mb-8 leading-relaxed", children: t("equipment.specializedRentals.description") }),
      /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-8 py-3 bg-white border-2 border-black text-black font-semibold hover:bg-black hover:text-white transition-colors duration-300", children: [
        t("equipment.bookStudio"),
        /* @__PURE__ */ jsx("span", { className: "ml-2", children: "→" })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "order-1 lg:order-2", children: /* @__PURE__ */ jsx("div", { className: "relative", children: /* @__PURE__ */ jsx("div", { className: "bg-gray-900 rounded-lg overflow-hidden shadow-2xl", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/eqi-1.png",
        alt: t("equipment.specializedRentals.imageAlt"),
        className: "w-full h-full object-cover aspect-video"
      }
    ) }) }) })
  ] }) }) });
};
const ConnectCollaborateSection = () => {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx("section", { className: "bg-black text-white py-32 px-4", children: /* @__PURE__ */ jsxs("div", { className: "max-w-4xl mx-auto text-center", children: [
    /* @__PURE__ */ jsxs("div", { className: "mb-16", children: [
      /* @__PURE__ */ jsx("h2", { className: "text-3xl md:text-7xl font-black leading-tight mb-8", children: t("equipment.connectCollaborate.title") }),
      /* @__PURE__ */ jsx("p", { className: "text-xl md:text-2xl text-gray-300 leading-relaxed max-w-3xl mx-auto", children: t("equipment.connectCollaborate.description") })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "mt-12", children: /* @__PURE__ */ jsxs("button", { className: "inline-flex items-center px-12 py-4 bg-white text-black font-bold text-lg hover:bg-gray-100 transition-colors duration-300", children: [
      t("equipment.applyForMembership"),
      /* @__PURE__ */ jsx("span", { className: "ml-3", children: "→" })
    ] }) }),
    /* @__PURE__ */ jsx("div", { className: "mt-24 pt-12 border-t border-gray-800", children: /* @__PURE__ */ jsx("p", { className: "text-gray-500 text-sm", children: t("equipment.copyright") }) })
  ] }) });
};
function Equipment() {
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx(HeroSection, {}),
    /* @__PURE__ */ jsx(ConsistentGearSection, {}),
    /* @__PURE__ */ jsx(InterconnectedSection, {}),
    /* @__PURE__ */ jsx(StockedEssentialsSection, {}),
    /* @__PURE__ */ jsx(ExpertGuidanceSection, {}),
    /* @__PURE__ */ jsx(SpecializedRentalsSection, {}),
    /* @__PURE__ */ jsx(ConnectCollaborateSection, {})
  ] });
}
const route20 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Equipment,
  meta: meta$8
}, Symbol.toStringTag, { value: "Module" }));
const meta$7 = () => {
  return [
    { title: "Events | Nomadics" },
    { name: "description", content: "Upcoming creative events and workshops" }
  ];
};
function Events$1() {
  return /* @__PURE__ */ jsxs("div", { className: "container mx-auto px-4 py-8", children: [
    /* @__PURE__ */ jsx("h1", { className: "text-4xl font-bold mb-6", children: "Events" }),
    /* @__PURE__ */ jsx("p", { className: "text-lg text-muted-foreground", children: "Discover upcoming events, workshops, and gatherings." })
  ] });
}
const route21 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Events$1,
  meta: meta$7
}, Symbol.toStringTag, { value: "Module" }));
const servicesData = [
  {
    title: "SOCIAL MEDIA STRATEGY",
    description: "Whether you're filming a commercial, live-streaming a podcast, or editing a short film, our studios provide the space and tools you need to deliver professional results. Unlike other generic rental spaces, we're built specifically for creators who demand both flexibility and quality.",
    image: "https://images.unsplash.com/photo-1515378791036-0648a3ef77b2",
    // Replace with your image
    reverse: false
  },
  {
    title: "CONTENT CREATION (IMAGES, VIDEOS, COPY)",
    description: "Whether you're filming a commercial, live-streaming a podcast, or editing a short film, our studios provide the space and tools you need to deliver professional results. Unlike other generic rental spaces, we're built specifically for creators who demand both flexibility and quality.",
    image: "https://images.unsplash.com/photo-1465101046530-73398c7f28ca",
    // Replace with your image
    reverse: true
  },
  {
    title: "PAID AD CAMPAIGNS",
    description: "Whether you're filming a commercial, live-streaming a podcast, or editing a short film, our studios provide the space and tools you need to deliver professional results. Unlike other generic rental spaces, we're built specifically for creators who demand both flexibility and quality.",
    image: "https://images.unsplash.com/photo-1519125323398-675f0ddb6308",
    // Replace with your image
    reverse: false
  },
  {
    title: "ANALYTICS AND INSIGHTS",
    description: "Whether you're filming a commercial, live-streaming a podcast, or editing a short film, our studios provide the space and tools you need to deliver professional results. Unlike other generic rental spaces, we're built specifically for creators who demand both flexibility and quality.",
    image: "https://images.unsplash.com/photo-1506744038136-46273834b3fb",
    // Replace with your image
    reverse: true
  }
];
function services() {
  return /* @__PURE__ */ jsxs("div", { className: "bg-black text-white min-h-screen font-sans", children: [
    /* @__PURE__ */ jsxs("section", { className: "max-w-7xl mx-auto px-6 py-20", children: [
      /* @__PURE__ */ jsxs("h1", { className: "text-6xl md:text-7xl font-black mb-8 leading-tight", children: [
        "GROW YOUR AUDIENCE,",
        /* @__PURE__ */ jsx("br", {}),
        "BUILD YOUR BRAND."
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-col md:flex-row md:items-center gap-8 mb-16", children: [
        /* @__PURE__ */ jsx("div", { className: "md:w-1/2" }),
        /* @__PURE__ */ jsx("div", { className: "md:w-1/2 text-lg text-gray-200", children: "Our team offers a complete suite of social media and marketing services, including content creation, post-scheduling, analytics, and campaign management. Whether you're an emerging artist or a seasoned professional, we help you stand out in a crowded digital landscape." })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-col md:flex-row gap-6 justify-center mb-16", children: [
        /* @__PURE__ */ jsx("button", { className: "border border-white rounded-full px-10 py-6 text-lg font-medium text-white hover:bg-white hover:text-black transition w-full md:w-auto", children: "EXPAND YOUR REACH" }),
        /* @__PURE__ */ jsx("button", { className: "border border-white rounded-full px-10 py-6 text-lg font-medium text-white hover:bg-white hover:text-black transition w-full md:w-auto", children: "GENERATE PASSIVE INCOME" }),
        /* @__PURE__ */ jsx("button", { className: "border border-white rounded-full px-10 py-6 text-lg font-medium text-white hover:bg-white hover:text-black transition w-full md:w-auto", children: "SIMPLIFY MONETIZATION" })
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "bg-white text-black py-20", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto px-6 grid gap-20", children: servicesData.map((service, idx) => /* @__PURE__ */ jsx(
      "div",
      {
        className: `grid md:grid-cols-2 gap-0 items-stretch`,
        children: idx % 2 === 0 ? /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx("div", { className: "flex items-center justify-center p-8", children: /* @__PURE__ */ jsx(
            "img",
            {
              src: service.image,
              alt: service.title,
              className: "rounded-xl w-full h-64 object-cover shadow-lg"
            }
          ) }),
          /* @__PURE__ */ jsxs("div", { className: "flex flex-col justify-center p-8", children: [
            /* @__PURE__ */ jsx("h2", { className: "text-3xl font-bold mb-4", children: service.title }),
            /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 leading-relaxed", children: service.description })
          ] })
        ] }) : /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsxs("div", { className: "flex flex-col justify-center p-8", children: [
            /* @__PURE__ */ jsx("h2", { className: "text-3xl font-bold mb-4", children: service.title }),
            /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-700 leading-relaxed", children: service.description })
          ] }),
          /* @__PURE__ */ jsx("div", { className: "flex items-center justify-center p-8", children: /* @__PURE__ */ jsx(
            "img",
            {
              src: service.image,
              alt: service.title,
              className: "rounded-xl w-full h-64 object-cover shadow-lg"
            }
          ) })
        ] })
      },
      service.title
    )) }) }),
    /* @__PURE__ */ jsxs("section", { className: "bg-black text-white py-24 text-center", children: [
      /* @__PURE__ */ jsxs("h2", { className: "text-5xl md:text-6xl font-black mb-8", children: [
        "JOIN US TO LEARN HOW TO",
        /* @__PURE__ */ jsx("br", {}),
        "GET STARTED WITH ONLINE SERVICES."
      ] }),
      /* @__PURE__ */ jsx("p", { className: "text-lg text-gray-200 mb-8 max-w-2xl mx-auto", children: "Be part of something bigger. Nomadic Studios is more than just a place to work – it's a vibrant network of creatives who inspire and support each other." }),
      /* @__PURE__ */ jsx("button", { className: "bg-white text-black font-semibold px-10 py-5 rounded shadow hover:bg-gray-200 transition text-lg", children: "APPLY FOR MEMBERSHIP →" })
    ] })
  ] });
}
const route22 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: services
}, Symbol.toStringTag, { value: "Module" }));
const meta$6 = () => {
  return [
    { title: "Events - Nomadics Studios" },
    {
      name: "description",
      content: "Discover and join exciting events at Nomadics Studios. Connect with creative professionals and explore new opportunities."
    }
  ];
};
function Events() {
  const { t } = useLanguage();
  const [events, setEvents] = useState([]);
  const [locations, setLocations] = useState([]);
  const [selectedLocation, setSelectedLocation] = useState(null);
  const [selectedDate, setSelectedDate] = useState("");
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const formatEventDate = (dateString) => {
    const date = new Date(dateString);
    const day = date.getDate().toString();
    const month = date.toLocaleDateString("en-US", { month: "short" }).toUpperCase();
    return { day, month };
  };
  const formatEventTime = (timeString) => {
    return timeString;
  };
  const generateSlug = (title) => {
    return title.toLowerCase().replace(/[^a-z0-9]+/g, "-").replace(/(^-|-$)/g, "");
  };
  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const [locationsResponse, eventsResponse] = await Promise.all([
          locationsAPI.getLocations(),
          eventsAPI.getAll(selectedDate, selectedLocation || void 0)
        ]);
        if (locationsResponse.status && locationsResponse.data) {
          const uniqueLocations = locationsResponse.data.filter(
            (location, index, self) => index === self.findIndex((l) => l.id === location.id && l.location === location.location)
          );
          setLocations(uniqueLocations);
        }
        if (eventsResponse.status && eventsResponse.data) {
          setEvents(eventsResponse.data);
        } else {
          setError("Failed to fetch events");
        }
      } catch (err) {
        setError("An error occurred while fetching data");
        console.error("Error fetching data:", err);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, [selectedDate, selectedLocation]);
  const handleLocationFilter = (locationId) => {
    setSelectedLocation(locationId);
  };
  const handleDateFilter = (date) => {
    setSelectedDate(date);
  };
  if (loading) {
    return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx("main", { className: "bg-white min-h-screen pb-24", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto pt-8 px-4", children: /* @__PURE__ */ jsx("div", { className: "flex justify-center items-center h-64", children: /* @__PURE__ */ jsx("div", { className: "text-gray-500", children: t("events.loading", "Loading events...") }) }) }) }) });
  }
  if (error) {
    return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx("main", { className: "bg-white min-h-screen pb-24", children: /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto pt-8 px-4", children: /* @__PURE__ */ jsx("div", { className: "flex justify-center items-center h-64", children: /* @__PURE__ */ jsxs("div", { className: "text-red-500", children: [
      t("events.error", "Error"),
      ": ",
      error
    ] }) }) }) }) });
  }
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("main", { className: "bg-white min-h-screen pb-24", children: [
    /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto pt-8 px-4", children: /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 md:grid-cols-2 gap-6 mb-8", children: [
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium text-gray-700 mb-2", children: t("events.filters.location", "LOCATION") }),
        /* @__PURE__ */ jsxs("div", { className: "flex flex-wrap gap-3", children: [
          /* @__PURE__ */ jsx(
            "button",
            {
              className: `px-5 py-2 rounded-full border font-medium text-sm transition ${selectedLocation === null ? "bg-black text-white" : "bg-white text-black border-black hover:bg-gray-100"}`,
              onClick: () => handleLocationFilter(null),
              children: t("events.filters.allLocations", "All Locations")
            }
          ),
          locations.map((location) => /* @__PURE__ */ jsx(
            "button",
            {
              className: `px-5 py-2 rounded-full border font-medium text-sm transition ${selectedLocation === location.id ? "bg-black text-white" : "bg-white text-black border-black hover:bg-gray-100"}`,
              onClick: () => handleLocationFilter(location.id),
              children: location.location
            },
            location.id
          ))
        ] })
      ] }),
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx("label", { className: "block text-sm font-medium text-gray-700 mb-2", children: t("events.filters.date", "DATE") }),
        /* @__PURE__ */ jsx(
          "input",
          {
            type: "date",
            value: selectedDate,
            onChange: (e) => handleDateFilter(e.target.value),
            className: "px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-black"
          }
        )
      ] })
    ] }) }),
    /* @__PURE__ */ jsx("div", { className: "max-w-7xl mx-auto px-4", children: events.length === 0 ? /* @__PURE__ */ jsx("div", { className: "text-center py-12", children: /* @__PURE__ */ jsx("div", { className: "text-gray-500", children: t("events.noEvents", "No events found matching your filters.") }) }) : /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8", children: events.map((event) => {
      const { day, month } = formatEventDate(event.event_date);
      const slug = generateSlug(event.event_title);
      return /* @__PURE__ */ jsxs(
        Link,
        {
          to: `/events-details/${slug}`,
          className: "group block bg-white rounded-lg shadow-sm overflow-hidden hover:shadow-xl transition-all duration-300 transform hover:scale-[1.02] cursor-pointer",
          children: [
            /* @__PURE__ */ jsxs("div", { className: "relative overflow-hidden", children: [
              /* @__PURE__ */ jsx(
                "img",
                {
                  src: event.thumbnail,
                  alt: event.event_title,
                  className: "w-full h-40 object-cover transition-transform duration-500 group-hover:scale-110",
                  onError: (e) => {
                    e.currentTarget.src = "/event.png";
                  }
                }
              ),
              /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-300" }),
              /* @__PURE__ */ jsx("div", { className: "absolute top-3 right-3", children: /* @__PURE__ */ jsx("span", { className: `px-2 py-1 text-xs font-medium rounded-full ${event.event_type === "online" ? "bg-green-500 text-white" : "bg-blue-500 text-white"}`, children: event.event_type }) }),
              event.public_private === "private" && /* @__PURE__ */ jsx("div", { className: "absolute top-3 left-3", children: /* @__PURE__ */ jsx("span", { className: "px-2 py-1 text-xs font-medium rounded-full bg-gray-800 text-white", children: "Private" }) })
            ] }),
            /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-4 px-4 py-3 group-hover:bg-gray-50 transition-colors duration-300", children: [
              /* @__PURE__ */ jsxs("div", { className: "text-center min-w-[40px]", children: [
                /* @__PURE__ */ jsx("div", { className: "text-lg font-bold text-black group-hover:text-black/80 transition-colors", children: day }),
                /* @__PURE__ */ jsx("div", { className: "text-xs text-gray-500 group-hover:text-gray-600 transition-colors", children: month })
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "flex-1", children: [
                /* @__PURE__ */ jsx("div", { className: "font-semibold text-black group-hover:text-black/90 transition-colors", children: event.event_title }),
                /* @__PURE__ */ jsx("div", { className: "text-xs text-gray-500 group-hover:text-gray-600 transition-colors", children: formatEventTime(event.event_time) }),
                /* @__PURE__ */ jsx("div", { className: "text-xs text-gray-500 group-hover:text-gray-600 transition-colors mt-1", children: event.address }),
                event.ticket_price > 0 && /* @__PURE__ */ jsxs("div", { className: "text-xs font-medium text-green-600 mt-1", children: [
                  "$",
                  event.ticket_price
                ] })
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "text-center", children: [
                /* @__PURE__ */ jsxs("div", { className: "text-xs text-gray-500", children: [
                  event.interested_count + event.going_count,
                  " interested"
                ] }),
                /* @__PURE__ */ jsx("div", { className: "opacity-0 group-hover:opacity-100 transition-opacity duration-300", children: /* @__PURE__ */ jsx("svg", { className: "w-5 h-5 text-gray-400", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M9 5l7 7-7 7" }) }) })
              ] })
            ] })
          ]
        },
        event.event_id
      );
    }) }) })
  ] }) });
}
const route23 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Events,
  meta: meta$6
}, Symbol.toStringTag, { value: "Module" }));
const meta$5 = () => {
  return [
    { title: "People | Nomadics" },
    { name: "description", content: "Meet our creative community" }
  ];
};
function People() {
  return /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx(PeopleSection, {}) });
}
const route24 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: People,
  meta: meta$5
}, Symbol.toStringTag, { value: "Module" }));
const meta$4 = () => {
  return [
    { title: "Sign Up | Nomadics" },
    { name: "description", content: "Join the global creative community" }
  ];
};
function Signup() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(SignupFirst, {}) });
}
const route25 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Signup,
  meta: meta$4
}, Symbol.toStringTag, { value: "Module" }));
const heroActions = [
  {
    label: "Apply for membership",
    href: "/apply",
    variant: "fill-white",
    icon: true
  },
  {
    label: "Explore services",
    href: "/services",
    variant: "outline-white"
  }
];
function WelcomeSection() {
  return /* @__PURE__ */ jsx(
    HeroSection$1,
    {
      backgroundImage: "/images/home/welcome.png",
      title: "WHAT YOU NEED <br /> TO CREATE",
      subtitle: {
        mobile: "We empower creators to transform ideas into reality wherever they are",
        desktop: "Nomadic Studios empowers creators to transform ideas into reality wherever they are"
      },
      children: /* @__PURE__ */ jsx(HeroActions, { actions: heroActions })
    }
  );
}
function Carousel({
  children,
  showArrows = true,
  className = ""
}) {
  const scrollerRef = useRef(null);
  function scrollByCards(dir) {
    const el = scrollerRef.current;
    if (!el) return;
    const card = el.querySelector("[data-card]");
    const amount = card ? card.offsetWidth + 24 : el.clientWidth * 0.9;
    el.scrollBy({
      left: dir === "next" ? amount : -amount,
      behavior: "smooth"
    });
  }
  return /* @__PURE__ */ jsxs("div", { className: `relative ${className}`, children: [
    /* @__PURE__ */ jsx(
      "div",
      {
        ref: scrollerRef,
        className: "flex snap-x snap-mandatory gap-6 overflow-x-auto pb-4 [-ms-overflow-style:none] [scrollbar-width:none] [&::-webkit-scrollbar]:hidden",
        "aria-label": "Carousel items",
        children
      }
    ),
    showArrows && /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsxs("div", { className: "hidden lg:flex absolute -top-[120px] right-0 items-center gap-4", children: [
        /* @__PURE__ */ jsx(
          "button",
          {
            className: "w-[60px] h-[60px] rounded-full border border-neutral-900 opacity-50 flex items-center justify-center transition-opacity hover:opacity-100 !text-black",
            "aria-label": "Previous",
            onClick: () => scrollByCards("prev"),
            children: /* @__PURE__ */ jsx(ArrowLeft, { className: "h-5 w-5" })
          }
        ),
        /* @__PURE__ */ jsx(
          "button",
          {
            className: "w-[60px] h-[60px] rounded-full border border-neutral-900 flex items-center justify-center transition-opacity hover:opacity-75 !text-black",
            "aria-label": "Next",
            onClick: () => scrollByCards("next"),
            children: /* @__PURE__ */ jsx(ArrowRight, { className: "h-5 w-5" })
          }
        )
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "lg:hidden mt-4 flex justify-end items-center gap-3", children: [
        /* @__PURE__ */ jsx(
          "button",
          {
            className: "w-10 h-10 rounded-full border border-neutral-900 opacity-50 flex items-center justify-center transition-opacity hover:opacity-100",
            "aria-label": "Previous",
            onClick: () => scrollByCards("prev"),
            children: /* @__PURE__ */ jsx(ArrowLeft, { className: "h-4 w-4" })
          }
        ),
        /* @__PURE__ */ jsx(
          "button",
          {
            className: "w-10 h-10 rounded-full border border-neutral-900 flex items-center justify-center transition-opacity hover:opacity-75",
            "aria-label": "Next",
            onClick: () => scrollByCards("next"),
            children: /* @__PURE__ */ jsx(ArrowRight, { className: "h-4 w-4" })
          }
        )
      ] })
    ] })
  ] });
}
function StudioCard({
  image,
  title,
  description,
  href,
  className = ""
}) {
  return /* @__PURE__ */ jsxs(
    "article",
    {
      "data-card": "studio",
      className: `flex flex-col gap-4 lg:gap-6 ${className}`,
      children: [
        /* @__PURE__ */ jsxs(
          "a",
          {
            href,
            className: "relative aspect-[4/3] w-full overflow-hidden group",
            children: [
              /* @__PURE__ */ jsx(
                "img",
                {
                  src: image,
                  alt: title,
                  className: "h-full w-full object-cover",
                  loading: "lazy"
                }
              ),
              /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-neutral-900/30 opacity-0 group-hover:opacity-100 transition-opacity backdrop-blur-[10px]" }),
              /* @__PURE__ */ jsxs("button", { className: "absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 px-4 lg:px-6 py-3 lg:py-[14px] border border-white/30 rounded-full opacity-0 group-hover:opacity-100 transition-opacity flex items-center gap-2", children: [
                /* @__PURE__ */ jsx("span", { className: "text-white text-button-small lg:text-button-big tracking-[0.02em] uppercase", children: "View" }),
                /* @__PURE__ */ jsx(ArrowRight, { className: "h-4 lg:h-5 w-4 lg:w-5 text-white" })
              ] })
            ]
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "flex flex-col lg:flex-row lg:justify-between lg:items-start gap-2 lg:gap-4", children: /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-1 lg:gap-2 flex-1", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex items-center justify-between", children: [
            /* @__PURE__ */ jsx("h3", { className: "text-heading-5 lg:text-heading-4 tracking-[-0.01em] !text-neutral-900", children: title }),
            /* @__PURE__ */ jsx(
              "a",
              {
                href,
                className: "lg:hidden flex items-center justify-center w-8 h-8  border border-neutral-900 shrink-0 hover:bg-neutral-900 hover:text-white transition-colors",
                "aria-label": `View ${title}`,
                children: /* @__PURE__ */ jsx(ArrowRight, { className: "h-4 w-4" })
              }
            )
          ] }),
          /* @__PURE__ */ jsx("p", { className: "text-body-3 !text-neutral-800/80", children: description })
        ] }) })
      ]
    }
  );
}
function StudiosSection({
  title,
  subtitle,
  items,
  isCarousel = false
}) {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-[120px]", children: [
    /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-6 lg:gap-10", children: [
      /* @__PURE__ */ jsx("div", { className: "flex flex-col lg:flex-row lg:items-end gap-4 lg:gap-6", children: /* @__PURE__ */ jsx(
        SectionHeader,
        {
          eyebrow: "STUDIO TYPES",
          title,
          className: "flex-1 max-w-[991px]"
        }
      ) }),
      /* @__PURE__ */ jsx("div", { className: "flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4", children: subtitle && /* @__PURE__ */ jsx("p", { className: "text-body-3 lg:text-body-1 text-neutral-800 max-w-[763px]", children: subtitle }) })
    ] }),
    isCarousel ? /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx("div", { className: "mt-6 flex flex-col gap-6 lg:hidden", children: items.map((item) => /* @__PURE__ */ jsx(
        StudioCard,
        {
          image: item.image,
          title: item.title,
          description: item.description,
          href: item.href
        },
        item.id
      )) }),
      /* @__PURE__ */ jsx(Carousel, { className: "mt-6 hidden lg:block", children: items.map((item) => /* @__PURE__ */ jsx(
        StudioCard,
        {
          image: item.image,
          title: item.title,
          description: item.description,
          href: item.href,
          className: "snap-start shrink-0 w-[426.67px]"
        },
        item.id
      )) })
    ] }) : /* @__PURE__ */ jsx("div", { className: "mt-6 flex flex-col gap-6 lg:grid lg:grid-cols-1 md:lg:grid-cols-2 lg:grid-cols-3", children: items.map((item) => /* @__PURE__ */ jsx(
      StudioCard,
      {
        image: item.image,
        title: item.title,
        description: item.description,
        href: item.href
      },
      item.id
    )) })
  ] }) });
}
const studioTypes = [
  {
    id: "1",
    image: "/images/home/studios/video-studios.png",
    title: "VIDEO STUDIOS",
    description: "Whether you're filming, livestreaming, or editing, our video studios are built for every stage of production.",
    href: "/studios/video"
  },
  {
    id: "2",
    image: "/images/home/studios/music-studios.png",
    title: "MUSIC STUDIOS",
    description: "Spaces to make your music heard. Treated rooms, pro interfaces, and backline on request.",
    href: "/studios/music"
  },
  {
    id: "3",
    image: "/images/home/studios/photo-studios.png",
    title: "PHOTO STUDIOS",
    description: "Our photography studios are equipped for everything from product shoots to portrait sessions.",
    href: "/studios/photo"
  }
];
function StudioTypesSection() {
  return /* @__PURE__ */ jsx(
    StudiosSection,
    {
      title: "STUDIOS TAILORED TO YOUR NEEDS",
      subtitle: "Our studios are designed with versatility in mind, providing flexible spaces that adapt to any creative pursuit.",
      items: studioTypes,
      isCarousel: true
    }
  );
}
function Accordion({
  ...props
}) {
  return /* @__PURE__ */ jsx(AccordionPrimitive.Root, { "data-slot": "accordion", ...props });
}
function AccordionItem({
  className,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    AccordionPrimitive.Item,
    {
      "data-slot": "accordion-item",
      className: cn("border-b last:border-b-0", className),
      ...props
    }
  );
}
function AccordionTrigger({
  className,
  children,
  ...props
}) {
  return /* @__PURE__ */ jsx(AccordionPrimitive.Header, { className: "flex", children: /* @__PURE__ */ jsxs(
    AccordionPrimitive.Trigger,
    {
      "data-slot": "accordion-trigger",
      className: cn(
        "focus-visible:border-ring focus-visible:ring-ring/50 flex flex-1 items-start justify-between gap-4 rounded-md py-4 text-left text-sm font-medium transition-all outline-none hover:underline focus-visible:ring-[3px] disabled:pointer-events-none disabled:opacity-50 [&[data-state=open]>svg]:rotate-180",
        className
      ),
      ...props,
      children: [
        children,
        /* @__PURE__ */ jsx(ChevronDownIcon, { className: "text-muted-foreground pointer-events-none size-4 shrink-0 translate-y-3 transition-transform duration-200 translate-x-[-8px]" })
      ]
    }
  ) });
}
function AccordionContent({
  className,
  children,
  ...props
}) {
  return /* @__PURE__ */ jsx(
    AccordionPrimitive.Content,
    {
      "data-slot": "accordion-content",
      className: "data-[state=closed]:animate-accordion-up data-[state=open]:animate-accordion-down overflow-hidden text-sm",
      ...props,
      children: /* @__PURE__ */ jsx("div", { className: cn("pt-0 pb-4", className), children })
    }
  );
}
function LocationAccordion({ locations }) {
  return /* @__PURE__ */ jsx(Accordion, { type: "single", collapsible: true, className: "w-full", defaultValue: "1", children: locations.map((location, index) => {
    const formattedIndex = String(index + 1).padStart(2, "0");
    return /* @__PURE__ */ jsxs(
      AccordionItem,
      {
        value: location.id,
        className: "border-b border-white/10",
        children: [
          /* @__PURE__ */ jsx(AccordionTrigger, { className: "hover:no-underline py-8 group [&[data-state=open]>div>svg]:rotate-180", children: /* @__PURE__ */ jsxs(
            "div",
            {
              className: "grid grid-cols-1 md:grid-cols-2 \n     items-center w-full",
              children: [
                /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-8", children: [
                  /* @__PURE__ */ jsx("span", { className: "text-heading-3 font-light text-white/30 min-w-[4rem]", children: formattedIndex }),
                  /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("h3", { className: "text-heading-5 lg:text-heading-4 tracking-[-0.01em] text-white uppercase text-left", children: location.name }) })
                ] }),
                /* @__PURE__ */ jsx("span", { className: "text-body-3 text-left  text-white/60 uppercase tracking-wider ", children: location.meta }),
                /* @__PURE__ */ jsx("div", {})
              ]
            }
          ) }),
          /* @__PURE__ */ jsxs(AccordionContent, { className: "pb-8 pl-[6rem]", children: [
            /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 lg:grid-cols-2 gap-4 mb-6", children: [
              /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden rounded-lg", children: /* @__PURE__ */ jsx(
                "img",
                {
                  src: location.image,
                  alt: `${location.name}, ${location.country}`,
                  className: "h-full w-full object-cover max-w-[500px]",
                  loading: "lazy"
                }
              ) }),
              /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden rounded-lg", children: /* @__PURE__ */ jsx(
                "img",
                {
                  src: location.image,
                  alt: "Studio interior",
                  className: "h-full w-full object-cover max-w-[500px]",
                  loading: "lazy"
                }
              ) })
            ] }),
            /* @__PURE__ */ jsx("p", { className: "text-body-3 text-white/80 mb-6 ", children: location.content }),
            /* @__PURE__ */ jsx("div", { className: "flex items-center justify-end", children: /* @__PURE__ */ jsx(
              Button,
              {
                asChild: true,
                variant: "outline-white",
                size: "lg",
                className: "bg-transparent border-white/30 text-white hover:bg-white/10 hover:border-white/50 uppercase tracking-wider font-medium ",
                children: /* @__PURE__ */ jsxs("a", { href: location.href, className: "flex items-center gap-2", children: [
                  "EXPLORE ",
                  location.name,
                  " STUDIOS",
                  /* @__PURE__ */ jsx(ArrowRight, { className: "h-4 w-4" })
                ] })
              }
            ) })
          ] })
        ]
      },
      location.id
    );
  }) });
}
const locationsData = [
  {
    id: "1",
    name: "BALI",
    country: "INDONESIA",
    meta: "VIDEO STUDIO / MUSIC STUDIO",
    content: "Bali is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/bali",
    image: "/images/home/locations/bali.png"
  },
  {
    id: "2",
    name: "BANGKOK",
    country: "THAILAND",
    meta: "VIDEO STUDIO / MUSIC STUDIO",
    content: "Bangkok is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/bangkok",
    image: "/images/home/locations/bali.png"
  },
  {
    id: "3",
    name: "MEXICO",
    country: "",
    meta: "PHOTOGRAPHY STUDIO / MUSIC STUDIO",
    content: "Mexico is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/mexico",
    image: "/images/home/locations/bali.png"
  },
  {
    id: "4",
    name: "COSTA RICA",
    country: "",
    meta: "VIDEO STUDIO / MUSIC STUDIO",
    content: "Costa Rica is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/costa-rica",
    image: "/images/home/locations/bali.png"
  },
  {
    id: "5",
    name: "NEW YORK",
    country: "",
    meta: "PHOTOGRAPHY STUDIO / FASHION STUDIO",
    content: "New York is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/new-york",
    image: "/images/home/locations/bali.png"
  },
  {
    id: "6",
    name: "MUMBAI",
    country: "",
    meta: "MUSIC STUDIO / ART STUDIO / FASHION STUDIO",
    content: "Mumbai is a city of innovation, culture, and creativity, and Nomadic Studios is proud to be at the heart of it. Our space here is designed to empower creators across disciplines, providing the tools, resources, and inspiration needed to take your work to the next level. Whether you're a local professional looking for a versatile studio or a traveling creator seeking a place to call your creative home, you'll find everything you need right here.",
    href: "/locations/mumbai",
    image: "/images/home/locations/bali.png"
  }
];
function LocationsSection() {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-neutral-900 text-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-24", children: [
    /* @__PURE__ */ jsx("div", { className: "mb-10 lg:mb-16", children: /* @__PURE__ */ jsx(
      SectionHeader,
      {
        eyebrow: "COMMUNITIES",
        title: "SEE OUR LOCATIONS",
        className: "flex-1 max-w-[991px] [&_span]:text-white/60 [&_h2]:text-white"
      }
    ) }),
    /* @__PURE__ */ jsx(LocationAccordion, { locations: locationsData })
  ] }) });
}
function EquipmentAccordion({ equipment: equipment2 }) {
  return /* @__PURE__ */ jsx(Accordion, { type: "single", collapsible: true, className: "w-full", defaultValue: "1", children: equipment2.map((item) => {
    return /* @__PURE__ */ jsxs(
      AccordionItem,
      {
        value: item.id,
        className: "border-b border-white/10",
        children: [
          /* @__PURE__ */ jsx(AccordionTrigger, { className: "hover:no-underline py-6 group [&[data-state=open]>svg]:rotate-180", children: /* @__PURE__ */ jsx(
            "h3",
            {
              className: "text-body-2 lg:text-heading-5 tracking-[-0.01em] text-white uppercase text-left pr-4",
              children: item.title
            }
          ) }),
          /* @__PURE__ */ jsxs(AccordionContent, { className: "pb-6 flex flex-col gap-4", children: [
            /* @__PURE__ */ jsx("p", { className: "text-body-3 text-white/80", children: item.content }),
            /* @__PURE__ */ jsx("div", { className: "flex", children: /* @__PURE__ */ jsx(
              Button,
              {
                variant: "outline-white",
                size: "lg",
                className: "bg-transparent border-white/30 text-white hover:bg-white/10 hover:border-white/50 uppercase tracking-wider font-medium",
                arrowRight: true,
                children: "BOOK A STUDIO"
              }
            ) })
          ] })
        ]
      },
      item.id
    );
  }) });
}
const equipmentData = [
  {
    id: "1",
    title: "CONSISTENT GEAR, WHEREVER YOU ARE",
    content: "Familiar tools make all the difference in achieving professional results. We equip our spaces globally with the gear you trust to maintain consistency in your work no matter which location you visit—one can rely on getting what you want—not to learning new setups."
  },
  {
    id: "2",
    title: "INTERCONNECTED FOR SEAMLESS COLLABORATION",
    content: "Our equipment ecosystems are designed to work together seamlessly, enabling efficient workflows and real-time collaboration across different creative disciplines and locations."
  },
  {
    id: "3",
    title: "SPECIALIZED RENTALS FOR UNIQUE NEEDS",
    content: "Beyond our standard equipment roster, we offer specialized rental options for unique project requirements, ensuring you have access to the exact tools needed for your creative vision."
  },
  {
    id: "4",
    title: "STOCKED WITH ESSENTIALS",
    content: "From cameras and audio equipment to lighting and post-production tools, our studios come fully stocked with industry-standard equipment maintained to the highest standards."
  },
  {
    id: "5",
    title: "EXPERT GUIDANCE ON-SITE",
    content: "Our technical staff is always available to provide guidance on equipment usage, troubleshooting, and optimization to ensure your creative process runs smoothly."
  }
];
function EquipmentSection() {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-neutral-900 text-white", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-24", children: [
    /* @__PURE__ */ jsx("div", { className: "mb-10 lg:mb-16", children: /* @__PURE__ */ jsx(
      SectionHeader,
      {
        eyebrow: "EQUIPMENT",
        title: "THE EQUIPMENT YOU NEED TO CREATE",
        className: "flex-1 max-w-[991px] [&_span]:text-white/60 [&_h2]:text-white"
      }
    ) }),
    /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 lg:grid-cols-2 gap-8 lg:gap-16 items-stretch", children: [
      /* @__PURE__ */ jsx("div", { className: "flex flex-col", children: /* @__PURE__ */ jsx("div", { className: "aspect-[624/748] overflow-hidden rounded-lg mb-8 flex-1", children: /* @__PURE__ */ jsx(
        "img",
        {
          src: "/images/home/equipment/studio-setup.png",
          alt: "Professional studio equipment setup",
          className: "h-full w-full object-cover",
          loading: "lazy"
        }
      ) }) }),
      /* @__PURE__ */ jsx("div", { className: "flex flex-col", children: /* @__PURE__ */ jsx("div", { className: "flex-1 flex flex-col justify-center", children: /* @__PURE__ */ jsx(EquipmentAccordion, { equipment: equipmentData }) }) })
    ] })
  ] }) });
}
function EventsSection() {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-white", children: /* @__PURE__ */ jsx("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-24", children: /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 gap-8 lg:gap-16 items-start", children: [
    /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsxs("div", { className: "mb-6 lg:mb-8 flex  gap-4", children: [
      /* @__PURE__ */ jsx(
        SectionHeader,
        {
          eyebrow: "EVENTS",
          title: "WORKSHOPS & NETWORKING",
          layout: "vertical",
          className: "[&_span]:text-black/60 [&_h2]:text-black mt-auto"
        }
      ),
      /* @__PURE__ */ jsxs("div", { className: "*:flex flex-col gap-4", children: [
        /* @__PURE__ */ jsx("p", { className: "text-body-3 lg:text-body-2 !text-black/80 max-w-[500px]", children: "Expand your knowledge and spark new ideas through our exclusive events. Workshops, discussions, and networking mixers connect you with creative minds, helping you grow and gain inspiration along the way." }),
        /* @__PURE__ */ jsx("div", { className: "flex mt-8", children: /* @__PURE__ */ jsx(
          Button,
          {
            variant: "outline-black",
            size: "lg",
            arrowRight: true,
            className: "!text-black",
            children: "SEE UPCOMING EVENTS"
          }
        ) })
      ] })
    ] }) }),
    /* @__PURE__ */ jsx(EventsGrid, { data: [] })
  ] }) }) });
}
function SupportTile({
  icon,
  title,
  description,
  href,
  className
}) {
  const Component = href ? "a" : "div";
  return /* @__PURE__ */ jsx(
    Component,
    {
      href,
      className: cn(
        "group relative flex flex-col items-left justify-center p-6 lg:p-12 ",
        "border border-white/20 rounded-lg",
        "transition-all duration-300",
        "hover:border-white/40 hover:bg-white/5",
        "h-[200px]",
        className
      ),
      children: /* @__PURE__ */ jsxs("div", { className: "flex flex-col items-left space-y-2", children: [
        /* @__PURE__ */ jsx("div", { className: "text-white/80 transition-transform duration-300 group-hover:scale-105", children: icon }),
        /* @__PURE__ */ jsx("h3", { className: "text-heading-4 transition-transform duration-300 group-hover:scale-105", children: title })
      ] })
    }
  );
}
const UploadIcon = () => /* @__PURE__ */ jsx(
  "svg",
  {
    width: "34",
    height: "33",
    viewBox: "0 0 34 33",
    fill: "none",
    xmlns: "http://www.w3.org/2000/svg",
    children: /* @__PURE__ */ jsx(
      "path",
      {
        d: "M17 2.08203C11.9375 2.08203 7.375 4.76953 4.875 9.08203C2.3125 13.457 2.3125 18.7695 4.875 23.082C7.375 27.457 11.9375 30.082 17 30.082C22 30.082 26.5625 27.457 29.0625 23.082C31.625 18.7695 31.625 13.457 29.0625 9.08203C26.5625 4.76953 22 2.08203 17 2.08203ZM17 32.082C11.25 32.082 6 29.082 3.125 24.082C0.25 19.1445 0.25 13.082 3.125 8.08203C6 3.14453 11.25 0.0820312 17 0.0820312C22.6875 0.0820312 27.9375 3.14453 30.8125 8.08203C33.6875 13.082 33.6875 19.1445 30.8125 24.082C27.9375 29.082 22.6875 32.082 17 32.082ZM17 7.08203C17.5625 7.08203 18.125 7.33203 18.5 7.70703L24.1875 12.957C24.6875 13.3945 25 14.0195 25 14.707C25 16.0195 23.875 17.082 22.5625 17.082H21V22.1445C21 23.7695 19.625 25.1445 18 25.1445H16C14.3125 25.1445 13 23.7695 13 22.1445V17.1445H11.375C10.0625 17.1445 9 16.0195 9 14.707C9 14.082 9.25 13.3945 9.75 12.957L15.4375 7.70703C15.875 7.33203 16.375 7.08203 17 7.08203ZM16.8125 9.20703L11.125 14.457C11 14.5195 11 14.582 11 14.707C11 14.957 11.125 15.082 11.375 15.082H14C14.5 15.082 15 15.582 15 16.082V22.1445C15 22.6445 15.4375 23.1445 16 23.1445H18C18.5 23.1445 19 22.6445 19 22.1445V16.1445C19 15.582 19.4375 15.1445 20 15.1445H22.5625C22.8125 15.1445 23 14.957 23 14.707C23 14.6445 22.9375 14.5195 22.875 14.457L17.1875 9.20703C17.125 9.14453 17.0625 9.08203 17 9.08203C16.875 9.08203 16.8125 9.14453 16.8125 9.20703Z",
        fill: "white"
      }
    )
  }
);
const ScaleIcon = () => /* @__PURE__ */ jsx(
  "svg",
  {
    width: "40",
    height: "34",
    viewBox: "0 0 40 34",
    fill: "none",
    xmlns: "http://www.w3.org/2000/svg",
    children: /* @__PURE__ */ jsx(
      "path",
      {
        d: "M32.9375 1.76953C33.125 2.33203 32.875 2.89453 32.3125 3.08203L23.9375 5.83203C23.625 7.39453 22.5 8.58203 21 8.95703V31.082H33C33.5625 31.082 34 31.582 34 32.082C34 32.6445 33.5625 33.082 33 33.082H20C19.4375 33.082 19 32.6445 19 32.082V8.95703C18.375 8.83203 17.8125 8.51953 17.3125 8.08203L8.3125 11.082C7.8125 11.207 7.25 10.957 7.0625 10.457C6.875 9.89453 7.1875 9.33203 7.6875 9.14453L16.1875 6.33203C16.0625 5.95703 16 5.51953 16 5.08203C16 2.89453 17.8125 1.08203 20 1.08203C21.75 1.08203 23.25 2.20703 23.8125 3.76953L31.6875 1.14453C32.25 0.957031 32.8125 1.26953 32.9375 1.76953ZM27.5 21.7695C28.5625 22.582 30.125 23.082 32 23.082C33.8125 23.082 35.375 22.582 36.4375 21.7695C37.5 20.957 38 20.0195 38 19.082H26V19.0195V19.082C26 20.0195 26.4375 20.957 27.5 21.7695ZM32 8.33203L26.625 17.082H37.3125L32 8.33203ZM24 19.082V19.0195C24 18.082 24.25 17.207 24.6875 16.3945L30.4375 6.95703C30.75 6.45703 31.375 6.08203 32 6.08203C32.5625 6.08203 33.1875 6.45703 33.5 6.95703L39.25 16.3945C39.6875 17.207 40 18.082 40 19.0195V19.082C40 22.3945 36.375 25.082 32 25.082C27.5625 25.082 24 22.3945 24 19.082ZM2 27.082C2 28.0195 2.4375 28.957 3.5 29.7695C4.5625 30.582 6.125 31.082 8 31.082C9.8125 31.082 11.375 30.582 12.4375 29.7695C13.5 28.957 14 28.0195 14 27.082H2V27.0195V27.082ZM2.625 25.082H13.3125L8 16.332L2.625 25.082ZM8 33.082C3.5625 33.082 0 30.3945 0 27.082V27.0195C0 26.082 0.25 25.207 0.6875 24.3945L6.4375 14.957C6.75 14.457 7.375 14.082 8 14.082C8.5625 14.082 9.1875 14.457 9.5 14.957L15.25 24.3945C15.6875 25.207 16 26.082 16 27.0195V27.082C16 30.3945 12.375 33.082 8 33.082ZM20 7.08203C21.125 7.08203 22 6.20703 22 5.08203C22 4.01953 21.125 3.08203 20 3.08203C18.9375 3.08203 18 4.01953 18 5.08203C18 6.20703 18.9375 7.08203 20 7.08203Z",
        fill: "white"
      }
    )
  }
);
const CreditCardIcon = () => /* @__PURE__ */ jsx(
  "svg",
  {
    width: "32",
    height: "32",
    viewBox: "0 0 32 32",
    fill: "none",
    xmlns: "http://www.w3.org/2000/svg",
    children: /* @__PURE__ */ jsx(
      "path",
      {
        d: "M10 5.08203C10 5.26953 10.0625 5.51953 10.5 5.89453C10.5625 5.95703 10.6875 6.08203 10.75 6.14453C9.8125 6.14453 8.875 6.26953 8 6.39453V5.08203C8 4.14453 8.5 3.39453 9.1875 2.76953C9.875 2.20703 10.8125 1.70703 11.875 1.33203C14 0.582031 16.875 0.0820312 20 0.0820312C23.125 0.0820312 25.9375 0.582031 28.0625 1.33203C29.125 1.70703 30.0625 2.20703 30.75 2.76953C31.4375 3.39453 32 4.14453 32 5.08203V11.582V18.707C32 19.707 31.4375 20.5195 30.75 21.082C30.125 21.707 29.1875 22.207 28.125 22.6445C27.4375 22.8945 26.75 23.082 26 23.2695V21.207C26.5 21.082 26.9375 20.8945 27.375 20.7695C28.3125 20.3945 29 20.0195 29.4375 19.6445C29.875 19.207 30 18.957 30 18.707V14.582C29.4375 14.957 28.8125 15.2695 28.125 15.5195C27.4375 15.7695 26.75 15.957 26 16.1445V14.082C26.5 13.957 26.9375 13.832 27.375 13.6445C28.3125 13.2695 29 12.8945 29.4375 12.5195C29.875 12.1445 30 11.832 30 11.6445V8.01953C29.4375 8.33203 28.8125 8.64453 28.125 8.89453C27 9.26953 25.75 9.58203 24.375 9.76953C24.25 9.64453 24.125 9.51953 24 9.45703C23.375 8.89453 22.625 8.39453 21.875 8.01953C24.0625 7.89453 25.9375 7.51953 27.4375 6.95703C28.3125 6.64453 29 6.26953 29.4375 5.89453C29.875 5.51953 30 5.26953 30 5.08203C30 4.95703 29.875 4.64453 29.4375 4.26953C29 3.95703 28.375 3.51953 27.4375 3.20703C25.5625 2.51953 22.9375 2.08203 20 2.08203C17 2.08203 14.375 2.51953 12.5625 3.20703C11.625 3.58203 10.9375 3.95703 10.5 4.33203C10.0625 4.70703 9.9375 4.95703 9.9375 5.08203H10ZM2 13.082C2 13.2695 2.0625 13.5195 2.5 13.8945C2.9375 14.2695 3.625 14.6445 4.5625 15.0195C6.375 15.6445 9 16.082 12 16.082C14.9375 16.082 17.5625 15.6445 19.4375 15.0195C20.3125 14.6445 21 14.2695 21.4375 13.8945C21.875 13.5195 22 13.2695 22 13.082C22 12.957 21.875 12.707 21.4375 12.332C21 11.957 20.375 11.582 19.4375 11.207C17.5625 10.5195 14.9375 10.082 12 10.082C9 10.082 6.375 10.5195 4.5625 11.207C3.625 11.582 2.9375 11.957 2.5 12.332C2.0625 12.707 1.9375 12.957 1.9375 13.082H2ZM0 13.082C0 12.1445 0.5 11.3945 1.1875 10.7695C1.875 10.207 2.8125 9.70703 3.875 9.33203C6 8.58203 8.875 8.08203 12 8.08203C15.125 8.08203 17.9375 8.58203 20.0625 9.33203C21.125 9.70703 22.0625 10.207 22.75 10.7695C23.4375 11.3945 24 12.1445 24 13.082V19.582V26.707C24 27.707 23.4375 28.457 22.75 29.082C22.125 29.707 21.1875 30.207 20.125 30.6445C18 31.457 15.125 31.8945 12 31.8945C8.875 31.8945 5.9375 31.457 3.8125 30.6445C2.75 30.207 1.875 29.707 1.1875 29.082C0.5 28.5195 0 27.707 0 26.707V19.582V13.082ZM22 16.0195C21.4375 16.332 20.8125 16.6445 20.0625 16.8945C17.9375 17.6445 15.125 18.082 12 18.082C8.875 18.082 6 17.6445 3.875 16.8945C3.1875 16.6445 2.5 16.332 2 16.0195V19.582C2 19.832 2.0625 20.082 2.5 20.5195C2.9375 20.8945 3.625 21.2695 4.5625 21.6445C6.375 22.332 9 22.7695 12 22.7695C14.9375 22.7695 17.5625 22.332 19.375 21.6445C20.3125 21.2695 21 20.8945 21.4375 20.5195C21.875 20.082 22 19.832 22 19.582V16.0195ZM2 26.707C2 26.8945 2.0625 27.207 2.5 27.6445C2.9375 28.0195 3.625 28.3945 4.5625 28.7695C6.375 29.457 9 29.8945 12 29.8945C14.9375 29.8945 17.5625 29.457 19.375 28.7695C20.3125 28.3945 21 28.0195 21.4375 27.6445C21.875 27.207 22 26.957 22 26.707V22.582C21.4375 22.957 20.8125 23.2695 20.125 23.5195C18 24.332 15.125 24.7695 12 24.7695C8.875 24.7695 5.9375 24.332 3.8125 23.5195C3.125 23.2695 2.5 22.957 2 22.582V26.707Z",
        fill: "white"
      }
    )
  }
);
const DollarSign = () => /* @__PURE__ */ jsx(
  "svg",
  {
    width: "32",
    height: "33",
    viewBox: "0 0 32 33",
    fill: "none",
    xmlns: "http://www.w3.org/2000/svg",
    children: /* @__PURE__ */ jsx(
      "path",
      {
        d: "M12.5625 9.08203L12.25 9.33203C8.8125 11.5195 2 16.8945 2 26.082C2 28.332 3.75 30.082 6 30.082H26C28.1875 30.082 30 28.332 30 26.082C30 16.8945 23.125 11.5195 19.6875 9.33203L19.375 9.08203H12.5625ZM18.6875 7.08203H18.75L19.125 6.51953L22.1875 2.08203H9.75L12.8125 6.51953L13.25 7.08203H18.6875ZM9.5 8.76953C10.125 8.33203 10.6875 7.95703 11.1875 7.64453L10 5.95703L7.5625 2.45703C6.9375 1.45703 7.625 0.0820312 8.8125 0.0820312H23.125C24.3125 0.0820312 25.0625 1.45703 24.375 2.45703L21.9375 5.95703L20.8125 7.64453C21.25 7.95703 21.8125 8.33203 22.4375 8.76953C26.25 11.582 32 17.207 32 26.082C32 29.3945 29.3125 32.082 26 32.082H6C2.6875 32.082 0 29.3945 0 26.082C0 17.207 5.6875 11.582 9.5 8.76953ZM17 13.582V14.707C17.5 14.7695 18 14.8945 18.5 15.0195C19 15.1445 19.3125 15.707 19.1875 16.207C19.0625 16.7695 18.5 17.082 17.9375 16.957C17.25 16.7695 16.5625 16.6445 15.9375 16.582C15.4375 16.582 14.875 16.707 14.5 16.957C14.125 17.1445 14 17.3945 14 17.7695C14 17.957 14.0625 18.1445 14.4375 18.3945C14.875 18.6445 15.4375 18.832 16.25 19.082H16.3125C17 19.2695 17.875 19.582 18.5625 19.957C19.3125 20.457 19.9375 21.207 20 22.3945C20 23.582 19.375 24.457 18.5625 24.957C18.0625 25.2695 17.5 25.457 17 25.5195V26.582C17 27.1445 16.5 27.582 16 27.582C15.4375 27.582 15 27.1445 15 26.582V25.5195C14.25 25.3945 13.625 25.1445 13.0625 24.957C12.9375 24.8945 12.75 24.832 12.625 24.832C12.125 24.6445 11.875 24.082 12 23.582C12.1875 23.0195 12.75 22.7695 13.3125 22.8945C13.4375 22.957 13.5625 23.0195 13.75 23.082C14.5625 23.332 15.25 23.582 16 23.6445C16.5625 23.6445 17.125 23.5195 17.5 23.2695C17.8125 23.082 18 22.832 17.9375 22.3945C17.9375 22.1445 17.875 21.8945 17.5 21.707C17.0625 21.3945 16.4375 21.207 15.6875 20.957H15.5625C14.875 20.7695 14.0625 20.5195 13.375 20.082C12.625 19.6445 12 18.8945 12 17.7695C12 16.582 12.625 15.707 13.4375 15.207C13.9375 14.957 14.4375 14.7695 15 14.707V13.582C15 13.082 15.4375 12.582 16 12.582C16.5 12.582 17 13.082 17 13.582Z",
        fill: "white"
      }
    )
  }
);
const supportData = [
  {
    id: "amplify",
    icon: /* @__PURE__ */ jsx(UploadIcon, {}),
    title: "AMPLIFY YOUR REACH"
  },
  {
    id: "protect",
    icon: /* @__PURE__ */ jsx(ScaleIcon, {}),
    title: "PROTECT WHAT'S YOURS"
  },
  {
    id: "finances",
    icon: /* @__PURE__ */ jsx(CreditCardIcon, {}),
    title: "MASTER YOUR FINANCES"
  },
  {
    id: "monetize",
    icon: /* @__PURE__ */ jsx(DollarSign, {}),
    title: "MONETIZE YOUR CREATIVITY"
  }
];
function SupportSection() {
  return /* @__PURE__ */ jsx("section", { className: "relative w-full bg-neutral-900", children: /* @__PURE__ */ jsxs("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 py-14 lg:py-24", children: [
    /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-1 lg:grid-cols-2 gap-8 lg:gap-16", children: [
      /* @__PURE__ */ jsx("div", { className: "flex flex-col justify-center", children: /* @__PURE__ */ jsx(
        SectionHeader,
        {
          eyebrow: "ADDITIONAL SERVICES",
          title: "SUPPORT YOU NEED",
          layout: "vertical",
          className: "[&_span]:text-white/60 [&_h2]:text-white"
        }
      ) }),
      /* @__PURE__ */ jsx("div", { className: "flex flex-col justify-center", children: /* @__PURE__ */ jsx("p", { className: "text-body-3 lg:text-body-2 text-white/80", children: "Nomadic Studios supports your creative journey with a full range of services designed to simplify your workload, protect your work, and elevate your craft. From managing your online presence to securing your intellectual property and ensuring your finances are in order, we provide the expertise you need to thrive." }) })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 md:grid-cols-2 gap-6 mt-12 lg:mt-16", children: supportData.map((item) => /* @__PURE__ */ jsx(SupportTile, { ...item }, item.id)) })
  ] }) });
}
const meta$3 = () => {
  return [
    { title: "Nomadics Studios - Your Creative Space Awaits" },
    {
      name: "description",
      content: "Connect creators with premium studio spaces, professional equipment, and a thriving creative community."
    }
  ];
};
function Index() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("div", { className: "relative", children: [
    /* @__PURE__ */ jsx(WelcomeSection, {}),
    /* @__PURE__ */ jsx(StudioTypesSection, {}),
    /* @__PURE__ */ jsx(LocationsSection, {}),
    /* @__PURE__ */ jsx(
      PeopleSection,
      {
        title: "THE PEOPLE TO HELP YOU CREATE",
        buttonText: "FIND YOUR CREATIVE SUPPORT",
        eyebrow: "OUR TEAM"
      }
    ),
    /* @__PURE__ */ jsx(EquipmentSection, {}),
    /* @__PURE__ */ jsx(EventsSection, {}),
    /* @__PURE__ */ jsx(SupportSection, {}),
    /* @__PURE__ */ jsx(MembershipSection, {})
  ] }) });
}
const route26 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Index,
  meta: meta$3
}, Symbol.toStringTag, { value: "Module" }));
const meta$2 = () => {
  return [
    { title: "Login | Nomadics" },
    { name: "description", content: "Sign in to your account" }
  ];
};
function Login() {
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsx(SignIn, {}) });
}
const route27 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Login,
  meta: meta$2
}, Symbol.toStringTag, { value: "Module" }));
const meta$1 = () => {
  return [
    { title: "Home - Nomadics Studios" },
    {
      name: "description",
      content: "Connect creators with premium studio spaces, professional equipment, and a thriving creative community."
    }
  ];
};
function Home() {
  const { t } = useLanguage();
  return /* @__PURE__ */ jsx(ProtectedRoute, { children: /* @__PURE__ */ jsx("div", { className: "min-h-screen flex items-center justify-center bg-black text-white", children: /* @__PURE__ */ jsxs("div", { className: "text-center px-4", children: [
    /* @__PURE__ */ jsx("div", { className: "mb-8", children: /* @__PURE__ */ jsx(
      "img",
      {
        src: "/images/logo-white.png",
        alt: "Nomadics Studios",
        className: "h-16 lg:h-20 w-auto mx-auto mb-8"
      }
    ) }),
    /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold mb-6 tracking-wider", children: t("home.comingSoon", "COMING SOON") }),
    /* @__PURE__ */ jsxs("p", { className: "text-lg lg:text-xl text-white/80 mb-8 max-w-2xl mx-auto leading-relaxed", children: [
      t("home.description", "We're building something amazing for creators worldwide."),
      /* @__PURE__ */ jsx("br", {}),
      t("home.subDescription", "Premium studio spaces, professional equipment, and a thriving creative community.")
    ] }),
    /* @__PURE__ */ jsx("div", { className: "flex flex-col sm:flex-row gap-4 justify-center items-center", children: /* @__PURE__ */ jsx("div", { className: "text-sm text-white/60 uppercase tracking-widest", children: t("home.stayTuned", "Stay tuned for updates") }) })
  ] }) }) });
}
const route28 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Home,
  meta: meta$1
}, Symbol.toStringTag, { value: "Module" }));
const styles = `
  .line-clamp-3 {
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  .line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
  }
  
  /* Custom date input styling */
  input[type="date"]::-webkit-calendar-picker-indicator {
    opacity: 0;
    position: absolute;
    right: 8px;
    width: 20px;
    height: 20px;
    cursor: pointer;
  }
  
  input[type="date"] {
    color-scheme: light;
  }
`;
const meta = () => {
  return [
    { title: "News - Nomadics Studios" },
    {
      name: "description",
      content: "Stay updated with the latest news, insights, and stories from the creative community at Nomadics Studios."
    }
  ];
};
const newsArticles = [
  {
    id: 1,
    title: "Travel as a failure",
    category: "Travel",
    author: "Akanksha Bhartaure",
    publishedAt: "5 min ago",
    excerpt: "A story about a trip to Barcelona and the lessons learned from unexpected challenges along the way.",
    image: "/bw (1).png",
    slug: "travel-as-a-failure"
  },
  {
    id: 2,
    title: "Travel as a fresh perspective",
    category: "Travel",
    author: "Super Admin",
    publishedAt: "2 hours ago",
    excerpt: "A story about travel and how it opens our minds to new creative possibilities and fresh perspectives.",
    image: "/image.png",
    slug: "travel-as-a-fresh-perspective"
  },
  {
    id: 3,
    title: "Creative Studios: The Future of Work",
    category: "Studios",
    author: "John Smith",
    publishedAt: "1 day ago",
    excerpt: "Discover how modern creative studios are reshaping the future of work for artists, filmmakers, and content creators worldwide.",
    image: "/image 210.png",
    slug: "creative-studios-future-of-work"
  },
  {
    id: 4,
    title: "Equipment Innovation in 2024",
    category: "Equipment",
    author: "Tech Team",
    publishedAt: "2 days ago",
    excerpt: "Explore the latest equipment innovations that are transforming creative workflows and production quality in professional studios.",
    image: "/eqi-1.png",
    slug: "equipment-innovation-2024"
  },
  {
    id: 5,
    title: "Building Creative Communities",
    category: "Community",
    author: "Community Team",
    publishedAt: "3 days ago",
    excerpt: "Learn how Nomadics Studios is fostering connections and collaboration among creators from diverse backgrounds and disciplines.",
    image: "/bw.png",
    slug: "building-creative-communities"
  },
  {
    id: 6,
    title: "Sustainable Production Practices",
    category: "Sustainability",
    author: "Green Team",
    publishedAt: "1 week ago",
    excerpt: "Discover our commitment to environmentally conscious production methods and how studios can reduce their carbon footprint.",
    image: "/service-1.png",
    slug: "sustainable-production-practices"
  }
];
function NewsCard({ article, featured = false }) {
  return /* @__PURE__ */ jsx(
    Link,
    {
      to: `/news-details/${article.slug}`,
      className: "group block transition-all duration-300 hover:transform hover:scale-[1.02]",
      children: /* @__PURE__ */ jsxs("article", { className: "bg-white shadow-lg hover:shadow-xl transition-shadow duration-300 overflow-hidden h-full flex flex-col", children: [
        /* @__PURE__ */ jsx("div", { className: "aspect-[4/3] overflow-hidden bg-neutral-100", children: /* @__PURE__ */ jsx(
          "img",
          {
            src: article.image,
            alt: article.title,
            className: "h-full w-full object-cover transition-transform duration-500 group-hover:scale-105",
            onError: (e) => {
              e.currentTarget.src = "/placeholder-news.jpg";
            }
          }
        ) }),
        /* @__PURE__ */ jsxs("div", { className: "p-6 flex-1 flex flex-col", children: [
          /* @__PURE__ */ jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsx("span", { className: "inline-block px-3 py-1 bg-gray-100 text-gray-600 text-xs font-medium uppercase tracking-wider rounded", children: article.category }) }),
          /* @__PURE__ */ jsx("h3", { className: "text-xl font-bold text-black mb-4 group-hover:text-black/80 transition-colors leading-tight", children: article.title }),
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 text-sm leading-relaxed mb-6 flex-1 line-clamp-3", children: article.excerpt }),
          /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 pt-4 border-t border-gray-100", children: [
            /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-2", children: [
              /* @__PURE__ */ jsx("div", { className: "w-8 h-8 bg-gray-200 rounded-full flex items-center justify-center", children: /* @__PURE__ */ jsx("span", { className: "text-xs font-semibold text-gray-600", children: article.author.charAt(0) }) }),
              /* @__PURE__ */ jsx("span", { className: "text-sm font-medium text-gray-700", children: article.author })
            ] }),
            /* @__PURE__ */ jsx("span", { className: "text-xs text-gray-400", children: "•" }),
            /* @__PURE__ */ jsx("span", { className: "text-xs text-gray-500", children: article.publishedAt })
          ] })
        ] })
      ] })
    }
  );
}
function NewsPage() {
  const [selectedDate, setSelectedDate] = useState("");
  const [searchTags, setSearchTags] = useState("");
  const filteredArticles = useMemo(() => {
    let filtered = newsArticles;
    if (searchTags.trim()) {
      const searchTerms = searchTags.toLowerCase().split(" ");
      filtered = filtered.filter(
        (article) => searchTerms.some(
          (term) => article.title.toLowerCase().includes(term) || article.category.toLowerCase().includes(term) || article.excerpt.toLowerCase().includes(term)
        )
      );
    }
    if (selectedDate) {
      filtered = filtered.slice(0, 3);
    }
    return filtered;
  }, [searchTags, selectedDate]);
  return /* @__PURE__ */ jsx(PublicRoute, { children: /* @__PURE__ */ jsxs("div", { className: "relative bg-white min-h-screen", children: [
    /* @__PURE__ */ jsx("style", { dangerouslySetInnerHTML: { __html: styles } }),
    /* @__PURE__ */ jsxs("section", { className: "relative h-[30vh] lg:h-[35vh] overflow-hidden", children: [
      /* @__PURE__ */ jsxs("div", { className: "absolute inset-0", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: "/bw.png",
            alt: "Discover our news",
            className: "h-full w-full object-cover"
          }
        ),
        /* @__PURE__ */ jsx("div", { className: "absolute inset-0 bg-gradient-to-b from-black/20 via-black/40 to-black/60" })
      ] }),
      /* @__PURE__ */ jsx("div", { className: "relative z-10 h-full flex items-end", children: /* @__PURE__ */ jsx("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14 pb-12 lg:pb-16", children: /* @__PURE__ */ jsxs("div", { className: "max-w-4xl", children: [
        /* @__PURE__ */ jsx("h1", { className: "text-4xl lg:text-6xl font-bold text-white mb-6 uppercase tracking-tight", children: "Discover our news" }),
        /* @__PURE__ */ jsxs("div", { className: "flex flex-col sm:flex-row gap-4 mt-8 mb-4", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex-1 relative", children: [
            /* @__PURE__ */ jsx(
              "input",
              {
                type: "date",
                value: selectedDate,
                onChange: (e) => setSelectedDate(e.target.value),
                placeholder: "mm/dd/yyyy",
                className: "w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
              }
            ),
            /* @__PURE__ */ jsx("div", { className: "absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none", children: /* @__PURE__ */ jsx("svg", { className: "w-4 h-4 text-gray-500", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" }) }) })
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex-1 relative", children: [
            /* @__PURE__ */ jsx(
              "input",
              {
                type: "text",
                value: searchTags,
                onChange: (e) => setSearchTags(e.target.value),
                placeholder: "Search By Tags",
                className: "w-full px-4 py-3 bg-white backdrop-blur-sm border border-white/20 text-black placeholder:text-gray-500 focus:outline-none focus:bg-white focus:border-gray-300 text-sm rounded-lg transition-all"
              }
            ),
            /* @__PURE__ */ jsx("div", { className: "absolute right-3 top-1/2 transform -translate-y-1/2 pointer-events-none", children: /* @__PURE__ */ jsx("svg", { className: "w-4 h-4 text-gray-500", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" }) }) })
          ] })
        ] }),
        (searchTags || selectedDate) && /* @__PURE__ */ jsxs("div", { className: "mt-4 text-white/80 text-sm", children: [
          filteredArticles.length === 0 ? "No articles found matching your search criteria" : `Showing ${filteredArticles.length} article${filteredArticles.length !== 1 ? "s" : ""}`,
          searchTags && /* @__PURE__ */ jsxs("span", { className: "ml-2", children: [
            'for "',
            searchTags,
            '"'
          ] }),
          (searchTags || selectedDate) && /* @__PURE__ */ jsx(
            "button",
            {
              onClick: () => {
                setSearchTags("");
                setSelectedDate("");
              },
              className: "ml-3 text-white/60 hover:text-white underline",
              children: "Clear filters"
            }
          )
        ] })
      ] }) }) })
    ] }),
    /* @__PURE__ */ jsx("section", { className: "bg-neutral-50 py-12 lg:py-16", children: /* @__PURE__ */ jsx("div", { className: "mx-auto w-full max-w-[1328px] px-4 lg:px-14", children: filteredArticles.length > 0 ? /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx("div", { className: "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-8 mb-16", children: filteredArticles.map((article) => /* @__PURE__ */ jsx(NewsCard, { article }, article.id)) }),
      !searchTags && !selectedDate && filteredArticles.length >= 4 && /* @__PURE__ */ jsx("div", { className: "text-center", children: /* @__PURE__ */ jsx(
        Button,
        {
          variant: "outline-black",
          size: "lg",
          arrowRight: true,
          className: "text-black",
          children: "Load More Articles"
        }
      ) })
    ] }) : (
      /* No Results State */
      /* @__PURE__ */ jsx("div", { className: "text-center py-16", children: /* @__PURE__ */ jsxs("div", { className: "max-w-md mx-auto", children: [
        /* @__PURE__ */ jsx("svg", { className: "w-16 h-16 text-gray-300 mx-auto mb-6", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 1, d: "M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" }) }),
        /* @__PURE__ */ jsx("h3", { className: "text-xl font-semibold text-gray-700 mb-2", children: "No articles found" }),
        /* @__PURE__ */ jsx("p", { className: "text-gray-500 mb-6", children: "Try adjusting your search criteria or browse all articles" }),
        /* @__PURE__ */ jsx(
          Button,
          {
            variant: "outline-black",
            size: "md",
            onClick: () => {
              setSearchTags("");
              setSelectedDate("");
            },
            children: "Show All Articles"
          }
        )
      ] }) })
    ) }) })
  ] }) });
}
const route29 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: NewsPage,
  meta
}, Symbol.toStringTag, { value: "Module" }));
const serverManifest = { "entry": { "module": "/assets/entry.client-CHstyiZt.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/components-QUj63gnA.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes": { "root": { "id": "root", "parentId": void 0, "path": "", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": true, "module": "/assets/root-B9FR2DJg.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/components-QUj63gnA.js", "/assets/index-DXk8Kqbs.js", "/assets/button-BoTqdS8w.js", "/assets/index-DiTe4udN.js", "/assets/chevron-down-BrAWRPud.js", "/assets/api-D8cydhK-.js", "/assets/useLanguage-BWE53ESH.js"], "css": [] }, "routes/events-details.$eventId": { "id": "routes/events-details.$eventId", "parentId": "root", "path": "events-details/:eventId", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/events-details._eventId-YnYfQGxR.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/button-BoTqdS8w.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/index-DXk8Kqbs.js", "/assets/components-QUj63gnA.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js"], "css": [] }, "routes/additional-information": { "id": "routes/additional-information", "parentId": "root", "path": "additional-information", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/additional-information-Bcf4mpbi.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/upload-profile-picture": { "id": "routes/upload-profile-picture", "parentId": "root", "path": "upload-profile-picture", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/upload-profile-picture-DWjMgj3R.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/book-studio.$studioId": { "id": "routes/book-studio.$studioId", "parentId": "root", "path": "book-studio/:studioId", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": true, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/book-studio._studioId-nMuP0krM.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/PlanYourVisitSection-BHLgHSt1.js", "/assets/button-BoTqdS8w.js", "/assets/index-tHycW7fo.js", "/assets/PeopleToolbar-DQRXogMq.js", "/assets/components-QUj63gnA.js", "/assets/chevron-down-BrAWRPud.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/locations.$location": { "id": "routes/locations.$location", "parentId": "root", "path": "locations/:location", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": true, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/locations._location-XoHf7MqV.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/MembershipSection-Cj06OQ5i.js", "/assets/PeopleToolbar-DQRXogMq.js", "/assets/ImageGrid-DIiTkV5D.js", "/assets/index-DXk8Kqbs.js", "/assets/EventsGrid-Cxe9-0hB.js", "/assets/PlanYourVisitSection-BHLgHSt1.js", "/assets/PeopleSection-01VErWqJ.js", "/assets/components-QUj63gnA.js", "/assets/button-BoTqdS8w.js", "/assets/index-tHycW7fo.js", "/assets/chevron-down-BrAWRPud.js"], "css": [] }, "routes/news-details.$slug": { "id": "routes/news-details.$slug", "parentId": "root", "path": "news-details/:slug", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/news-details._slug-DeefZ1gP.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/button-BoTqdS8w.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/index-DXk8Kqbs.js", "/assets/components-QUj63gnA.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js"], "css": [] }, "routes/reset-password-otp": { "id": "routes/reset-password-otp", "parentId": "root", "path": "reset-password-otp", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/reset-password-otp-CPexl0AW.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/account-settings": { "id": "routes/account-settings", "parentId": "root", "path": "account-settings", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/account-settings-DYjIhk1s.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/otp-verification": { "id": "routes/otp-verification", "parentId": "root", "path": "otp-verification", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/otp-verification-m9usgb4g.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/RouteProtection-C3OLxQmC.js"], "css": [] }, "routes/progressier[.]js": { "id": "routes/progressier[.]js", "parentId": "root", "path": "progressier.js", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": true, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/progressier_._js-l0sNRNKZ.js", "imports": [], "css": [] }, "routes/profile - Copy": { "id": "routes/profile - Copy", "parentId": "root", "path": "profile - Copy", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/profile - Copy-S_hEyD3p.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js"], "css": [] }, "routes/reset-password": { "id": "routes/reset-password", "parentId": "root", "path": "reset-password", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/reset-password-DDto4Z45.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/RouteProtection-C3OLxQmC.js"], "css": [] }, "routes/signup-details": { "id": "routes/signup-details", "parentId": "root", "path": "signup-details", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/signup-details-BXEEG4Zq.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/components-QUj63gnA.js", "/assets/RouteProtection-C3OLxQmC.js"], "css": [] }, "routes/equipment.org": { "id": "routes/equipment.org", "parentId": "routes/equipment", "path": "org", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/equipment.org-B-IFnz2H.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js"], "css": [] }, "routes/profile.$uuid": { "id": "routes/profile.$uuid", "parentId": "root", "path": "profile/:uuid", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/profile._uuid-S_hEyD3p.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js"], "css": [] }, "routes/studios.$type": { "id": "routes/studios.$type", "parentId": "root", "path": "studios/:type", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": true, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/studios._type-8h8bnF9d.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/MembershipSection-Cj06OQ5i.js", "/assets/ImageGrid-DIiTkV5D.js", "/assets/PeopleSection-01VErWqJ.js", "/assets/components-QUj63gnA.js", "/assets/button-BoTqdS8w.js", "/assets/index-tHycW7fo.js", "/assets/chevron-down-BrAWRPud.js", "/assets/PeopleToolbar-DQRXogMq.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/login-email": { "id": "routes/login-email", "parentId": "root", "path": "login-email", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/login-email-B8mEg8bk.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/RouteProtection-C3OLxQmC.js"], "css": [] }, "routes/news.$slug": { "id": "routes/news.$slug", "parentId": "routes/news", "path": ":slug", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/news._slug-CNwtKB_-.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/button-BoTqdS8w.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/index-DXk8Kqbs.js", "/assets/components-QUj63gnA.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js"], "css": [] }, "routes/StudioPage": { "id": "routes/StudioPage", "parentId": "root", "path": "StudioPage", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/StudioPage-yH2B2QD2.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js"], "css": [] }, "routes/equipment": { "id": "routes/equipment", "parentId": "root", "path": "equipment", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/equipment-CZ1_egvf.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-tHycW7fo.js"], "css": [] }, "routes/events.bk": { "id": "routes/events.bk", "parentId": "routes/events", "path": "bk", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/events.bk-B6hxFJsD.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js"], "css": [] }, "routes/services": { "id": "routes/services", "parentId": "root", "path": "services", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/services-Cz6Q81C_.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js"], "css": [] }, "routes/events": { "id": "routes/events", "parentId": "root", "path": "events", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/events-BR3FBRvM.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/useLanguage-BWE53ESH.js", "/assets/components-QUj63gnA.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/people": { "id": "routes/people", "parentId": "root", "path": "people", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/people-BYauzoPd.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/PeopleSection-01VErWqJ.js", "/assets/index-tHycW7fo.js", "/assets/button-BoTqdS8w.js", "/assets/PeopleToolbar-DQRXogMq.js", "/assets/chevron-down-BrAWRPud.js"], "css": [] }, "routes/signup": { "id": "routes/signup", "parentId": "root", "path": "signup", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/signup-BSyLo08R.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/api-D8cydhK-.js"], "css": [] }, "routes/_index": { "id": "routes/_index", "parentId": "root", "path": void 0, "index": true, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/_index-BKFlflNj.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/MembershipSection-Cj06OQ5i.js", "/assets/PeopleToolbar-DQRXogMq.js", "/assets/index-tHycW7fo.js", "/assets/chevron-down-BrAWRPud.js", "/assets/index-DiTe4udN.js", "/assets/button-BoTqdS8w.js", "/assets/PeopleSection-01VErWqJ.js", "/assets/EventsGrid-Cxe9-0hB.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/components-QUj63gnA.js", "/assets/index-DXk8Kqbs.js", "/assets/api-D8cydhK-.js"], "css": [] }, "routes/login": { "id": "routes/login", "parentId": "root", "path": "login", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/login-BM-fUZeq.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/button-BoTqdS8w.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-DXk8Kqbs.js", "/assets/RouteProtection-C3OLxQmC.js"], "css": [] }, "routes/home": { "id": "routes/home", "parentId": "root", "path": "home", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/home-CKqkdYd1.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/useLanguage-BWE53ESH.js", "/assets/index-tHycW7fo.js", "/assets/api-D8cydhK-.js", "/assets/index-DXk8Kqbs.js"], "css": [] }, "routes/news": { "id": "routes/news", "parentId": "root", "path": "news", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/news-BrVuf3N3.js", "imports": ["/assets/jsx-runtime-0DLF9kdB.js", "/assets/button-BoTqdS8w.js", "/assets/RouteProtection-C3OLxQmC.js", "/assets/index-tHycW7fo.js", "/assets/components-QUj63gnA.js", "/assets/api-D8cydhK-.js", "/assets/index-DXk8Kqbs.js"], "css": [] } }, "url": "/assets/manifest-7efd9e67.js", "version": "7efd9e67" };
const mode = "production";
const assetsBuildDirectory = "build\\client";
const basename = "/";
const future = { "v3_fetcherPersist": true, "v3_relativeSplatPath": true, "v3_throwAbortReason": true, "v3_routeConfig": false, "v3_singleFetch": true, "v3_lazyRouteDiscovery": true, "unstable_optimizeDeps": false };
const isSpaMode = false;
const publicPath = "/";
const entry = { module: entryServer };
const routes = {
  "root": {
    id: "root",
    parentId: void 0,
    path: "",
    index: void 0,
    caseSensitive: void 0,
    module: route0
  },
  "routes/events-details.$eventId": {
    id: "routes/events-details.$eventId",
    parentId: "root",
    path: "events-details/:eventId",
    index: void 0,
    caseSensitive: void 0,
    module: route1
  },
  "routes/additional-information": {
    id: "routes/additional-information",
    parentId: "root",
    path: "additional-information",
    index: void 0,
    caseSensitive: void 0,
    module: route2
  },
  "routes/upload-profile-picture": {
    id: "routes/upload-profile-picture",
    parentId: "root",
    path: "upload-profile-picture",
    index: void 0,
    caseSensitive: void 0,
    module: route3
  },
  "routes/book-studio.$studioId": {
    id: "routes/book-studio.$studioId",
    parentId: "root",
    path: "book-studio/:studioId",
    index: void 0,
    caseSensitive: void 0,
    module: route4
  },
  "routes/locations.$location": {
    id: "routes/locations.$location",
    parentId: "root",
    path: "locations/:location",
    index: void 0,
    caseSensitive: void 0,
    module: route5
  },
  "routes/news-details.$slug": {
    id: "routes/news-details.$slug",
    parentId: "root",
    path: "news-details/:slug",
    index: void 0,
    caseSensitive: void 0,
    module: route6
  },
  "routes/reset-password-otp": {
    id: "routes/reset-password-otp",
    parentId: "root",
    path: "reset-password-otp",
    index: void 0,
    caseSensitive: void 0,
    module: route7
  },
  "routes/account-settings": {
    id: "routes/account-settings",
    parentId: "root",
    path: "account-settings",
    index: void 0,
    caseSensitive: void 0,
    module: route8
  },
  "routes/otp-verification": {
    id: "routes/otp-verification",
    parentId: "root",
    path: "otp-verification",
    index: void 0,
    caseSensitive: void 0,
    module: route9
  },
  "routes/progressier[.]js": {
    id: "routes/progressier[.]js",
    parentId: "root",
    path: "progressier.js",
    index: void 0,
    caseSensitive: void 0,
    module: route10
  },
  "routes/profile - Copy": {
    id: "routes/profile - Copy",
    parentId: "root",
    path: "profile - Copy",
    index: void 0,
    caseSensitive: void 0,
    module: route11
  },
  "routes/reset-password": {
    id: "routes/reset-password",
    parentId: "root",
    path: "reset-password",
    index: void 0,
    caseSensitive: void 0,
    module: route12
  },
  "routes/signup-details": {
    id: "routes/signup-details",
    parentId: "root",
    path: "signup-details",
    index: void 0,
    caseSensitive: void 0,
    module: route13
  },
  "routes/equipment.org": {
    id: "routes/equipment.org",
    parentId: "routes/equipment",
    path: "org",
    index: void 0,
    caseSensitive: void 0,
    module: route14
  },
  "routes/profile.$uuid": {
    id: "routes/profile.$uuid",
    parentId: "root",
    path: "profile/:uuid",
    index: void 0,
    caseSensitive: void 0,
    module: route15
  },
  "routes/studios.$type": {
    id: "routes/studios.$type",
    parentId: "root",
    path: "studios/:type",
    index: void 0,
    caseSensitive: void 0,
    module: route16
  },
  "routes/login-email": {
    id: "routes/login-email",
    parentId: "root",
    path: "login-email",
    index: void 0,
    caseSensitive: void 0,
    module: route17
  },
  "routes/news.$slug": {
    id: "routes/news.$slug",
    parentId: "routes/news",
    path: ":slug",
    index: void 0,
    caseSensitive: void 0,
    module: route18
  },
  "routes/StudioPage": {
    id: "routes/StudioPage",
    parentId: "root",
    path: "StudioPage",
    index: void 0,
    caseSensitive: void 0,
    module: route19
  },
  "routes/equipment": {
    id: "routes/equipment",
    parentId: "root",
    path: "equipment",
    index: void 0,
    caseSensitive: void 0,
    module: route20
  },
  "routes/events.bk": {
    id: "routes/events.bk",
    parentId: "routes/events",
    path: "bk",
    index: void 0,
    caseSensitive: void 0,
    module: route21
  },
  "routes/services": {
    id: "routes/services",
    parentId: "root",
    path: "services",
    index: void 0,
    caseSensitive: void 0,
    module: route22
  },
  "routes/events": {
    id: "routes/events",
    parentId: "root",
    path: "events",
    index: void 0,
    caseSensitive: void 0,
    module: route23
  },
  "routes/people": {
    id: "routes/people",
    parentId: "root",
    path: "people",
    index: void 0,
    caseSensitive: void 0,
    module: route24
  },
  "routes/signup": {
    id: "routes/signup",
    parentId: "root",
    path: "signup",
    index: void 0,
    caseSensitive: void 0,
    module: route25
  },
  "routes/_index": {
    id: "routes/_index",
    parentId: "root",
    path: void 0,
    index: true,
    caseSensitive: void 0,
    module: route26
  },
  "routes/login": {
    id: "routes/login",
    parentId: "root",
    path: "login",
    index: void 0,
    caseSensitive: void 0,
    module: route27
  },
  "routes/home": {
    id: "routes/home",
    parentId: "root",
    path: "home",
    index: void 0,
    caseSensitive: void 0,
    module: route28
  },
  "routes/news": {
    id: "routes/news",
    parentId: "root",
    path: "news",
    index: void 0,
    caseSensitive: void 0,
    module: route29
  }
};
const build = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  assets: serverManifest,
  assetsBuildDirectory,
  basename,
  entry,
  future,
  isSpaMode,
  mode,
  publicPath,
  routes
}, Symbol.toStringTag, { value: "Module" }));
export {
  assetsBuildDirectory as a,
  build as b,
  basename as c,
  entry as e,
  future as f,
  isSpaMode as i,
  mode as m,
  publicPath as p,
  routes as r,
  serverManifest as s
};
