import { Link, useLocation, useNavigate } from "@remix-run/react";
import { Button } from "./ui/button";
import { MenuIcon } from "lucide-react";
import { useEffect, useState } from "react";
import {
  NavigationMenu,
  NavigationMenuContent,
  NavigationMenuItem,
  NavigationMenuLink,
  NavigationMenuList,
  NavigationMenuTrigger,
  navigationMenuTriggerStyle,
} from "./ui/navigation-menu";
import { Sheet, SheetContent, SheetHeader, SheetTrigger } from "./ui/sheet";
import { LanguageSelector } from "./LanguageSelector";
import { ProfileImage } from "./shared";
// Dropdown components are implemented natively
import {
  getUserData,
  isAuthenticated,
  handleLogout,
  API_BASE_URL,
} from "~/lib/api";
import { useLanguage } from "~/hooks/useLanguage";

interface Notification {
  id: number;
  user: string;
  type: string;
  reference_id: number;
  action: string;
  content: string;
  status: number;
  created_at: string;
  updated_at: string | null;
}

interface AppHeaderProps {
  theme?: "dark" | "white";
}

export function AppHeader({ theme = "dark" }: AppHeaderProps) {
  const location = useLocation();
  const navigate = useNavigate();
  const [isScrolled, setIsScrolled] = useState(false);
  const [isHovered, setIsHovered] = useState(false);
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [user, setUser] = useState<any>(null);
  const [showProfileDropdown, setShowProfileDropdown] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [showNotificationModal, setShowNotificationModal] = useState(false);
  const [autoLoginLoading, setAutoLoginLoading] = useState(false);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [notificationsLoading, setNotificationsLoading] = useState(false);
  const [hasUnreadNotifications, setHasUnreadNotifications] = useState(false);
  const [unreadCount, setUnreadCount] = useState(0);
  const { t, currentLanguage } = useLanguage();

  // Read auth token from localStorage when running in the browser
  const authToken =
    typeof window !== "undefined"
      ? localStorage.getItem("authToken") ||
        localStorage.getItem("auth_token") ||
        localStorage.getItem("token")
      : null;

  // Determine dashboard access from multiple possible places:
  // - authToken may be a JSON string containing { user: { dashboard_access } }
  // - `user` state (from getUserData()) may have `dashboard_access`
  // - localStorage `user_data` may include dashboard access
  const hasDashboardAccess = (() => {
    try {
      if (typeof authToken === "string") {
        // Try parse token if it's a JSON payload
        try {
          const parsed = JSON.parse(authToken as string);
          if (parsed) {
            if (parsed.user && parsed.user.dashboard_access !== undefined) {
              return !!parsed.user.dashboard_access;
            }
            if (parsed.dashboard_access !== undefined) {
              return !!parsed.dashboard_access;
            }
          }
        } catch (e) {
          // not JSON, ignore
        }
      }

      if (user && user.dashboard_access !== undefined) {
        return !!user.dashboard_access;
      }

      if (typeof window !== "undefined") {
        const ud = localStorage.getItem("user_data");
        if (ud) {
          try {
            const parsedUd = JSON.parse(ud);
            if (parsedUd.dashboard_access !== undefined)
              return !!parsedUd.dashboard_access;
            if (parsedUd.user && parsedUd.user.dashboard_access !== undefined)
              return !!parsedUd.user.dashboard_access;
          } catch (e) {
            // ignore
          }
        }
      }
    } catch (e) {
      // swallow
    }

    return false;
  })();

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 0);
    };

    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  useEffect(() => {
    const checkAuthStatus = () => {
      const authenticated = isAuthenticated();
      setIsLoggedIn(authenticated);

      if (authenticated) {
        const userData = getUserData();
        setUser(userData);
        // Fetch notifications on mount when user is logged in
        fetchNotifications();
      } else {
        setUser(null);
      }
    };

    checkAuthStatus();

    // Listen for points update event
    const handlePointsUpdate = (event: CustomEvent) => {
      if (event.detail && event.detail.points !== undefined) {
        const userData = getUserData();
        if (userData) {
          userData.point_balance = event.detail.points;
          userData.points = event.detail.points;
          setUser({ ...userData });
        }
      }
    };

    window.addEventListener(
      "userPointsUpdated",
      handlePointsUpdate as EventListener
    );

    return () => {
      window.removeEventListener(
        "userPointsUpdated",
        handlePointsUpdate as EventListener
      );
    };
    // Check auth status on mount and when route changes only
    // No need for interval polling
  }, [location.pathname]); // Re-check when route changes

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (showProfileDropdown) {
        setShowProfileDropdown(false);
      }
    };

    document.addEventListener("click", handleClickOutside);
    return () => document.removeEventListener("click", handleClickOutside);
  }, [showProfileDropdown]);

  const handleLogoutClick = async () => {
    try {
      await handleLogout();
      setIsLoggedIn(false);
      setUser(null);
      // Redirect to home page after logout
      window.location.href = "/";
    } catch (error) {
      console.error("Logout error:", error);
      // Even if API call fails, clear local state and redirect
      setIsLoggedIn(false);
      setUser(null);
      window.location.href = "/";
    }
  };

  const handleWalletClick = () => {
    navigate("/account-settings?tab=points");
  };

  const fetchNotifications = async () => {
    if (!authToken) return;

    setNotificationsLoading(true);
    try {
      const response = await fetch(`${API_BASE_URL}/notifications`, {
        method: "POST",
        headers: {
          Authorization: `Bearer ${authToken}`,
          "Content-Type": "application/json",
        },
      });

      if (response.ok) {
        const data = await response.json();
        if (data.status && data.data) {
          setNotifications(data.data);
          // Check for unread notifications (status 0)
          const unreadCount = data.data.filter(
            (n: Notification) => n.status === 0
          ).length;
          setUnreadCount(unreadCount);
          setHasUnreadNotifications(unreadCount > 0);
        }
      } else {
        console.error("Failed to fetch notifications");
      }
    } catch (error) {
      console.error("Error fetching notifications:", error);
    } finally {
      setNotificationsLoading(false);
    }
  };

  const handleNotificationClick = () => {
    setShowNotificationModal(!showNotificationModal);
  };

  const formatNotificationTime = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffInHours = Math.floor(
      (now.getTime() - date.getTime()) / (1000 * 60 * 60)
    );

    if (diffInHours < 1) {
      return "Just now";
    } else if (diffInHours < 24) {
      return `${diffInHours}h ago`;
    } else {
      const diffInDays = Math.floor(diffInHours / 24);
      return `${diffInDays}d ago`;
    }
  };

  const getNotificationIcon = (type: string) => {
    switch (type) {
      case "booking":
        return (
          <svg
            className="w-4 h-4 text-blue-500"
            fill="currentColor"
            viewBox="0 0 20 20"
          >
            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
        );
      case "point":
        return (
          <svg
            className="w-4 h-4 text-green-500"
            fill="currentColor"
            viewBox="0 0 20 20"
          >
            <path d="M4 4a2 2 0 00-2 2v4a2 2 0 002 2V6h10a2 2 0 00-2-2H4zm2 6a2 2 0 012-2h8a2 2 0 012 2v4a2 2 0 01-2 2H8a2 2 0 01-2-2v-4zm6 4a2 2 0 100-4 2 2 0 000 4z" />
          </svg>
        );
      default:
        return (
          <svg
            className="w-4 h-4 text-gray-500"
            fill="currentColor"
            viewBox="0 0 20 20"
          >
            <path
              fillRule="evenodd"
              d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
              clipRule="evenodd"
            />
          </svg>
        );
    }
  };
  const navItems = [
    {
      label: t("nav.studios"),
      href: "/studios",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.videoStudios"), href: "/studios/video" },
        { label: t("nav.musicStudios"), href: "/studios/music" },
        { label: t("nav.photoStudios"), href: "/studios/photo" },
        { label: t("nav.artStudios"), href: "/studios/art" },
        { label: t("nav.fashionStudios"), href: "/studios/fashion" },
        { label: t("nav.makerStudios"), href: "/studios/maker" },
      ],
    },
    {
      label: t("nav.locations"),
      href: "/locations",
      hasDropdown: true,
      dropdownItems: [
        { label: t("nav.bali"), href: "/locations/bali" },
        { label: t("nav.bangkok"), href: "/locations/bangkok" },
        { label: t("nav.mexico"), href: "/locations/mexico" },
        { label: t("nav.costaRica"), href: "/locations/costa-rica" },
        { label: t("nav.newYork"), href: "/locations/new-york" },
        { label: t("nav.mumbai"), href: "/locations/mumbai" },
      ],
    },
    { label: t("nav.people"), href: "/people" },
    { label: t("nav.equipment"), href: "/equipment" },
    { label: t("nav.events"), href: "/events" },
    { label: t("nav.services"), href: "/services" },
    { label: t("nav.news"), href: "/news" },
  ];

  const isDark = theme === "dark";
  const bgColor = isDark ? "bg-black" : "bg-white";
  const textColor = isDark ? "text-white" : "text-black";
  const borderColor = isDark ? "border-white/20" : "border-black/20";
  const hoverBg = isDark ? "hover:bg-white/10" : "hover:bg-black/10";
  const logoSrc = isDark ? "/images/logo-white.png" : "/images/logo-black.png";

  return (
    <header
      className={`fixed top-0 left-0 right-0 ${bgColor} z-50 transition-all duration-300 ${
        isScrolled && !isHovered
          ? `${isDark ? "bg-black/90" : "bg-white/90"} backdrop-blur-sm`
          : bgColor
      }`}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      <div className="max-w-[1400px] mx-auto px-4 lg:px-14">
        <div className="flex items-center justify-between h-16 lg:h-20 min-w-0">
          {/* Logo */}
          <Link to={isLoggedIn ? "/home" : "/"} className="block flex-shrink-0">
            <img
              src={logoSrc}
              alt={t("common.logo")}
              className="h-10 lg:h-12 w-auto"
            />
          </Link>

          {/* Desktop Navigation */}
          <div className="hidden lg:flex items-center gap-4 flex-1 justify-center mx-4 min-w-0">
            <NavigationMenu viewport={true} className="max-w-3xl">
              <NavigationMenuList className="gap-1">
                {navItems.map((item) => (
                  <NavigationMenuItem key={item.href}>
                    {item.hasDropdown ? (
                      <>
                        <NavigationMenuTrigger>
                          {item.label}
                        </NavigationMenuTrigger>
                        <NavigationMenuContent
                          className={`${bgColor} border ${borderColor} rounded-md overflow-hidden`}
                        >
                          <ul className="grid w-[600px] gap-3 p-4 grid-cols-2">
                            {item.dropdownItems?.map((dropdownItem) => (
                              <li key={dropdownItem.href}>
                                <NavigationMenuLink asChild>
                                  <Link
                                    to={dropdownItem.href}
                                    className={`block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors ${hoverBg} ${
                                      isDark
                                        ? "hover:text-white focus:text-white text-white/90"
                                        : "hover:text-black focus:text-black text-black/90"
                                    } ${
                                      isDark
                                        ? "focus:bg-white/10"
                                        : "focus:bg-black/10"
                                    }`}
                                  >
                                    <div className="text-sm lg:text-base 2xl:text-lg font-medium leading-none uppercase tracking-[0.02em]">
                                      {dropdownItem.label}
                                    </div>
                                  </Link>
                                </NavigationMenuLink>
                              </li>
                            ))}
                          </ul>
                        </NavigationMenuContent>
                      </>
                    ) : (
                      <Link to={item.href}>
                        <NavigationMenuLink
                          className={navigationMenuTriggerStyle()}
                        >
                          {item.label}
                        </NavigationMenuLink>
                      </Link>
                    )}
                  </NavigationMenuItem>
                ))}
              </NavigationMenuList>
            </NavigationMenu>
          </div>

          {/* Mobile Navigation */}
          <div className="xl:hidden">
            <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
              <SheetTrigger asChild>
                <button className={`${textColor} ${hoverBg} p-2 rounded`}>
                  <MenuIcon className="h-6 w-6" />
                  <span className="sr-only">{t("header.openMenu")}</span>
                </button>
              </SheetTrigger>
              <SheetContent
                side="right"
                className={`${bgColor} border-l ${borderColor} ${textColor} w-full sm:w-96 overflow-y-auto`}
                onInteractOutside={(e) => {
                  // Allow closing when clicking outside
                }}
              >
                <div className="px-4 sm:px-8 py-6 sm:py-12 min-h-full">
                  <SheetHeader className="p-0 mb-6 sm:mb-8">
                    <div className="flex items-center justify-between">
                      <img
                        src={logoSrc}
                        alt={t("common.companyName")}
                        className="h-8 sm:h-10"
                      />
                    </div>
                  </SheetHeader>

                  <div
                    className={`space-y-3 sm:space-y-4 mb-6 sm:mb-8 pb-6 sm:pb-8 border-b ${
                      isDark ? "border-white/10" : "border-black/10"
                    }`}
                  >
                    {!isLoggedIn ? (
                      <>
                        <Button
                          size="sm"
                          asChild
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          <Link
                            to="/login"
                            onClick={() => setMobileMenuOpen(false)}
                          >
                            {t("auth.login")}
                          </Link>
                        </Button>
                        <Button
                          size="sm"
                          asChild
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          <Link
                            to="/signup"
                            onClick={() => setMobileMenuOpen(false)}
                          >
                            {t("auth.signUp")}
                          </Link>
                        </Button>
                      </>
                    ) : (
                      <div className="space-y-3 sm:space-y-4">
                        <div className="flex items-center gap-3 p-2.5 sm:p-3 rounded-lg bg-white/5">
                          <ProfileImage
                            src={user?.profile_pic}
                            alt={t("header.profile")}
                            className="w-10 h-10 sm:w-12 sm:h-12 rounded-full object-cover"
                          />
                          <div>
                            <p
                              className={`font-medium text-sm sm:text-base lg:text-lg 2xl:text-xl ${textColor}`}
                            >
                              {user?.fname && user?.lname
                                ? `${user.fname} ${user.lname}`
                                : user?.email || "User"}
                            </p>
                            <p
                              className={`text-xs sm:text-sm lg:text-base 2xl:text-lg ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {user?.point_balance || user?.points || 0} pts
                            </p>
                          </div>
                        </div>

                        {/* Mobile Action Buttons */}
                        <div className="flex justify-between items-center gap-2 sm:gap-4 p-2.5 sm:p-3 rounded-lg bg-white/5">
                          <button
                            className="flex flex-col items-center gap-1"
                            onClick={() => {
                              setMobileMenuOpen(false);
                              handleNotificationClick();
                            }}
                          >
                            <div className="relative p-0.5">
                              <svg
                                className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor}`}
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                              >
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
                                />
                              </svg>
                              {unreadCount > 0 && (
                                <div
                                  className={`absolute -top-0.5 -right-0.5 min-w-[16px] h-4 bg-red-500 rounded-full flex items-center justify-center px-0.5 ${
                                    isDark
                                      ? "border-2 border-black"
                                      : "border-2 border-white"
                                  }`}
                                >
                                  <span className="text-white text-[9px] font-bold leading-none">
                                    {unreadCount}
                                  </span>
                                </div>
                              )}
                            </div>
                            <span
                              className={`text-xs ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.notifications")}
                            </span>
                          </button>

                          <button
                            className="flex flex-col items-center gap-1 opacity-50 cursor-not-allowed"
                            disabled
                          >
                            <svg
                              className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor} opacity-50`}
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                              />
                            </svg>
                            <span
                              className={`text-xs opacity-50 ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.chat")}
                            </span>
                          </button>

                          <button
                            className="flex flex-col items-center gap-1"
                            onClick={() => {
                              setMobileMenuOpen(false);
                              handleWalletClick();
                            }}
                          >
                            <svg
                              className={`w-5 h-5 sm:w-6 sm:h-6 ${textColor}`}
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"
                              />
                            </svg>
                            <span
                              className={`text-xs ${
                                isDark ? "text-white/60" : "text-black/60"
                              }`}
                            >
                              {t("header.wallet")}
                            </span>
                          </button>
                        </div>

                        {/* Account Settings and Dashboard for Mobile */}
                        <div className="space-y-2 sm:space-y-3">
                          <Link
                            to="/account-settings"
                            onClick={() => setMobileMenuOpen(false)}
                            className={`flex items-center gap-3 p-2.5 sm:p-3 rounded-lg ${hoverBg} transition-colors`}
                          >
                            <svg
                              className="w-5 h-5 sm:w-6 sm:h-6"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"
                              />
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                              />
                            </svg>
                            <span className="text-sm sm:text-base font-medium">
                              {t("header.accountSettings")}
                            </span>
                          </Link>

                          {hasDashboardAccess && (
                            <button
                              onClick={(e) => {
                                e.preventDefault();
                                e.stopPropagation();

                                if (!authToken) {
                                  alert(
                                    "No auth token found. Please log in again."
                                  );
                                  return;
                                }

                                try {
                                  setAutoLoginLoading(true);

                                  // Create form and submit immediately
                                  const form = document.createElement("form");
                                  form.method = "POST";
                                  form.action =
                                    "https://dashboard.nomadicstudios.net/auto-login";
                                  form.target = "_blank";
                                  form.style.display = "none";

                                  const input = document.createElement("input");
                                  input.type = "hidden";
                                  input.name = "token";
                                  input.value = authToken;

                                  form.appendChild(input);
                                  document.body.appendChild(form);

                                  // Submit form before closing menu
                                  form.submit();

                                  // Close menu after submission
                                  setTimeout(() => {
                                    setMobileMenuOpen(false);
                                    setAutoLoginLoading(false);
                                  }, 100);
                                } catch (err) {
                                  console.error(
                                    "Auto-login submit error:",
                                    err
                                  );
                                  alert("Auto-login failed. Please try again.");
                                  setAutoLoginLoading(false);
                                }
                              }}
                              className={`flex items-center gap-3 p-2.5 sm:p-3 rounded-lg ${hoverBg} transition-colors w-full text-left`}
                              disabled={autoLoginLoading}
                            >
                              <svg
                                className="w-5 h-5 sm:w-6 sm:h-6"
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                              >
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M12 15v2"
                                />
                                <rect
                                  x="4"
                                  y="11"
                                  width="16"
                                  height="8"
                                  rx="2"
                                  strokeWidth={2}
                                  stroke="currentColor"
                                  fill="none"
                                />
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M7 11V9a5 5 0 0110 0v2"
                                />
                              </svg>
                              <span className="text-sm sm:text-base font-medium">
                                {autoLoginLoading
                                  ? "Signing in..."
                                  : "Dashboard"}
                              </span>
                            </button>
                          )}
                        </div>

                        <Button
                          size="sm"
                          onClick={() => {
                            setMobileMenuOpen(false);
                            handleLogoutClick();
                          }}
                          variant={isDark ? "outline-white" : "outline-black"}
                          className="w-full h-10 sm:h-12 text-sm sm:text-base font-medium"
                        >
                          {t("auth.signOut")}
                        </Button>
                      </div>
                    )}
                  </div>

                  <nav className="space-y-4 sm:space-y-6 pb-8 sm:pb-10">
                    {navItems.map((item) => (
                      <div key={item.href} className="space-y-2 sm:space-y-3">
                        <Link
                          to={item.href}
                          onClick={() => setMobileMenuOpen(false)}
                          className={`block ${textColor} ${
                            isDark
                              ? "hover:text-white/70"
                              : "hover:text-black/70"
                          } transition-colors text-base sm:text-lg font-medium uppercase tracking-wide`}
                        >
                          {item.label}
                        </Link>
                        {item.hasDropdown && item.dropdownItems && (
                          <div
                            className={`pl-3 sm:pl-4 space-y-1.5 sm:space-y-2 border-l ${
                              isDark ? "border-white/10" : "border-black/10"
                            }`}
                          >
                            {item.dropdownItems.map((dropdownItem) => (
                              <Link
                                key={dropdownItem.href}
                                to={dropdownItem.href}
                                onClick={() => setMobileMenuOpen(false)}
                                className={`block ${
                                  isDark
                                    ? "text-white/60 hover:text-white"
                                    : "text-black/60 hover:text-black"
                                } transition-colors text-xs sm:text-sm uppercase tracking-wide`}
                              >
                                {dropdownItem.label}
                              </Link>
                            ))}
                          </div>
                        )}
                      </div>
                    ))}
                  </nav>

                  {/* Language Selector for Mobile */}
                  <div
                    className={`mt-8 pt-6 border-t ${
                      isDark ? "border-white/10" : "border-black/10"
                    }`}
                  >
                    <div className="flex items-center justify-between">
                      <span className={`text-sm font-medium ${textColor}`}>
                        {t(
                          "languageSelector.languageAndRegion",
                          "Language & Region"
                        )}
                      </span>
                      <LanguageSelector />
                    </div>
                  </div>
                </div>
              </SheetContent>
            </Sheet>
          </div>

          {/* Desktop Auth Buttons / Profile */}
          <div className="hidden xl:flex items-center gap-3 flex-shrink-0 min-w-fit">
            {!isLoggedIn ? (
              <>
                <Button
                  className={`${
                    isDark ? "!text-white/90" : "!text-black/90"
                  } whitespace-nowrap ${
                    currentLanguage === "sp" ? "min-w-[120px]" : ""
                  }`}
                  asChild
                  variant={isDark ? "fill-black" : "outline-black"}
                  responsive
                >
                  <Link to="/login">{t("auth.login")}</Link>
                </Button>
                <Button
                  className={`${
                    isDark ? "!text-white/90" : "!text-black/90"
                  } whitespace-nowrap ${
                    currentLanguage === "sp" ? "min-w-[120px]" : ""
                  }`}
                  asChild
                  variant={isDark ? "fill-black" : "outline-black"}
                  responsive
                >
                  <Link to="/signup">{t("auth.signUp")}</Link>
                </Button>
              </>
            ) : (
              <div className="flex items-center gap-6">
                {/* Notification Bell */}
                <button
                  className="relative p-1"
                  onClick={handleNotificationClick}
                >
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
                    />
                  </svg>
                  {unreadCount > 0 && (
                    <div className="absolute -top-0.5 -right-0.5 min-w-[18px] h-[18px] bg-red-500 rounded-full flex items-center justify-center px-0.5 border-2 border-black">
                      <span className="text-white text-[10px] font-bold leading-none">
                        {unreadCount}
                      </span>
                    </div>
                  )}
                </button>

                {/* Chat Icon */}
                {/* <button className="" onClick={() => navigate("/discuss")}> */}
                <button className="opacity-50 cursor-not-allowed" disabled>
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                    />
                  </svg>
                </button>

                {/* Points */}
                <button
                  className={`flex items-center gap-2 ${textColor} hover:opacity-80 transition-opacity`}
                  onClick={handleWalletClick}
                >
                  <span className="text-sm font-medium">
                    {user?.point_balance || user?.points || 0} pts
                  </span>
                </button>

                {/* Wallet Icon */}
                <button className="" onClick={handleWalletClick}>
                  <svg
                    className={`w-6 h-6 ${textColor}`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"
                    />
                  </svg>
                </button>

                {/* Profile Dropdown */}
                <div className="relative">
                  <button
                    onClick={() => setShowProfileDropdown(!showProfileDropdown)}
                    className="w-10 h-10 rounded-full overflow-hidden focus:outline-none focus:ring-2 focus:ring-white/20"
                  >
                    <ProfileImage
                      src={user?.profile_pic}
                      alt={t("header.profile")}
                      className="w-full h-full object-cover"
                    />
                  </button>

                  {showProfileDropdown && (
                    <div
                      className={`absolute right-0 mt-2 w-56 sm:w-64 ${bgColor} border ${borderColor} rounded-lg shadow-lg z-50`}
                    >
                      {/* User Name Header */}
                      <div className="px-3 sm:px-4 py-2.5 sm:py-3 border-b border-gray-200">
                        <p
                          className={`font-medium text-sm sm:text-base ${textColor}`}
                        >
                          {user?.fname && user?.lname
                            ? `${user.fname} ${user.lname}`
                            : user?.email || "User"}
                        </p>
                      </div>

                      {/* Menu Items */}
                      <div className="py-1">
                        <Link
                          to="/home"
                          className={`px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor} block`}
                          onClick={() => setShowProfileDropdown(false)}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("nav.home")}
                          </span>
                        </Link>

                        <Link
                          to="/account-settings"
                          className={`px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor} block`}
                          onClick={() => setShowProfileDropdown(false)}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"
                            />
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("header.accountSettings")}
                          </span>
                        </Link>

                        {hasDashboardAccess && (
                          <button
                            onClick={() => {
                              setShowProfileDropdown(false);
                              if (!authToken) {
                                alert("No auth token found.");
                                return;
                              }
                              try {
                                setAutoLoginLoading(true);
                                // Create a hidden form and submit it so the browser performs a
                                // full navigational POST to the SSO endpoint. This avoids CORS
                                // preflight and allows the backend to set cookies and redirect.
                                const form = document.createElement("form");
                                form.method = "POST";
                                form.action =
                                  "https://dashboard.nomadicstudios.net/auto-login";
                                form.style.display = "none";
                                const input = document.createElement("input");
                                input.type = "hidden";
                                input.name = "token";
                                input.value = authToken;
                                form.appendChild(input);
                                document.body.appendChild(form);
                                form.submit();
                              } catch (err) {
                                console.error("Auto-login submit error:", err);
                                alert("Auto-login failed. Please try again.");
                              } finally {
                                setAutoLoginLoading(false);
                              }
                            }}
                            className={`px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor} w-full text-left`}
                            disabled={autoLoginLoading}
                          >
                            <svg
                              className="w-4 h-4 sm:w-5 sm:h-5"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M12 15v2"
                              />
                              <rect
                                x="4"
                                y="11"
                                width="16"
                                height="8"
                                rx="2"
                                strokeWidth={2}
                                stroke="currentColor"
                                fill="none"
                              />
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M7 11V9a5 5 0 0110 0v2"
                              />
                            </svg>
                            <span className="text-sm sm:text-base">
                              {autoLoginLoading ? "Signing in..." : "Dashboard"}
                            </span>
                          </button>
                        )}

                        <button
                          onClick={() => {
                            setShowProfileDropdown(false);
                            handleLogoutClick();
                          }}
                          className={`w-full text-left px-3 sm:px-4 py-2.5 sm:py-3 ${hoverBg} cursor-pointer flex items-center gap-2.5 sm:gap-3 ${textColor} hover:${textColor}`}
                        >
                          <svg
                            className="w-4 h-4 sm:w-5 sm:h-5"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"
                            />
                          </svg>
                          <span className="text-sm sm:text-base">
                            {t("auth.signOut")}
                          </span>
                        </button>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Notification Modal */}
      {showNotificationModal && (
        <div
          className="absolute top-full right-0 z-50 mt-2 mr-4 lg:mr-8"
          style={{ width: "400px", maxWidth: "90vw" }}
        >
          <div
            className={`${
              isDark
                ? "bg-black/95 border-white/20 text-white"
                : "bg-white/95 border-black/20 text-black"
            } backdrop-blur-lg border rounded-2xl shadow-2xl max-h-96 overflow-hidden`}
          >
            <div className="p-4 border-b border-current/10">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <h3 className="text-lg font-medium">Notifications</h3>
                  {unreadCount > 0 && (
                    <span className="inline-flex items-center justify-center px-2 py-0.5 text-xs font-bold text-white bg-red-500 rounded-full">
                      {unreadCount}
                    </span>
                  )}
                </div>
                <button
                  onClick={() => setShowNotificationModal(false)}
                  className={`p-1 rounded-lg ${
                    isDark ? "hover:bg-white/10" : "hover:bg-black/10"
                  } transition-colors`}
                >
                  <svg
                    className="w-5 h-5"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M6 18L18 6M6 6l12 12"
                    />
                  </svg>
                </button>
              </div>
            </div>

            <div className="overflow-y-auto max-h-80">
              {notificationsLoading ? (
                <div className="flex items-center justify-center py-8">
                  <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-current"></div>
                  <span className="ml-2 text-sm">Loading...</span>
                </div>
              ) : notifications.length > 0 ? (
                <div className="divide-y divide-current/10">
                  {notifications.map((notification) => (
                    <div
                      key={notification.id}
                      className={`p-4 hover:${
                        isDark ? "bg-white/5" : "bg-black/5"
                      } transition-colors ${
                        notification.status === 0 ? "bg-blue-500/10" : ""
                      }`}
                    >
                      <div className="flex items-start gap-3">
                        <div className="flex-shrink-0 mt-1">
                          {getNotificationIcon(notification.type)}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div
                            className="text-sm text-current/90 leading-relaxed"
                            dangerouslySetInnerHTML={{
                              __html: notification.content,
                            }}
                          />
                          <div className="flex items-center justify-between mt-2">
                            <span className="text-xs text-current/60">
                              {formatNotificationTime(notification.created_at)}
                            </span>
                            <div className="flex items-center gap-2">
                              <span
                                className={`inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium ${
                                  notification.type === "booking"
                                    ? "bg-blue-100 text-blue-800"
                                    : notification.type === "point"
                                    ? "bg-green-100 text-green-800"
                                    : "bg-gray-100 text-gray-800"
                                }`}
                              >
                                {notification.type}
                              </span>
                              {notification.status === 0 && (
                                <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                              )}
                            </div>
                          </div>
                          {notification.type === "booking" && (
                            <div className="mt-3">
                              <Button
                                size="sm"
                                variant={
                                  isDark ? "outline-white" : "outline-black"
                                }
                                className="w-full text-xs"
                                onClick={() => {
                                  setShowNotificationModal(false);
                                  navigate(
                                    `/booking-confirmed/${notification.reference_id}`
                                  );
                                }}
                              >
                                View Booking
                              </Button>
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <svg
                    className="w-12 h-12 mx-auto text-current/40 mb-2"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
                    />
                  </svg>
                  <p className="text-current/60 text-sm">
                    No notifications yet
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </header>
  );
}
