import React, { useState, useEffect } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

// Stripe types
declare global {
  interface Window {
    Stripe: any;
  }
}

const STRIPE_KEY = "pk_test_51NpyhWBfA6yrUxU0IRBFtBsOIk1GXsmQlD1Z8RouELcyISQFBOSfCG5cDohyKBIZPkrHWbSFYcOWdDU2cc9fcjih00CpUxQIAN";
const API_BASE_URL = "https://dashboard.nomadicstudios.net/api";

const AddCreditCard: React.FC = () => {
  const navigate = useNavigate();
  const location = useLocation();
  const { t } = useLanguage();
  const [isVisible, setIsVisible] = useState(false);
  const [nameOnCard, setNameOnCard] = useState("");
  const [stripe, setStripe] = useState<any>(null);
  const [cardElement, setCardElement] = useState<any>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");

  // Get data from previous page or localStorage
  const getPlanData = () => {
    const storedData = typeof window !== 'undefined' ? localStorage.getItem('signup_plan_data') : null;
    if (storedData) {
      const parsed = JSON.parse(storedData);
      return {
        plan: parsed.tierData?.name || parsed.plan || parsed.userData?.plan,
        price: parsed.tierData?.price,
        uuid: parsed.uuid || location.state?.uuid,
        userData: parsed.userData || location.state?.userData
      };
    }
    return {
      plan: location.state?.selectedPlan || location.state?.userData?.plan,
      price: location.state?.selectedPrice,
      uuid: location.state?.uuid,
      userData: location.state?.userData
    };
  };
  
  const planData = getPlanData();
  const uuid = planData.uuid;
  const userData = planData.userData;
  const selectedPlan = planData.plan || 'STARVING ARTIST';
  const selectedPrice = planData.price ? `${planData.price}` : '1149';
  
  // Check if it's a free/Guest plan
  const isFreePlan = selectedPlan.toLowerCase().includes('guest') || planData.price === 0 || selectedPrice === '0';

  // Debug: Log plan data
  useEffect(() => {
    console.log('AddCreditCard - Plan Data:', { selectedPlan, selectedPrice, uuid, isFreePlan });
  }, [selectedPlan, selectedPrice, uuid, isFreePlan]);

  useEffect(() => {
    // Redirect if no UUID
    if (!uuid) {
      navigate('/signup');
      return;
    }
    
    setIsVisible(true);
  }, [uuid, navigate]);

  // Load Stripe and initialize elements
  useEffect(() => {
    let mounted = true;
    
    const loadStripe = async () => {
      try {
        // Load Stripe script if not already loaded
        if (!window.Stripe) {
          console.log('Loading Stripe script...');
          const script = document.createElement('script');
          script.src = 'https://js.stripe.com/v3/';
          script.async = true;
          document.head.appendChild(script);
          
          await new Promise<void>((resolve, reject) => {
            script.onload = () => {
              console.log('Stripe script loaded successfully');
              resolve();
            };
            script.onerror = () => reject(new Error('Failed to load Stripe'));
          });
        }

        if (!mounted) return;

        console.log('Initializing Stripe with key:', STRIPE_KEY.substring(0, 20) + '...');
        const stripeInstance = window.Stripe(STRIPE_KEY);
        const elementsInstance = stripeInstance.elements();
        
        const cardElementInstance = elementsInstance.create('card', {
          style: {
            base: {
              fontSize: '16px',
              color: '#ffffff',
              backgroundColor: 'transparent',
              fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
              fontWeight: '400',
              lineHeight: '24px',
              padding: '12px 16px',
              '::placeholder': {
                color: 'rgba(255, 255, 255, 0.6)',
              },
              iconColor: '#ffffff',
            },
            invalid: {
              color: '#fa755a',
              iconColor: '#fa755a'
            },
            complete: {
              color: '#ffffff',
            }
          },
          hidePostalCode: true
        });

        if (!mounted) return;

        setStripe(stripeInstance);
        setCardElement(cardElementInstance);

        // Mount card element after DOM is ready
        setTimeout(() => {
          if (!mounted) return;
          
          const cardContainer = document.getElementById('card-element');
          if (cardContainer) {
            console.log('Mounting Stripe Card Element...');
            cardElementInstance.mount('#card-element');
            
            // Add event listeners for better UX
            cardElementInstance.on('ready', () => {
              console.log('Stripe Card Element is ready and mounted');
            });
            
            cardElementInstance.on('change', (event: any) => {
              if (event.error) {
                setErrorMessage(event.error.message);
              } else {
                setErrorMessage('');
              }
            });
          } else {
            console.error('Card element container not found');
          }
        }, 1000);
      } catch (error) {
        console.error('Error loading Stripe:', error);
        setErrorMessage('Failed to load payment system. Please refresh the page.');
      }
    };

    loadStripe();
    
    // Cleanup function
    return () => {
      mounted = false;
    };
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!stripe || !cardElement) {
      setErrorMessage("Stripe is not loaded properly. Please refresh the page.");
      return;
    }

    if (!nameOnCard.trim()) {
      setErrorMessage("Name is required field.");
      return;
    }

    setIsProcessing(true);
    setErrorMessage("");

    try {
      // Create payment method directly
      const { paymentMethod, error } = await stripe.createPaymentMethod({
        type: 'card',
        card: cardElement,
        billing_details: {
          name: nameOnCard
        }
      });

      if (error) {
        console.error('Stripe error:', error);
        setErrorMessage(error.message);
        setIsProcessing(false);
      } else {
        // Successfully created payment method, now save to API
        await savePaymentMethod(paymentMethod.id);
      }
    } catch (error) {
      console.error('Payment processing error:', error);
      setErrorMessage("An error occurred while processing your payment method.");
      setIsProcessing(false);
    }
  };

  const savePaymentMethod = async (paymentMethodId: string) => {
    try {
      // Get the signup token from localStorage
      const signupToken = localStorage.getItem('signup_token');
      
      if (!signupToken) {
        setErrorMessage('Authentication token not found. Please try signing up again.');
        setIsProcessing(false);
        return;
      }
      
      const formData = new FormData();
      formData.append('name', nameOnCard);
      formData.append('token', paymentMethodId);
      formData.append('type', 'signup');

      const response = await fetch(`${API_BASE_URL}/add-payment-method`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${signupToken}`,
        },
        body: formData
      });

      const result = await response.json();

      if (response.ok && result.status) {
        // Navigate to additional information page
        navigate('/additional-information', {
          state: {
            uuid,
            userData
          }
        });
      } else {
        setErrorMessage(result.message || result.error || "Failed to save payment method. Please try again.");
        setIsProcessing(false);
      }
    } catch (error) {
      console.error('API error:', error);
      setErrorMessage("Failed to save payment method. Please try again.");
      setIsProcessing(false);
    }
  };

  const handleSkip = () => {
    // Navigate to additional information page without payment method
    navigate('/additional-information', {
      state: {
        uuid,
        userData
      }
    });
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0 py-12">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            ADD PAYMENT METHOD
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            Secure your account with a payment method
          </p>
        </div>

        {/* Payment Card */}
        <div className={`w-full max-w-lg transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            
            {/* Selected Plan Info */}
            <div className="mb-8 text-center">
              <div className="inline-flex items-center bg-orange-500/20 border border-orange-400/30 rounded-xl px-6 py-3 mb-2">
                <div className="text-orange-400 text-sm font-semibold uppercase tracking-wider">
                  SELECTED PLAN: {selectedPlan}
                </div>
              </div>
              <div className="text-white text-2xl font-bold">
                {selectedPrice}/month
              </div>
            </div>

            <style>{`
              /* Custom styles for Stripe Elements */
              .StripeElement {
                height: 40px;
                padding: 10px 12px;
                background-color: transparent !important;
                color: white !important;
                border: none;
              }
              
              .StripeElement--focus {
                box-shadow: none;
              }
              
              .StripeElement--invalid {
                border-color: #fa755a;
              }
              
              .StripeElement--webkit-autofill {
                background-color: transparent !important;
              }
              
              #card-element .StripeElement {
                width: 100%;
                height: auto;
                padding: 12px 0;
              }
            `}</style>

            {/* Error Message */}
            {errorMessage && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {errorMessage}
              </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Name on Card */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  Name on Card
                </label>
                <input
                  type="text"
                  placeholder="John Doe"
                  value={nameOnCard}
                  onChange={(e) => setNameOnCard(e.target.value)}
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                  required
                  disabled={isProcessing}
                />
              </div>

              {/* Stripe Card Element */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  Card Information
                </label>
                <div 
                  id="card-element"
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus-within:ring-2 focus-within:ring-orange-400 focus-within:border-orange-400 backdrop-blur-sm transition-all"
                  style={{
                    minHeight: '56px',
                    display: 'block',
                    backgroundColor: 'rgba(255, 255, 255, 0.1)',
                    position: 'relative'
                  }}
                >
                  {/* Loading state */}
                  {!cardElement && (
                    <div className="flex items-center justify-center h-full">
                      <span className="text-white/40 text-sm">Loading payment form...</span>
                    </div>
                  )}
                </div>
                <p className="text-white/40 text-xs">
                  Enter your card number, expiry date, and CVC. All fields are in one input.
                </p>
              </div>

              {/* Submit Button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full uppercase tracking-wide"
                arrowRight={!isProcessing}
                disabled={isProcessing || !stripe}
              >
                {isProcessing ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    Processing...
                  </>
                ) : (
                  'Add Card & Continue'
                )}
              </Button>

              {/* Skip Button - Only show for free/Guest plan */}
              {isFreePlan && (
                <Button
                  type="button"
                  variant="outline-white"
                  size="md"
                  className="w-full uppercase tracking-wide"
                  onClick={handleSkip}
                  disabled={isProcessing}
                >
                  Skip for now
                </Button>
              )}
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AddCreditCard;
