import React, { useState, useEffect } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';
import { authAPI, locationsAPI, type Location } from '~/lib/api';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

const AdditionalInformation: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  // Get UUID from previous page
  const uuid = location.state?.uuid;
  const userData = location.state?.userData;

  const [formData, setFormData] = useState({
    home_city: '',
    creator_interest: [] as string[],
    profile_link: '',
    instagram_link: '',
    facebook_link: '',
    twitter_link: '',
    sm_link: '',
    referred_by: ''
  });

  // Creative interests options
  const creativeInterests = [
    'Fine Art',
    'Fashion', 
    'Music',
    'Photography',
    'Video',
    'Maker Space'
  ];

  // State for locations
  const [locations, setLocations] = useState<Location[]>([]);
  const [locationsLoading, setLocationsLoading] = useState(true);

  useEffect(() => {
    // Redirect if no UUID
    if (!uuid) {
      navigate('/signup');
      return;
    }
    
    setIsVisible(true);
    
    // Fetch locations
    const fetchLocations = async () => {
      try {
        setLocationsLoading(true);
        const response = await locationsAPI.getLocations();
        if (response.status && response.data) {
          // Remove duplicates based on id and location name
          const uniqueLocations = response.data.filter((location, index, self) => 
            index === self.findIndex(l => l.id === location.id && l.location === location.location)
          );
          setLocations(uniqueLocations);
        } else {
          console.error('Failed to fetch locations:', response.error);
        }
      } catch (error) {
        console.error('Error fetching locations:', error);
      } finally {
        setLocationsLoading(false);
      }
    };
    
    fetchLocations();
  }, [uuid, navigate]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };

  const handleInterestToggle = (interest: string) => {
    setFormData(prev => ({
      ...prev,
      creator_interest: prev.creator_interest.includes(interest)
        ? prev.creator_interest.filter(i => i !== interest)
        : [...prev.creator_interest, interest]
    }));
    if (error) setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.home_city.trim()) {
      setError(t("auth.additionalInfo.errors.homeCityRequired", "Home city is required"));
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const response = await authAPI.saveAdditionalInfo({
        uuid,
        home_city: formData.home_city, // Send location ID as selected
        creator_interest: formData.creator_interest,
        profile_link: formData.profile_link || undefined,
        instagram_link: formData.instagram_link || undefined,
        facebook_link: formData.facebook_link || undefined,
        twitter_link: formData.twitter_link || undefined,
        sm_link: formData.sm_link || undefined,
        referred_by: formData.referred_by || undefined
      });

      if (response.status) {
        // Clear the signup token after successful signup completion
        if (typeof window !== "undefined") {
          localStorage.removeItem('signup_token');
        }
        
        // Success! Navigate to login with success message
        navigate('/login', {
          state: {
            message: t("auth.additionalInfo.success", "Account created successfully! Please login to continue."),
            type: 'success'
          }
        });
      } else {
        setError(response.error || t("auth.additionalInfo.errors.saveFailed", "Failed to save information. Please try again."));
      }
    } catch (err) {
      console.error('Additional info save error:', err);
      setError(err instanceof Error ? err.message : t("auth.additionalInfo.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <>
      <style>{`
        select option {
          background-color: #1f2937 !important;
          color: white !important;
          padding: 8px !important;
        }
        select option:hover {
          background-color: #374151 !important;
        }
        select option:checked {
          background-color: #f97316 !important;
        }
      `}</style>
      <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0 py-12">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.additionalInfo.title", "We would like to know you more")}
          </h1>
        </div>

        {/* Form Card */}
        <div className={`w-full max-w-2xl transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            
            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Home City */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.additionalInfo.homeCity", "What is your home city?")} *
                </label>
                <div className="relative">
                  <select
                    name="home_city"
                    value={formData.home_city}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-gray-800/90 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all appearance-none cursor-pointer hover:bg-gray-700/90"
                    required
                    disabled={isLoading || locationsLoading}
                    style={{ 
                      colorScheme: 'dark'
                    }}
                  >
                    <option value="" className="bg-gray-800 text-gray-400">
                      {locationsLoading ? t("auth.additionalInfo.loadingLocations", "Loading locations...") : t("auth.additionalInfo.selectLocation", "Select a location")}
                    </option>
                    {locations.map((location) => (
                      <option 
                        key={`${location.id}-${location.location}`}
                        value={location.id.toString()} 
                        className="bg-gray-800 text-white py-2"
                      >
                        {location.location}
                      </option>
                    ))}
                  </select>
                  <svg className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-white/60 pointer-events-none" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </div>
              </div>

              {/* Creative Interests */}
              <div className="space-y-4">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.additionalInfo.creativeInterests", "What are you creating?")}
                </label>
                <div className="grid grid-cols-2 gap-4">
                  {creativeInterests.map((interest) => (
                    <div
                      key={interest}
                      className={`flex items-center p-4 rounded-xl border-2 cursor-pointer transition-all ${
                        formData.creator_interest.includes(interest)
                          ? 'border-orange-400 bg-orange-500/20'
                          : 'border-white/20 bg-white/5 hover:bg-white/10'
                      }`}
                      onClick={() => handleInterestToggle(interest)}
                    >
                      <div className={`w-5 h-5 rounded border-2 transition-all duration-200 ${
                        formData.creator_interest.includes(interest)
                          ? 'border-orange-400 bg-orange-400'
                          : 'border-white/40'
                      }`}>
                        {formData.creator_interest.includes(interest) && (
                          <svg className="w-3 h-3 text-white ml-0.5 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                            <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                          </svg>
                        )}
                      </div>
                      <span className="text-white ml-3 font-medium">{interest}</span>
                    </div>
                  ))}
                </div>
              </div>

              {/* Link to Profile */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.additionalInfo.profileLink", "Link to profile")}
                </label>
                <input
                  type="url"
                  name="profile_link"
                  placeholder={t("auth.additionalInfo.linkPlaceholder", "Link")}
                  value={formData.profile_link}
                  onChange={handleInputChange}
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                  disabled={isLoading}
                />
              </div>

              {/* Social Media Links */}
              <div className="space-y-4">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.additionalInfo.socialMedia", "Link to social media")}
                </label>
                <div className="space-y-3">
                  <input
                    type="url"
                    name="instagram_link"
                    placeholder={t("auth.additionalInfo.instagramLink", "Link to Instagram")}
                    value={formData.instagram_link}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    disabled={isLoading}
                  />
                  <input
                    type="url"
                    name="facebook_link"
                    placeholder={t("auth.additionalInfo.facebookLink", "Link to Facebook")}
                    value={formData.facebook_link}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    disabled={isLoading}
                  />
                  <input
                    type="url"
                    name="twitter_link"
                    placeholder={t("auth.additionalInfo.twitterLink", "Link to Twitter")}
                    value={formData.twitter_link}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    disabled={isLoading}
                  />
                  <input
                    type="url"
                    name="sm_link"
                    placeholder={t("auth.additionalInfo.smLink", "Link to SM")}
                    value={formData.sm_link}
                    onChange={handleInputChange}
                    className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    disabled={isLoading}
                  />
                </div>
              </div>

              {/* Referral */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">
                  {t("auth.additionalInfo.referredBy", "Please provide the name of the member who referred you")}
                </label>
                <input
                  type="text"
                  name="referred_by"
                  placeholder={t("auth.additionalInfo.namePlaceholder", "Name")}
                  value={formData.referred_by}
                  onChange={handleInputChange}
                  className="w-full px-4 py-4 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                  disabled={isLoading}
                />
              </div>

              {/* Submit Button */}
              <div className="pt-6">
                <Button
                  type="submit"
                  variant="fill-white"
                  size="lg"
                  className="w-full"
                  disabled={isLoading}
                  arrowRight={!isLoading}
                >
                  {isLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                      {t("auth.additionalInfo.saving", "Saving...")}
                    </>
                  ) : (
                    t("auth.additionalInfo.save", "Save")
                  )}
                </Button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
    </>
  );
};

export default AdditionalInformation;