import React, { useState, useEffect } from "react";
import { useNavigate, useLocation } from "@remix-run/react";
import { authAPI, handleAuthSuccess } from "~/lib/api";
import { Button } from "../ui/button";
import { useLanguage } from "~/hooks/useLanguage";

const OtpVerification: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [otp, setOtp] = useState(["", "", "", "", "", ""]);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [resendTimer, setResendTimer] = useState(30);
  const [canResend, setCanResend] = useState(false);

  // Get data from previous page
  const phoneNumber = location.state?.phoneNumber || "+1 XXX XXX XXXX";
  const requestType = location.state?.requestType || "login";
  const signupData = location.state?.signupData;
  const returnState = location.state?.returnState;

  useEffect(() => {
    setIsVisible(true);

    // Start countdown timer
    const timer = setInterval(() => {
      setResendTimer((prev) => {
        if (prev <= 1) {
          setCanResend(true);
          clearInterval(timer);
          return 0;
        }
        return prev - 1;
      });
    }, 1000);

    return () => clearInterval(timer);
  }, []);

  const handleOtpChange = (index: number, value: string) => {
    if (value.length > 1) return; // Only allow single digit

    const newOtp = [...otp];
    newOtp[index] = value;
    setOtp(newOtp);

    // Auto-focus next input
    if (value && index < 5) {
      const nextInput = document.getElementById(`otp-${index + 1}`);
      nextInput?.focus();
    }

    // Clear error when user starts typing
    if (error) setError(null);
  };

  const handleKeyDown = (
    index: number,
    e: React.KeyboardEvent<HTMLInputElement>
  ) => {
    // Handle backspace
    if (e.key === "Backspace" && !otp[index] && index > 0) {
      const prevInput = document.getElementById(`otp-${index - 1}`);
      prevInput?.focus();
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const otpString = otp.join("");

    if (otpString.length !== 6) {
      setError(t("auth.otp.errors.incompleteCode", "Please enter the complete 6-digit code"));
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      let response;
      
      if (requestType === "signup" && signupData) {
        // For signup, include all the required signup data
        response = await authAPI.verifyOTP(
          phoneNumber,
          otpString,
          requestType,
          signupData
        );
      } else {
        // For login, just verify the OTP
        response = await authAPI.verifyOTP(phoneNumber, otpString, requestType);
      }

      if (response.status) {
        if (requestType === "signup") {
          // Save the token for profile picture upload
          if (response.token) {
            localStorage.setItem('signup_token', response.token);
          }
          
          // Save signup data including plan info to localStorage for persistence
          if (signupData || returnState?.selectedTier) {
            localStorage.setItem('signup_plan_data', JSON.stringify({
              plan: signupData?.plan,
              tierData: returnState?.selectedTier,
              uuid: response.uuid,
              userData: signupData
            }));
          }
          
          // After successful signup, redirect to profile picture upload
          navigate("/upload-profile-picture", {
            state: {
              uuid: response.uuid,
              userData: signupData,
              tierData: returnState?.selectedTier
            }
          });
        } else {
          // For login, redirect to home or back to booking
          handleAuthSuccess(response);
          
          // Check if there's a pending booking
          const pendingBooking = typeof window !== "undefined" 
            ? sessionStorage.getItem("pendingBooking") 
            : null;
          
          setTimeout(() => {
            if (pendingBooking) {
              const bookingData = JSON.parse(pendingBooking);
              // Redirect back to the profile page to continue booking
              window.location.href = bookingData.returnUrl || "/home";
            } else {
              // Regular login, go to home
              window.location.href = "/home"; // Full reload to reset state
            }
          }, 100);
        }
      } else {
        // Handle error from response
        let errorMsg = t("auth.otp.errors.invalidCode", "Invalid verification code. Please try again.");
        
        if (response.error) {
          if (typeof response.error === 'string') {
            errorMsg = response.error;
          } else if (typeof response.error === 'object') {
            // Handle validation errors
            const errors = Object.values(response.error).flat();
            errorMsg = errors.join(', ');
          }
        } else if (response.message) {
          errorMsg = response.message;
        }
        
        setError(errorMsg);
      }
    } catch (err) {
      console.error("OTP verification error:", err);
      setError(
        err instanceof Error
          ? err.message
          : t("auth.otp.errors.verificationFailed", "Verification failed. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };

  const handleResend = async () => {
    if (!canResend) return;

    setIsLoading(true);
    setError(null);

    try {
      // Use the phone number as received from previous page (already formatted with country code)
      const response = await authAPI.sendOTP(phoneNumber);

      if (response.status) {
        setResendTimer(30);
        setCanResend(false);
        setOtp(["", "", "", "", "", ""]);

        // Restart timer
        const timer = setInterval(() => {
          setResendTimer((prev) => {
            if (prev <= 1) {
              setCanResend(true);
              clearInterval(timer);
              return 0;
            }
            return prev - 1;
          });
        }, 1000);
      } else {
        setError(response.error || t("auth.otp.errors.resendFailed", "Failed to resend code. Please try again."));
      }
    } catch (err) {
      console.error("Resend OTP error:", err);
      setError(
        err instanceof Error
          ? err.message
          : t("auth.otp.errors.resendFailed", "Failed to resend code. Please try again.")
      );
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div
          className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60"
          style={{ animationDelay: "0s", animationDuration: "3s" }}
        />
        <div
          className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40"
          style={{ animationDelay: "1s" }}
        />
        <div
          className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30"
          style={{ animationDelay: "2s" }}
        />
        <div
          className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50"
          style={{ animationDelay: "1.5s", animationDuration: "4s" }}
        />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Back Button */}
        <div
          className={`w-full max-w-md mb-4 transform transition-all duration-1000 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <Button
            type="button"
            variant="outline-white"
            size="sm"
            onClick={() => {
              if (requestType === 'signup' && returnState) {
                // Navigate back to signup page with saved form data
                navigate('/signup-details', { state: returnState });
              } else {
                navigate(-1);
              }
            }}
            className="flex items-center gap-2"
          >
            <svg
              className="w-4 h-4"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={2}
                d="M15 19l-7-7 7-7"
              />
            </svg>
            {t("common.back", "Back")}
          </Button>
        </div>

        {/* Hero Title */}
        <div
          className={`text-center mb-12 transform transition-all duration-1000 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.otp.title", "CONFIRM YOUR PHONE")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.otp.subtitle", "Please input 6-digit code we just sent to")} {phoneNumber}
          </p>
        </div>

        {/* OTP Card */}
        <div
          className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${
            isVisible ? "translate-y-0 opacity-100" : "translate-y-10 opacity-0"
          }`}
        >
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-8">
              {/* OTP Input Fields */}
              <div className="flex justify-center space-x-3">
                {otp.map((digit, index) => (
                  <input
                    key={index}
                    id={`otp-${index}`}
                    type="text"
                    inputMode="numeric"
                    maxLength={1}
                    value={digit}
                    onChange={(e) => handleOtpChange(index, e.target.value)}
                    onKeyDown={(e) => handleKeyDown(index, e)}
                    className="w-12 h-14 text-center text-2xl font-bold bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all"
                    disabled={isLoading}
                    autoComplete="off"
                  />
                ))}
              </div>

              {/* Resend Section */}
              <div className="text-center">
                {!canResend ? (
                  <p className="text-white/60 text-sm">
                    {t("auth.otp.resendTimer", "Didn't get an SMS? Resend in")} {resendTimer} {t("auth.otp.seconds", "seconds")}
                  </p>
                ) : (
                  <Button
                    type="button"
                    variant="outline-white"
                    size="sm"
                    onClick={handleResend}
                    disabled={isLoading}
                  >
                    {t("auth.otp.resendCode", "Resend code")}
                  </Button>
                )}
              </div>

              {/* Submit Button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full"
                disabled={isLoading || otp.some((digit) => !digit)}
                arrowRight={!isLoading}
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    {t("auth.otp.verifying", "Verifying...")}
                  </>
                ) : (
                  t("auth.otp.submit", "Submit")
                )}
              </Button>

              {/* Back to Login */}
              <div className="text-center text-white/60 text-sm pt-4">
                {t("auth.otp.wrongNumber", "Wrong phone number?")}{" "}
                <Button
                  type="button"
                  variant="outline-white"
                  size="sm"
                  className="ml-2"
                  onClick={() => navigate("/login")}
                >
                  {t("auth.otp.backToLogin", "Login")}
                </Button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default OtpVerification;
