import React, { useState, useEffect } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';
import { authAPI } from '~/lib/api';
import { Button } from '../ui/button';
import { useLanguage } from '~/hooks/useLanguage';

const ResetPasswordOtp: React.FC = () => {
  const { t } = useLanguage();
  const navigate = useNavigate();
  const location = useLocation();
  const [isVisible, setIsVisible] = useState(false);
  const [otp, setOtp] = useState(['', '', '', '', '', '']);
  const [passwords, setPasswords] = useState({
    password: '',
    confirmPassword: ''
  });
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [resendTimer, setResendTimer] = useState(30);
  const [canResend, setCanResend] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);

  // Get email and mobile from previous page
  const email = location.state?.email || '';
  const mobile_number = location.state?.mobile_number || '';
  const requestType = location.state?.requestType || 'reset-password';

  useEffect(() => {
    console.log('ResetPasswordOtp mounted with state:', { email, mobile_number, requestType });
    
    // Redirect if no email (mobile_number is optional now)
    if (!email) {
      console.warn('No email found, redirecting to reset-password');
      navigate('/reset-password');
      return;
    }

    setIsVisible(true);

    // Start countdown timer
    const timer = setInterval(() => {
      setResendTimer((prev) => {
        if (prev <= 1) {
          setCanResend(true);
          clearInterval(timer);
          return 0;
        }
        return prev - 1;
      });
    }, 1000);

    return () => clearInterval(timer);
  }, [email, mobile_number, navigate]);

  const handleOtpChange = (index: number, value: string) => {
    if (value.length > 1) return; // Only allow single digit

    const newOtp = [...otp];
    newOtp[index] = value;
    setOtp(newOtp);

    // Auto-focus next input
    if (value && index < 5) {
      const nextInput = document.getElementById(`otp-${index + 1}`);
      nextInput?.focus();
    }

    // Clear error when user starts typing
    if (error) setError(null);
  };

  const handleOtpKeyDown = (index: number, e: React.KeyboardEvent) => {
    // Handle backspace
    if (e.key === 'Backspace' && !otp[index] && index > 0) {
      const prevInput = document.getElementById(`otp-${index - 1}`);
      prevInput?.focus();
    }
  };

  const handlePasswordChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setPasswords(prev => ({
      ...prev,
      [name]: value
    }));
    if (error) setError(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);

    // Validate OTP
    const otpString = otp.join('');
    if (otpString.length !== 6) {
      setError(t("auth.resetPasswordOtp.errors.incompleteOtp", "Please enter the complete 6-digit OTP"));
      setIsLoading(false);
      return;
    }

    // Validate passwords
    if (passwords.password !== passwords.confirmPassword) {
      setError(t("auth.resetPasswordOtp.errors.passwordMismatch", "Passwords do not match"));
      setIsLoading(false);
      return;
    }

    if (passwords.password.length < 8) {
      setError(t("auth.resetPasswordOtp.errors.passwordTooShort", "Password must be at least 8 characters long"));
      setIsLoading(false);
      return;
    }

    try {
      // Get mobile number from state, localStorage, or use email as fallback
      let mobileNumber = mobile_number;
      
      if (!mobileNumber) {
        // Try to get from localStorage (user data)
        const userData = localStorage.getItem('user_data');
        if (userData) {
          try {
            const user = JSON.parse(userData);
            mobileNumber = user.mobile_number || user.phone_number || '';
          } catch (e) {
            console.error('Error parsing user data:', e);
          }
        }
      }
      
      console.log('Verifying OTP with:', { email, mobileNumber, otp: otpString, requestType });
      
      const response = await authAPI.verifyOTP(
        mobileNumber || email, // Use email as fallback if mobile_number not available
        otpString,
        requestType,
        {
          password: passwords.password,
          confirm_password: passwords.confirmPassword,
          email: email // Also send email in payload
        }
      );

      console.log('OTP verification response:', response);

      if (response.status) {
        // Show success alert
        alert(response.message || t("auth.resetPasswordOtp.success", "Password reset successfully!"));
        
        // Password reset successful - navigate to login
        navigate('/login', {
          state: {
            message: t("auth.resetPasswordOtp.success", "Password reset successfully!"),
            type: 'success'
          }
        });
      } else {
        // Handle error response
        let errorMessage = t("auth.resetPasswordOtp.errors.resetFailed", "Failed to reset password. Please try again.");
        
        if (response.error) {
          if (typeof response.error === 'string') {
            errorMessage = response.error;
          } else if (typeof response.error === 'object') {
            // Extract first error message from nested object
            const errorKey = Object.keys(response.error)[0];
            if (errorKey && Array.isArray(response.error[errorKey])) {
              errorMessage = response.error[errorKey][0];
            } else if (errorKey) {
              errorMessage = response.error[errorKey];
            }
          }
        }
        
        setError(errorMessage);
      }
    } catch (err) {
      console.error('Reset password OTP verification error:', err);
      setError(err instanceof Error ? err.message : t("auth.resetPasswordOtp.errors.unexpected", "An unexpected error occurred. Please try again."));
    } finally {
      setIsLoading(false);
    }
  };

  const handleResendOTP = async () => {
    if (!canResend) return;

    try {
      setIsLoading(true);
      const response = await authAPI.forgotPassword(email);
      
      if (response.status) {
        setCanResend(false);
        setResendTimer(30);
        setError(null);

        // Start countdown timer again
        const timer = setInterval(() => {
          setResendTimer((prev) => {
            if (prev <= 1) {
              setCanResend(true);
              clearInterval(timer);
              return 0;
            }
            return prev - 1;
          });
        }, 1000);

        // Show success message
        setError(null);
      } else {
        setError(response.error || t("auth.resetPasswordOtp.errors.resendFailed", "Failed to resend OTP"));
      }
    } catch (err) {
      setError(t("auth.resetPasswordOtp.errors.resendFailed", "Failed to resend OTP"));
    } finally {
      setIsLoading(false);
    }
  };

  const formatTime = (seconds: number) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  return (
    <div className="min-h-screen bg-[#030303] relative overflow-hidden">
      {/* Animated Background */}
      <div className="absolute inset-0">
        <img
          src="/images/home/welcome.png"
          alt=""
          className="absolute inset-0 h-full w-full object-cover opacity-40"
        />
        <div className="absolute inset-0 bg-gradient-to-br from-[#030303]/80 via-[#030303]/60 to-[#030303]/80" />
      </div>

      {/* Floating Elements Animation */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-10 w-2 h-2 bg-orange-400 rounded-full animate-bounce opacity-60" style={{ animationDelay: '0s', animationDuration: '3s' }} />
        <div className="absolute top-40 right-16 w-1 h-1 bg-white rounded-full animate-pulse opacity-40" style={{ animationDelay: '1s' }} />
        <div className="absolute bottom-32 left-20 w-3 h-3 border border-orange-400 rounded-full animate-ping opacity-30" style={{ animationDelay: '2s' }} />
        <div className="absolute bottom-20 right-10 w-2 h-2 bg-white rounded-full animate-bounce opacity-50" style={{ animationDelay: '1.5s', animationDuration: '4s' }} />
      </div>

      {/* Main Content */}
      <div className="relative z-10 min-h-screen flex flex-col items-center justify-center px-4 lg:px-0">
        {/* Hero Title */}
        <div className={`text-center mb-12 transform transition-all duration-1000 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <h1 className="text-4xl lg:text-6xl font-bold text-white uppercase tracking-tight mb-4">
            {t("auth.resetPasswordOtp.title", "RESET PASSWORD")}
          </h1>
          <p className="text-lg lg:text-xl text-white/80 font-light">
            {t("auth.resetPasswordOtp.subtitle", "Enter the OTP sent to your mobile and set new password")}
          </p>
        </div>

        {/* OTP Verification Card */}
        <div className={`w-full max-w-md transform transition-all duration-1000 delay-300 ${isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'}`}>
          <div className="bg-white/10 backdrop-blur-lg rounded-2xl border border-white/20 p-8 shadow-2xl">
            <h2 className="text-2xl font-semibold mb-8 text-white text-center uppercase tracking-wider">
              {t("auth.resetPasswordOtp.cardTitle", "Verify & Reset")}
            </h2>

            {/* Error Message */}
            {error && (
              <div className="mb-6 p-4 bg-red-500/20 border border-red-500/30 text-red-200 rounded-xl text-sm backdrop-blur-sm">
                {error}
              </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-6">
              {/* OTP Input */}
              <div className="space-y-4">
                <div className="text-center">
                  <p className="text-white/80 text-sm mb-2">{t("auth.resetPasswordOtp.enterOtp", "Enter 6-digit OTP")}</p>
                  <p className="text-white/60 text-xs">
                    {t("auth.resetPasswordOtp.sentTo", "Sent to")} {mobile_number ? mobile_number.replace(/(\d{2})(\d{4})(\d{4})/, '$1****$3') : t("auth.resetPasswordOtp.registeredNumber", "your registered mobile number")}
                  </p>
                </div>

                <div className="flex justify-center space-x-3">
                  {otp.map((digit, index) => (
                    <input
                      key={index}
                      id={`otp-${index}`}
                      type="text"
                      inputMode="numeric"
                      pattern="[0-9]*"
                      maxLength={1}
                      value={digit}
                      onChange={(e) => handleOtpChange(index, e.target.value)}
                      onKeyDown={(e) => handleOtpKeyDown(index, e)}
                      className="w-12 h-12 text-center text-xl font-bold bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 text-white backdrop-blur-sm transition-all"
                      disabled={isLoading}
                    />
                  ))}
                </div>

                {/* Resend OTP */}
                <div className="text-center">
                  {canResend ? (
                    <Button
                      type="button"
                      variant="outline-white"
                      size="sm"
                      onClick={handleResendOTP}
                      disabled={isLoading}
                    >
                      {t("auth.resetPasswordOtp.resendOtp", "Resend OTP")}
                    </Button>
                  ) : (
                    <p className="text-white/60 text-sm">
                      {t("auth.resetPasswordOtp.resendIn", "Resend OTP in")} {formatTime(resendTimer)}
                    </p>
                  )}
                </div>
              </div>

              {/* New Password */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.resetPasswordOtp.newPassword", "New Password")}</label>
                <div className="relative">
                  <input
                    type={showPassword ? "text" : "password"}
                    name="password"
                    placeholder={t("auth.resetPasswordOtp.newPasswordPlaceholder", "Enter new password")}
                    value={passwords.password}
                    onChange={handlePasswordChange}
                    className="w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                    minLength={8}
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors"
                  >
                    {showPassword ? (
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21" />
                      </svg>
                    ) : (
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                      </svg>
                    )}
                  </button>
                </div>
              </div>

              {/* Confirm Password */}
              <div className="space-y-2">
                <label className="text-white/80 text-sm font-medium uppercase tracking-wider">{t("auth.resetPasswordOtp.confirmPassword", "Confirm New Password")}</label>
                <div className="relative">
                  <input
                    type={showConfirmPassword ? "text" : "password"}
                    name="confirmPassword"
                    placeholder={t("auth.resetPasswordOtp.confirmPasswordPlaceholder", "Confirm new password")}
                    value={passwords.confirmPassword}
                    onChange={handlePasswordChange}
                    className="w-full px-4 py-4 pr-12 bg-white/10 border border-white/20 rounded-xl focus:outline-none focus:ring-2 focus:ring-orange-400 focus:border-orange-400 placeholder-white/60 text-white backdrop-blur-sm transition-all"
                    required
                    disabled={isLoading}
                    minLength={8}
                  />
                  <button
                    type="button"
                    onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-white/60 hover:text-white/80 transition-colors"
                  >
                    {showConfirmPassword ? (
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21" />
                      </svg>
                    ) : (
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                      </svg>
                    )}
                  </button>
                </div>
              </div>

              {/* Submit Button */}
              <Button
                type="submit"
                variant="fill-white"
                size="lg"
                className="w-full"
                disabled={isLoading}
                arrowRight={!isLoading}
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-black mr-3"></div>
                    {t("auth.resetPasswordOtp.resetting", "Resetting Password...")}
                  </>
                ) : (
                  t("auth.resetPasswordOtp.resetPasswordButton", "Reset Password")
                )}
              </Button>

              {/* Back to Reset Password */}
              <div className="text-center pt-4">
                <Button
                  type="button"
                  variant="outline-white"
                  size="sm"
                  onClick={() => navigate('/reset-password')}
                >
                  {t("auth.resetPasswordOtp.backToReset", "Back to Reset Password")}
                </Button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ResetPasswordOtp;